/* -*- c-file-style: "GNU" -*- */
/*
 * Copyright (C) CNRS, INRIA, Université Bordeaux 1, Télécom SudParis
 * See COPYING in top-level directory.
 */

#include <opcodes.h>
#include <tracing.h>
#include <isize.h>
#include <memory.h>
#include <binary.h>
#include <errors.h>
#include <stdlib.h>


#ifdef __x86_64__
#ifdef HAVE_LIBOPCODE
#include "arch/x86_64/hijack.c"
#endif	/* HAVE_LIBOPCODES */
#endif

#ifndef HAVE_CHECK_INSTRUCTIONS
/* check_instructions is not defined.
 * define it as a dummy function that always allow the insertion of a trampoline
 */

int check_instructions(void* bin, pid_t child, word_uint sym_addr, word_uint reloc_addr,
		       size_t length) {
  return 1;
}
#endif


ssize_t hijack_code_small_jump(void* bin, pid_t child, word_uint sym_addr,
                               word_uint sym_size, word_uint reloc_addr,
                               word_uint orig_addr, word_uint repl_addr) {

#define HIJACK_CODE_FAIL_SMALL_JUMP(test) if(test) {	\
    free(trampoline);					\
    free(trampoline2);					\
    free(trampoline3);					\
    return -1;						\
  }

  /* sym_addr: address of the original function (in which we install a call to eztrace)
   * reloc_addr: address of the block of data that jumps to eztrace
   * repl_addr: address of the eztrace version of the function
   * orig_addr: address of the block of data that replays the overwritten opcodes and jumps to the original function
   */
  /* the goal here is to change the child process so that when the application calls the sym_addr function,
   * the following behavior occurs:
   * call sym_addr: (small) jump to reloc_addr (<-- small_jump)
   * reloc_addr:  (long) jump to repl_addr (<-- long_jump)
   * repl_addr: perform eztrace stuff (recording events, etc.)
   * repl_addr: call orig_addr
   *   orig_addr: replay the overwritten opcodes
   *   orig_addr: (long) jump to sym_addr+1 (<-- back_jump)
   *   sym_addr: process stuff (it is the application function)
   *   sym_addr: ret -> go back to repl_addr
   * repl_addr: perform eztrace stuff (recording events, etc.)
   * repl_addr: ret -> go back to the application
   */

  // If the size is not available, then we suppose we have enough room
  if (sym_size <= 0) {
    sym_size = MAX_TRAMPOLINE_SIZE;
  }
  uint8_t* trampoline = (uint8_t*) malloc(MAX_TRAMPOLINE_SIZE);
  uint8_t* trampoline2 = (uint8_t*) malloc(MAX_TRAMPOLINE_SIZE);
  uint8_t* trampoline3 = (uint8_t*) malloc(sym_size);

  // Building an intermediate trampoline for space reason (a long jump in the reloc space so that the trampoline is a short jump)
  ssize_t trampoline3_size = generate_trampoline(trampoline3,
                                                 MAX_TRAMPOLINE_SIZE,
                                                 reloc_addr, repl_addr);
  HIJACK_CODE_FAIL_SMALL_JUMP(trampoline3_size < 0);

  // Building the trampoline. This can fail if we don't have enough space in the hijacked symbol
  ssize_t trampoline_size = generate_trampoline(trampoline, sym_size, sym_addr,
                                                reloc_addr);
  HIJACK_CODE_FAIL_SMALL_JUMP(trampoline_size < 0);

  // And now ptrace insertion...

  // Copy the original function into the dest process. BUG? We should make sure all calls and jump stays correct... (how?)
  ssize_t over = get_overridden_size(bin, child, sym_addr, trampoline_size); // get the size to override

  HIJACK_CODE_FAIL_SMALL_JUMP(   (!check_instructions(bin, child, sym_addr, reloc_addr, trampoline_size)) ||
				 over < trampoline_size
				 || over > sym_size);

  ssize_t trampoline2_size = generate_trampoline(
      trampoline2, sym_size, reloc_addr + over + trampoline3_size,
      sym_addr + over); // create the trampoline back to the original address
  HIJACK_CODE_FAIL_SMALL_JUMP(trampoline2_size < 0);

  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tinserting trampoline from relocated code (0x%lx) to interception function (0x%lx)\n",
      (unsigned long) reloc_addr, (unsigned long) repl_addr);
  trace_write(child, reloc_addr, trampoline3, trampoline3_size); // Insert the long jump
  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tcopying overwritten code (%d bytes) from original code (0x%lx) to relocated position (0x%lx)\n",
      over, (unsigned long) sym_addr,
      (unsigned long) (reloc_addr + trampoline3_size));
  trace_copy(child, sym_addr, reloc_addr + trampoline3_size, over); // insert the relocated code
  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tinserting trampoline from relocated code (0x%lx) to original position (0x%lx)\n",
      (unsigned long) (reloc_addr + trampoline3_size + over),
      (unsigned long) (sym_addr + over));
  trace_write(child, reloc_addr + trampoline3_size + over, trampoline2,
              trampoline2_size); // insert the trampoline back to the original function

  // Insert the trampoline into the dest process
  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tinserting trampoline from original position (0x%lx) to relocated code (0x%lx)\n",
      (unsigned long) sym_addr, (unsigned long) reloc_addr);
  trace_write(child, sym_addr, trampoline, trampoline_size);
  free(trampoline);
  free(trampoline2);
  free(trampoline3);

  // Now only the orig_addr needs to be written with the reloc_addr
  reloc_addr += trampoline3_size;
  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tcopying relocated code addr (0x%lx) to original symbol (0x%lx)\n",
      (unsigned long) reloc_addr, (unsigned long) orig_addr);
  trace_write(child, orig_addr, (uint8_t*) (&reloc_addr), sizeof(reloc_addr));

  // Return the size of the inserted code
  return over + trampoline3_size + trampoline2_size;
}

ssize_t hijack_code_long_jump(void* bin, pid_t child, word_uint sym_addr,
                              word_uint sym_size, word_uint reloc_addr,
                              word_uint orig_addr, word_uint repl_addr) {

  /* sym_addr: address of the original function (in which we install a call to eztrace)
   * reloc_addr: address of the block of data that jumps to eztrace
   * repl_addr: address of the eztrace version of the function
   * orig_addr: address of the callback in eztrace that permits to call the user-defined function
   */

  /* the goal here is to change the child process so that when the application calls the sym_addr function,
   * the following behavior occurs:
   * call sym_addr: jump to repl_addr (<-- first_jump)
   * repl_addr: perform eztrace stuff (recording events, etc.)
   * repl_addr: call *orig_addr
   *   *orig_addr: replay the overwritten opcodes
   *   *orig_addr: (long) jump to sym_addr+1 (<-- back_jump)
   *   sym_addr: process stuff (it is the application function)
   *   sym_addr: ret -> go back to repl_addr
   * repl_addr: perform eztrace stuff (recording events, etc.)
   * repl_addr: ret -> go back to the application
   */

  // If the size is not available, then we assume we have enough room
  if (sym_size <= 0) {
    sym_size = MAX_TRAMPOLINE_SIZE;
  }

#define HIJACK_CODE_FAIL_LONG_JUMP(test) if(test) {	\
      free(first_jump);				\
      free(back_jump);				\
      return -1;				\
    }

  uint8_t *first_jump = (uint8_t*) malloc(sym_size);
  uint8_t* back_jump = (uint8_t*) malloc(MAX_TRAMPOLINE_SIZE);

  // Building the first jump from sym_addr to repl_addr
  ssize_t first_jump_size = generate_trampoline(first_jump, sym_size, sym_addr,
                                                repl_addr);
  HIJACK_CODE_FAIL_LONG_JUMP(first_jump_size < 0);

  // Copy the original function into the dest process. BUG? We should make sure all calls and jump stays correct... (how?)
  ssize_t over = get_overridden_size(bin, child, sym_addr, first_jump_size); // get the size to override
  HIJACK_CODE_FAIL_LONG_JUMP(over < first_jump_size || over > sym_size);

  // create the trampoline back to the original address
  ssize_t back_jump_size = generate_trampoline(back_jump, MAX_TRAMPOLINE_SIZE,
                                               reloc_addr + over,
                                               sym_addr + over);
  HIJACK_CODE_FAIL_LONG_JUMP(back_jump_size < 0);

  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tcopying overwritten code (%d bytes) from original code (0x%lx) to relocated position (0x%lx)\n",
      over, (unsigned long) sym_addr, (unsigned long) (reloc_addr));

  /* write the trampoline to the child memory */
  trace_copy(child, sym_addr, reloc_addr, over); // insert the relocated code

  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tinserting trampoline from relocated code (0x%lx) to interception function (0x%lx)\n",
      (unsigned long) sym_addr, (unsigned long) repl_addr);

  trace_write(child, sym_addr, first_jump, first_jump_size); // Insert the long jump

  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tinserting trampoline from relocated code (0x%lx) to original position (0x%lx)\n",
      (unsigned long) (reloc_addr + over), (unsigned long) (sym_addr + over));
  trace_write(child, reloc_addr + over, back_jump, back_jump_size); // insert the trampoline back to the original function

  free(first_jump);
  free(back_jump);

  // Now only the orig_addr needs to be written with the reloc_addr
#ifdef __arm__
  /* we need to switch to Thumb momde */
  reloc_addr |= 0x01;
#endif
  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_ALL,
      "(hijack)\tcopying relocated code addr (0x%lx) to original symbol (0x%lx)\n",
      (unsigned long) reloc_addr, (unsigned long) orig_addr);
  trace_write(child, orig_addr, (uint8_t*) (&reloc_addr), sizeof(reloc_addr));

  // Return the size of the inserted code
  return over + back_jump_size;
}

ssize_t hijack_code(void* bin, pid_t child, word_uint sym_addr,
                    word_uint sym_size, word_uint reloc_addr,
                    word_uint orig_addr, word_uint repl_addr) {
#if __arm__
  return hijack_code_long_jump(bin, child, sym_addr, sym_size, reloc_addr, orig_addr, repl_addr);
#else
  return hijack_code_small_jump(bin, child, sym_addr, sym_size, reloc_addr,
                                orig_addr, repl_addr);
#endif
}

ssize_t hijack(void* bin, pid_t child, zzt_symbol *toHijack, zzt_symbol *orig,
               zzt_symbol *repl) {
  // Compute the addresses from the symbols
  word_uint addr =
    (word_uint) (toHijack->symbol_offset + toHijack->section_addr);
  word_uint sym_size = (word_uint) (toHijack->symbol_size);
  word_uint orig_addr = (word_uint) (orig->symbol_offset + orig->section_addr);
  word_uint repl_addr = (word_uint) (repl->symbol_offset + repl->section_addr);

  pptrace_debug(
      PPTRACE_DEBUG_LEVEL_DEBUG,
      "Hijacking symbol %s (0x%lx) with %s (0x%lx) using %s (0x%lx) as function pointer...\n",
      toHijack->symbol_name, (unsigned long) addr, repl->symbol_name,
      (unsigned long) repl_addr, orig->symbol_name, (unsigned long) orig_addr);
  pptrace_debug(PPTRACE_DEBUG_LEVEL_DEBUG,
                "Allocating buffer for relocating bytes... ");
  word_uint reloc_addr = allocate_buffer(child, MAX_TRAMPOLINE_SIZE);
  if (reloc_addr != 0) {
    pptrace_debug(PPTRACE_DEBUG_LEVEL_DEBUG, "ok (0x%lx)\n",
                  (unsigned long) reloc_addr);
    ssize_t hijack_size = hijack_code(bin, child, addr, sym_size, reloc_addr,
                                      orig_addr, repl_addr);

    if (hijack_size > 0 && hijack_size != MAX_TRAMPOLINE_SIZE) {
      correct_buffer_allocation(child, MAX_TRAMPOLINE_SIZE, hijack_size);
    }
    pptrace_debug(PPTRACE_DEBUG_LEVEL_DEBUG, "Symbol hijacked...\n");
    return hijack_size;
  }
  pptrace_debug(PPTRACE_DEBUG_LEVEL_DEBUG, "failed!\n");
  return -1;
}
