/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: tiex.c
*/

#include "include.h"

/********************************************************************
*
*  Function: curvature_forces()
*
*  Purpose:  Forces as function of curvatures.
*            Finds mean and gaussian curvature at each vertex,
*            and normal.
*
*/

struct teix_gvert  *tgverts;

void curvature_forces_init(mode,mi)
int mode;
struct method_instance *mi;
{ 
  vertex_id v[3];
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe;
  int fixcount;
  int i,j;
  REAL area;
  REAL side[3][MAXCOORD];
  REAL ss[3];
  REAL st[3];
  REAL angle;
  REAL c;
  struct teix_gvert  *gv,*vc[3];
  REAL normal[MAXCOORD];

  if ( tgverts ) myfree((char*)tgverts);
  tgverts = (struct teix_gvert*)mycalloc(web.skel[VERTEX].max_ord+1,
                                            sizeof(struct teix_gvert));

  /* accumulate angles and stuff around vertices */
  FOR_ALL_FACETS(f_id)
  { fe = get_facet_fe(f_id);
    fixcount = 0;
    for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
    { e_id = get_fe_edge(fe);
      get_edge_side(e_id,side[i]);
      v[i] = get_edge_tailv(e_id);
      if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
    }
    for ( i = 0 ; i < FACET_VERTS ; i++ )
    { ss[i] = SDIM_dot(side[i],side[i]);
      st[i] = SDIM_dot(side[i],side[(i+2)%3]);
    }
    area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
    cross_prod(side[0],side[1],normal);
    for ( i = 0 ; i < FACET_VERTS ; i++ )
    { REAL dd = ss[i]*ss[(i+2)%3];
      gv = tgverts + loc_ordinal(v[i]);
      if ( dd > 0.0 )
      { c = -st[i]/sqrt(ss[i]*ss[(i+2)%3]);
        if ( fabs(c) <= 1.0 )
        { angle = acos(c);
          gv->angle += angle;
        }
      } 
      gv->area  += area/3;
      gv->star_area  += area/(3-fixcount);
      for ( j = 0 ; j < SDIM ; j++ )
      gv->normal[j] += normal[j];
    }

    /* for mean curvature */

    for ( i = 0 ; i < FACET_VERTS ; i++ )
      vc[i] = tgverts + loc_ordinal(v[i]);

    for ( i = 0 ; i < SDIM ; i++ )
    { 
       vc[0]->force[i] += (ss[1]*side[2][i]-st[2]*side[1][i])/4/area;
       vc[1]->force[i] += (ss[2]*side[0][i]-st[0]*side[2][i])/4/area;
       vc[2]->force[i] += (ss[0]*side[1][i]-st[1]*side[0][i])/4/area;
    }
  }
}
 
REAL curvature_forces_energy(v_info)
struct qinfo *v_info;
{ return 0.0; /* fake it */
}

REAL curvature_forces(v_info)
struct qinfo *v_info;
{ REAL K; /* gauss curvature of vertex */
  REAL H; /* mean curvature of vertex, average of sectional curvatures */
  struct teix_gvert *vg = tgverts + loc_ordinal(v_info->id);
  REAL norm;
  REAL f; /* function of H and K */
  int i;

  if ( get_vattr(v_info->id) & (FIXED|BOUNDARY) ) return 0.0;

  /* normalize normal */
  norm = sqrt(SDIM_dot(vg->normal,vg->normal));
  for ( i = 0 ; i < SDIM ; i++ )
     vg->normal[i] /= norm;

  K = (2*M_PI - vg->angle)/vg->area;

  H = SDIM_dot(vg->force,vg->normal)/2/vg->area;

  /* fix here for function of H and K */
  f =  (H>0.0) ? exp(K) : -exp(K);    /* to detect misorientation */
  /* end of fixing */

  /* force as multiple of normal */
  for ( i = 0 ; i < SDIM ; i++ )
     v_info->grad[0][i] = -f*vg->normal[i];
 
  return 0.0;  /* fake energy */
}

/****************************************************************************
*
*  Function: vertex_mean_curvature()
*
*  Purpose: calculate mean curvature at a vertex, as force divided by area
*/

REAL vertex_mean_curvature(v_id)
vertex_id v_id;
{ facet_id f_id, start_f;
  REAL meanc;
  REAL force[MAXCOORD],projf[MAXCOORD];
  REAL area = 0.0,farea;
  REAL s1[MAXCOORD],s2[MAXCOORD];
  int i;

  if ( vfacet_timestamp < top_timestamp )  make_vfacet_lists(); 
  for ( i = 0 ; i < MAXCOORD ; i++ ) 
    force[i] = 0.0;
  f_id = start_f = get_first_vertex_facet(v_id);
  if ( !valid_id(f_id) ) return 0.0;
  do 
  { facetedge_id fe;
    REAL s11,s12,s22;
    fe = get_facet_fe(f_id);
    while ( !equal_id(v_id,get_fe_headv(fe)) )
      fe = get_next_edge(fe);
    get_edge_side(get_fe_edge(fe),s1); 
    get_edge_side(get_fe_edge(get_prev_edge(fe)),s2); 
    s11 = dot(s1,s1,SDIM);
    s12 = dot(s1,s2,SDIM);
    s22 = dot(s2,s2,SDIM);
    farea = sqrt(s11*s22-s12*s12);
    for ( i = 0 ; i < SDIM ; i++ ) 
      force[i] += 1/farea*(s1[i]*s22 - s12*s2[i])/2;
    area += farea/2;
    f_id = get_next_vertex_facet(v_id,f_id);
  } while ( !equal_element(start_f,f_id) );

  force_project(force,v_id,projf);
  meanc = sqrt(dot(projf,projf,SDIM))/(area/3)/2; /* divide by 2 for mean */
  return meanc;
}
