/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/********************************************************************
*
*  File: storage.h
*
*  Purpose:  Header file defining details of storage implementation.
*                All machine-dependent gory details should be here
*                (for inclusion in other source files that need to know)
*                or in storage.c (purely private details).
*
*      This version has element ids typed as longs.
*      Also implements id as offset from element base.
*      All elements of same type in one memory block, extended 
*        as needed.  To be used on at least 32 bit machines.
*/

/* using offset as id is 15% faster overall than using ordinal */

#define NUMELEMENTS 5

/* these values used for identification and as index into skeleton info */
#define  VERTEX    0
#define  EDGE      1
#define  FACET     2
#define  BODY      3
#define  FACETEDGE 4 

#ifdef xelement
/*****************************************************************
*
*  Universal element identifier.  Don't want to use straight
*  pointer since base changes when realloced.
*  (not actually used; just for documentation)
*/

typedef struct xelement_id {
     unsigned int  type : 3;    /* see enum below */
     unsigned int  valid: 1;    /* valid id bit */
     unsigned int  sign : 1;    /* set for reverse orientation */
     unsigned int  offset: 27;    /* offset from block start */
     } xelement_id;
#endif

/* masks for fields */
#define TYPEMASK    0xE0000000
#define VALIDMASK  0x10000000
#define SIGNMASK    0x08000000
#define OFFSETMASK 0x07FFFFFF

/* shifts for fields */
#define TYPESHIFT  29
#define VALIDSHIFT 28
#define SIGNSHIFT  27

#define NULLID 0L 

/* to get type of an element */
#define id_type(id)  ((int)(((id)&TYPEMASK)>>TYPESHIFT))

/* to give switched orientation of first if that of second is inverted */
#define same_sign(id1,id2)     ((id1) ^ ((id2) & SIGNMASK))

/* number of elements to allocate memory for at one time */
#define BATCHSIZE 100

/* outside storage.*, element_id structure is not visible; acts like long */     
typedef unsigned long
      element_id, vertex_id, edge_id, facet_id, body_id, facetedge_id; 

/* Macros for getting structure pointer from id */
#define vptr(v_id)   ((struct vertex *)(vibase[(v_id)&OFFSETMASK]))
#define eptr(e_id)   ((struct edge   *)(eibase[(e_id)&OFFSETMASK]))
#define fptr(f_id)   ((struct facet  *)(fibase[(f_id)&OFFSETMASK]))
#define bptr(b_id)   ((struct body   *)(bibase[(b_id)&OFFSETMASK]))
#define feptr(fe_id) ((struct facetedge *)(feibase[(fe_id)&OFFSETMASK]))
#define ordinal(id)  (valid_id(id) ? (int)((id) & OFFSETMASK) : -1 )

/* Macros for manipulating and testing element ids */
#define edge_inverse(id)   inverse_id(id)
#define facet_inverse(id)  inverse_id(id)
#define fe_inverse(id)     inverse_id(id)
#define invert(id)         ((id) ^= SIGNMASK)
#define equal_id(a,b)      ((a)==(b))
#define equal_element(a,b) (((a)|SIGNMASK) == ((b)|SIGNMASK))
#define valid_id(id)       ((id)&VALIDMASK)
#define inverted(id)       ((id)&SIGNMASK)
#define inverse_id(id)     ((id) ^ SIGNMASK)
#define positive_id(id)    ((id) & ~SIGNMASK)
typedef int ORDTYPE;       /* element numbering type */

typedef struct element *INDIRECT_TYPE; 
extern struct blocklist_struct
     { struct element *blockptr; /* allocated block */
       int start_ord;            /* ordinal of first element */
       int count;                /* elements in block */
     } *blocklist[NUMELEMENTS];
extern int blockcount[NUMELEMENTS];  /* how many blocks allocated */
extern int blockmax[NUMELEMENTS];  /* length of blocklist */

/* individual indirect block pointer arrays */
extern INDIRECT_TYPE *ibase[NUMELEMENTS];
extern int ialloc[NUMELEMENTS]; /* allocated length of ibase's */
extern INDIRECT_TYPE *vibase;
extern INDIRECT_TYPE *eibase;
extern INDIRECT_TYPE *fibase;
extern INDIRECT_TYPE *bibase;
extern INDIRECT_TYPE *feibase;

