/*************************************************************
*  This file is part of the Surface Evolver source code.      *
*  Programmer:  Ken Brakke, brakke@susqu.edu                     *
*************************************************************/

/******************************************************************
*
*  File:    popfilm.c
*
*  Purpose:  Convert all tangent cones to minimal cones in
*                soapfilm model.  
*/

#include "include.h"

/*********************************************************************
*
*  Function: popfilm()
*
*  Purpose:  Overall control of tangent cone minimization.
*                Creates sorted list of vertex-facet incidences
*                sorted on vertex.  Analyzes tangent cone for each
*                vertex and calls minimizer for the non-minimal.
*/


/* comparison routine for qsort */
static int vfcomp(a,b)
struct verfacet *a,*b;
{
  if ( a->v_id < b->v_id ) return -1;
  if ( a->v_id > b->v_id ) return 1;
  return 0;
}

int popfilm()
{
  int popped = 0;

  popped += boundary_pull_film();
  popped += edgepop_film();
  sprintf(msg,"Edges popped: %d\n",popped); outstring(msg);
  popped = verpop_film();
  
  return popped;
}

/********************************************************************
*
*  Function: boundary_pull_film()
*
*  Purpose:  Readjusts edges on boundaries and constraints for
*                proper tangent cones and such.
*/

int boundary_pull_film()
{
  edge_id e_id;
  facetedge_id fe_id;
  int popped = 0;

  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
     {
        int facet_count = 0;

        if ( !(get_eattr(e_id) & (FIXED|BOUNDARY|CONSTRAINT)) ) continue;

        /* count free facets */
        while ( generate_edge_fe(e_id,&fe_id) ) 
         if ( !(get_fattr(get_fe_facet(fe_id)) & (FIXED|BOUNDARY|CONSTRAINT)) )
            facet_count++;

        if ( facet_count < 2 ) continue;
        /* Edges may either pull jointly away from the boundary
            if their net pull exceeds 1, or else they will separate
            on boundary.  Actually, they can almost always separate
            along the boundary if there is pi/2 contact angle.
            But if constraint has energy, then there is critical
            angle for separating.
         */

erroutstring("Popping of multiple facets on boundary edge not yet working.\n");
     }
  
  return popped;
}

/********************************************************************
*
*  Function: edgepop_film()
*
*  Purpose:  Reconfigure all edges with more than 3 facets into
*                triple edges.
*/

REAL new_displacement[MAXCOORD]; /* one normal of splittin wedge */

int  edgepop_film()
{
  edge_id e_id;
  int i,k;
  REAL midnormal[MAXCOORD];
  int popped = 0;

  /* Loop through all edges, popping as you go.  New edges created
      during popping will be scanned to see if they are still
      pop-worthy.
    */

  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
     {
        int facet_count = 0;
        facetedge_id fe_id,fe,key_fe=0,new_key;
        REAL side[MAXCOORD],sideA[MAXCOORD],sideB[MAXCOORD];
        REAL normA,normB,normalA[MAXCOORD],normalB[MAXCOORD];
        REAL maxcos,newcos;
		  int didsplit;

        /* already did boundaries */
        if ( get_eattr(e_id) & (FIXED|BOUNDARY|CONSTRAINT) ) continue;

        /* count facets around edge */
        while ( generate_edge_fe(e_id,&fe_id) ) facet_count++;

        /* only pop if more than 3 facets */
        if ( facet_count < 4 ) continue;

        if ( verbose_flag )
        { sprintf(msg,"Popping edge %d\n",ordinal(e_id)+1);
          outstring(msg);
        }

        /* find narrowest wedge to pull out */
        get_edge_side(e_id,side);
        maxcos = -2.0;    /* for finding minimum angle between facets */

        /* find first facet normal */
        fe = get_edge_fe(e_id);
        get_fe_side(get_next_edge(fe),sideA);
        cross_prod(side,sideA,normalA);
        normA = sqrt(SDIM_dot(normalA,normalA));
        for ( i = 0 ; i < facet_count ; i++ )
          {
             /* get next normal */
             fe = get_next_facet(fe);
             get_fe_side(get_next_edge(fe),sideB);
             cross_prod(side,sideB,normalB);
             normB = sqrt(SDIM_dot(normalB,normalB));
             newcos = SDIM_dot(normalA,normalB)/normA/normB;
             if ( newcos > maxcos )
                { key_fe = fe;
                  maxcos = newcos;
                  for ( k = 0 ; k < SDIM ; k++ )
                     midnormal[k] = normalA[k]/normA + normalB[k]/normB;
                  cross_prod(midnormal,side,new_displacement);
                }
             /* set up for next angle */
             normA = normB;
             memcpy((char *)normalA,(char *)normalB,sizeof(normalA));
          }
        /* now a little check just to make sure things happened as planned */
        if ( maxcos < 0.0 )
        {
         sprintf(errmsg,"Can't find less than right angle between facets on poppable edge %d.\n",
          ordinal(e_id)+1);
         kb_error(1299,errmsg, RECOVERABLE);

        }

        didsplit = 0;

        /* try propagating split forward */
        new_key = key_fe;
        while ( try_prop(&new_key,key_fe) ) 
          { didsplit = 1;
             if ( verbose_flag )
             { sprintf(msg," Propagating pop to edge %d\n",ordinal(get_fe_edge(key_fe))+1);
                outstring(msg);
             }
             popped++;
          }

        /* try propagating split backward */
        new_key = inverse_id(get_prev_facet(key_fe));
        while ( try_prop(&new_key,inverse_id(get_prev_facet(key_fe))) ) 
          { didsplit = 1;
             if ( verbose_flag )
             { sprintf(msg," Propagating pop to edge %d\n",ordinal(get_fe_edge(key_fe))+1);
                outstring(msg);
             }
             popped++;
          }

        /* if can't split forward or backward, divide edge and split */
        if ( !didsplit )
          {
             edge_refine(e_id);
             /* will split when new edge gets scanned */
          }

     }

  return popped;
}

/***********************************************************************
*  
*  Function: try_prop()
*
*  Purpose:  Sees if vertex at head of *pass_key can be split.  
*                If it can, it does.  If split can be propagated
*                further, it returns the next edge to try in *pass_key.
*                If the vertex was split, it returns 1, else 0.
*                Does not split FIXED vertices. Or BOUNDARY vertices.
*/

int try_prop(pass_key,start_key)
facetedge_id *pass_key;
facetedge_id  start_key; /* to check for complete loop */
{
  int splitflag;
  int propflag;
  facetedge_id wing_fe;
  facetedge_id flip_fe;
  facetedge_id new_key=0;
  facetedge_id key_fe = *pass_key;
  edge_id e_id = get_fe_edge(key_fe);
  edge_id next_e=0;

  if ( !valid_id(*pass_key) ) return 0;

  if ( get_vattr(get_fe_headv(key_fe)) & (FIXED|BOUNDARY) ) return 0;

  /* don't even try if valence is already <= 3 */
  /*if ( get_edge_valence(e_id) <= 3 ) return 0; */

  /* swing forward on one side until find multiple edge */
  flip_fe = inverse_id(key_fe);
  for (;;)
     {
        wing_fe = get_next_edge(inverse_id(flip_fe));
      if ( equal_id(wing_fe,start_key) )
      { /* have made complete loop around */
          splitflag = 1;
          propflag = 0;
          break;
      }
        flip_fe = get_next_facet(wing_fe);
        if ( equal_id(wing_fe,flip_fe) )
          { /* dead end on wing, so OK to split vertex maybe */
             splitflag = 1;
             propflag = 0;
             break;
          }
        if ( !equal_id(wing_fe,get_next_facet(flip_fe)) )
          { /* have found multiple edge */
             next_e = get_fe_edge(wing_fe);
             if ( !equal_element(e_id,next_e) )
                { /* have found legitimate next edge */
                  splitflag = 1;
                  propflag = 1;
                  new_key = wing_fe;
                  break;
                }
             if ( equal_element(wing_fe,get_prev_facet(key_fe)) )
                { /* have cirque, so OK to split, but no propagation */
                  splitflag = 1;
                  propflag = 0;
                }
             else
                { /* have cirque crossing proposed split */
                  /* which ends propagation */
                  splitflag = 0;
                  propflag = 0;
                }
             break;
          }
     }
  if ( !splitflag ) return 0;

  /* swing forward on other side until find multiple edge */
  flip_fe = inverse_id(get_prev_facet(key_fe));
  for (;;)
     {
        wing_fe = get_next_edge(inverse_id(flip_fe));
        flip_fe = get_next_facet(wing_fe);
        if ( equal_id(wing_fe,flip_fe) )
          { /* dead end on wing, so OK to split vertex maybe */
             propflag = 0;
             break;
          }
        if ( !equal_id(wing_fe,get_next_facet(flip_fe)) )
          { /* have found multiple edge */
             if ( !equal_id(get_fe_edge(wing_fe),next_e) )
                {
                  /* can't split; have serious nonminimal vertex */
                  splitflag = 0;
                  propflag = 0;
                  break;
                }
             else /* check normals */
                {
                }
             break;
          }
      }

  if ( !propflag ) new_key = NULLFACETEDGE;

  if ( splitflag ) versplit(key_fe,new_key);

  *pass_key = new_key;


  return splitflag;
}

/**********************************************************************
*
*  Function: versplit()
*
*  Purpose:  Split a wedge of facets off a vertex along two edges.
*                Old edges go with split wedge (old edges now proper
*                minimal cones) and new edges have rest of old facets
*                and await scanning for minimality.  Arguments are
*                facetedges for the two edges inside the wedge; if 
*                second is null, means there is no second edge.
*                This is essentially the reverse of eliminating an edge.
*/

void versplit(fe_a,fe_b)
facetedge_id fe_a,fe_b;
{
  vertex_id old_v = get_fe_headv(fe_a);
  vertex_id new_v;
  edge_id new_e,new_a,new_b=0,e_id;
  edge_id old_a = get_fe_edge(fe_a);
  edge_id old_b = get_fe_edge(fe_b);
  facetedge_id fe_aa = get_prev_facet(fe_a);
  facetedge_id fe_bb = get_prev_facet(fe_b);
  facetedge_id fe_a_old,fe_a_new,fe_a_e;
  facetedge_id fe_b_old=0,fe_b_new=0,fe_b_e=0;
  facet_id new_fa,new_fb=0;
  body_id b_id;
  int i;
  REAL *x;
  int halfflag;
  facetedge_id fe;

  /* create a new vertex, which will be split away with wedge */
  new_v = dup_vertex(old_v);
  x = get_coord(new_v);
  for ( i = 0 ; i < SDIM ; i++ )
     x[i] += 0.01*new_displacement[i];
  
  /* create new edge between vertices */
  new_e = new_edge(old_v,new_v,NULLID);
  if ( get_vattr(old_v) & CONSTRAINT )
     set_e_conmap(new_e,get_v_constraint_map(old_v));
  if ( get_vattr(old_v) & BOUNDARY )
     set_edge_boundary_num(new_e,get_boundary(old_v)->num);
  
  /* create new edges split off from wedge */
  new_a = dup_edge(old_a);
  insert_vertex_edge(get_edge_tailv(old_a),new_a);
  insert_vertex_edge(old_v,inverse_id(new_a));
  if ( valid_id(fe_b) )
     {
        new_b = dup_edge(old_b);
        insert_vertex_edge(old_v,new_b);
        insert_vertex_edge(get_edge_headv(old_b),inverse_id(new_b));
     }
 
  /* reset edge endpoints coming into new_v */
  fe = fe_a;
  halfflag = 0;
  do
     { e_id = get_fe_edge(fe);
        remove_vertex_edge(old_v,inverse_id(e_id));
        set_edge_headv(e_id,new_v);
        fe = get_next_facet(inverse_id(get_next_edge(fe)));
        if ( equal_id(fe,get_next_facet(fe)) )
          { halfflag = 1; /* need to go back an do other side of wedge */
             break;
          }
     } while ( !equal_id(fe,fe_a) );
  if ( halfflag )
     { fe = fe_aa;
        do
          {
             e_id = get_fe_edge(fe);
             remove_vertex_edge(old_v,inverse_id(e_id));
             set_edge_headv(e_id,new_v);
             fe = get_prev_facet(inverse_id(get_next_edge(fe)));
             if ( equal_id(fe,get_next_facet(fe)) )
                  break;
          } while ( !equal_id(fe,fe_aa) );
     }

  /* create two new facets */
  new_fa = new_facet();
  if ( valid_id(fe_b) )
     {
        new_fb = new_facet();
     }

  /* create new facet-edges */
  fe_a_e = new_facetedge(new_fa,new_e);
  fe_a_new = new_facetedge(new_fa,new_a);
  fe_a_old  = new_facetedge(new_fa,inverse_id(old_a));
  if ( valid_id(fe_b) )
     {
        fe_b_e = new_facetedge(new_fb,inverse_id(new_e));
        fe_b_new = new_facetedge(new_fb,new_b);
        fe_b_old  = new_facetedge(new_fb,inverse_id(old_b));
     }

  /* link elements to facet-edges */
  set_edge_fe(old_a,inverse_id(fe_a_old)); 
  set_edge_fe(new_a,fe_a_new);
  set_edge_fe(new_e,fe_a_e);
  set_facet_fe(new_fa,fe_a_new);
  if ( valid_id(fe_b) )
     {
        set_edge_fe(old_b,inverse_id(fe_b_old)); 
        set_edge_fe(new_b,fe_b_new);
        set_facet_fe(new_fb,fe_b_new);
     }

  /* link edges around new facets */
  set_next_edge(fe_a_e,fe_a_old);
  set_next_edge(fe_a_old,fe_a_new);
  set_next_edge(fe_a_new,fe_a_e);
  set_prev_edge(fe_a_old,fe_a_e);
  set_prev_edge(fe_a_new,fe_a_old);
  set_prev_edge(fe_a_e,fe_a_new);
  if ( valid_id(fe_b) )
     {
        set_next_edge(fe_b_e,fe_b_new);
        set_next_edge(fe_b_old,fe_b_e);
        set_next_edge(fe_b_new,fe_b_old);
        set_prev_edge(fe_b_old,fe_b_new);
        set_prev_edge(fe_b_new,fe_b_e);
        set_prev_edge(fe_b_e,fe_b_old);
     }

  /* link facets around edges */
  
  /* new middle edge */
  if ( valid_id(fe_b) )
     {  
        set_next_facet(fe_a_e,inverse_id(fe_b_e));
        set_next_facet(fe_b_e,inverse_id(fe_a_e));
        set_prev_facet(fe_a_e,inverse_id(fe_b_e));
        set_prev_facet(fe_b_e,inverse_id(fe_a_e));
     }
  else
     {
        set_next_facet(fe_a_e,fe_a_e);
        set_prev_facet(fe_a_e,fe_a_e);
     }
    
  /* facet A old edge */
  set_next_facet(fe_a_old,inverse_id(fe_a));
  set_prev_facet(fe_a_old,inverse_id(fe_aa));

  /* facet A new edge */
  set_next_facet(fe_a_new,get_next_facet(fe_a));
  set_prev_facet(fe_a_new,get_prev_facet(fe_aa));

  /* reset old links */
  set_next_facet(get_prev_facet(fe_a_old),fe_a_old);
  set_prev_facet(get_next_facet(fe_a_old),fe_a_old);
  set_next_facet(get_prev_facet(fe_a_new),fe_a_new);
  set_prev_facet(get_next_facet(fe_a_new),fe_a_new);

  /* reset edges of facet-edges around new edge */
  generate_edge_fe_init();
  while ( generate_edge_fe(new_a,&fe) )
     set_fe_edge(fe,new_a);

  if ( valid_id(fe_b) )
     {  
        /* facet B old edge */
        set_next_facet(fe_b_old,inverse_id(fe_b));
        set_prev_facet(fe_b_old,inverse_id(fe_bb));

        /* facet B new edge */
        set_next_facet(fe_b_new,get_next_facet(fe_b));
        set_prev_facet(fe_b_new,get_prev_facet(fe_bb));

        /* reset old links */
        set_next_facet(get_prev_facet(fe_b_old),fe_b_old);
        set_prev_facet(get_next_facet(fe_b_old),fe_b_old);
        set_next_facet(get_prev_facet(fe_b_new),fe_b_new);
        set_prev_facet(get_next_facet(fe_b_new),fe_b_new);

        /* reset edges of facet-edges around new edge */
        generate_edge_fe_init();
        while ( generate_edge_fe(new_b,&fe) )
          set_fe_edge(fe,new_b);
     }

  /* set new facet bodies */
  fe = get_next_facet(fe_a_new);
  b_id = get_facet_body(get_fe_facet(fe));
  set_facet_body(inverse_id(new_fa),b_id);
  if ( valid_id(fe_b) )
     set_facet_body(inverse_id(new_fb),b_id);
  fe = inverse_id(get_prev_facet(fe_a_new));
  b_id = get_facet_body(get_fe_facet(fe));
  set_facet_body(new_fa,b_id);
  if ( valid_id(fe_b) )
     set_facet_body(new_fb,b_id);
}

/**********************************************************************

     Vertex popping.

*************************************************************************/

#define OTHERCONE      0
#define WIREBOUNDARY  1
#define PLANECONE      2
#define WIREWING        3
#define WIRECORNER     4
#define TRIPLE_EDGER  5
#define TETRAHEDRAL    6
static  facetedge_id *felist;  /* for cell-ordered list */


/*************************************************************************
*
*  Function: verpop_film()
*
*  Purpose:  Pop vertices with non-minimal tangent cones.  Assumes
*                all edges are at most triple edges.
*/

int verpop_film()
{
  int popcount = 0;

  while ( find_vertex_to_pop() )
     popcount++;

  return popcount;
}

/************************************************************************
*
*  function: find_vertex_to_pop
*
*  purpose:  Looks through all vertices and finds one to pop.  Whole
*                process restarts after each pop since popping makes
*                search lists obsolete.
*
*                Returns number popped (0 or 1)
*/

int find_vertex_to_pop()
{
  struct verfacet *vflist;
  facetedge_id fe;
  facet_id f_id;
  int count,maxcount;
  int spot,dups,conetype;
  int popcount = 0;  


  /* Allocate list, with room for each facet thrice */
  maxcount = 3*web.skel[FACET].count;
  vflist = (struct verfacet *)temp_calloc(sizeof(struct verfacet),maxcount);  

  /* create unsorted list */
  count = 0;
  FOR_ALL_FACETS(f_id)
     {
        vflist[count].f_id = f_id;
        fe = get_facet_fe(f_id);
        vflist[count].v_id = get_fe_tailv(fe);
        count++;
        fe = get_next_edge(fe);    
        vflist[count].f_id = f_id;
        vflist[count].v_id = get_fe_tailv(fe);
        count++;
        fe = get_next_edge(fe);    
        vflist[count].f_id = f_id;
        vflist[count].v_id = get_fe_tailv(fe);
        count++;
        if ( count > maxcount )
          {
             kb_error(1300,"Internal error: Not enough structures allocated for vertex-facet list.\n",
                RECOVERABLE);
             return 0;
          }
     }

  /* sort by vertex order */
  qsort((char *)vflist,count,sizeof(struct verfacet),FCAST vfcomp);

  /* go through list and pop appropriate vertices */
  for ( spot = 0 ; spot < count ; spot += dups )
     {
        vertex_id v_id = vflist[spot].v_id;

        /* find how many successive duplicates of current vertex */
        for ( dups = 1 ; spot+dups < count ; dups++ )
          if ( !equal_id(vflist[spot+dups].v_id,v_id) ) break;

        if ( get_vattr(v_id) & FIXED ) continue;

        conetype = cone_analyze(vflist + spot,dups);
        if ( conetype == OTHERCONE )
          { popcount += pop_vertex(v_id,dups);
          }
        temp_free((char *)felist);
        if ( popcount ) break;  /* one vertex at a time */
     }
             
  temp_free((char *)vflist);
  return popcount;
}

/**************************************************************************
*
*  function: cone_analyze()
*
*  purpose: figure out what type of tangent cone a vertex has.
*
*/

#define CELLMAX 30
#define ARCMAX  60

static struct cell { int start;     /* arc list start in arclist */
                  int festart;  /* starting place in felist */
                  int num;        /* number of arcs                */
                  int fenum;     /* number of facetedges */
                  REAL area;     /* external angle deficit     */
                  body_id b_id; /* which body, if any */
                } cell[CELLMAX];
static struct arc  { int start;     /* edge list start in felist */
                  int num;        /* number of edges              */
                  int valence;  /* number of arcs into head node */
/*                int headtype; */  /* type of head node */
                } arclist[ARCMAX];
static  int cells;

int figure_type ARGS((int));

int figure_type(arcs)
int arcs;
{ int k,type;

  /* Classifying cone */
  switch ( cells )
     {
        case 1: /* wire boundary */
                  type = WIREBOUNDARY;
                  break;

        case 2: switch ( arcs )
                    {
                      case 2: /* internal plane */
                         type = PLANECONE;
                         break;
                      case 4: /* two wings on wire boundary */
                         type = WIREWING;  /* minimality depends on angle */
                         break;
                      default: /* n boundary wires meet at center */
                         type = WIRECORNER;
                         break;
                    }
                  break;

        case 3: switch ( arcs )
                    {
                      case 6: /* triple edge */
                         type = TRIPLE_EDGER;
                         for ( k = 0 ; k < cells ; k++ )
                            if ( cell[k].num != 2 ) type = OTHERCONE;
                         break;
                      default:
                         type = OTHERCONE;
                         break;
                    }
                  break;

        case 4: switch ( arcs )
                    {
                      case 12: /* maybe tetrahedral cone */
                         type = TETRAHEDRAL;
                         for ( k = 0 ; k < cells ; k++ )
                            if ( cell[k].num != 3 ) type = OTHERCONE;
                         break;
                      default:
                         type = OTHERCONE;
                         break;
                    }
                  break;

         default:
                    type = OTHERCONE;
                    break;
      }
  return type;
}

int cone_analyze(vf,count)
struct verfacet *vf;
int count;
{
  int type;  /* final classification */
  facetedge_id fe,nextfe,ray,nextray;
  vertex_id v_id = vf->v_id;
  int cellstart,arcstart,cellsides;
  int arclen,valence;
  int arcs;
  int k,j;
  int nodeflag;  /* whether any interesting node has been found for cell */

  /* First, set up network structure of edges opposite central vertex */
  felist = (facetedge_id *)temp_calloc(sizeof(facetedge_id),2*count);

  /* fill with outside edges of facets */
  for ( k = 0 ; k < count ; k++ )
     {
        fe = get_facet_fe(vf[k].f_id);
        if ( equal_id(v_id,get_fe_tailv(fe)) )
          fe = get_next_edge(fe);
        else if ( equal_id(v_id,get_fe_headv(fe)) )
          fe = get_prev_edge(fe);
        felist[2*k] = fe;
        felist[2*k+1] = inverse_id(fe);
     }

  /* order into arcs and cell boundaries */
  k = 0;        /* current edge number */
  cells = 0;    /* current cell number */
  arcs = 0;     /* current arc number */
  while ( k < 2*count )
     {
CELLSTART:
        /* starting new cell */
        if ( k >= 2*count ) break;
        if ( cells >= CELLMAX )
          kb_error(1301,"pop: Too many cells in cone_analyze().\n",RECOVERABLE);

        nodeflag = 0;  /* whether interesting node found on cell */
        cellstart = k;
        cellsides = 0;
        cell[cells].start = arcs;

ARCSTART:
        /* starting new arc */
        if ( arcs >= ARCMAX )
          kb_error(1302,"pop: Too many arcs in cone_analyze().\n",RECOVERABLE);

        arcstart = k;
        arclist[arcs].start = arcstart;
ARCRESTART:
        arclen = 0;

        for (k = arcstart; k < 2*count ;k++)
        {
        /* starting next edge */
        arclen++;
        fe = felist[k];
        nextray = ray = get_next_edge(fe);
        for ( valence = 1 ; ; valence++ )
          { nextray = get_next_facet(nextray);
             if ( equal_id(nextray,ray) ) break;
          }
        nextray = get_next_facet(ray);
        nextfe = get_next_edge(inverse_id(nextray));
        if ( equal_id(nextfe,felist[cellstart]) )  /* have gone around */
         {
            arclist[arcs].num = arclen;
            arclist[arcs].valence = valence;
            arcs++;
            cellsides++;
            cell[cells].num = cellsides;
            cells++;
            k++;
            goto CELLSTART;
         }

        /* linear search list until nextfe found */
        for ( j = k+1 ; j < 2*count ; j++ )
          if ( equal_id(nextfe,felist[j]) ) break;
        if ( !equal_id(nextfe,felist[j]) ) 
          { kb_error(1303,"Internal error: cone_analyze: can't find nextfe in felist.\n",WARNING);
             return 1;
          }
        if ( j > k+1 ) /* swap into place */
          {
             felist[j] = felist[k+1];
             felist[k+1] = nextfe;
          }

        /* see if node between is interesting */
        if ( (valence != 2))
          { /* have interesting node */
             if ( nodeflag == 0 ) 
                { /* first interesting node for cell */
                  /* reset felist so cell and arc start are same */
                  felist[k] = felist[arcstart];
                  felist[arcstart] = nextfe;
                  felist[k+1] = fe;
                  nodeflag = 1;
                  goto ARCRESTART;
                }
             arclist[arcs].num = arclen;
             arclist[arcs].valence = valence;
             arcs++;
             cellsides++;
             k++;
             goto ARCSTART;
          }
        }
     }

  type = figure_type(arcs);  /* so stupid SUN can optimize */

  return  type;
}


/*******************************************************************
*
*  Function:  pop_vertex()
*
*  Purpose:    pops one non-minimal vertex using info found by
*                 cone_analyze().  Finds areas of each face.  All
*                 face cones are pulled out to sphere except largest
*                 area.  Special treatment for face with disjoint
*                 boundary.
*/

int pop_vertex(v_id,count)
vertex_id v_id;
int count;  /* number of entries in vflist */
{
  int i,j,k,m;
  struct arc *ar;
  int fenum;
  facetedge_id fe;
  REAL cosdef;
  REAL prevnormal[MAXCOORD],prevnorm;
  REAL thisnormal[MAXCOORD],thisnorm;
  REAL maxarea = 0.0;
  int  bigcell=0;
  REAL total_area = 0.0;
  REAL total_angle,angle;
  REAL side[MAXCOORD],ray[MAXCOORD];  
  REAL *vx,newx[MAXCOORD];
  edge_id ray_e;
  facetedge_id ray_fe;
  vertex_id new_v;

  if ( verbose_flag )
  { sprintf(msg,"Popping vertex %d\n",ordinal(v_id)+1);
     outstring(msg);
  }

  /* fix up cell structures */
  for ( i = 0 ; i < cells ; i++ )
     { cell[i].fenum = 0;
        ar = arclist + cell[i].start;
        cell[i].festart = ar->start;
        for ( j = 0 ; j < cell[i].num ; j++, ar++ )
          cell[i].fenum += ar->num;
     }

  /* see if any cells are totally detachable */
  for ( i = 0 ; i < cells ; i++ )
     { vertex_id newv;
        ar = arclist + cell[i].start;
        if ( (cell[i].num != 1) || (ar->valence != 2) ) continue;
        /* now have one */
        newv = dup_vertex(v_id);
        for ( j = 0 ; j < ar->num ; j++ )
          { fe = felist[ar->start+j];
             ray_e = get_fe_edge(get_next_edge(fe));
             remove_vertex_edge(v_id,inverse_id(ray_e));
             set_edge_headv(ray_e,newv);
          }
        return 1;    /* safe to do only one at a time */
     }

  /* calculate areas of each cell */
  for ( i = 0 ; i < cells ; i++ )
     {
        total_angle = 0.0;
        fenum = cell[i].festart;
        fe = felist[fenum+cell[i].fenum-1];
        cell[i].b_id = get_facet_body(get_fe_facet(inverse_id(fe)));
        get_fe_side(fe,side);
        get_fe_side(get_prev_edge(fe),ray);
        cross_prod(ray,side,prevnormal);
        prevnorm = sqrt(SDIM_dot(prevnormal,prevnormal));
        for ( k = 0 ; k < cell[i].fenum ; k++,fenum++ )
          { 
             fe = felist[fenum];
             get_fe_side(fe,side);
             get_fe_side(get_prev_edge(fe),ray);
             cross_prod(ray,side,thisnormal);
             thisnorm = sqrt(SDIM_dot(thisnormal,thisnormal));
             cosdef = SDIM_dot(prevnormal,thisnormal)/
                             prevnorm/thisnorm;
             if ( cosdef > 1.0 ) angle = 0.0;
             else if ( cosdef < -1.0 ) angle = M_PI;
             else angle = acos(cosdef);
             if ( SDIM_dot(side,prevnormal) > 0.0 )
                total_angle += angle; 
             else
                total_angle -= angle; 

             /* set up for next loop */
             prevnorm = thisnorm;
             memcpy((char *)prevnormal,(char *)thisnormal,sizeof(prevnormal));
          }

      cell[i].area = 2*M_PI - total_angle;
      total_area += cell[i].area;
      if ( cell[i].area > maxarea ) { bigcell = i; maxarea = cell[i].area; }
     }                  

  if ( total_area < 4*M_PI - 0.00001 )
  { sprintf(errmsg,"Solid angle deficit %g found around vertex %d. Not popped.\n",
      (double)(4*M_PI-total_area), ordinal(v_id)+1);
     kb_error(3395,errmsg,WARNING);
     return 0;
  }
  if ( total_area > 4*M_PI + 0.00001 )
  { sprintf(errmsg,"Solid angle excess %g found around vertex %d. Not popped.\n",
      (double)(total_area-4*M_PI), ordinal(v_id)+1);
     kb_error(3395,errmsg,WARNING);
     return 0;
  }
  

  /* Now go around covering over all cells but bigcell or cells
      bigger than hemisphere */

  /* First, put new endpoint on all edges coming into v_id */
  vx = get_coord(v_id);
  for ( j = 0 ; j < 2*count ; j++ )
     { REAL rayvec[MAXCOORD];
        fe = felist[j];
        ray_fe = get_prev_edge(fe);
        if ( !equal_id(v_id,get_fe_tailv(ray_fe)) ) continue; /* done this */
        ray_e = get_fe_edge(ray_fe);
        get_edge_side(ray_e,rayvec);
        for ( m = 0 ; m < SDIM ; m++ )
          newx[m] = 0.5*rayvec[m] + vx[m];
        new_v = new_vertex(newx,ray_e);
        remove_vertex_edge(v_id,ray_e);
        set_edge_tailv(ray_e,new_v);
     }

  /*  divide up old facets */
  for ( j = 0 ; j < 2*count ; j++ )
     { facetedge_id new_fe,new_fe_next,new_fe_prev;
        facetedge_id inray_fe;
        vertex_id v1,v2;
        edge_id new_e;
     
        fe = felist[j];
        ray_fe = get_prev_edge(fe);
        inray_fe = get_next_edge(fe);
        v1 = get_fe_tailv(ray_fe);
        v2 = get_fe_headv(inray_fe);

        /*  Each facet gets covered twice. First time we put in edge;
             second time triangulate the quadrilateral.  This way we
             can substitute the new inner fe for the old outer fe
             in felist.
         */
        if ( equal_id(inray_fe,get_prev_edge(ray_fe)) )
          { /* first time around */
             
             /* put in new edge */
             new_e = new_edge(v2,v1,get_fe_facet(fe));
             new_fe = new_facetedge(get_fe_facet(fe),new_e);
             set_edge_fe(new_e,new_fe);
             set_next_edge(inray_fe,new_fe);
             set_prev_edge(new_fe,inray_fe);
             set_next_edge(new_fe,ray_fe);
             set_prev_edge(ray_fe,new_fe);

             /* stub fe's for cells on each side */
             new_fe_prev = new_facetedge(NULLFACET,new_e);
             new_fe_next = new_facetedge(NULLFACET,new_e);
             set_next_facet(new_fe,new_fe_next);
             set_next_facet(new_fe_next,new_fe_prev);
             set_next_facet(new_fe_prev,new_fe);
             set_prev_facet(new_fe,new_fe_prev);
             set_prev_facet(new_fe_prev,new_fe_next);
             set_prev_facet(new_fe_next,new_fe);
             
             felist[j] = inverse_id(new_fe_next);    /* new inner fe */
          }
        else
          { felist[j] = inverse_id(get_next_facet(get_next_edge(inray_fe)));
             face_triangulate(get_fe_facet(fe),4);
          }
      }
          
        
  /* now connect up insides of cells */
  for ( i = 0 ; i < cells ; i++ )
     {
        facet_id new_f=0,old_f;
        facetedge_id prev_fe=0;  /* last in edge chain */

        fenum = cell[i].festart;
        if ( (i != bigcell) && (cell[i].area < 2*M_PI) )
         { prev_fe = felist[fenum+cell[i].fenum-1];
            old_f = facet_inverse(get_fe_facet(get_next_facet(prev_fe)));
            new_f = dup_facet(old_f);
            set_facet_body(new_f,cell[bigcell].b_id);
            set_facet_body(inverse_id(new_f),cell[i].b_id);
            set_facet_fe(new_f,prev_fe);
         }

        for ( k = 0 ; k < cell[i].fenum ; k++,fenum++ )
          { 
             fe = felist[fenum];
             if ( (i == bigcell) || (cell[i].area >= 2*M_PI) )
                { /* have to excise fe */
                  set_prev_facet(get_next_facet(fe),get_prev_facet(fe));
                  set_next_facet(get_prev_facet(fe),get_next_facet(fe));
                  free_element(fe);
                }
             else
                { /* hook into edge loop around facet */
                  set_prev_edge(fe,prev_fe);
                  set_next_edge(prev_fe,fe);
                  set_fe_facet(fe,new_f);
                  prev_fe = fe;
                }
          }

        /* if necessary, triangulate new cell facet */
        if ( (i != bigcell) && (cell[i].area < 2*M_PI) ) 
          if ( cell[i].fenum > 3 )
             face_triangulate(new_f,cell[i].fenum);
     }


  /* free original central vertex */
  free_element(v_id);

  return 1;
}

