/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <signal.h>
#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>

#include "ef-backend.h"
#include "ef-msg-queue.h"

EFMessageQueue *queue;

static void
msg_loop ()
{
	EFMessage *msg;

	g_return_if_fail (queue != NULL);

	while (1) {
		msg = ef_msg_queue_receive (queue);

		/* Note: not very useful to print an error message as we are in daemon mode */
		if (msg) {
			ef_backend_set_governor (msg->type);

			if (msg->type == GOV_USERSPACE)
				ef_backend_set_frequency (msg->frequency);

			g_free (msg);
		} else
			syslog (LOG_ERR, "failed to receive a message: %m");
	}
}

static void
sig_handler (int signal)
{
	ef_msg_queue_free (queue);

	exit (100);
}

int
main (int    argc,
      char **argv)
{
	if (getuid ()) {
		fprintf (stderr, _("Not running as root. Please run emifreqd with root priviledges.\n"));
		exit (1);
	}

	if (!ef_backend_is_available ()) {
		fprintf (stderr, _("CpuFreq support not available. Check sysfs is mounted and your CPU-specific module is loaded or built in the kernel.\n"));
		exit (2);
	}

	signal (SIGTERM, sig_handler);
	signal (SIGQUIT, sig_handler);
	signal (SIGINT,  sig_handler);

	if ((queue = ef_msg_queue_new (TRUE)) == NULL) {
		perror (_("Error while creating the message queue"));
		exit (3);
	}

	openlog ("emifreqd", LOG_PID, LOG_DAEMON);

	daemon (0, 0);

	msg_loop (queue);

	g_assert_not_reached ();

	ef_msg_queue_free (queue);

	return 0;
}
