/* @source cacheensembl application
**
** Prepares an EMBOSS cache file for an Ensembl server
**
** @author Copyright (C) Michael Schuster
** @modified $Date: 2011/07/08 16:29:51 $ by $Author: rice $
** @version $Revision: 1.2 $
** @@
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
******************************************************************************/




/* ==================================================================== */
/* ========================== include files =========================== */
/* ==================================================================== */

#include "emboss.h"




/* ==================================================================== */
/* ============================ constants ============================= */
/* ==================================================================== */




/* ==================================================================== */
/* ======================== global variables ========================== */
/* ==================================================================== */




/* ==================================================================== */
/* ============================== data ================================ */
/* ==================================================================== */




/* ==================================================================== */
/* ==================== function prototypes =========================== */
/* ==================================================================== */




/* ==================================================================== */
/* ======================== private functions ========================= */
/* ==================================================================== */

static int cacheensembl_stringcompare(const void* P1, const void* P2);

static void cacheensembl_stringdelete(void** PP1, void* cl);




/* @prog cacheensembl *********************************************************
**
** Prepares an EMBOSS cache file for an Ensembl server
**
******************************************************************************/

int main(int argc, char** argv)
{
    AjPFile outf = NULL;
    AjPFile cachef = NULL;

    AjIList iterator = NULL;
    AjPList aliases  = NULL;
    AjPList dbas     = NULL;
    AjPList species  = NULL;

    AjPStr alias   = NULL;
    AjPStr dbname  = NULL;
    AjPStr spname  = NULL;
    AjPStr svrname = NULL;
    AjPStr svrurl  = NULL;
    AjPStr dbcurl  = NULL;

    AjPTime svrtime = NULL;

    EnsEDatabaseadaptorGroup dbag = ensEDatabaseadaptorGroupNULL;

    EnsPDatabaseadaptor dba = NULL;

    EnsPDatabaseconnection dbc = NULL;

    embInit("cacheensembl", argc, argv);
    ensInit();

    svrname = ajAcdGetString("servername");
    outf    = ajAcdGetOutfile("outfile");
    cachef = ajAcdGetOutfile("cachefile");

    dbcurl = ajStrNew();
    svrurl = ajStrNew();
    dbname = ajStrNew();

    ajNamSvrGetUrl(svrname, &svrurl);

    if(!svrurl)
        ajFatal("Could not resolve server name '%S'.", svrname);

    dbc = ensDatabaseconnectionNewUrl(svrurl);
    ensRegistryLoadDatabaseconnection(dbc);
    ensDatabaseconnectionDel(&dbc);

    /* Write the server file header. */

    svrtime = ajTimeNewTodayFmt("cachefile");
    ajFmtPrintF(cachef, "# %S %D\n", ajFileGetNameS(cachef), svrtime);
    ajTimeDel(&svrtime);

    ajFmtPrintF(cachef,
                "# Automatically generated by cacheensembl "
                "for server '%S'.\n\n",
                svrname);

    /*
    ** Get all Ensembl Database Adaptor objects and write them as
    ** EMBOSS Database definitions.
    */

    aliases = ajListstrNew();
    dbas    = ajListNew();
    species = ajListstrNew();

    ensRegistryRetrieveAllSpecies(species);

    while(ajListstrPop(species, &spname))
    {
        ensRegistryGetAllDatabaseadaptors(ensEDatabaseadaptorGroupNULL,
                                          spname,
                                          dbas);

        while(ajListPop(dbas, (void**) &dba))
        {
            dbag = ensDatabaseadaptorGetGroup(dba);

            if(dbag == ensEDatabaseadaptorGroupNULL)
            {
                ajDebug("cacheensembl main got unexpected "
                        "Ensembl Database Adaptor Group %d.\n",
                        dbag);

                continue;
            }

            ajStrAssignS(&dbname, ensDatabaseadaptorGetSpecies(dba));

            if(dbag != ensEDatabaseadaptorGroupCore)
            {
                ajStrAppendC(&dbname, "_");
                ajStrAppendC(&dbname, ensDatabaseadaptorGroupToChar(dbag));
            }

            dbc = ensDatabaseadaptorGetDatabaseconnection(dba);

            ensDatabaseconnectionFetchUrl(dbc, &dbcurl);

            if(outf)
                ajFmtPrintF(outf, "%S\n", dbname);

            ajFmtPrintF(cachef, "DBNAME %S [\n", dbname);
            ajFmtPrintF(cachef, "  release: \"%s\"\n", ensSoftwareGetVersion());
            ajFmtPrintF(cachef, "  server:  \"%S\"\n", svrname);
            ajFmtPrintF(cachef, "  url:     \"%S\"\n", dbcurl);
            ajFmtPrintF(cachef, "]\n");
            ajFmtPrintF(cachef, "\n");

            if(dbag != ensEDatabaseadaptorGroupCore)
                continue;

            ensRegistryAliasFetchAllbySpecies(
                ensDatabaseadaptorGetSpecies(dba),
                aliases);

            /*
            ** Format all aliases to lower case,
            ** sort them alphabetically and remove duplicates.
            */

            iterator = ajListIterNew(aliases);
            while(!ajListIterDone(iterator))
            {
                alias = ajListstrIterGet(iterator);
                ajStrFmtLower(&alias);
            }
            ajListIterDel(&iterator);

            ajListSortUnique(aliases,
                             cacheensembl_stringcompare,
                             cacheensembl_stringdelete);

            alias = NULL;
            if(ajListGetLength(aliases) > 0)
            {
                while(ajListstrPop(aliases, &alias))
                {
                    /*
                    ** Reject any aliases with other than alpha-numeric
                    ** characters like white space.
                    */

                    if(ajStrIsAlnum(alias))
                        ajFmtPrintF(cachef,
                                    "ALIAS %S %S\n",
                                    alias,
                                    ensDatabaseadaptorGetSpecies(dba));

                    ajStrDel(&alias);
                }

                ajFmtPrintF(cachef, "\n");
            }

            /* Ensembl Database Adaptor objects *must not* be deleted. */
        }

        ajStrDel(&spname);
    }

    ajListstrFree(&aliases);
    ajListFree(&dbas);

    ajStrDel(&dbcurl);
    ajStrDel(&svrurl);
    ajStrDel(&dbname);
    ajStrDel(&svrname);

    ajFileClose(&outf);
    ajFileClose(&cachef);

    embExit();

    return EXIT_SUCCESS;
}




/* @funcstatic cacheensembl_stringcompare *************************************
**
** Comparison function to sort AJAX String objects case-insensitively in
** ascending order.
**
** @param [r] P1 [const void*] AJAX String 1
** @param [r] P2 [const void*] AJAX String 2
** @see ajListSortUnique
**
** @return [int] The comparison function returns an integer less than,
**               equal to, or greater than zero if the first argument is
**               considered to be respectively less than, equal to, or
**               greater than the second.
** @@
******************************************************************************/

static int cacheensembl_stringcompare(const void* P1, const void* P2)
{
    const AjPStr string1 = NULL;
    const AjPStr string2 = NULL;

    string1 = *(AjPStr const*) P1;
    string2 = *(AjPStr const*) P2;

#if AJFALSE
    if(ajDebugTest("cacheensembl_stringcompare"))
        ajDebug("cacheensembl_stringcompare\n"
                "  string1 %u\n"
                "  string2 %u\n",
                string1,
                string2);
#endif

    return ajStrCmpCaseS(string1, string2);
}




/* @funcstatic cacheensembl_stringdelete **************************************
**
** ajListSortUnique nodedelete function to delete AJAX String objects
** that are redundant.
**
** @param [r] PP1 [void**] AJAX String address 1
** @param [r] cl [void*] Standard. Passed in from ajListSortUnique
** @see ajListSortUnique
**
** @return [void]
** @@
******************************************************************************/

static void cacheensembl_stringdelete(void** PP1, void* cl)
{
    if(!PP1)
        return;

    (void) cl;

    ajStrDel((AjPStr*) PP1);

    *PP1 = NULL;

    return;
}
