extern "C" {
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <pthread.h>
#include "map.h"

static unsigned long const bitrate_table[5][15] = {
  /* MPEG-1 */
  { 0,  32000,  64000,  96000, 128000, 160000, 192000, 224000,  /* Layer I   */
       256000, 288000, 320000, 352000, 384000, 416000, 448000 },
  { 0,  32000,  48000,  56000,  64000,  80000,  96000, 112000,  /* Layer II  */
       128000, 160000, 192000, 224000, 256000, 320000, 384000 },
  { 0,  32000,  40000,  48000,  56000,  64000,  80000,  96000,  /* Layer III */
       112000, 128000, 160000, 192000, 224000, 256000, 320000 },

  /* MPEG-2 LSF */
  { 0,  32000,  48000,  56000,  64000,  80000,  96000, 112000,  /* Layer I   */
       128000, 144000, 160000, 176000, 192000, 224000, 256000 },
  { 0,   8000,  16000,  24000,  32000,  40000,  48000,  56000,  /* Layers    */
        64000,  80000,  96000, 112000, 128000, 144000, 160000 } /* II & III  */
};

static unsigned int const samplerate_table[3] = { 44100, 48000, 32000 };

int DecodeHeader(char *current, struct hdr *hdr)
{
    unsigned long h;
    int protection=0;
    int padding=0;

    if(hdr)
        memset((char *)hdr, 0, sizeof(*hdr));
    
    h=((*(unsigned char *)current) << 24) | ((*((unsigned char *)current+1)) << 16) | ((*((unsigned char *)current+2)) << 8) | (*((unsigned char *)current+3));

    if((h & 0xffe00000) != 0xffe00000)
    {
        return(-1);
    }
    
    if((h & 0x00180000) == 0x00080000)
    {
        return(-1);
    }
    
    if(hdr)
    {
        if((h & 0x00180000) == 0x00100000)
            hdr->mpeg_version=3;
        else if((h & 0x00180000) == 0x00180000)
            hdr->mpeg_version=1;
        else
            hdr->mpeg_version=2;
    }

    if((h & 0x00060000) == 0)
    {
        return(-1);
    }
    
    if(hdr)
    {
        if((h & 0x00060000) == 0x00060000)
            hdr->layer=1;
        else if((h & 0x00060000) == 0x00040000)
            hdr->layer=2;
        else
            hdr->layer=3;
    }

    if(!(h & 0x00010000))
        protection=1;
    
    if((h & 0x0000f000) == 0x0000f000)
    {
        return(-1);
    }
    
    if(hdr)
    {
        if(hdr->mpeg_version == 2)
            hdr->bitrate=bitrate_table[3+(hdr->layer >> 1)][(h & 0x0000f000) >> 12];
        else
            hdr->bitrate=bitrate_table[hdr->layer-1][(h & 0x0000f000) >> 12];
    }

    if((h & 0x00000c00) == 0x00000c00)
    {
        return(-1);
    }
    
    if(hdr)
    {
        hdr->samplerate=samplerate_table[(h & 0x00000c00) >> 10];
        if(hdr->mpeg_version == 2)
            hdr->samplerate/=2;
        else if(hdr->mpeg_version == 3)
            hdr->samplerate/=4;
        
        padding=h & 0x00000200 ? 1 : 0;

        hdr->padding=padding;

        if(hdr->layer == 1)
        {
            hdr->framesize=384;
            hdr->framelength=(12*hdr->bitrate/hdr->samplerate+padding)*4;
        }
        else
        {
            hdr->framesize=1152;
            hdr->framelength=144*hdr->bitrate/hdr->samplerate+padding;
        }
        if(protection)
            hdr->framelength+=2;
    }

    if(hdr)
    {
        if((h & 0x000000c0) == 0x000000c0)
            hdr->channels=1;
        else
            hdr->channels=2;
    }

//printf("Version %d\nLayer %d\nbr %d\nsr %d\nfl %d fs %d\n", hdr->mpeg_version, hdr->layer, hdr->bitrate, hdr->samplerate, hdr->framelength, hdr->framesize);
    return(0);
}

int DecodeChain(char *current, char *buffer, int bytes_in_buffer)
{
    struct hdr hdr;
    while(current <= (buffer+bytes_in_buffer-4))
    {
        if(DecodeHeader(current, &hdr) < 0)
		{
//			printf("Bad sub frame\n");
            return(-1);
		}
        if(!hdr.framelength)
		{
            return(0);
		}
        current+=hdr.framelength;
    }
    return(0);
}
}
