#ifndef _JACK_H_INCLUDED
#define _JACK_H_INCLUDED

#include <qobject.h>
#include <qfile.h>
#include <glib.h>
#include <pthread.h>
#include <samplerate.h>
#include "wavelet.h"
#include "RingBuffer.hpp"
#include "jacklow.h"

class JackPlayer;
class Wavelet;
class Pump;
class QWidget;
class Client;
class Port;
class Effect;

extern "C" {
#include <jack/jack.h>
}

typedef struct tagBuffer
{
	short *left;
	short *right;
	int nsamples;
} BUFFER;

class QImage;

class StreamSource : public QObject
{
Q_OBJECT

public:
	StreamSource(QWidget *owner);
	~StreamSource();

	virtual int get_buffer(char *buffer, int max);

	virtual unsigned long frame_position();
	virtual unsigned long total_frames();

	virtual int play(unsigned long start_frame, unsigned long frames, bool loop=false);
	virtual void exit_loop(void);
	virtual int stop();
	virtual int attach(QString name);
	virtual int detach();
	virtual int is_playing();
	virtual void set_pitch(double);
	virtual void generate_wave(void);
	virtual void get_data(void);
	virtual void set_transport_position(unsigned long position);
	virtual void prefetch();
	virtual void decode_video();
	virtual void get_wave_pixmap(QPixmap *pm, int display_width);
	virtual void lock_frame();
	virtual void unlock_frame();
	virtual unsigned char **get_frame();
	virtual bool has_video();
	virtual void set_preview_framebuffer(int width, int height, unsigned char **ext_framebuf);
	virtual bool writeMetadata(const char *file);
	virtual QString artist();
	virtual QString title();
	virtual QString display_info();
	virtual bool can_do_multiple_reads();

	unsigned long max_frames;
	unsigned long frame_pos;
	unsigned long last_start;
	int sample_rate;
	QString filename;
	int wavelets;
	Wavelet **wave;
	virtual StreamSource *get_source(void);
	MainWnd *app_window;
	int max_wavelet_value;
	int peak_value;
protected:
	double max_agc;
	double s_agc;
	StreamSource *wave_source;
	pthread_t tid;
	virtual bool event(QEvent *ev);
	pthread_mutex_t control_lock;
	double pitch;
	unsigned long last_samples;
	bool loop_play;
	int playing;
	unsigned long play_samples;
	QString m_artist;
	QString m_title;

	void calcAgc(char *buf, int max);
	void bump_frame_position(long by);
	JackPlayer *player;
friend class JackPlayer;
friend class Pump;
signals:
	void frame_changed(unsigned);
	void play_done(void);
	void agc_change(double a);
};

class JackPlayer : public QObject, public Processor
{
Q_OBJECT

public:
	JackPlayer(Client *client, int numbase);
	~JackPlayer();
	void set_stream(StreamSource *stream);
	MainWnd *app_window;
	void setAttenuation(double a);
	double attenuation();
	void setGain(double g);
	double gain();
	void setAgc(double a);
	double agc();
	void addEffect(Effect *);
	void removeEffect(Effect *);
	void addPostFaderEffect(Effect *);
	void removePostFaderEffect(Effect *);
	Client *client;
	static long player_callback(void *thisp, float **data);
protected:
	QValueList<Effect *> effects;
	QValueList<Effect *> postfader_effects;
	unsigned long total_consumed;
	double pitch;
	double resample;
	double att;
	double ga;
	double s_agc;
	double tone_value;
	virtual int process(jack_nframes_t nsamples, void *arg);
	Port *left;
	Port *right;
	StreamSource *source;
	void set_pitch_percent(double);
	void set_tone_value(double);
	RingBuffer *buffer;
	float frame_buffer[4096];
	int sample_src;
	int sample_count;
	SRC_STATE *src_state;
friend class StreamSource;
friend class Pump;
};


#endif
