/*
 * discover.c --
 *
 *      Hardware detection utility.
 *
 * Copyright (C) 2000 Progeny Linux Systems, Inc.
 *
 * discover is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * discover is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Written by Ian Murdock <imurdock@progeny.com>.
 */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>

#include <errno.h>
#include <getopt.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <discover.h>

static void error(int status, int errnum, const char *format, ...);
static void output(FILE *stream, char *format, char *vendor, char *model,
                   char *module, char *device, char *server);
static void process_bus_list(char *list, int value);
static char *return_xserver(char *string);
static char *return_xdriver(char *string);
static void usage(int status);
static void *xmalloc(size_t size);

#define MALLOC          xmalloc

#define FLAG_ENABLE     1
#define FLAG_ENABLE_ALL 2
#define FLAG_DISABLE    3
#define FLAG_DISABLE_ALL 4
#define FLAG_VENDOR     5
#define FLAG_MODEL      6
#define FLAG_MODULE     7
#define FLAG_DEVICE     8
#define FLAG_XSERVER    9
#define FLAG_XDRIVER    10

static struct option options[] =
{
    { "enable",         1,      NULL,   FLAG_ENABLE },
    { "enable-all",     0,      NULL,   FLAG_ENABLE_ALL },
    { "disable",        1,      NULL,   FLAG_DISABLE },
    { "disable-all",    0,      NULL,   FLAG_DISABLE_ALL },
    { "format",         1,      NULL,   'f' },
    { "vendor",         0,      NULL,   FLAG_VENDOR },
    { "model",          0,      NULL,   FLAG_MODEL },
    { "module",         0,      NULL,   FLAG_MODULE },
    { "device",         0,      NULL,   FLAG_DEVICE },
    { "xserver",        0,      NULL,   FLAG_XSERVER },
    { "xdriver",        0,      NULL,   FLAG_XDRIVER },
    { NULL,             0,      NULL,   0 }
};

static char *program_name = NULL;

enum { TRUE = 1, FALSE = 0 };

static int scan_pci = TRUE;
static int scan_isa = FALSE;
static int scan_pcmcia = TRUE;
static int scan_usb = TRUE;
static int scan_ide = TRUE;
static int scan_scsi = TRUE;
static int scan_parallel = TRUE;
static int scan_serial = TRUE;

#define OPTION_ALL      "pci,isa,pcmcia,usb,ide,scsi,parallel,serial"


int
main (int argc, char **argv)
{
    struct cards_lst *lst = NULL;
    struct bus_lst *bus = NULL;
    struct bridge_info *bridge_item, *bridge = NULL;
    struct identf_info *ide_item, *ide = NULL;
    struct scsintf_info *scsi_item, *scsi = NULL;
    struct scsintf_info *ide_scsi = NULL;
    struct usbntf_info *usb_item, *usb = NULL;
    struct cdrom_info *cdrom_item, *cdrom = NULL;
    struct ethernet_info *eth_item, *eth = NULL;
    struct soundcard_info *sound_item, *sound = NULL;
    struct video_info *video_item, *video = NULL;
    char *format = NULL;
    int c, i, index;

    program_name = argv[0];

    while (1) {
        c = getopt_long(argc, argv, "f:", options, &index);
        if (c == -1) {
            break;
        }

        switch (c) {
          case FLAG_ENABLE:     /* --enable */
              process_bus_list(optarg, TRUE);
              break;
          case FLAG_ENABLE_ALL: /* --enable-all */
              process_bus_list(OPTION_ALL, TRUE);
              break;
          case FLAG_DISABLE:    /* --disable */
              process_bus_list(optarg, FALSE);
              break;
          case FLAG_DISABLE_ALL: /* --disable-all */
              process_bus_list(OPTION_ALL, FALSE);
              break;
          case 'f':             /* --format */
              if (format) {
                  usage(1);
              }
              format = optarg;
                break;
          case FLAG_VENDOR:     /* ---vendor */
              if (format) {
                  usage(1);
              }
              format = "%V\n";
              break;
          case FLAG_MODEL:      /* ---model */
              if (format) {
                  usage(1);
              }
              format = "%M\n";
              break;
          case FLAG_MODULE:     /* ---module */
              if (format) {
                  usage(1);
              }
              format = "%m\n";
              break;
          case FLAG_DEVICE:     /* ---device */
              if (format) {
                  usage(1);
              }
              format = "%d\n";
              break;
          case FLAG_XSERVER:    /* ---xserver */
              if (format) {
                  usage(1);
              }
              format = "%S\n";
              break;
          case FLAG_XDRIVER:    /* ---xdriver */
              if (format) {
                  usage(1);
              }
              format = "%D\n";
              break;
          default:
              usage(1);
        }
    }

    if (!format) {
        format = "%V %M\n";
    }

    sync();

    /* Initialize the detect library. */
    lst = init_lst(PATH_PCI_LST, PATH_PCMCIA_LST, PATH_USB_LST);

    /* There's no way to limit the bus scan done by bus_detect(),
       so we do it ourselves rather than calling bus_detect(). */
    bus = MALLOC(sizeof(*bus));
    memset(bus, 0, sizeof(*bus));
    if (scan_pci) {
        bus->pci = pci_detect(lst);
    }
    if (scan_isa) {
        bus->isa = isa_detect(lst);
    }
    if (scan_pcmcia) {
        bus->pcmcia = pcmcia_detect(lst);
    }
    if (scan_usb) {
        bus->usb = usb_detect(lst);
    }
    if (scan_ide) {
        bus->ide = ide_detect();
    }
    if (scan_scsi) {
        bus->scsi = scsi_detect();
    }
    if (scan_parallel) {
        bus->parallel = parallel_detect();
    }
    if (scan_serial) {
        bus->serial = serial_detect(lst);
    }

    for (i = optind; i < argc; i++) {
        if (strcmp(argv[i], "bridge") == 0) {
            /*
             * Bridges:
             */
            if (bridge == NULL) {
                bridge = bridge_detect(bus);
            }
            for (bridge_item = bridge; bridge_item;
                 bridge_item = bridge_item->next) {
                output(stdout, format, bridge_item->vendor, bridge_item->model,
                       bridge_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "cdrom") == 0) {
            /*
             * CD-ROM drives:
             */
            if (cdrom == NULL) {
                cdrom = cdrom_detect(bus);
            }
            for (cdrom_item = cdrom; cdrom_item;
                 cdrom_item = cdrom_item->next) {
                output(stdout, format, cdrom_item->vendor, cdrom_item->model,
                       cdrom_item->module, cdrom_item->device, NULL);
            }
        } else if (strcmp(argv[i], "ethernet") == 0) {
            /*
             * Ethernet cards:
             */
            if (eth == NULL) {
                eth = ethernet_detect(bus);
            }
            for (eth_item = eth; eth_item; eth_item = eth_item->next) {
                output(stdout, format, eth_item->vendor, eth_item->model,
                       eth_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "ide") == 0) {
            /*
             * IDE devices:
             */
            if (ide == NULL) {
                ide = ideinterface_detect(bus);
            }
            for (ide_item = ide; ide_item; ide_item = ide_item->next) {
                output(stdout, format, ide_item->vendor, ide_item->model,
                       ide_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "scsi") == 0) {
            /*
             * SCSI devices:
             */
            if (scsi == NULL) {
                scsi = scsiinterface_detect(bus);

                /*
                 * Check if we should add a driver reference for IDE CD-ROMs
                 * via ide-scsi:
                 */
                if (cdrom == NULL) {
                    cdrom = cdrom_detect(bus);
                }
                for (cdrom_item = cdrom; cdrom_item;
                     cdrom_item = cdrom_item->next) {
                    if (cdrom_item->bus == IDE) {
                        ide_scsi = (struct scsintf_info *)
                            xmalloc(sizeof(struct scsintf_info));
                        ide_scsi->vendor = "Linux";
                        ide_scsi->model = "IDE-SCSI emulation layer";
                        ide_scsi->module = "ide-scsi";
                        ide_scsi->next = NULL;
                        break;
                    }
                }
                if (ide_scsi != NULL) {
                    if (!scsi) {
                        scsi = ide_scsi;
                    } else {
                        for (scsi_item = scsi; scsi_item && scsi_item->next;
                             scsi_item = scsi_item->next) ;
                        scsi_item->next = ide_scsi;
                    }
                }
            }

            for (scsi_item = scsi; scsi_item; scsi_item = scsi_item->next) {
                output(stdout, format, scsi_item->vendor, scsi_item->model,
                       scsi_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "sound") == 0) {
            /*
             * Sound cards:
             */
            if (sound == NULL) {
                sound = soundcard_detect(bus);
            }
            for (sound_item = sound; sound_item;
                 sound_item = sound_item->next) {
                output(stdout, format, sound_item->vendor, sound_item->model,
                       sound_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "usb") == 0) {
            /*
             * USB devices:
             */
            if (usb == NULL) {
                usb = usbinterface_detect(bus);
            }
            for (usb_item = usb; usb_item; usb_item = usb_item->next) {
                output(stdout, format, usb_item->vendor, usb_item->model,
                       usb_item->module, NULL, NULL);
            }
        } else if (strcmp(argv[i], "video") == 0) {
            /*
             * Video cards:
             */
            if (video == NULL) {
                video = video_detect(bus);
            }
            for (video_item = video; video_item;
                 video_item = video_item->next) {
                output(stdout, format, video_item->vendor, video_item->model,
                       NULL, NULL, video_item->server);
            }
        } else {
            error(1, 0, "Unknown or unsupported device `%s'", argv[i]);
        }
    }

    return 0;
}


/* Print the error message indicated by FORMAT.  If ERRNUM is
   set, then print the corresponding error string.  If STATUS
   is set, then exit STATUS after printing. */
static void
error(int status, int errnum, const char *format, ...)
{
    va_list arguments;

    va_start(arguments, format);
    vfprintf(stderr, format, arguments);
    va_end(arguments);

    if (errnum) {
        fprintf(stderr, ": %s\n", strerror(errnum));
    } else {
        fprintf(stderr, "\n");
    }

    if (status) {
        exit(status);
    }
}


/* Generate output as defined by FORMAT. */
static void
output(FILE *stream, char *format, char *vendor, char *model, char *module,
       char *device, char *server)
{
    char *xserver, *xdriver;
    int i;

    for (i = 0; i < strlen(format); i++) {
        if (format[i] == '%') {
            switch (format[++i]) {
              case 'V':
                  if (vendor) {
                      fputs(vendor, stream);
                  }
                  break;
              case 'M':
                  if (model) {
                      fputs(model, stream);
                  }
                  break;
              case 'm':
                  if (module &&
                      (strcmp(module, "ignore") != 0) &&
                      (strcmp(module, "unknown") != 0))
                  {
                      fputs(module, stream);
                  }
                  break;
              case 'd':
                  if (device) {
                      fputs(device, stream);
                  }
                  break;
              case 'S':
                  xserver = return_xserver(server);
                  if (xserver) {
                      fputs(xserver, stream);
                  }
                  break;
              case 'D':
                  xdriver = return_xdriver(server);
                  if (xdriver) {
                      fputs(xdriver, stream);
                  }
                  break;
              default:
                  error(1, 0, "`%c': invalid conversion specifier", format[i]);
            }
        } else if (format[i] == '\\') {
            /* XXX: Incorporate remaining escape sequences. */
            switch(format[++i]) {
              case 'n':
                  fputc('\n', stdout);
                  break;
              case 't':
                  fputc('\t', stdout);
                  break;
              default:
                  fputc('\\', stdout);
                  fputc(format[i], stdout);
                  break;
            }
        } else {
            fputc(format[i], stdout);
        }
    }
}


static char *
return_xserver(char *string)
{
    char *p, *s, *t;
    if (!string) {
        return NULL;
    }
    p = strdup(string);
    s = p;
    t = strchr(s, ':');
    if (!t) {
        error(0, 0, "parse error reading X server string `%s'", string);
        return "unknown";
    }
    s = ++t;
    t = strchr(s, '(');
    if (t) {
        *t = '\0';
    }
    return s;
}


static char *
return_xdriver(char *string)
{
    char *p, *s, *t;
    if (!string) {
        return NULL;
    }
    p = strdup(string);
    s = p;
    t = strchr(s, ':');
    if (!t) {
        error(0, 0, "parse error reading X server string `%s'", string);
        return "unknown";
    }
    s = ++t;
    t = strchr(s, '(');
    if (!t) {
        return NULL;
    }
    s = ++t;
    t = strchr(s, ')');
    if (!t) {
        error(0, 0, "parse error reading X server string `%s'", string);
        return "unknown";
    }
    *t = '\0';
    return s;
}


/* Process a bus list.  Used by the --enable and --disable
   switches.  The format of the list is "BUS,BUS,BUS,..."
   If BUS is present in the list, set scan_BUS to VALUE. */
static void
process_bus_list(char *list, int value)
{
    char *s, *t;

    if (!list) {
        return;
    }

    /* strtok munges its first argument, so pass it a copy */
    s = strdup(list);

    t = strtok(s, ",");
    while (t) {
        if (strcmp(t, "pci") == 0) {
            scan_pci = value;
        } else if (strcmp(t, "isa") == 0) {
            scan_isa = value;
        } else if (strcmp(t, "pcmcia") == 0) {
            scan_pcmcia = value;
        } else if (strcmp(t, "usb") == 0) {
            scan_usb = value;
        } else if (strcmp(t, "ide") == 0) {
            scan_ide = value;
        } else if (strcmp(t, "scsi") == 0) {
            scan_scsi = value;
        } else if (strcmp(t, "parallel") == 0) {
            scan_parallel = value;
        } else if (strcmp(t, "serial") == 0) {
            scan_serial = value;
        } else {
            error(1, 0, "Unknown or unsupported bus `%s'", t);
        }

        t = strtok(NULL, ",");
    }

    free(s);
}


/* Print usage message and exit. */
static void
usage(int status)
{
    printf("Usage: %s [OPTIONS] DEVICE [DEVICE...]\n\
\n\
      --enable=BUS[,BUS...]     Enable probing of BUS.\n\
      --enable-all              Enable probing of all buses.\n\
      --disable=BUS[,BUS...]    Disable probing of BUS.\n\
      --disable-all             Disable probing of all buses.\n\
  -f, --format=STRING           Format output according to STRING. STRING
                                is a printf-like string, where %%V expands
                                to the vendor, %%M expands to the model,
                                %%m expands to the module to be loaded, %%d
                                expands to the path of the device file,
                                %%S expands to the name of the X server
                                to be used, and %%D expands to the name
                                of X server driver to be used.\n\
      --vendor                  Print vendor.\n\
      --model                   Print model.\n\
      --module                  Print module to be loaded.\n\
      --device                  Print path of device file.\n\
      --xserver                 Print X server.\n\
      --xdriver                 Print X server driver.\n", program_name);
    exit(0);
}


/* Allocate memory and check return value. */
static void *
xmalloc(size_t size)
{
    void *p;
    p = malloc(size);
    if (!p) {
        error(1, 0, "Out of memory");
    }
    return p;
}

/* vim:set ai tw=0 et sw=4 softtabstop=4: */
