#include <glib.h>
#include <gnome.h>
#include "appsupport.h"
#include "appglobals.h"
#include "groups.h"
#include "users.h"

#ifdef FREEBSD
#include <des.h>
#else
#include <crypt.h>
#endif /* FREEBSD */

#include <sys/time.h>
#include <unistd.h>
#include "md5.h"
#include "defaults.h"

//standard attributes to check on creation/modification
gchar *attributestocheck[] = { "uid", "uidnumber", "gidnumber", "givenname",
  "initials", "sn", "cn", "homedirectory",
  "loginshell", "mail", "maillocaladdress",
  "mailroutingaddress", "mailhost",
  "gecos", "userPassword", "shadowLastChange",
  "shadowmin", "shadowmax", "shadowwarning", "shadowinactive",
  "shadowexpire", "title", "physicaldeliveryofficename",
  "ou", "l",
  "telephonenumber", "facsimiletelephonenumber", "o", "homephone",
  "mobile",
#ifdef USE_TRUSTMODE
  "trustModel",
#endif
  NULL
};


// This is from 'pam_ldap.c' by Luke Howard, <lukeh@padl.com>
// Used without explicit permission.
/* i64c - convert an integer to a radix 64 character */
static int
i64c (int i)
{
  const char *base64 =
    "./01234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  if (i < 0)
    i = 0;
  else if (i > 63)
    i = 63;

  return base64[i];
}

// This is from 'pam_ldap.c' by Luke Howard, <lukeh@padl.com>
// Used with permission granted in the 'GNU Library General Public License'.
static char *
_get_salt (char salt[3])
{
  int i;
  int j;

  srand (time (NULL));

  for (j = 0; j < 2; j++)
    {
      i = rand () % 3;
      switch (i)
	{
	case 0:
	  i = (rand () % (57 - 46)) + 46;
	  break;
	case 1:
	  i = (rand () % (90 - 65)) + 65;
	  break;
	case 2:
	  i = (rand () % (122 - 97)) + 97;
	  break;
	}
      salt[j] = i;
    }
  salt[2] = '\0';
  return (salt);
}

static char *
_get_md5_salt (char saltbuf[16])
{
  md5_state_t state;
  md5_byte_t digest[16];
  struct timeval tv;
  int i;

  md5_init (&state);
  gettimeofday (&tv, NULL);
  md5_append (&state, (unsigned char *) &tv, sizeof (tv));
  i = getpid ();
  md5_append (&state, (unsigned char *) &i, sizeof (i));
  i = clock ();
  md5_append (&state, (unsigned char *) &i, sizeof (i));
  md5_append (&state, (unsigned char *) saltbuf, sizeof (saltbuf));
  md5_finish (&state, digest);

  strcpy (saltbuf, "$1$");
  for (i = 0; i < 8; i++)
    saltbuf[i + 3] = i64c (digest[i] & 0x3f);

  saltbuf[i + 3] = '\0';

  return saltbuf;
}

diradmin_user *
diradmin_user_new (gchar * dn,
		   gchar * uid,
		   gchar * uidnumber,
		   gchar * gidnumber,
		   gchar * givenname,
		   gchar * sn,
		   gchar * cn,
		   gchar * homedirectory, gchar * loginshell, GList * allowed)
{

  diradmin_user *conn = g_new (diradmin_user, 1);
  conn->diradmin_user_data = NULL;
  conn->diradmin_user_allowedservers = NULL;
  conn->diradmin_user_objectclasses = NULL;

  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "dn", dn);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "uid", uid);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "uidnumber",
			      uidnumber);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "gidnumber",
			      gidnumber);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "givenname",
			      givenname);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "sn", sn);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "cn", cn);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "homedirectory",
			      homedirectory);
  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "loginshell",
			      loginshell);

  diradmin_user_set_allowedservers (conn, allowed);

  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses, g_strdup ("top"));
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses, g_strdup ("account"));
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup ("posixAccount"));
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup ("shadowAccount"));
#ifdef USE_TRUSTMODE
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup ("trustAccount"));
#endif
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses, g_strdup ("person"));
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup ("organizationalPerson"));
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup ("inetOrgPerson"));

  return conn;
}

diradmin_user *
diradmin_user_duplicate (diradmin_user * tobeduped)
{

  diradmin_user *conn = g_new (diradmin_user, 1);
  conn->diradmin_user_data = NULL;
  conn->diradmin_user_allowedservers = NULL;
  conn->diradmin_user_objectclasses = NULL;

  conn->diradmin_user_data =
    pairs_list_duplicate (tobeduped->diradmin_user_data);

  diradmin_user_set_allowedservers (conn,
				    tobeduped->diradmin_user_allowedservers);
  diradmin_user_set_objectclasses (conn,
				   tobeduped->diradmin_user_objectclasses);

  return conn;

}


//these three functions receive pointers to pointers as arguments for the datalist
void
diradmin_user_destroy (diradmin_user * todestroy)
{
  if (todestroy)
    {
      pairs_list_destroy (todestroy->diradmin_user_data);
      diradmin_user_set_allowedservers (todestroy, NULL);
      diradmin_user_set_objectclasses (todestroy, NULL);
      g_free (todestroy);
    }

}

gchar *
diradmin_user_get_attribute (diradmin_user * conn, gchar * attr)
{
//  g_print("getting attribute "); g_print(attr);
//  g_print(" to "); g_print(to); g_print("\n");
  g_assert (conn);
  g_assert (attr);
  return pairs_list_get_attribute (conn->diradmin_user_data, attr);
}

void
diradmin_user_set_attribute (diradmin_user * conn, gchar * attr, gchar * to)
{

  g_assert (conn);
  g_assert (attr);
//  g_print("setting attribute "); g_print(attr);
//  g_print(" to "); g_print(to); g_print("\n");

  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, attr, to);

}

void
diradmin_user_remove_attribute (diradmin_user * conn, gchar * attr)
{
  g_assert (conn);
  g_assert (attr);
  conn->diradmin_user_data =
    pairs_list_remove_attribute (conn->diradmin_user_data, attr);
}

void
diradmin_user_set_password (diradmin_user * conn, gchar * to)
{
  char seed[2], buf[96], saltbuf[16], md5crypt[64];
  g_print ("set password called with password: %s\n", to);
  if (preferences.passwordcrypttype == CRYPT)
    {
      snprintf (buf, sizeof buf, "{crypt}%s", crypt (to, _get_salt (seed)));
      diradmin_user_set_attribute (conn, "userPassword", buf);
    }
  else if (preferences.passwordcrypttype == MD5)
    {
      strncpy (md5crypt, to, sizeof (md5crypt) - 1);
      _get_md5_salt (saltbuf);
      g_print (crypt (md5crypt, saltbuf));
      snprintf (buf, sizeof buf, "{crypt}%s", crypt (md5crypt, saltbuf));
      diradmin_user_set_attribute (conn, "userPassword", buf);
    }
  else
    {
      diradmin_user_set_attribute (conn, "userPassword", to);
    }
  //this line of code is taken from pam_ldap.c
  snprintf (buf, sizeof buf, "%ld", time (NULL) / (60 * 60 * 24));
  diradmin_user_set_attribute (conn, "shadowLastChange", buf);
  g_print ("set password set attribute: %s\n",
	   diradmin_user_get_attribute (conn, "userPassword"));

}

diradmin_user *
diradmin_user_new_from_ldap (connection_profile * usethisone, char *userdn)
{
  //conn should already be connected, or else undefined behaviour!!!

  diradmin_user *user = NULL;

  int ldap_errors;

  LDAPMessage *searchresults = NULL;
  LDAPMessage *entry = NULL;

  char **value_collection = NULL;

  gchar *filter;

  int i = 0;

  gchar *attribute;
  gchar *dn;
  BerElement *attributehandler;

  user = g_new (diradmin_user, 1);
  user->diradmin_user_data = NULL;
  user->diradmin_user_allowedservers = NULL;
  user->diradmin_user_objectclasses = NULL;


  //check 4 connection
  g_assert (connection_profile_get_ldap_handler (usethisone));

  filter =
    g_strconcat ("(&(objectClass=posixAccount)(uid=", userdn, "))", NULL);

  //look data up
  ldap_errors =
    ldap_search_s (connection_profile_get_ldap_handler (usethisone),
		   connection_profile_get_treeroot (usethisone),
		   LDAP_SCOPE_SUBTREE, filter, NULL, 0, &searchresults);
  if (ldap_errors)
    {

      //any error?
      g_print ("LDAP error while creating a diradmin_user structure for ");
      g_print (userdn);
      g_print (": ");
      g_print (ldap_err2string (ldap_errors));
      g_print ("\n");
    }
  else
    {
      //get only first entry
      entry =
	ldap_first_entry (connection_profile_get_ldap_handler (usethisone),
			  searchresults);

      // loop thru attribute values

      dn =
	ldap_get_dn (connection_profile_get_ldap_handler (usethisone), entry);
      diradmin_user_set_attribute (user, "dn", dn);
      g_free (dn);


      attribute =
	ldap_first_attribute (connection_profile_get_ldap_handler
			      (usethisone), entry, &attributehandler);

      g_assert (attribute);

      while (attribute)
	{
//       g_print(attribute);g_print(" <- AAAAAAAAA \n");
	  value_collection =
	    ldap_get_values (connection_profile_get_ldap_handler (usethisone),
			     entry, attribute);
#ifdef USE_TRUSTMODE
	  if (g_strcasecmp (attribute, "accessTo") == 0)
#else
	  if (g_strcasecmp (attribute, "host") == 0)
#endif
	    {
	      i = 0;
	      while (value_collection[i])
		{
		  diradmin_user_add_allowedserver (user, value_collection[i]);
		  i++;
		}
	    }
	  else if (g_strcasecmp (attribute, "objectclass") == 0)
	    {
	      i = 0;
	      while (value_collection[i])
		{
		  diradmin_user_add_objectclass (user, value_collection[i]);
		  i++;
		}
	    }
	  else
	    {
	      g_assert (value_collection[0]);
	      diradmin_user_set_attribute (user, attribute,
					   value_collection[0]);
	    }
	  ldap_value_free (value_collection);
	  attribute =
	    ldap_next_attribute (connection_profile_get_ldap_handler
				 (usethisone), entry, attributehandler);
	}

      ldap_msgfree (searchresults);
    }
  g_free (filter);

  return (user);
}




GList *
diradmin_user_get_allowedservers (diradmin_user * conn)
{
  //return a reference to the allowedservers list.  shouldnt be manipulated outside.
  return g_list_first (conn->diradmin_user_allowedservers);
}

void
diradmin_user_add_allowedserver (diradmin_user * conn, gchar * allowedserver)
{
  diradmin_user_remove_allowedserver (conn, allowedserver);
  conn->diradmin_user_allowedservers =
    g_list_append (conn->diradmin_user_allowedservers,
		   g_strdup (allowedserver));
}

void
diradmin_user_remove_allowedserver (diradmin_user * conn, gchar
				    * allowedserver)
{
  GList *iterator = NULL;
  iterator = g_list_find_custom (conn->diradmin_user_allowedservers,
				 allowedserver, (GCompareFunc) g_strcasecmp);
  if (iterator)
    {
      conn->diradmin_user_allowedservers =
	g_list_remove_link (conn->diradmin_user_allowedservers, iterator);
      g_free (iterator->data);
      g_list_free_1 (iterator);
    }
}

gboolean
diradmin_user_has_allowedserver (diradmin_user * conn, gchar * allowedserver)
{
  gchar *data;
  GList *iterator = NULL;
  iterator = g_list_first (conn->diradmin_user_allowedservers);
  while (iterator)
    {
      data = iterator->data;
      if (g_strcasecmp (allowedserver, iterator->data) == 0)
	{
	  return (TRUE);
	}
      iterator = g_list_next (iterator);
    }
  return (FALSE);
}

void
diradmin_user_set_allowedservers (diradmin_user * conn, GList
				  * allowedserverlist)
{
  //removes all allowedservers in the list.  that means frees its allocated storage.

  GList *newallowedservers = NULL;
  GList *allowedservers = NULL;


  //free the old allowedservers list
  allowedservers = g_list_first (diradmin_user_get_allowedservers (conn));
  while (allowedservers)
    {
      g_free (allowedservers->data);
      allowedservers = g_list_next (allowedservers);
    }
  g_list_free (g_list_first (diradmin_user_get_allowedservers (conn)));

  //make a copy of the passed allowedservers list into newallowedservers
  allowedservers = g_list_first (allowedserverlist);
  while (allowedservers)
    {
      newallowedservers =
	g_list_append (newallowedservers, g_strdup (allowedservers->data));
      allowedservers = g_list_next (allowedservers);
    }

  conn->diradmin_user_allowedservers = newallowedservers;
}

//objectclasses
GList *
diradmin_user_get_objectclasses (diradmin_user * conn)
{
  //return a reference to the allowedservers list.  shouldnt be manipulated outside.
  return g_list_first (conn->diradmin_user_objectclasses);
}

void
diradmin_user_add_objectclass (diradmin_user * conn, gchar * allowedserver)
{
  diradmin_user_remove_objectclass (conn, allowedserver);
  conn->diradmin_user_objectclasses =
    g_list_append (conn->diradmin_user_objectclasses,
		   g_strdup (allowedserver));
}

void
diradmin_user_remove_objectclass (diradmin_user * conn, gchar * allowedserver)
{
  GList *iterator = NULL;
  iterator = g_list_find_custom (conn->diradmin_user_objectclasses,
				 allowedserver, (GCompareFunc) g_strcasecmp);
  if (iterator)
    {
      conn->diradmin_user_objectclasses =
	g_list_remove_link (conn->diradmin_user_objectclasses, iterator);
      g_free (iterator->data);
      g_list_free_1 (iterator);
    }
}

gboolean
diradmin_user_has_objectclass (diradmin_user * conn, gchar * allowedserver)
{
  gchar *data;
  GList *iterator = NULL;
  iterator = g_list_first (conn->diradmin_user_objectclasses);
  while (iterator)
    {
      data = iterator->data;
      if (g_strcasecmp (allowedserver, iterator->data) == 0)
	{
	  return (TRUE);
	}
      iterator = g_list_next (iterator);
    }
  return (FALSE);
}

void
diradmin_user_set_objectclasses (diradmin_user * conn, GList
				 * allowedserverlist)
{

  //removes all allowedservers in the list.  that means frees its allocated storage.

  GList *newallowedservers = NULL;
  GList *allowedservers = NULL;

  //free the old allowedservers list
  allowedservers = g_list_first (diradmin_user_get_objectclasses (conn));
  while (allowedservers)
    {
      g_free (allowedservers->data);
      allowedservers = g_list_next (allowedservers);
    }
  g_list_free (g_list_first (conn->diradmin_user_objectclasses));

  //make a copy of the passed allowedservers list into newallowedservers
  allowedservers = g_list_first (allowedserverlist);
  while (allowedservers)
    {
      newallowedservers =
	g_list_append (newallowedservers, g_strdup (allowedservers->data));
      allowedservers = g_list_next (allowedservers);
    }

  conn->diradmin_user_objectclasses = newallowedservers;
}



LDAPMod **
diradmin_user_generate_ldapdiff (diradmin_user * oldone,
				 diradmin_user * newone)
{

  LDAPMod **diffs;
  LDAPMod *curmod = NULL;
  gchar **values;
  gint curmod_counter = 0;
  GList *listone = NULL;
  gint currattribute_counter = 0;
  gchar *oldattr = NULL;
  gchar *newattr = NULL;
  g_assert (oldone);
  g_assert (newone);

  diffs = g_new0 (LDAPMod *, sizeof (attributestocheck)
		  + g_list_length (diradmin_user_get_objectclasses (oldone))
		  +
		  +g_list_length (diradmin_user_get_objectclasses (newone))
		  +
		  +g_list_length (diradmin_user_get_allowedservers (oldone))
		  +
		  g_list_length (diradmin_user_get_allowedservers (newone)));

  while (attributestocheck[currattribute_counter])
    {
      oldattr =
	diradmin_user_get_attribute (oldone,
				     attributestocheck
				     [currattribute_counter]);
      newattr =
	diradmin_user_get_attribute (newone,
				     attributestocheck
				     [currattribute_counter]);
      if (g_strcasecmp
	  (attributestocheck[currattribute_counter], "userPassword") != 0
	  &&
	  g_strcasecmp
	  (attributestocheck[currattribute_counter], "shadowLastChange") != 0
	  && oldattr != NULL && newattr == NULL)
	{
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_DELETE;
	  curmod->mod_type =
	    g_strdup (attributestocheck[currattribute_counter]);
	  curmod->mod_values = NULL;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      else if (oldattr == NULL && newattr != NULL)
	{
	  g_print ("\replacing old attribute %s with %s\n", oldattr, newattr);
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_REPLACE;
	  curmod->mod_type =
	    g_strdup (attributestocheck[currattribute_counter]);
	  values[0] = g_strdup (newattr);
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      else if (oldattr && newattr && g_strcasecmp (oldattr, newattr) != 0)
	{
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_REPLACE;
	  curmod->mod_type =
	    g_strdup (attributestocheck[currattribute_counter]);
	  values[0] = g_strdup (newattr);
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      currattribute_counter++;
    }

  g_assert (diffs);

  listone = diradmin_user_get_allowedservers (oldone);
  while (listone)
    {
      if (diradmin_user_has_allowedserver (newone, listone->data) == FALSE)
	{
	  //generar mod de eliminacion
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_DELETE;
#ifdef USE_TRUSTMODE
	  curmod->mod_type = g_strdup ("accessTo");
#else
	  curmod->mod_type = g_strdup ("host");
#endif
	  values[0] = g_strdup (listone->data);
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      listone = g_list_next (listone);
    }
  listone = diradmin_user_get_allowedservers (newone);
  while (listone)
    {
      if (diradmin_user_has_allowedserver (oldone, listone->data) == FALSE)
	{
	  //generar mod de eliminacion
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_ADD;
#ifdef USE_TRUSTMODE
	  curmod->mod_type = g_strdup ("accessTo");
#else
	  curmod->mod_type = g_strdup ("host");
#endif
	  values[0] = g_strdup (listone->data);
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      listone = g_list_next (listone);
    }

  // if the old user has email policy and the new doesn't, then kick it out
  if (diradmin_user_has_objectclass (oldone, "inetlocalmailrecipient"))
    {
      if (diradmin_user_has_objectclass (newone, "inetlocalmailrecipient") ==
	  FALSE)
	{
	  g_print ("\nRemoving object class inetlocalmailrecipient\n");
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_DELETE;
	  curmod->mod_type = g_strdup ("objectclass");
	  values[0] = g_strdup ("inetlocalmailrecipient");
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
    }

  listone = diradmin_user_get_objectclasses (newone);
  while (listone)
    {
      if (diradmin_user_has_objectclass (oldone, listone->data) == FALSE)
	{
	  //generar mod de adicion
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_ADD;
	  curmod->mod_type = g_strdup ("objectclass");
	  values[0] = g_strdup (listone->data);
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      listone = g_list_next (listone);
    }
  if (curmod_counter == 0)
    {
      g_free (diffs);
      diffs = NULL;
    }
  else
    diffs[curmod_counter] = NULL;

  return diffs;

}

diradmin_user *
create_user_struct_from_dialogbox (connection_profile * conn,
				   GtkWidget * dialogbox, gchar * dn)
{

  diradmin_user *newuser;
  gint iterator;
  gchar *uid = NULL;
  gchar *uidnumber = NULL;
  gchar *gidnumber = NULL;
  gchar *givenname = NULL;
  gchar *sn = NULL;
  gchar *cn = NULL;
  gchar *homedirectory = NULL;
  gchar *loginshell = NULL;
  gchar *userpassword = NULL;
  gchar *gecos = NULL;
  gchar *mail = NULL;
  gchar *mailhost = NULL;
  gchar *mailroutingaddress = NULL;
  gchar *shadowmin = NULL;
  gchar *shadowmax = NULL;
  gchar *shadowwarning = NULL;
  gchar *shadowinactive = NULL;
  char buf[32];
  time_t shadowexpiretime;

  GtkCList *allowedservers = NULL;
  gchar *allowedserver = NULL;

  g_print ("called: create_user_struct_from_dialogbox\n");

  allowedservers = (GtkCList *) lookup_widget (dialogbox, "allowedservers");

  uid = gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "uid")));
  uidnumber =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "uidnumber")));
  gidnumber =
    cn_to_gidnumber (conn,
		     gtk_entry_get_text (GTK_ENTRY
					 (lookup_widget
					  (dialogbox, "gidnumber"))));
  sn = gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "sn")));
  cn = gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "cn")));
  givenname =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "givenname")));
  homedirectory =
    gtk_entry_get_text (GTK_ENTRY
			(lookup_widget (dialogbox, "homedirectory")));
  loginshell =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "loginshell")));
  mail = gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "mail")));
  mailhost =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "mailhost")));
  mailroutingaddress =
    gtk_entry_get_text (GTK_ENTRY
			(lookup_widget (dialogbox, "mailroutingaddress")));

  userpassword =
    gtk_entry_get_text (GTK_ENTRY
			(lookup_widget (dialogbox, "userpassword")));
  gecos = gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "cn")));

  newuser =
    diradmin_user_new (dn, uid, uidnumber, gidnumber, givenname, sn, cn,
		       homedirectory, loginshell, NULL);
  g_free (gidnumber);		//mallocced by the function

  diradmin_user_set_attribute (newuser, "gecos", gecos);
  diradmin_user_set_attribute (newuser, "mail", mail);

  //shadow password setting attributes
  shadowmin =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "shadowmin")));
  shadowmax =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget (dialogbox, "shadowmax")));
  shadowwarning =
    gtk_entry_get_text (GTK_ENTRY
			(lookup_widget (dialogbox, "shadowwarning")));
  shadowinactive =
    gtk_entry_get_text (GTK_ENTRY
			(lookup_widget (dialogbox, "shadowinactive")));
  diradmin_user_set_attribute (newuser, "shadowmin", shadowmin);
  diradmin_user_set_attribute (newuser, "shadowmax", shadowmax);
  diradmin_user_set_attribute (newuser, "shadowwarning", shadowwarning);
  diradmin_user_set_attribute (newuser, "shadowinactive", shadowinactive);

//organizationalperson
  diradmin_user_set_attribute (newuser, "initials",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox,
						     "initials"))));
  diradmin_user_set_attribute (newuser, "title",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox, "title"))));
  diradmin_user_set_attribute (newuser, "physicaldeliveryofficename",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox,
						     "physicaldeliveryofficename"))));
  diradmin_user_set_attribute (newuser, "ou",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox, "ou"))));
  diradmin_user_set_attribute (newuser, "l",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox, "l"))));
  diradmin_user_set_attribute (newuser, "telephonenumber",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox,
						     "telephonenumber"))));
  diradmin_user_set_attribute (newuser, "facsimiletelephonenumber",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox,
						     "facsimiletelephonenumber"))));
  diradmin_user_set_attribute (newuser, "o",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox, "o"))));
  diradmin_user_set_attribute (newuser, "homephone",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox,
						     "homephone"))));
  diradmin_user_set_attribute (newuser, "mobile",
			       gtk_entry_get_text (GTK_ENTRY
						   (lookup_widget
						    (dialogbox, "mobile"))));

  if (GTK_TOGGLE_BUTTON
      (lookup_widget (dialogbox, "shadowexpirecheck"))->active)
    {
      shadowexpiretime =
	gnome_date_edit_get_date ((GnomeDateEdit *)
				  lookup_widget (dialogbox, "shadowexpire"));
      snprintf (buf, sizeof buf, "%ld", shadowexpiretime / (60 * 60 * 24));
      diradmin_user_set_attribute (newuser, "shadowexpire", buf);

    }


  if (strlen (userpassword) > 0)
    diradmin_user_set_password (newuser, userpassword);

  if (GTK_TOGGLE_BUTTON
      (lookup_widget (dialogbox, "logontoallservers"))->active)
    {
#ifdef USE_TRUSTMODE
        g_print
	("create_user_struct_from_dialogbox: trustModel: fullaccess\n\n");
	diradmin_user_add_objectclass (newuser, "trustAccount");
	diradmin_user_set_attribute (newuser, "trustModel", "fullaccess");
#endif
    }
  else
    {
#ifdef USE_TRUSTMODE
      g_print ("create_user_struct_from_dialogbox: trustModel: byserver\n\n");
      diradmin_user_add_objectclass (newuser, "trustAccount");
      diradmin_user_set_attribute (newuser, "trustModel", "byserver");
#endif
      for (iterator = 0; iterator < allowedservers->rows; iterator++)
	{
	  gtk_clist_get_text (allowedservers, iterator, 0, &allowedserver);
	  diradmin_user_add_allowedserver (newuser, allowedserver);
	}
    }
  if (GTK_TOGGLE_BUTTON
      (lookup_widget (dialogbox, "enablemailpolicycontrol"))->active)
    {
      diradmin_user_add_objectclass (newuser, "inetlocalmailrecipient");
      diradmin_user_set_attribute (newuser, "maillocaladdress", mail);
      if (GTK_TOGGLE_BUTTON
	  (lookup_widget (dialogbox, "mailroutingaddressset"))->active)
	diradmin_user_set_attribute (newuser, "mailroutingaddress",
				     mailroutingaddress);
      if (GTK_TOGGLE_BUTTON
	  (lookup_widget (dialogbox, "mailhostset"))->active)
	diradmin_user_set_attribute (newuser, "mailhost", mailhost);
    }

  return (newuser);
}



LDAPMod **
diradmin_user_create_ldapdiff (diradmin_user * newone)
{

  LDAPMod **diffs;
  LDAPMod *curmod = NULL;
  gchar **values;
  gint curmod_counter = 0;
  GList *listone = NULL;
  gint currattribute_counter = 0;
  g_assert (newone);

  diffs = g_new0 (LDAPMod *, sizeof (attributestocheck)
		  +
		  g_list_length (diradmin_user_get_allowedservers (newone))
		  + g_list_length (diradmin_user_get_objectclasses (newone)));

  values = g_new0 (gchar *, 9);
  curmod = g_new0 (LDAPMod, 1);
  curmod->mod_op = LDAP_MOD_ADD;
  curmod->mod_type = g_strdup ("objectClass");
  values[0] = g_strdup ("top");
  values[1] = g_strdup ("account");
  values[2] = g_strdup ("posixAccount");
  values[3] = g_strdup ("shadowAccount");
  values[4] = g_strdup ("person");
  values[5] = g_strdup ("organizationalPerson");
  values[6] = g_strdup ("inetOrgPerson");
#ifdef USE_TRUSTMODE
  values[7] = g_strdup ("trustAccount");
  values[8] = NULL;
#else
  values[7] = NULL;
#endif
  curmod->mod_values = values;
  diffs[curmod_counter] = curmod;
  curmod_counter++;

  if (diradmin_user_has_objectclass (newone, "inetlocalmailrecipient"))
    {
      values = g_new0 (gchar *, 2);
      curmod = g_new0 (LDAPMod, 1);
      curmod->mod_op = LDAP_MOD_ADD;
      curmod->mod_type = g_strdup ("objectClass");
      values[0] = g_strdup ("inetlocalmailrecipient");
      values[1] = NULL;
      curmod->mod_values = values;
      diffs[curmod_counter] = curmod;
      curmod_counter++;
    }


  while (attributestocheck[currattribute_counter])
    {
      if (diradmin_user_get_attribute
	  (newone, attributestocheck[currattribute_counter]) != NULL)
	{
	  values = g_new0 (gchar *, 2);
	  curmod = g_new0 (LDAPMod, 1);
	  curmod->mod_op = LDAP_MOD_ADD;
	  curmod->mod_type =
	    g_strdup (attributestocheck[currattribute_counter]);
	  values[0] =
	    g_strdup (diradmin_user_get_attribute
		      (newone, attributestocheck[currattribute_counter]));
	  values[1] = NULL;
	  curmod->mod_values = values;
	  diffs[curmod_counter] = curmod;
	  curmod_counter++;
	}
      currattribute_counter++;
    }

  g_assert (diffs);

  listone = diradmin_user_get_allowedservers (newone);
  while (listone)
    {
      values = g_new0 (gchar *, 2);
      curmod = g_new0 (LDAPMod, 1);
      curmod->mod_op = LDAP_MOD_ADD;
#ifdef USE_TRUSTMODE
      curmod->mod_type = g_strdup ("accessTo");
#else
      curmod->mod_type = g_strdup ("host");
#endif
      values[0] = g_strdup (listone->data);
      values[1] = NULL;
      curmod->mod_values = values;
      diffs[curmod_counter] = curmod;
      curmod_counter++;
      listone = g_list_next (listone);
    }

  if (curmod_counter == 0)
    {
      g_free (diffs);
      diffs = NULL;
    }
  else
    diffs[curmod_counter] = NULL;

  return diffs;

}
