#! /usr/bin/perl -w

# debchange: update the debian changelog using your favorite visual editor
# Options:
# -i           generates a new changelog section increasing the Debian
#              release number
# -a           Adds an entry to the current changelog section
# -v version   generates a new changelog section with given version number
# -d, --fromdirname
#              Like -v, but takes version from directory name
# -p           preserve directory name
# Without any options, the package will look for an .upload file in the
# parent directory to determine whether or not the version number should
# be incremented or not.
#
# debchange -h prints a usage message.
#
# When creating a new changelog section, if either of the environment
# variables DEBEMAIL or EMAIL is set, debchange will use this as the
# uploader's email address (with the former taking precedence), and if
# DEBFULLNAME is set, it will use this as the uploader's full name.
# Otherwise, it will take the standard values for the current user or,
# failing that, just copy the values from the previous changelog entry.
#
# Originally by Christoph Lameter <clameter@debian.org>
# Modified extensively by Julian Gilbey <jdg@debian.org>
#
# Copyright 1999,2000 by Julian Gilbey 
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.003;  # We're using function prototypes
use Getopt::Long;
use File::Copy;
use Cwd;

# Predeclare functions
sub fatal($);

my $progname;
($progname=$0) =~ s|.*/||;

sub usage () {
    print <<"EOF";
Usage: $progname [options] [changelog entry]
Options:
  -i, --increment
         Increase the Debian release number, adding a new changelog entry
  -a, --append
         Append a new entry to the current changelog
  -v <version>, --newversion=<version>
         Add a new changelog entry with version number specified
  -d, --fromdirname
         Add a new changelog entry with version taken from the directory name
  -p, --preserve
         Preserve the directory name
  --no-preserve
         Do not preserve the directory name (default)
  -h, --help
         Display this help message and exit
  --version
         Display version information
  At most one of -a, -i and -v (or their long equivalents) may be used.
  With no options, one of -i or -a is chosen by looking for a .upload
  file in the parent directory and checking its contents.
EOF
}

sub version () {
    print <<"EOF";
This is $progname, from the Debian devscripts package, version ###VERSION###
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on code by Christoph Lameter.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later.
EOF
}

# Start by reading configuration files and then command line
# The next stuff is boilerplate

my @config_files = ('/etc/devscripts.conf', '~/.devscripts');
my %config_vars = (
		   'DEBCHANGE_PRESERVE' => 'no',
		   );

my $shell_cmd;
# Set defaults
foreach my $var (keys %config_vars) {
    $shell_cmd .= qq[$var="$config_vars{$var}";\n];
}
$shell_cmd .= 'for file in ' . join(" ",@config_files) . "; do\n";
$shell_cmd .= '[ -f $file ] && . $file; done;' . "\n";
# Read back values
foreach my $var (keys %config_vars) { $shell_cmd .= "echo \$$var;\n" }
my $shell_out = `/bin/sh -c '$shell_cmd'`;
@config_vars{keys %config_vars} = split /\n/, $shell_out, -1;

$opt_p = $config_vars{'DEBCHANGE_PRESERVE'} eq 'yes' ? 1 : 0;

# We use bundling so that the short option behaviour is the same as
# with older debchange versions.
Getopt::Long::Configure('bundling');
GetOptions("h", "help" => \$opt_h,
	   "version" => \$opt_version,
	   "i", "increment" => \$opt_i,
	   "a", "append" => \$opt_a,
	   "v=s", "newversion=s" => \$opt_v,
	   "d", "fromdirname" => \$opt_d,
	   "p", "preserve!" => \$opt_p)
    or die "Usage: $progname [options] [changelog entry]\nRun $progname --help for more details\n";
if ($opt_h) { usage; exit 0; }
if ($opt_version) { version; exit 0; }

# Only allow at most one non-help option
fatal "Only one of -a, -i, -v, -d is allowed; try $progname -h for more help"
    if ($opt_i?1:0) + ($opt_a?1:0) + ($opt_v?1:0) + ($opt_d?1:0) > 1;

# We'll process the rest of the command line later.

# Look for the debian changelog
until (-e "debian/changelog")
{
    chdir ".." or fatal "Can't chdir: $!";
    fatal "Cannot find debian/changelog anywhere!  Are you in the source code tree?"
	if cwd() eq "/";
}

# Clean up after old versions of debchange
if (-f "debian/RELEASED") {
    unlink("debian/RELEASED");
}

if ( -e "debian/changelog.dch" ) {
    fatal "The backup file debian/changelog.dch already exists --\n" .
		  "please move it before trying again";
}

sub BEGIN {
    # Initialise the variable
    $tmpchk=0;
}

sub END {
    unlink "debian/changelog.dch" or
	warn "Could not remove debian/changelog.dch"
	    if $tmpchk;
}

#####

# Find the current version number etc.
open PARSED, "dpkg-parsechangelog |"
    or fatal "Cannot execute dpkg-parsechangelog: $!";
while (<PARSED>) {
    chomp;
    if (/^(\S+):\s(.+?)\s*$/) { $changelog{$1}=$2; $last=$1; }
    elsif (/^(\S+):\s$/) { $changelog{$1}=''; $last=$1; }
    elsif (/^\s\.$/) { $changelog{$last}.="\n"; }
    elsif (/^\s(.+)$/) { $changelog{$last}.="$1\n"; }
    else {
	fatal "Don't understand dpkg-parsechangelog output:" . " $_";
    }
}

close PARSED
    or fatal "Problem executing dpkg-parsechangelog: $!";
if ($?) { fatal "dpkg-parsechangelog failed!" }

fatal "No version number in changelog!"
    unless exists $changelog{'Version'};

# Is this a native Debian package, i.e., does it have a - in the
# version number?
$VERSION=$changelog{'Version'};
($EPOCH) = ($VERSION =~ /^(\d+):/);
($SVERSION=$VERSION) =~ s/^\d+://;
($UVERSION=$SVERSION) =~ s/-[^-]*$//;

$PACKAGE=$changelog{'Source'};
($MAINTAINER,$EMAIL) = ($changelog{'Maintainer'} =~ /^([^<]+) <(.*)>/);

# Sanitise if necessary
if (exists $ENV{'DEBEMAIL'} and $ENV{'DEBEMAIL'} =~ /^(.*)\s+<(.*)>$/) {
    $ENV{'DEBFULLNAME'} ||= $1;
    $ENV{'DEBEMAIL'} = $2;
}
if (exists $ENV{'EMAIL'} and $ENV{'EMAIL'} =~ /^(.*)\s+<(.*)>$/) {
    $ENV{'DEBFULLNAME'} ||= $1;
    $ENV{'EMAIL'} = $2;
}

$MAINTAINER = $ENV{'DEBFULLNAME'} || do {
    my @pw = getpwuid $<;
    $pw[6] =~ s/,.*// if @pw;
    $pw[6]
    } || $MAINTAINER;
$EMAIL = $ENV{'DEBEMAIL'} || $ENV{'EMAIL'} || do {
    my $addr;
    if (open MAILNAME, '/etc/mailname') {
	chomp($addr = <MAILNAME>);
	close MAILNAME;
    }
    if (!$addr) {
	chomp($addr = `hostname --fqdn 2>/dev/null`);
	$addr = undef if $?;
    }
    if ($addr) {
	my $user = getpwuid $<;
	if (!$user) {
	    $addr = undef;
	}
	else {
	    $addr = "$user\@$addr";
	}
    }
    $addr
    } || $EMAIL;

#####

# Get a possible changelog entry from the command line
$SAVETEXT=$TEXT="@ARGV";

# Get the date
chomp($DATE=`822-date`);

# Are we going to have to figure things out for ourselves?
if (! $opt_i && ! $opt_v && ! $opt_d && ! $opt_a) {
    # Yes, we are
    @UPFILES = glob("../$PACKAGE\_$SVERSION\_*.upload");
    if (@UPFILES > 1) {
	fatal "Found more than one appropriate .upload file!\n" .
		      "Please use an explicit -a, -i or -v option instead.";
    }
    elsif (@UPFILES == 0) { $opt_a = 1 }
    else {
	open UPFILE, "<${UPFILES[0]}"
	    or fatal "Couldn't open .upload file for reading: $!\n" .
			"Please use an explicit -a, -i or -v option instead.";
	while (<UPFILE>) {
	    if (/^(s|Successfully uploaded) \Q$PACKAGE\E\_\Q$SVERSION\E\_[\w\-]+\.changes /) {
		$opt_i = 1;
		last;
	    }
	}
	close UPFILE
	    or fatal "Problems experienced reading .upload file: $!\n" .
			"Please use an explicit -a, -i or -v option instead.";
	if (! $opt_i) {
	    warn "A successful upload of the current version was not logged\n" .
		"in the upload log file; adding log entry to current version.";
	    $opt_a = 1;
	}
    }
}


# Open in anticipation....
open S, "debian/changelog" or fatal "Cannot open changelog: $!";
open O, ">debian/changelog.dch"
    or fatal "Cannot write to temporary file: $!";
# Note that we now have to remove it
$tmpchk=1;

if ($opt_i || $opt_v || $opt_d) {
    # Check that a given explicit version number is sensible.
    if ($opt_v || $opt_d) {
	if($opt_v) {
	    $NEW_VERSION=$opt_v;
	} else {
	    $pwd = `pwd`;
	    if ($pwd =~ m%.*/.*-([0-9][0-9a-zA-Z+\.]*)$%) {
		$NEW_VERSION=$1;
		if ($NEW_VERSION eq $UVERSION) {
		    # So it's a Debian-native package
		    if ($SVERSION eq $UVERSION) {
			fatal "New version taken from directory ($NEW_VERSION) is equal to\n" .
			    "the current version number ($UVERSION)!";
		    }
		    # So we just increment the Debian revision
		    warn "Warning: Incrementing Debian revision without altering upstream version number.\n";
		    $VERSION =~ /^(.*?)([a-yA-Y][a-zA-Z]*|\d*)$/;
		    my $end = $2;
		    if ($end eq '') {
			fatal "Cannot determine new Debian revision; please use -v option!";
		    }
		    $end++;
		    $NEW_VERSION="$1$end";
		} else {
		    $NEW_VERSION = "$EPOCH:$NEW_VERSION" if defined $EPOCH;
		    $NEW_VERSION .= "-1";
		}
	    } else {
		fatal "The directory name must be <package>-<version> for -d to work!\n" .
		    "No underscores allowed!";
	    }
	    # Don't try renaming the directory in this case!
	    $opt_p=1;
	}

	if (system("dpkg --compare-versions $VERSION lt $NEW_VERSION" .
		  " 2>/dev/null 1>&2")) {
	    fatal "New version specified ($NEW_VERSION) is less than\n" .
		    "the current version number ($VERSION)!";
	}

	($NEW_SVERSION=$NEW_VERSION) =~ s/^\d+://;
	($NEW_UVERSION=$NEW_SVERSION) =~ s/-[^-]*$//;
    }

    # We use the following criteria for the version and release number:
    # the last component of the version number is used as the
    # release number.  If this is not a Debian native package, then the
    # upstream version number is everything up to the final '-', not
    # including epochs.

    if (! $NEW_VERSION) {
	if ($VERSION =~ /(.*?)([a-yA-Y][a-zA-Z]*|\d*)$/i) {
	    $end=$2; $end++;
	    $NEW_VERSION = "$1$end";
	    ($NEW_SVERSION=$NEW_VERSION) =~ s/^\d+://;
	    ($NEW_UVERSION=$NEW_SVERSION) =~ s/-[^-]*$//;
	} else {
	    fatal "Error parsing version number: $VERSION";
	}
    }

    print O "$changelog{'Source'} ($NEW_VERSION) $changelog{'Distribution'}; ",
	"urgency=low\n\n";

    if ($TEXT) { write O } else { print O "  * \n"; $line=3; }
    print O "\n -- $MAINTAINER <$EMAIL>  $DATE\n\n";

    # Copy the old changelog file to the new one
    local $/ = undef;
    print O <S>;
}
else { # $opt_a = 1
    # This means we just have to generate a new * entry in changelog
    $NEW_VERSION=$VERSION;
    $NEW_SVERSION=$SVERSION;
    $NEW_UVERSION=$UVERSION;

    # The first lines are as we have already found
    print O $changelog{'Changes'};

    if ($TEXT) { write O; } else { print O "  * \n"; }
    print O "\n -- $MAINTAINER <$EMAIL>  $DATE\n";

    # Copy the rest of the changelog file to new one
    $line=-1;
    while (<S>) { $line++; last if /^ --/; }
    # Slurp the rest....
    local $/ = undef;
    print O <S>;
}

close S or fatal "Error closing debian/changelog: $!";
close O or fatal "Error closing temporary changelog: $!";

# Now Run the Editor
if (! $SAVETEXT) {  # $TEXT might have been modified by a write()
    my $mtime = (stat("debian/changelog.dch"))[9];
    defined $mtime or fatal
	"Error getting modification time of temporary changelog: $!";

    system("sensible-editor +$line debian/changelog.dch") == 0 or
    fatal "Error editing the changelog";

    my $newmtime = (stat("debian/changelog.dch"))[9];
    defined $newmtime or fatal
	"Error getting modification time of temporary changelog: $!";
    if ($mtime == $newmtime) {
	warn "Changelog unmodified; exiting.\n";
	exit 0;
    }
}

copy("debian/changelog.dch","debian/changelog") or
    fatal "Couldn't replace changelog with new changelog: $!";

if ($NEW_UVERSION ne $UVERSION && (cwd() =~ m%/\Q$PACKAGE\E-\Q$UVERSION\E$%)
    && !$opt_p) {
    if (move(cwd(), "../$PACKAGE-$NEW_UVERSION")) {
	warn "Warning: your current directory has been renamed to:\n../$PACKAGE-$NEW_UVERSION\n";
    } else {
	warn "Couldn't rename directory: $!";
    }
}

# Format for standard Debian changelogs
format O =
  * ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    $TEXT
 ~~ ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    $TEXT
.

sub fatal($) {
    my ($pack,$file,$line);
    ($pack,$file,$line) = caller();
    (my $msg = "$progname: fatal error at line $line:\n@_\n") =~ tr/\0//d;
    $msg =~ s/\n\n$/\n/;
    die $msg;
}
