#! /bin/bash

# debrelease: a devscripts wrapper around dupload which calls
#             dupload with the correct .changes file as parameter.
#             All command line options are passed onto dupload.
#
# Written and copyright 1999 by Julian Gilbey <jdg@debian.org> 
# Based on the original 'release' script by
#  Christoph Lameter <clameter@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

set -e

TEXTDOMAIN=devscripts
PROGNAME=`basename $0`
PKG_VERSION=@VERSION@

usage () {
    echo \
$"Usage: $PROGNAME [debrelease options] [dupload options]
  Run dupload on the newly created changes file.
  Debrelease options:
    -a_arch_        Search for .changes file made for Debian build _arch_
    -t_target_      Search for .changes file made for GNU _target_ arch
    --help          Show this message
    --version       Show version and copyright information"
}

version () {
    echo \
$"This is $PROGNAME, from the Debian devscripts package, version $PKG_VERSION
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on original code by Christoph Lameter.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later."
}

mustsetvar () {
    if [ "x$2" = x ]
    then
	echo >&2 $"$0: unable to determine $3"
	exit 1
    else
	# echo $"$0: $3 is $2"
	eval "$1=\"\$2\""
    fi
}

while [ $# != 0 ]
do
    case "$1" in
	-a*)
	    opt_a=1; arch="`echo \"$1\" | sed -e 's/^-a//'`" ;;
	--target)
	    shift; targetgnusystem="$1" ;;
	--target=*)
	    targetgnusystem="`echo \"$1\" | sed -e 's/^--target=//'`" ;;
	--help)	usage; exit 0 ;;
	--version)
		version; exit 0 ;;
	*)	DUPLOAD_OPTS="$DUPLOAD_OPTS $1"
    esac
    shift
done

# Look for .changes file via debian/changelog
until [ -f debian/changelog ]; do
    cd ..
    if [ `pwd` = "/" ]; then
	echo $"Cannot find debian/changelog anywhere." >&2
	exit 1
    fi
done

mustsetvar package "`dpkg-parsechangelog | sed -n 's/^Source: //p'`" \
    $"source package"
mustsetvar version "`dpkg-parsechangelog | sed -n 's/^Version: //p'`" \
    $"source version"

command -v dpkg-architecture > /dev/null 2>&1 && \
    eval `dpkg-architecture -a${arch} -t${targetgnusystem} -s`
test "${opt_a}" \
    || arch=`dpkg-architecture -a${arch} -t${targetgnusystem} \
	-qDEB_HOST_ARCH 2> /dev/null` && test "${arch}" \
    || mustsetvar arch "`dpkg --print-architecture`" $"build architecture"

sversion=`echo "$version" | perl -pe 's/^\d+://'`
pva="${package}_${sversion}${arch:+_${arch}}"
changes="../$pva.changes"

if [ ! -r "$changes" ]; then
    echo $"Could not read .changes file!"; exit 1
fi

exec dupload $DUPLOAD_OPTS "$changes"

echo $"Failed to exec dupload!" >&2
echo $"Aborting...." >&2
exit 1
