/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
using System;
using System.Collections;
using System.Reflection;
using Db4objects.Db4o.Query;
using Sharpen.Lang.Reflect;

namespace Sharpen.Lang
{
	public class Class
	{
		private static readonly IDictionary _typeToClassMap = new Hashtable();

		private static readonly IDictionary _typeNameToClassMap = new Hashtable();

		private static readonly IList PrimitiveTypes = new Type[] {
				typeof (DateTime), typeof (Decimal)
		};

		private Type _type;
		private String _name;
		private bool _primitive;

		public Class(Type type)
		{
			_type = type;
			_primitive = type.IsPrimitive || PrimitiveTypes.Contains(type);
		}

		public override bool Equals(object obj)
		{
			Class clazz = obj as Class;
			return clazz != null && clazz._type == _type;
		}

		public static Class ForName(String name)
		{
			if (null == name)
			{
				return null;
			}

			lock (_typeNameToClassMap.SyncRoot)
			{
				Class returnValue = (Class) _typeNameToClassMap[name];
				if (null != returnValue)
				{
					return returnValue;
				}

				try
				{
					Type t = TypeReference.FromString(name).Resolve();
					returnValue = GetClassForType(t);
					_typeNameToClassMap[name] = returnValue;
				}
				catch (Exception ex)
				{
					throw new TypeLoadException(name, ex);
				}
				return returnValue;
			}
		}

		public static Class GetClassForObject(object obj)
		{
			return GetClassForType(obj.GetType());
		}

		public static Class GetClassForType(Type forType)
		{
			if (forType == null)
			{
				return null;
            }

#if NET_2_0 || CF_2_0
            Type underlyingType = Nullable.GetUnderlyingType(forType);
            if (underlyingType != null)
            {
                forType = underlyingType;
            }
#endif
			
			lock (_typeToClassMap.SyncRoot)
			{
				Class clazz = (Class) _typeToClassMap[forType];
				if (clazz == null)
				{
					clazz = new Class(forType);
					_typeToClassMap[forType] = clazz;
				}
				return clazz;
			}
		}

		public Class GetComponentType()
		{
			return GetClassForType(_type.GetElementType());
		}
	    
		public String GetName()
		{
			if (_name == null)
			{
				_name = TypeReference.FromType(_type).GetUnversionedName();
			}
			return _name;
		}

		public Type GetNetType()
		{
			return _type;
		}

		public Class GetSuperclass()
		{
			return GetClassForType(_type.BaseType);
		}

		public static Type[] GetTypes(Class[] classes)
		{
			if (classes == null)
			{
				return new Type[] {};
			}
			Type[] types = new Type[classes.Length];
			for (int i = 0; i < types.Length; i++)
			{
				types[i] = classes[i].GetNetType();
			}
			return types;
		}

		public bool IsArray()
		{
			return _type.IsArray;
		}

		public bool IsAssignableFrom(Class clazz)
		{
			return _type.IsAssignableFrom(clazz._type);
		}

		public bool IsInstance(object obj)
		{
			if (obj == null)
			{
				return false;
			}
			if (_type.IsInterface)
			{
				return _type.IsAssignableFrom(obj.GetType());
			}
			return obj.GetType() == _type;
		}

		public bool IsInterface()
		{
			return _type.IsInterface;
		}

		public bool IsPrimitive()
		{
			return _primitive;
		}

		public Object NewInstance()
		{
			return Activator.CreateInstance(_type);
		}
	}
}
