#ifndef _SERVERPOOL_H
#define _SERVERPOOL_H 1

#include <stddef.h>
#include <string>
#include <vector>

#include "server.h"
#include "state.pb.h"
#include "stream.h"
#include "udpstream.h"

class Server;
class UDPStream;
struct ClientStats;
struct sockaddr_in6;

// Provides services such as load-balancing between a number of Server instances.
class ServerPool {
public:
	ServerPool(int num_servers);
	~ServerPool();

	// Fills streams() and clients().
	CubemapStateProto serialize();

	// Picks a server (round-robin) and allocates the given client to it.
	void add_client(int sock);
	void add_client_from_serialized(const ClientProto &client);

	// Adds the given stream to all the servers. Returns the stream index.
	int add_stream(const std::string &url, size_t backlog_size, Stream::Encoding encoding);
	int add_stream_from_serialized(const StreamProto &stream, const std::vector<int> &data_fds);
	void delete_stream(const std::string &url);
	int add_udpstream(const sockaddr_in6 &dst, int pacing_rate, int ttl, int multicast_iface_index);

	// Returns the stream index for the given URL (e.g. /foo.ts). Returns -1 on failure.
	int lookup_stream_by_url(const std::string &url) const;

	// Adds the given data to all the servers.
	void set_header(int stream_index,
	                const std::string &http_header,
	                const std::string &stream_header);
	void add_data(int stream_index, const char *data, size_t bytes, StreamStartSuitability suitable_for_stream_start);

	// Sets the max pacing rate for all the servers.
	void set_pacing_rate(int stream_index, uint32_t pacing_rate);

	// Changes the given stream's backlog size on all the servers.
	void set_backlog_size(int stream_index, size_t new_size);

	// Changes the given stream's encoding type on all the servers.
	void set_encoding(int stream_index, Stream::Encoding encoding);

	// Starts all the servers.
	void run();

	// Stops all the servers.
	void stop();

	std::vector<ClientStats> get_client_stats() const;

private:
	Server *servers;
	int num_servers, clients_added;

	// Our indexing is currently rather primitive; every stream_index in
	// [0, num_http_streams) maps to a HTTP stream (of which every Server
	// has exactly one copy), and after that, it's mapping directly into
	// <udp_streams>.
	int num_http_streams;
	std::vector<UDPStream *> udp_streams;

	ServerPool(const ServerPool &);
};

#endif  // !defined(_SERVERPOOL_H)
