/*  Copyright (C) 2002-2004 Gabriel Maldonado

    The gab library is free software; you can redistribute it
    and/or modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    The gab library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with the gab library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA

   Ported to csound5 by: Andres Cabrera
*/
#include "csdl.h"
#include <math.h>

#ifndef GAB_VECTORIAL_H
#define GAB_VECTORIAL_H

/* The follwoing from CsoundAV/newopcodes.h */
typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *kinterp, *ixmode, *outargs[VARGMAX];
} MTABLEI;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *kinterp, *ixmode, *outargs[VARGMAX];
        int nargs;
        MYFLT   xbmul;
        long  pfn, len;
        MYFLT   *ftable;
} MTABLE;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *ixmode, *inargs[VARGMAX];
} MTABLEIW;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *ixmode, *inargs[VARGMAX];
        int nargs;
        MYFLT   xbmul;
        long  pfn, len;
        MYFLT   *ftable;
} MTABLEW;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *inargs[VARGMAX];
} MTABIW;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *inargs[VARGMAX];
        int nargs;
  /* MYFLT      xbmul; */
        long  pfn, len;
        MYFLT   *ftable;
} MTABW;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *outargs[VARGMAX];
} MTABI;

typedef struct {
        OPDS    h;
        MYFLT   *xndx, *xfn, *outargs[VARGMAX];
        int nargs;
  /* MYFLT      xbmul; */
        long  pfn, len;
        MYFLT   *ftable;
} MTAB;

/* The following from CSoundAV/vectorial.h */
typedef struct {
        OPDS   h;
        MYFLT *ifn, *kval, *ielements;
        int elements;
        MYFLT *vector;
} VECTOROP;

typedef struct {
        OPDS   h;
        MYFLT *ifn1, *ifn2, *ielements;
        int elements;
        MYFLT *vector1, *vector2;
} VECTORSOP;


typedef struct {
        OPDS   h;
        MYFLT *ifn, *kmin, *kmax, *ielements;
        int elements;
        MYFLT *vector;
} VLIMIT;

typedef struct {
        OPDS    h;
        MYFLT   *ifn, *krange, *kcps, *ielements;
        AUXCH auxch;
        MYFLT *vector;
        int elements;
        long    phs;
        MYFLT   *num1;
} VRANDH;

typedef struct {
        OPDS    h;
        MYFLT   *ifn, *krange, *kcps, *ielements;
        AUXCH   auxch;
        MYFLT   *vector;
        int             elements;
        long    phs;
        MYFLT   *num1, *num2, *dfdmax;
} VRANDI;

/*  TSEG definition from H/vpvoc.h */
typedef struct {
        FUNC *function, *nxtfunction;
        MYFLT d;
        long   cnt;
} TSEG;

typedef struct {
        OPDS    h;
        MYFLT   *ioutfunc,*ielements,*argums[VARGMAX];
        TSEG    *cursegp;
        MYFLT *vector;
        int     elements;
        long    nsegs;
        AUXCH   auxch;
} VSEG;

typedef struct {
        OPDS    h;
        MYFLT   *ifn, *khtim, *ielements, *ifnInit;
        MYFLT   c1, c2, *yt1, *vector, prvhtim;
        int elements;
        AUXCH   auxch;
} VPORT;

typedef struct {
        OPDS    h;
        MYFLT   *ifnOut, *ifnIn, *ifnDel, *ielements, *imaxd, *istod;
        AUXCH   aux;
        MYFLT   **buf, *outvec, *invec, *dlyvec;
        long    *left, maxd;
        int             elements;
} VECDEL;



typedef struct {
        FUNC *function, *nxtfunction;
        double d;
} TSEG2;

typedef struct {
        OPDS    h;
        MYFLT   *kphase, *ioutfunc, *ielements,*argums[VARGMAX];
        TSEG2    *cursegp;
        MYFLT *vector;
        int     elements;
        long    nsegs;
        AUXCH   auxch;
} VPSEG;

typedef struct {
        OPDS    h;
        MYFLT   *kr, *kin, *kdel, *imaxd, *istod, *interp;
        AUXCH   aux;
        long    left, maxd;
} KDEL;

typedef struct {
        OPDS    h;
        MYFLT   *ktrig, *kreinit, *ioutFunc, *initStateFunc,
                *iRuleFunc, *ielements, *irulelen, *iradius;
        MYFLT   *currLine, *outVec, *initVec, *ruleVec;
        int     elements, NewOld, ruleLen;
        AUXCH   auxch;
} CELLA;

/* from uggab.h for vrandi, vrandh */
/* extern long holdrand; */

#define oneUp31Bit      (double) (4.656612875245796924105750827168e-10)
#define randGab   (MYFLT)((double) \
   (((holdrand = holdrand * 214013L + 2531011L) >> 1) & 0x7fffffff) * oneUp31Bit)
#define BiRandGab   (MYFLT)((double) \
        (holdrand = holdrand * -214013L + 2531011L) * oneUp31Bit)
#endif

void mtable_i(MTABLEI *p)
{
    FUNC *ftp;
    int j, nargs;
    MYFLT *table, xbmul = FL(0.0), **out = p->outargs;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtablei: incorrect table number");
      return;
    }
    table = ftp->ftable;
    nargs = p->INOCOUNT-4;
    if (*p->ixmode)
      xbmul = (MYFLT) (ftp->flen / nargs);

    if (*p->kinterp) {
      MYFLT     v1, v2 ;
      MYFLT fndx = (*p->ixmode) ? *p->xndx * xbmul : *p->xndx;
      long indx = (long) fndx;
      MYFLT fract = fndx - indx;
      for (j=0; j < nargs; j++) {
        v1 = table[indx * nargs + j];
        v2 = table[(indx + 1) * nargs + j];
        **out++ = v1 + (v2 - v1) * fract;
      }
    }
    else {
      long indx = (*p->ixmode) ? (long)(*p->xndx * xbmul) : (long) *p->xndx;
      for (j=0; j < nargs; j++)
        **out++ =  table[indx * nargs + j];
    }
    return;
}

void mtable_set(MTABLE *p)        /*  mtab by G.Maldonado */
{
    FUNC *ftp;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtable: incorrect table number");
      return;
    }
    p->ftable = ftp->ftable;
    p->nargs = p->INOCOUNT-4;
    p->len = ftp->flen / p->nargs;
    p->pfn = (long) *p->xfn;
    if (*p->ixmode)
      p->xbmul = (MYFLT) ftp->flen / p->nargs;
    return;
}

void mtable_k(MTABLE *p)
{
    int j, nargs = p->nargs;
    MYFLT **out = p->outargs;
    MYFLT *table;
    long len;
    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtable: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
      if (*p->ixmode)
        p->xbmul = (MYFLT) ftp->flen / nargs;
    }
    table= p->ftable;
    len = p->len;
    if (*p->kinterp) {
      MYFLT fndx;
      long indx;
      MYFLT fract;
      long indxp1;
      MYFLT     v1, v2 ;
      fndx = (*p->ixmode) ? *p->xndx * p->xbmul : *p->xndx;
      if (fndx >= len)
        fndx = (MYFLT) fmod(fndx, len);
      indx = (long) fndx;
      fract = fndx - indx;
      indxp1 = (indx < len-1) ? (indx+1) * nargs : 0;
      indx *=nargs;
      for (j=0; j < nargs; j++) {
        v1 = table[indx + j];
        v2 = table[indxp1 + j];
        **out++ = v1 + (v2 - v1) * fract;
      }
    }
    else {
      long indx = (*p->ixmode) ? ((long)(*p->xndx * p->xbmul) % len) * nargs :
                                 ((long) *p->xndx % len ) * nargs ;
      for (j=0; j < nargs; j++)
        **out++ =  table[indx + j];
    }
    return;
}

void mtable_a(MTABLE *p)
{
    int j, nargs = p->nargs;
    int nsmps = ksmps, ixmode = (int) *p->ixmode, k=0;
    MYFLT **out = p->outargs;
    MYFLT *table;
    MYFLT *xndx = p->xndx, xbmul;
    long len;

    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtable: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
      if (ixmode)
        p->xbmul = (MYFLT) ftp->flen / nargs;
    }
    table = p->ftable;
    len = p->len;
    xbmul = p->xbmul;
    if (*p->kinterp) {
      MYFLT fndx;
      long indx;
      MYFLT fract;
      long indxp1;
      do {
        MYFLT   v1, v2 ;
        fndx = (ixmode) ? *xndx++ * xbmul : *xndx++;
        if (fndx >= len)
          fndx = (MYFLT) fmod(fndx, len);
        indx = (long) fndx;
        fract = fndx - indx;
        indxp1 = (indx < len-1) ? (indx+1) * nargs : 0;
        indx *=nargs;
        for (j=0; j < nargs; j++) {
          v1 = table[indx + j];
          v2 = table[indxp1 + j];
          out[j][k] = v1 + (v2 - v1) * fract;

        }
        k++;
      } while(--nsmps);

    }
    else {
      do {
        long indx = (ixmode) ? ((long)(*xndx++ * xbmul)%len) * nargs :
                               ((long) *xndx++ %len) * nargs;
        for (j=0; j < nargs; j++) {
          out[j][k] =  table[indx + j];
        }
        k++;
      } while(--nsmps);
    }
    return;
}


void mtab_i(MTABI *p)
{
    FUNC *ftp;
    int j, nargs;
    long indx;
    MYFLT *table, **out = p->outargs;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtabi: incorrect table number");
      return;
    }
    table = ftp->ftable;
    nargs = p->INOCOUNT-2;

    indx = (long) *p->xndx;
    for (j=0; j < nargs; j++)
      **out++ =  table[indx * nargs + j];
    return;
}

void mtab_set(MTAB *p)    /* mtab by G.Maldonado */
{
    FUNC *ftp;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtable: incorrect table number");
      return;
    }
    p->ftable = ftp->ftable;
    p->nargs = p->INOCOUNT-2;
    p->len = ftp->flen / p->nargs;
    p->pfn = (long) *p->xfn;
    return;
}

void mtab_k(MTAB *p)
{
    int j, nargs = p->nargs;
    MYFLT **out = p->outargs;
    MYFLT *table;
    long len, indx;

    table= p->ftable;
    len = p->len;
    indx = ((long) *p->xndx % len ) * nargs ;
    for (j=0; j < nargs; j++)
      **out++ =  table[indx + j];
    return;
}

void mtab_a(MTAB *p)
{
    int j, nargs = p->nargs;
    int nsmps = ksmps, k=0;
    MYFLT **out = p->outargs;
    MYFLT *table;
    MYFLT *xndx = p->xndx;
    long len;
    table = p->ftable;
    len = p->len;
    do {
      long indx = ((long) *xndx++ %len) * nargs;
      for (j=0; j < nargs; j++) {
        out[j][k] =  table[indx + j];
      }
      k++;
    } while(--nsmps);
    return;
}

/* ////////// mtab end /////////////// */

void mtablew_i(MTABLEIW *p)
{
    FUNC *ftp;
    int j, nargs;
    long indx;
    MYFLT *table, xbmul = FL(0.0), **in = p->inargs;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtablewi: incorrect table number");
      return;
    }
    table = ftp->ftable;
    nargs = p->INOCOUNT-3;
    if (*p->ixmode)
      xbmul = (MYFLT) (ftp->flen / nargs);
    indx = (*p->ixmode) ? (long)(*p->xndx * xbmul) : (long) *p->xndx;
    for (j=0; j < nargs; j++)
      table[indx * nargs + j] = **in++;
    return;
}

void mtablew_set(MTABLEW *p)      /* mtabw by G.Maldonado */
{
    FUNC *ftp;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtabw: incorrect table number");
      return;
    }
    p->ftable = ftp->ftable;
    p->nargs = p->INOCOUNT-3;
    p->len = ftp->flen / p->nargs;
    p->pfn = (long) *p->xfn;
    if (*p->ixmode)
      p->xbmul = (MYFLT) ftp->flen / p->nargs;
    return;
}

void mtablew_k(MTABLEW *p)
{
    int j, nargs = p->nargs;
    MYFLT **in = p->inargs;
    MYFLT *table;
    long len, indx;
    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtabw: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
      if (*p->ixmode)
        p->xbmul = (MYFLT) ftp->flen / nargs;
    }
    table= p->ftable;
    len = p->len;
    indx = (*p->ixmode) ? ((long)(*p->xndx * p->xbmul) % len) * nargs :
                          ((long) *p->xndx % len ) * nargs ;
    for (j=0; j < nargs; j++)
      table[indx + j] = **in++;
    return;
}

void mtablew_a(MTABLEW *p)
{
    int j, nargs = p->nargs;
    int nsmps = ksmps, ixmode = (int) *p->ixmode, k=0;
    MYFLT **in = p->inargs;
    MYFLT *table;
    MYFLT *xndx = p->xndx, xbmul;
    long len;

    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtabw: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
      if (ixmode)
        p->xbmul = (MYFLT) ftp->flen / nargs;
    }
    table = p->ftable;
    len = p->len;
    xbmul = p->xbmul;
    do {
      long indx = (ixmode) ? ((long)(*xndx++ * xbmul)%len) * nargs :
                             ((long) *xndx++ %len) * nargs;
      for (j=0; j < nargs; j++) {
        table[indx + j] = in[j][k];
      }
      k++;
    } while(--nsmps);
    return;
}

/* ////////////////////////////////////// */

void mtabw_i(MTABIW *p)
{
    FUNC *ftp;
    int j, nargs;
    long indx;
    MYFLT *table, **in = p->inargs;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtabwi: incorrect table number");
      return;
    }
    table = ftp->ftable;
    nargs = p->INOCOUNT-2;
    indx = (long) *p->xndx;
    for (j=0; j < nargs; j++)
      table[indx * nargs + j] = **in++;
    return;
}

void mtabw_set(MTABW *p)  /* mtabw by G.Maldonado */
{
    FUNC *ftp;
    if ((ftp = ftnp2find( p->xfn)) == NULL) {
      initerror("mtabw: incorrect table number");
      return;
    }
    p->ftable = ftp->ftable;
    p->nargs = p->INOCOUNT-2;
    p->len = ftp->flen / p->nargs;
    p->pfn = (long) *p->xfn;
    return;
}

void mtabw_k(MTABW *p)
{
    int j, nargs = p->nargs;
    MYFLT **in = p->inargs;
    MYFLT *table;
    long len, indx;
    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtablew: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
    }
    table= p->ftable;
    len = p->len;
    indx = ((long) *p->xndx % len ) * nargs ;
    for (j=0; j < nargs; j++)
      table[indx + j] = **in++;
    return;
}

void mtabw_a(MTABW *p)
{
    int j, nargs = p->nargs;
    int nsmps = ksmps, k=0;
    MYFLT **in = p->inargs;
    MYFLT *table;
    MYFLT *xndx = p->xndx;
    long len;

    if (p->pfn != (long)*p->xfn) {
      FUNC *ftp;
      if ( (ftp = ftnp2find( p->xfn) ) == NULL) {
        perferror("mtabw: incorrect table number");
        return;
      }
      p->pfn = (long)*p->xfn;
      p->ftable = ftp->ftable;
      p->len = ftp->flen / nargs;
    }
    table = p->ftable;
    len = p->len;
    do {
      long indx = ((long) *xndx++ %len) * nargs;
      for (j=0; j < nargs; j++) {
        table[indx + j] = in[j][k];
      }
      k++;
    } while(--nsmps);
    return;
}


/* The following opcodes come from CsoundAV/vectorial.c */

void vectorOp_set(VECTOROP *p)
{
    FUNC        *ftp;
    if ((ftp = ftnp2find(p->ifn)) != NULL) {
      p->vector = ftp->ftable;
      p->elements = (int) *p->ielements;
    }
    if ( p->elements > ftp->flen ) {
      initerror("vectorop: invalid num of elements");
      return;
    }
    return;
}

void vadd(VECTOROP *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT value = *p->kval;

    do {
      *vector++ += value;
    } while (--elements);
    return;
}

void vmult(VECTOROP *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT value = *p->kval;

    do {
      *vector++ *= value;
    } while (--elements);
    return;
}

void vpow(VECTOROP *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT value = *p->kval;

    do {
      *vector++ = (MYFLT) pow (*vector, value);
    } while (--elements);
    return;
}

void vexp(VECTOROP *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT value = *p->kval;

    do {
      *vector++ = (MYFLT) pow (value, *vector);
    } while (--elements);
    return;
}
/* ------------------------- */

void vectorsOp_set(VECTORSOP *p)
{
    FUNC        *ftp1, *ftp2;
    if ((ftp1 = ftnp2find(p->ifn1)) != NULL) {
      p->vector1 = ftp1->ftable;
    }
    if ((ftp2 = ftnp2find(p->ifn2)) != NULL) {
      p->vector2 = ftp2->ftable;
    }
    p->elements = (int) *p->ielements;
    if ( p->elements > ftp1->flen || p->elements > ftp2->flen) {
      initerror("vectorop: invalid num of elements");
      return;
    }
    return;
}


void vcopy(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ = *vector2++;
    } while (--elements);
    return;
}

void vcopy_i(VECTORSOP *p)
{
    FUNC        *ftp1, *ftp2;
    MYFLT   *vector1 = NULL, *vector2 = NULL;
    unsigned long       elements;
    if ((ftp1 = ftnp2find(p->ifn1)) != NULL) {
      vector1 = ftp1->ftable;
    }
    if ((ftp2 = ftnp2find(p->ifn2)) != NULL) {
      vector2 = ftp2->ftable;
    }
    elements = (unsigned long) *p->ielements;
    if ( elements > ftp1->flen || elements > ftp2->flen) {
      initerror("vcopy_i: invalid num of elements");
      return;
    }

    do {
      *vector1++ = *vector2++;
    } while (--elements);
    return;
}



void vaddv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ += *vector2++;
    } while (--elements);
    return;
}

void vsubv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ -= *vector2++;
    } while (--elements);
    return;
}

void vmultv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ *= *vector2++;
    } while (--elements);
    return;
}

void vdivv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ /= *vector2++;
    } while (--elements);
    return;
}

void vpowv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ = (MYFLT) pow (*vector1, *vector2++);
    } while (--elements);
    return;
}

void vexpv(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ = (MYFLT) pow (*vector1, *vector2++);
    } while (--elements);
    return;
}

void vmap(VECTORSOP *p)
{
    int elements = p->elements;
    MYFLT *vector1 = p->vector1, *vector2 = p->vector2;

    do {
      *vector1++ = (vector2++)[(int)*vector1];
    } while (--elements);
    return;
}


void vlimit_set(VLIMIT *p)
{
    FUNC        *ftp;
    if ((ftp = ftnp2find(p->ifn)) != NULL) {
      p->vector = ftp->ftable;
      p->elements = (int) *p->ielements;
    }
    if ( p->elements >= ftp->flen ) {
      initerror("vectorop: invalid num of elements");
      return;
    }
    return;
}

void vlimit(VLIMIT *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT min = *p->kmin, max = *p->kmax;
    do {
      *vector++ = (*vector > min) ? ((*vector < max) ? *vector : max) : min;
    } while (--elements);
    return;
}

/*-----------------------------*/
void vport_set(VPORT *p)
{
    FUNC        *ftp;
    int elements;
    MYFLT *vector, *yt1,*vecInit  = NULL;

    if ((ftp = ftnp2find(p->ifn)) != NULL) {
      vector = (p->vector = ftp->ftable);
      elements = (p->elements = (int) *p->ielements);
      if ( elements >= ftp->flen )
        initerror("vport: invalid table length or num of elements");
      return;
    }
    else {
      initerror("vport: invalid table");
      return;
    }
    if (*p->ifnInit) {
      if ((ftp = ftnp2find(p->ifnInit)) != NULL) {
        vecInit = ftp->ftable;
        if ( elements >= ftp->flen ) {
          initerror("vport: invalid init table length or num of elements");
          return;
        }
      }
      else {
        initerror("vport: invalid init table");
        return;
      }
    }
    if (p->auxch.auxp == NULL)
      auxalloc(elements * sizeof(MYFLT), &p->auxch);
    yt1 = (p->yt1 = (MYFLT *) p->auxch.auxp);
    if (vecInit) {
      do {
        *yt1++ = *vecInit++;
      } while (--elements);
    } else {
      do {
        *yt1++ = FL(0.0);
      } while (--elements);
    }
    p->prvhtim = -FL(100.0);
    return;
}

void vport(VPORT *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector, *yt1 = p->yt1, c1, c2;
    if (p->prvhtim != *p->khtim) {
      p->c2 = (MYFLT)pow(0.5, (double)onedkr / *p->khtim);
      p->c1 = FL(1.0) - p->c2;
      p->prvhtim = *p->khtim;
    }
    c1 = p->c1;
    c2 = p->c2;
    do {
      *vector++ = (*yt1++ = c1 * *vector + c2 * *yt1);
    } while (--elements);
    return;
}

/*-------------------------------*/
void vwrap(VLIMIT *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT min = *p->kmin, max = *p->kmax;

    if (min >= max) {
      MYFLT average = (min+max)/2;
      do {
        *vector++ = average;
      } while (--elements);
    }
    else {
      do {
        if (*vector >= max)
          *vector++ = min + (MYFLT) fmod(*vector - min, fabs(min-max));
        else
          *vector++ = max - (MYFLT) fmod(max - *vector, fabs(min-max));
      } while (--elements);
    }
    return;
}

void vmirror(VLIMIT *p)
{
    int elements = p->elements;
    MYFLT *vector = p->vector;
    MYFLT min = *p->kmin, max = *p->kmax;

    if (min >= max) {
      MYFLT average = (min+max)* FL(0.50);
      do {
        *vector++ = average;
      } while (--elements);
    }
    else {
      do {
        while (!((*vector <= max) && (*vector >=min))) {
          if (*vector > max)
            *vector = max + max - *vector;
          else
            *vector = min + min - *vector;
        }
        vector++;
      } while (--elements);
    }
    return;
}

/* #include "newopcodes.h" */

void vrandh_set(VRANDH *p)
{
    FUNC        *ftp;
    int elements = 0;
    MYFLT *num1;
    if ((ftp = ftnp2find(p->ifn)) != NULL) {
      p->vector = ftp->ftable;
      elements = (p->elements = (int) *p->ielements);
    }
    else return;
    if ( p->elements >= ftp->flen ) {
      initerror("vectorop: invalid num of elements");
      return;
    }
    p->phs = 0;
    if (p->auxch.auxp == NULL)
      auxalloc(p->elements * sizeof(MYFLT), &p->auxch);
    num1 = (p->num1 = (MYFLT *) p->auxch.auxp);
    do {
      *num1++ = BiRandGab;
    } while (--elements);
    return;
}

void vrandh(VRANDH *p)
{
    MYFLT *vector = p->vector, *num1 = p->num1;
    MYFLT value = *p->krange;
    int elements = p->elements;

    do {
      *vector++ += *num1++ * value;
    } while (--elements);

    p->phs += (long)(*p->kcps * kicvt);
    if (p->phs >= MAXLEN) {
      p->phs &= PHMASK;
      elements = p->elements;
      vector = p->vector;
      num1 = p->num1;
      do {
        *num1++ = BiRandGab;
      } while (--elements);
    }
    return;
}

void vrandi_set(VRANDI *p)
{
    FUNC        *ftp;
    MYFLT fmaxlen = (MYFLT)MAXLEN;
    int elements = 0;
    MYFLT *dfdmax, *num1, *num2;
    if ((ftp = ftnp2find(p->ifn)) != NULL) {
      p->vector = ftp->ftable;
      elements = (p->elements = (int) *p->ielements);
    }
    else return;
    if ( p->elements >= ftp->flen ) {
      initerror("vectorop: invalid num of elements");
      return;
    }
    p->phs = 0;
    if (p->auxch.auxp == NULL)
      auxalloc(elements * sizeof(MYFLT) * 3, &p->auxch);
    num1 = (p->num1 = (MYFLT *) p->auxch.auxp);
    num2 = (p->num2 = &num1[elements]);
    dfdmax = (p->dfdmax = &num1[elements * 2]);

    do {
      *num1 = FL(0.0);
      *num2 = BiRandGab;
      *dfdmax++ = (*num2++ - *num1++) / fmaxlen;
    } while (--elements);
    return;
}


void vrandi(VRANDI *p)
{
    MYFLT *vector = p->vector, *num1 = p->num1, *num2, *dfdmax = p->dfdmax;
    MYFLT value = *p->krange;
    MYFLT fmaxlen = (MYFLT)MAXLEN;
    int elements = p->elements;

    do {
      *vector++ += (*num1++ + (MYFLT)p->phs * *dfdmax++) * value;
    } while (--elements);

    p->phs += (long)(*p->kcps * kicvt);
    if (p->phs >= MAXLEN) {
      p->phs &= PHMASK;
      elements = p->elements;
      vector = p->vector;
      num1 = p->num1;
      num2 = p->num2;
      dfdmax = p->dfdmax;

      do {
        *num1 = *num2;
        *num2 = BiRandGab ;
        *dfdmax++ = (*num2++ - *num1++) / fmaxlen;
      } while (--elements);
    }
    return;
}

void vecdly_set(VECDEL *p)
{
    FUNC        *ftp;
    int elements = (p->elements = (int) *p->ielements), j;
    long n;

    if ((ftp = ftnp2find(p->ifnOut)) != NULL) {
      p->outvec = ftp->ftable;
      elements = (p->elements = (int) *p->ielements);
      if ( elements >= ftp->flen ) {
        initerror("vecdelay: invalid num of elements");
        return;
      }
    }
    else {
      initerror("vecdly: invalid output table");
      return;
    }
    if ((ftp = ftnp2find(p->ifnIn)) != NULL) {
      p->invec = ftp->ftable;
      if (elements >= ftp->flen) {
        initerror("vecdelay: invalid num of elements");
        return;
      }
    }
    else {
      initerror("vecdly: invalid input table");
      return;
    }
    if ((ftp = ftnp2find(p->ifnDel)) != NULL) {
      p->dlyvec = ftp->ftable;
      if ( elements >= ftp->flen ) {
        initerror("vecdelay: invalid num of elements");
        return;
      }
    }
    else {
      initerror("vecdly: invalid delay table");
      return;
    }
    n = (p->maxd = (long) (*p->imaxd * ekr));
    if (n == 0) n = (p->maxd = 1);

    if (!*p->istod) {
      if (p->aux.auxp == NULL ||
          (int)(elements * sizeof(MYFLT *)
                + n * elements * sizeof(MYFLT)
                + elements * sizeof(long)) > p->aux.size) {
        auxalloc(elements * sizeof(MYFLT *)
                 + n * elements * sizeof(MYFLT)
                 + elements * sizeof(long),
                 &p->aux);
        p->buf= (MYFLT **) p->aux.auxp;
        for (j = 0; j < elements; j++) {
          p->buf[j] = (MYFLT *) (p->aux.auxp + sizeof(MYFLT *)* elements
                                 +sizeof(MYFLT ) * n * j);
        }
        p->left= (long *)  (p->aux.auxp +sizeof(MYFLT *)* elements
                            +sizeof(MYFLT ) * n * elements);
      }
      else {
        MYFLT **buf= p->buf;
        for (j = 0; j < elements; j++) {
          MYFLT *temp = buf[j];
          int count = n;
          do {
            *temp++ = FL(0.0);
          } while (--count);
          p->left[j] = 0;
        }
      }
    }
    return;
}


void vecdly(VECDEL *p)
{
    long maxd = p->maxd, *indx=p->left, v1, v2;
    MYFLT **buf = p->buf, fv1, fv2, *inVec = p->invec;
    MYFLT *outVec = p->outvec, *dlyVec = p->dlyvec;
    int elements = p->elements;
    if (buf==NULL) {
      initerror("vecdly: not initialized");
      return;
    }
    do {
      (*buf)[*indx] = *inVec++;
      fv1 = *indx - *dlyVec++ * ekr;
      while (fv1 < FL(0.0))     fv1 += (MYFLT)maxd;
      while (fv1 >= (MYFLT)maxd) fv1 -= (MYFLT)maxd;
      if (fv1 < maxd - 1) fv2 = fv1 + 1;
      else                fv2 = FL(0.0);
      v1 = (long)fv1;
      v2 = (long)fv2;
      *outVec++ = (*buf)[v1] + (fv1 - v1) * ((*buf)[v2]-(*buf)[v1]);
      ++buf;
      if (++(*indx) == maxd) *indx = 0;
      ++indx;
    } while (--elements);
    return;
}


void vseg_set(VSEG *p)
{
    TSEG        *segp;
    int nsegs;
    MYFLT       **argp, dur, *vector;
    FUNC *nxtfunc, *curfunc, *ftp;
    long        flength;

    nsegs = ((p->INCOUNT-2) >> 1);      /* count segs & alloc if nec */

    if ((segp = (TSEG *) p->auxch.auxp) == NULL) {
      auxalloc((long)(nsegs+1)*sizeof(TSEG), &p->auxch);
      p->cursegp = segp = (TSEG *) p->auxch.auxp;
      (segp+nsegs)->cnt = MAXPOS;
    }
    argp = p->argums;
    if ((nxtfunc = ftnp2find(*argp++)) == NULL)
      return;
    if ((ftp = ftnp2find(p->ioutfunc)) != NULL) {
      p->vector = ftp->ftable;
      p->elements = (int) *p->ielements;
    }
    if ( p->elements >= ftp->flen ) {
      initerror("vlinseg/vexpseg: invalid num. of elements");
      return;
    }
    vector = p->vector;
    flength = p->elements;

    do {
      *vector++ = FL(0.0);
    } while (--flength);

    if (**argp <= FL(0.0))  return; /* if idur1 <= 0, skip init  */
    p->cursegp = segp;              /* else proceed from 1st seg */
    segp--;
    do {
      segp++;           /* init each seg ..  */
      curfunc = nxtfunc;
      dur = **argp++;
      if ((nxtfunc = ftnp2find(*argp++)) == NULL) return;
      if (dur > FL(0.0)) {
        segp->d = dur * ekr;
        segp->function =  curfunc;
        segp->nxtfunction = nxtfunc;
        segp->cnt = (long) (segp->d + .5);
      }
      else break;               /*  .. til 0 dur or done */
    } while (--nsegs);
    segp++;
    segp->d = FL(0.0);
    segp->cnt = MAXPOS;         /* set last cntr to infin */
    segp->function =  nxtfunc;
    segp->nxtfunction = nxtfunc;
    return;
}

void vlinseg(VSEG *p)
{
    TSEG        *segp;
    MYFLT       *curtab, *nxttab,curval, nxtval, durovercnt=FL(0.0), *vector;
    long        flength, upcnt;
    if (p->auxch.auxp==NULL) {
      /* initerror(Str(X_1270,"tableseg: not initialized")); */
      initerror("tableseg: not initialized");
      return;
    }
    segp = p->cursegp;
    curtab = segp->function->ftable;
    nxttab = segp->nxtfunction->ftable;
    upcnt = (long)segp->d-segp->cnt;
    if (upcnt > 0)
      durovercnt = segp->d/upcnt;
    while (--segp->cnt < 0)
      p->cursegp = ++segp;
    flength = p->elements;
    vector = p->vector;
    do {
      curval = *curtab++;
      nxtval = *nxttab++;
      if (durovercnt > FL(0.0))
        *vector++ = (curval + ((nxtval - curval) / durovercnt));
      else
        *vector++ = curval;
    } while (--flength);
    return;
}

void vexpseg(VSEG *p)
{
    TSEG        *segp;
    MYFLT       *curtab, *nxttab,curval, nxtval, cntoverdur=FL(0.0), *vector;
    long        flength, upcnt;

    if (p->auxch.auxp==NULL) {
      /* initerror(Str(X_1271,"tablexseg: not initialized")); */
      initerror("tablexseg: not initialized");
      return;
    }
    segp = p->cursegp;
    curtab = segp->function->ftable;
    nxttab = segp->nxtfunction->ftable;
    upcnt = (long)segp->d-segp->cnt;
    if (upcnt > 0) cntoverdur = upcnt/ segp->d;
    while(--segp->cnt < 0)
      p->cursegp = ++segp;
    flength = p->elements;
    vector = p->vector;
    cntoverdur *= cntoverdur;
    do {
      curval = *curtab++;
      nxtval = *nxttab++;
      *vector++ = (curval + ((nxtval - curval) * cntoverdur));
    } while (--flength);
    return;
}

/* ----------------------------------------------- */
void vphaseseg_set(VPSEG *p)
{
    TSEG2       *segp;
    int nsegs,j;
    MYFLT       **argp,  *vector;
    double dur, durtot = 0.0, prevphs;
    FUNC *nxtfunc, *curfunc, *ftp;
    long        flength;

    nsegs = p->nsegs =((p->INCOUNT-3) >> 1);    /* count segs & alloc if nec */

    if ((segp = (TSEG2 *) p->auxch.auxp) == NULL) {
      auxalloc((long)(nsegs+1)*sizeof(TSEG), &p->auxch);
      p->cursegp = segp = (TSEG2 *) p->auxch.auxp;
      /* (segp+nsegs)->cnt = MAXPOS;  */
    }
    argp = p->argums;
    if ((nxtfunc = ftnp2find(*argp++)) == NULL)
      return;
    if ((ftp = ftnp2find(p->ioutfunc)) != NULL) {
      p->vector = ftp->ftable;
      p->elements = (int) *p->ielements;
    }
    if ( p->elements > ftp->flen ) {
      initerror("vphaseseg: invalid num. of elements");
      return;
    }
    vector = p->vector;
    flength = p->elements;

    do {
      *vector++ = FL(0.0);
    } while (--flength);

    if (**argp <= FL(0.0))  return; /* if idur1 <= 0, skip init  */
    /* p->cursegp = tempsegp =segp; */      /* else proceed from 1st seg */

    segp--;
    do {
      segp++;           /* init each seg ..  */
      curfunc = nxtfunc;
      dur = **argp++;
      if ((nxtfunc = ftnp2find(*argp++)) == NULL) return;
      if (dur > FL(0.0)) {
        durtot+=dur;
        segp->d = dur; /* * ekr; */
        segp->function = curfunc;
        segp->nxtfunction = nxtfunc;
        /* segp->cnt = (long) (segp->d + .5);  */
      }
      else break;               /*  .. til 0 dur or done */
    } while (--nsegs);
    segp++;

    segp->function =  nxtfunc;
    segp->nxtfunction = nxtfunc;
    nsegs = p->nsegs;

    segp = p->cursegp;

    for (j=0; j< nsegs; j++)
      segp[j].d /= durtot;

    for (j=nsegs-1; j>= 0; j--)
      segp[j+1].d = segp[j].d;

    segp[0].d = prevphs = 0.0;

    for (j=0; j<= nsegs; j++) {
      segp[j].d += prevphs;
      prevphs = segp[j].d;
    }
    return;
}

void vphaseseg(VPSEG *p)
{

    TSEG2       *segp = p->cursegp;
    double phase = *p->kphase, partialPhase = 0.0;
    int j, flength;
    MYFLT       *curtab = NULL, *nxttab = NULL, curval, nxtval, *vector;

    while (phase >= 1.0) phase -= 1.0;
    while (phase < 0.0) phase = 0.0;

    for (j = 0; j < p->nsegs; j++) {
      TSEG2 *seg = &segp[j], *seg1 = &segp[j+1];
      if (phase < seg1->d) {
        curtab = seg->function->ftable;
        nxttab = seg1->function->ftable;
        partialPhase = (phase - seg->d) / (seg1->d - seg->d);
        break;
      }
    }

    flength = p->elements;
    vector = p->vector;
    do {
      curval = *curtab++;
      nxtval = *nxttab++;
      *vector++ = (MYFLT) (curval + ((nxtval - curval) * partialPhase));
    } while (--flength);
    return;
}

/* ------------------------- */
void kdel_set(KDEL *p)
{
    unsigned long n;
    MYFLT *buf;
    n = (p->maxd = (long) (*p->imaxd * ekr));
    if (n == 0) n = (p->maxd = 1);

    if (!*p->istod) {
      if (p->aux.auxp == NULL || (int)(n*sizeof(MYFLT)) > p->aux.size)
        auxalloc(n * sizeof(MYFLT), &p->aux);
      else {
        buf = (MYFLT *)p->aux.auxp;
        do {
          *buf++ = FL(0.0);
        } while (--n);
      }
      p->left = 0;
    }
    return;
}

void kdelay(KDEL *p)
{
    long maxd = p->maxd, indx, v1, v2;
    MYFLT *buf = (MYFLT *)p->aux.auxp, fv1, fv2;

    if (buf==NULL) {
      initerror("vdelayk: not initialized");
      return;
    }

    indx = p->left;
    buf[indx] = *p->kin;
    fv1 = indx - *p->kdel * ekr;
    while (fv1 < FL(0.0))       fv1 += (MYFLT)maxd;
    while (fv1 >= (MYFLT)maxd) fv1 -= (MYFLT)maxd;
    if (*p->interp) { /*  no interpolation */
      *p->kr = buf[(long) fv1];
    }
    else {
      if (fv1 < maxd - 1) fv2 = fv1 + 1;
      else                fv2 = FL(0.0);
      v1 = (long)fv1;
      v2 = (long)fv2;
      *p->kr = buf[v1] + (fv1 - v1) * (buf[v2]-buf[v1]);
    }
    if (++(p->left) == maxd) p->left = 0;
    return;
}

/*  From fractals.c */
void ca_set(CELLA *p)
{
    FUNC        *ftp;
    int elements;
    MYFLT *currLine, *initVec = NULL;

    if ((ftp = ftnp2find(p->ioutFunc)) != NULL) {
      p->outVec = ftp->ftable;
      elements = (p->elements = (int) *p->ielements);
      if ( elements > ftp->flen ) {
        initerror("cella: invalid num of elements");
        return;
      }
    }
    else {
      initerror("cella: invalid output table");
      return;
    }
    if ((ftp = ftnp2find(p->initStateFunc)) != NULL) {
      initVec = (p->initVec = ftp->ftable);
      if ( elements > ftp->flen ) {
        initerror("cella: invalid num of elements");
        return;
      }
    }
    else {
      initerror("cella: invalid initial state table");
      return;
    }
    if ((ftp = ftnp2find(p->iRuleFunc)) != NULL) {
      p->ruleVec = ftp->ftable;
    }
    else {
      initerror("cella: invalid rule table");
      return;
    }
    if (p->auxch.auxp == NULL)
      auxalloc(elements * sizeof(MYFLT) * 2, &p->auxch);
    currLine = (p->currLine = (MYFLT *) p->auxch.auxp);
    p->NewOld = 0;
    p->ruleLen = (int) *p->irulelen;
    do {
      *currLine++ = *initVec++;
    } while (--elements);
    return;
}


void ca(CELLA *p)
{
    if (*p->kreinit) {
      MYFLT *currLine = p->currLine, *initVec = p->initVec;
      int elements =  p->elements;
      p->NewOld = 0;
      do {
        *currLine++ = *initVec++;
      } while (--elements);
    }
    if (*p->ktrig) {
      int j, elements = p->elements, jm1, ruleLen = p->ruleLen;
      MYFLT *actual, *previous, *outVec = p->outVec, *ruleVec = p->ruleVec;
      previous = &(p->currLine[elements * p->NewOld]);
      p->NewOld += 1;
      p->NewOld %= 2;
      actual   = &(p->currLine[elements * p->NewOld]);
      if (*p->iradius == 1) {
        for (j=0; j < elements; j++) {
          jm1 = (j < 1) ? elements-1 : j-1;
          outVec[j] = previous[j];
          actual[j] = ruleVec[ (int) (previous[jm1] + previous[j] +
                                      previous[(j+1) % elements] ) % ruleLen];
        }
      } else {
        int jm2;
        for (j=0; j < elements; j++) {
          jm1 = (j < 1) ? elements-1 : j-1;
          jm2 = (j < 2) ? elements-2 : j-2;
          outVec[j] = previous[j];
          actual[j] = ruleVec[ (int) (previous[jm2] +   previous[jm1] +
                                      previous[j] + previous[(j+1) % elements] +
                                      previous[(j+2) % elements])
                               % ruleLen];
        }
      }

    } else {
      int elements =  p->elements;
      MYFLT *actual = &(p->currLine[elements * !(p->NewOld)]), *outVec = p->outVec;
      do {
        *outVec++ = *actual++ ;
      } while (--elements);
    }
    return;
}


#define S sizeof

static OENTRY localops[] = {
  { "vtablei", S(MTABLEI),   1, "",   "iiiim", (SUBR)mtable_i,  NULL },
  { "vtablek", S(MTABLE),    3, "",   "kkkiz", (SUBR)mtable_set, (SUBR)mtable_k, NULL },
  { "vtablea", S(MTABLE),    5, "",   "akkiy", (SUBR)mtable_set, NULL, (SUBR)mtable_a },
  { "vtablewi", S(MTABLEIW), 1, "",   "iiim", (SUBR)mtablew_i,  NULL },
  { "vtablewk", S(MTABLEW),  3, "",   "kkiz", (SUBR)mtablew_set, (SUBR)mtablew_k, NULL },
  { "vtablewa", S(MTABLEW),  5, "",   "akiy", (SUBR)mtablew_set, NULL, (SUBR)mtablew_a },
  { "vtabi", S(MTABI),       1, "",   "iim", (SUBR)mtab_i,  NULL },
  { "vtabk", S(MTAB),        3, "",   "kiz", (SUBR)mtab_set, (SUBR)mtab_k, NULL },
  { "vtaba", S(MTAB),        5, "",  "aiy", (SUBR)mtab_set, NULL, (SUBR)mtab_a },
  { "vtabwi", S(MTABIW),     1, "",  "iim", (SUBR)mtabw_i,  NULL },
  { "vtabwk", S(MTABW),      3, "",  "kiz", (SUBR)mtabw_set, (SUBR)mtabw_k, NULL },
  { "vtabwa", S(MTABW),      5, "",  "aiy", (SUBR)mtabw_set, NULL, (SUBR)mtabw_a },

  { "vadd",S(VECTOROP),      3, "",  "iki", (SUBR)vectorOp_set, (SUBR) vadd    },
  { "vmult",S(VECTOROP),     3, "",  "iki", (SUBR)vectorOp_set, (SUBR) vmult   },
  { "vpow",S(VECTOROP),      3, "",  "iki", (SUBR)vectorOp_set, (SUBR) vpow    },
  { "vexp",S(VECTOROP),      3, "",  "iki", (SUBR)vectorOp_set, (SUBR) vexp    },
  { "vaddv",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR) vaddv  },
  { "vsubv",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR) vsubv  },
  { "vmultv",S(VECTORSOP),   3, "",  "iii", (SUBR)vectorsOp_set, (SUBR) vmultv },
  { "vdivv",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR)vdivv   },
  { "vpowv",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR)vpowv   },
  { "vexpv",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR) vexpv  },
  { "vcopy",S(VECTORSOP),    3, "",  "iii", (SUBR)vectorsOp_set, (SUBR)vcopy   },
  { "vcopy_i",S(VECTORSOP),  1, "",  "iii", (SUBR)vcopy_i           },
  { "vmap",S(VECTORSOP),     3, "",  "iii", (SUBR)vectorsOp_set, (SUBR)vmap    },
  { "vlimit",S(VLIMIT),      3, "",  "ikki",(SUBR)vlimit_set, (SUBR)vlimit     },
  { "vwrap",S(VLIMIT),       3, "",  "ikki",(SUBR)vlimit_set, (SUBR) vwrap     },
  { "vmirror",S(VLIMIT),     3, "",  "ikki",(SUBR)vlimit_set, (SUBR)vmirror     },
  { "vlinseg",S(VSEG),       3, "",  "iin", (SUBR)vseg_set,   (SUBR)vlinseg     },
  { "vexpseg",S(VSEG),       3, "",  "iin", (SUBR)vseg_set,   (SUBR)vexpseg     },
  { "vrandh",S(VRANDH),      3, "",  "ikki",(SUBR)vrandh_set, (SUBR) vrandh     },
  { "vrandi",S(VRANDI),      3, "",  "ikki",(SUBR)vrandi_set, (SUBR)vrandi      },
  { "vport",S(VPORT),        3, "",  "ikio",(SUBR)vport_set,  (SUBR)vport       },
  { "vecdelay",S(VECDEL),    3, "",  "iiiiio",(SUBR)vecdly_set, (SUBR)vecdly    },
  { "vdelayk",S(KDEL),       3, "k", "kkioo",(SUBR)kdel_set,  (SUBR)kdelay      },
  { "vcella",S(CELLA),       3, "",  "kkiiiiip",(SUBR)ca_set, (SUBR)ca          },

};

long opcode_size(void)
{
    return sizeof(localops);
}

OENTRY *opcode_init(GLOBALS *xx)
{
    pcglob = xx;
    return localops;
}

