/*
 *  config-table and mount-table utilities for cryptmount
 *  $Revision: 202 $, $Date: 2008-06-07 10:29:31 +0100 (Sat, 07 Jun 2008) $
 *  (C)Copyright 2005-2008, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "armour.h"
#include "cryptmount.h"
#include "delegates.h"
#include "tables.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


enum                /* config-file tokens */
{
    T_IDENT,
    T_LBRACE, T_OPT, T_RBRACE,
    T_ERROR
};


struct statfile     /* information about status file */
{
    int version;        /* file format version */
    FILE *fp;           /* file handle */
};


static cment_t *parse_stream(FILE *fp, const char *cfgname);


/*  The format for the status-file (version 0) is as follows:
    # [TITLE]           - one-line text header
    0                   - integer file-format version number
    n,[TARGET],uid      - target-name length, target-name, uid of mounter
    ...                 - further target records
*/


cment_t *alloc_cment(const cment_t *prototype)
    /* allocate storage for a target-structure */
{   cment_t *cment;

#define DFLT_OR_PROTO(dflt, proto) \
        (prototype == NULL ? (dflt) : (proto))

    cment = (cment_t*)malloc(sizeof(cment_t));

    cment->ident = NULL;
    cment->flags = DFLT_OR_PROTO(FLG_DEFAULTS, prototype->flags);

    cment->dev = NULL;
    cment->start = 0; cment->length = -1;
    cment->dir = NULL;
    cment->fstype = DFLT_OR_PROTO(NULL, cm_strdup(prototype->fstype));
    cment->fsoptions = DFLT_OR_PROTO(NULL, cm_strdup(prototype->fsoptions));
    cment->loopdev = NULL;

    cment->cipher = DFLT_OR_PROTO(cm_strdup(DFLT_CIPHER),
                                cm_strdup(prototype->cipher));
    cment->ivoffset = 0;

    cment->key.format = DFLT_OR_PROTO(NULL, cm_strdup(prototype->key.format));
    cment->key.filename = NULL;
    cment->key.digestalg = DFLT_OR_PROTO(NULL, cm_strdup(prototype->key.digestalg));
    cment->key.cipheralg = DFLT_OR_PROTO(NULL, cm_strdup(prototype->key.cipheralg));
    cment->key.maxlen = DFLT_OR_PROTO(-1L, prototype->key.maxlen);
    cment->key.retries = DFLT_OR_PROTO(1U, prototype->key.retries);

    cment->nx = NULL;

#undef DFLT_OR_PROTO

    return cment;
}


const cment_t *get_cment(const cment_t *head, const char *ident)
    /* find info-structure for target of given name */
{   const cment_t *itr,*ent=NULL;

    for (itr=head; itr!=NULL && ent==NULL; itr=itr->nx) {
        if (strcmp(ident, itr->ident) == 0) {
            ent = itr;
        }
    }

    return ent;
}


void free_cment(cment_t *cment)
    /* relinquish storage for a target-structure */
{
    if (cment->ident != NULL) free((void*)cment->ident);

    if (cment->dev != NULL) free((void*)cment->dev);
    if (cment->dir != NULL) free((void*)cment->dir);
    if (cment->fstype != NULL) free((void*)cment->fstype);
    if (cment->fsoptions != NULL) free((void*)cment->fsoptions);
    if (cment->loopdev != NULL) free((void*)cment->loopdev);

    if (cment->cipher != NULL) free((void*)cment->cipher);

    if (cment->key.format != NULL) free((void*)cment->key.format);
    if (cment->key.filename != NULL) free((void*)cment->key.filename);
    if (cment->key.digestalg != NULL) free((void*)cment->key.digestalg);
    if (cment->key.cipheralg != NULL) free((void*)cment->key.cipheralg);

    free((void*)cment);
}


static void append(char c, char **buff, unsigned *pos, unsigned *bufflen)
    /* append character to string, reallocating memory as necessary */
{   unsigned newlen;

    if (*pos >= *bufflen) {
        newlen = (*bufflen) * 2 + 64;
        *buff = (char*)realloc(*buff, (size_t)newlen);
        *bufflen = newlen;
    }

    (*buff)[*pos] = c;
    ++(*pos);
}


static int proc_string(void *ptr, const char *src)
    /* process var=val entry in config-table for "string" type */
{   char **addr=(char**)ptr;

    *addr = (char*)realloc(*addr, (size_t)(strlen(src) + 1));
    strcpy(*addr, src);

    return (*addr == NULL);
}


#define PROC_VARVAL(FN_NAME, TYPE, FMT)                         \
    static int FN_NAME(void *ptr, const char *src)              \
        /* process var=val entry in config-table for "TYPE" */  \
    {   TYPE qv, *addr=(TYPE*)ptr;                              \
        if (sscanf(src, FMT, &qv) == 1) {                       \
            *addr = qv; return 0;                               \
        }                                                       \
        return 1;                                               \
    }

PROC_VARVAL(proc_unsigned, unsigned, "%u")
PROC_VARVAL(proc_long, long, "%ld")
PROC_VARVAL(proc_int64, int64_t, "%" SCNi64)

#undef PROC_VARVAL


int proc_flags(void *ptr, const char *src)
    /* convert string of configuration-switches into binary flags */
{   unsigned *flags=(unsigned*)ptr;
    struct opt_t {
        const char *str;
        unsigned andmask, ormask; };
    struct opt_t opts[] = {
        { "defaults",   0U,             FLG_DEFAULTS },
        { "user",       ~0U,            FLG_USER },
        { "nouser",     ~FLG_USER,      0U },
        { "fsck",       ~0U,            FLG_FSCK },
        { "nofsck",     ~FLG_FSCK,      0U },
        { "mkswap",     ~0U,            FLG_MKSWAP },
        { "nomkswap",   ~FLG_MKSWAP,    0U },
        { NULL, ~0U, 0U } };
    unsigned idx,len;

    if (src == NULL) return 0;

    for (;;) {
        for (len=0; src[len]!='\0' && src[len]!=','; ++len);
        for (idx=0; opts[idx].str!=NULL; ++idx) {
            if (strncmp(src, opts[idx].str, (size_t)len) == 0) {
                *flags = (*flags & opts[idx].andmask) | opts[idx].ormask;
                break;
            }
        }
        if (opts[idx].str == NULL) {
            fprintf(stderr, "bad option \"%s\"\n", src);
            return 1;
        }

        if (src[len] == '\0') break;
        src += len + 1;
    }

    return 0;
}



static void read_token(char *buff, unsigned *t_state, cment_t *cment)
    /* process token (word) from configuration-file while parsing */
{   struct tokinfo_t {
        const char *name;
        int varoffset;
        int (*proc)(void *var, const char *val);
    } *tok;
#define OFFSET(x) (int)((char*)&((cment_t*)NULL)->x - (char*)NULL)
    struct tokinfo_t toktable[] = {
        { "flags",          OFFSET(flags),          proc_flags },
        { "dev",            OFFSET(dev),            proc_string },
        { "dir",            OFFSET(dir),            proc_string },
        { "startsector",    OFFSET(start),          proc_int64 },
        { "numsectors",     OFFSET(length),         proc_int64 },
        { "fstype",         OFFSET(fstype),         proc_string },
        { "fsoptions",      OFFSET(fsoptions),      proc_string },
        { "loop",           OFFSET(loopdev),        proc_string },
        { "cipher",         OFFSET(cipher),         proc_string },
        { "ivoffset",       OFFSET(ivoffset),       proc_int64 },
        { "keyformat",      OFFSET(key.format),     proc_string },
        { "keyfile",        OFFSET(key.filename),   proc_string },
        { "keyhash",        OFFSET(key.digestalg),  proc_string },
        { "keycipher",      OFFSET(key.cipheralg),  proc_string },
        { "keymaxlen",      OFFSET(key.maxlen),     proc_long },
        { "passwdretries",  OFFSET(key.retries),    proc_unsigned },
        { NULL, 0, NULL }
    };
#undef OFFSET
    char *eq;

    switch (*t_state) {
        case T_IDENT:
            (void)proc_string((void*)&cment->ident, buff);
            *t_state = T_LBRACE;
            break;
        case T_LBRACE:
            *t_state = (strcmp(buff, "{") == 0 ? T_OPT : T_ERROR);
            break;
        case T_OPT:
            if (strcmp(buff, "}") == 0) {
                if (cment->key.format == NULL
                  && cment->key.cipheralg != NULL
                  && strcmp(cment->key.cipheralg, "none") == 0) {
                    fprintf(stderr, _("cryptmount: using \"keycipher=none\" is deprecated - please use \"keyformat=raw\" instead\n"));
                    cment->key.format = cm_strdup("raw");
                }
                *t_state = T_RBRACE; break;
            }
            if ((eq = strchr(buff,'=')) == NULL) {
                *t_state = T_ERROR; break;
            }
            *eq = '\0'; ++eq;

            /* delegate processing to specific token-processor from table: */
            for (tok=toktable; tok->name!=NULL; ++tok) {
                if (strcmp(tok->name, buff) == 0) {
                    (*tok->proc)((void*)((char*)cment + tok->varoffset), eq);
                    break;
                }
            }
            if (tok->name == NULL) {
                fprintf(stderr, _("unrecognized option \"%s\"\n"), buff);
                *t_state = T_ERROR;
            }
            break;
        default:
            break;
    }
}


cment_t *parse_stream(FILE *fp, const char *cfgname)
    /* convert config-file information into linked-list of target-structures */
{   enum { C_SPACE, C_WORD, C_COMMENT };
    int ch,lineno=1,literal=0;
    unsigned pos=0,bufflen=0;
    unsigned c_state,t_state;
    char *buff=NULL;
    cment_t *head=NULL, **cmp=&head, *cment=NULL, *prototype=NULL;

    c_state = C_SPACE;
    t_state = T_IDENT;
    cment = alloc_cment(prototype);

    while (!feof(fp) && t_state != T_ERROR) {
        ch = fgetc(fp);
        if (ch == (int)'\n') ++lineno;

        if (literal && ch == (int)'\n') {    /* ignore escaped end-of-line */
            literal = 0;
            continue;
        }

        if (!literal && ch == (int)'#') c_state = C_COMMENT;
        if (!literal && ch == (int)'\\') {
            /* treat next character literally */
            literal = 1;
            continue;
        }

        switch (c_state) {
            case C_SPACE:
                if (literal || !isspace(ch)) {
                    pos = 0;
                    append((char)ch, &buff, &pos, &bufflen);
                    c_state = C_WORD;
                }
                break;
            case C_WORD:
                if (literal || !isspace(ch)) {
                    append((char)ch, &buff, &pos, &bufflen);
                } else {
                    append('\0', &buff, &pos, &bufflen);
                    read_token(buff, &t_state, cment);
                    c_state = C_SPACE;
                }
                break;
            case C_COMMENT:
                if (!literal && ch == '\n') c_state = C_SPACE;
                break;
            default:
                break;
        }

        literal = 0;
        if (t_state == T_RBRACE) {
            /* parsing has reached end of target definition */

            if (strcmp(cment->ident, "_DEFAULTS_") == 0) {
                /* new target definition is set of default values */
                if (prototype != NULL) free_cment(prototype);
                prototype = cment;
            } else {
                /* new target definition is genuine target */
                *cmp = cment;
                cmp = &cment->nx;
            }

            cment = alloc_cment(prototype);
            t_state = T_IDENT;
        }
    }

    if (t_state == T_ERROR) {
        fprintf(stderr, _("configuration error near %s:%d\n"),
                cfgname, lineno);
    }

    if (prototype != NULL) free_cment(prototype);
    if (cment != NULL) free_cment(cment);
    if (buff != NULL) free((void*)buff);

    return head;
}


cment_t *parse_config(const char *cfgname)
    /* convert config-file into linked-list of target-structures */
{   FILE *fp;
    cment_t *head=NULL;

    fp = fopen(cfgname, "r");
    if (fp == NULL) {
        fprintf(stderr, "failed to open \"%s\"\n", cfgname);
        return NULL;
    }

    head = parse_stream(fp, cfgname);
    fclose(fp);

    return head;
}


cment_t *parse_config_fd(int fd)
    /* convert input-stream config-data into target-structures */
{   FILE *fp=NULL;
    cment_t *head=NULL;
    char label[64];

    fp = fdopen(fd, "r");
    if (fp == NULL) {
        fprintf(stderr, "failed to read input-stream %d\n", fd);
        return NULL;
    }

    snprintf(label, sizeof(label), "stream-%d", fd);
    head = parse_stream(fp, label);

    return head;
}


void free_config(cment_t **head)
    /* free all entries in target-config list */
{   cment_t *cmx;

    if (head == NULL) return;

    while ((cmx = *head) != NULL) {
        *head = cmx->nx;
        free_cment(cmx);
    }
}



tgtstat_t *alloc_tgtstatus(const struct cment *ent)
    /* create new status record for given target */
{   tgtstat_t *ts;

    ts = (tgtstat_t*)malloc(sizeof(tgtstat_t));
    ts->ident = NULL;
    ts->uid = 0;
    ts->nx = NULL;

    if (ent != NULL) {
        ts->ident = (char*)malloc((size_t)(strlen(ent->ident) + 1));
        strcpy(ts->ident, ent->ident);
    }

    return ts;
}


void free_tgtstatus(tgtstat_t *ts)
    /* free storage of target-status record (or list thereof) */
{   tgtstat_t *tx=NULL;

    while ((tx = ts) != NULL) {
        ts = tx->nx;
        if (tx->ident != NULL) free((void*)tx->ident);
        free((void*)tx);
    }
}


struct statfile *statfile_open(const char *fname, const char *mode)
    /* prepare status-file for reading/writing */
{   struct statfile *sf=NULL;
    char buff[256];
    FILE *fp;

    if ((fp = fopen(fname, mode)) != NULL) {
        sf = (struct statfile*)malloc(sizeof(struct statfile));
        sf->fp = fp;

        if (mode[0] == 'w') {
            sf->version = 0;        /* format-version for new files */
            fprintf(fp,"# auto-generated by cryptmount - do not edit\n");
            fprintf(fp, "%d\n", sf->version);
        } else {
            (void)fgets(buff, (int)sizeof(buff), fp);
            fscanf(fp, "%d", &sf->version);
        }
    }

    return sf;
}


tgtstat_t *statfile_read(struct statfile *sf)
    /* read information about next target from status-file */
{   tgtstat_t *ts=NULL;
    char *ident=NULL;
    int len;
    unsigned long uid;

    if (sf == NULL) goto bail_out;

    if (fscanf(sf->fp, "%d,", &len) != 1) goto bail_out;

    ident = (char*)malloc((size_t)(len + 1));
    if (fread((void*)ident, (size_t)(len + 1), (size_t)1, sf->fp) != 1) {
        goto bail_out;
    }
    ident[len] = '\0';

    if (fscanf(sf->fp, "%lu", &uid) != 1) goto bail_out;
    if (feof(sf->fp)) goto bail_out;

    ts = alloc_tgtstatus(NULL);
    ts->ident = ident;
    ts->uid = uid;
    ident = NULL;

  bail_out:

    if (ident != NULL) free((void*)ident);

    return ts;
}


void statfile_write(struct statfile *sf, const tgtstat_t *stat)
{
    fprintf(sf->fp, "%u,", (unsigned)strlen(stat->ident));
    fprintf(sf->fp, "%s,", stat->ident);
    fprintf(sf->fp, "%lu\n", stat->uid);
}


void statfile_close(struct statfile *sf)
{
    fclose(sf->fp);
    free((void*)sf);
}


tgtstat_t *get_tgtstatus(const cment_t *cment)
    /* find mount/owner status of given target */
{   char *fname=NULL;
    tgtstat_t *ts=NULL;
    struct statfile *sf;
    int badlock;

    (void)cm_path(&fname, "cmstatus");
    badlock = cm_mutex_lock();
    sf = statfile_open(fname, "r");
    if (sf == NULL) goto bail_out;

    while ((ts = statfile_read(sf)) != NULL) {
        if (strcmp(cment->ident, ts->ident) == 0) break;
        else free_tgtstatus(ts);
    }

    statfile_close(sf);

  bail_out:

    if (!badlock) cm_mutex_unlock();
    if (fname != NULL) free((void*)fname);

    return ts;
}


tgtstat_t *get_all_tgtstatus()
    /* find list of mount/owner status for all mounted targets */
{   char *fname=NULL;
    tgtstat_t *ts=NULL, *head=NULL, **sfp=&head;
    struct statfile *sf;
    int badlock;

    (void)cm_path(&fname, "cmstatus");
    badlock = cm_mutex_lock();
    sf = statfile_open(fname, "r");
    if (sf == NULL) goto bail_out;

    while ((ts = statfile_read(sf)) != NULL) {
        *sfp = ts;
        sfp = &ts->nx;
    }

    statfile_close(sf);

  bail_out:

    if (!badlock) cm_mutex_unlock();
    if (fname != NULL) free((void*)fname);

    return head;
}


int put_tgtstatus(const cment_t *cment, const tgtstat_t *newstat)
{   char *newfname=NULL,*oldfname=NULL;
    struct statfile *sfin,*sfout;
    tgtstat_t *ts=NULL;
    struct stat sbuff;
    int badlock, eflag=0;

    (void)cm_path(&oldfname, "cmstatus");
    (void)cm_path(&newfname, "cmstatus-temp");
    badlock = cm_mutex_lock();

    sfout = statfile_open(newfname, "w");
    if (sfout == NULL) {
        eflag = 1; goto bail_out;
    }

    if (stat(oldfname, &sbuff) == 0) {
        sfin = statfile_open(oldfname, "r");
        if (sfin == NULL) {
            statfile_close(sfout);
            unlink(newfname);
            eflag = 1; goto bail_out;
        }

        /* copy most entries from existing status-file: */
        while ((ts = statfile_read(sfin)) != NULL) {
            if (strcmp(cment->ident, ts->ident) != 0) {
                statfile_write(sfout, ts);
            }
            free_tgtstatus(ts);
        }
        statfile_close(sfin);
    }

    /* add new entry onto end of new status-file: */
    if (newstat != NULL) {
        statfile_write(sfout, newstat);
    }
    statfile_close(sfout);

    /* overwrite old status-file: */
    rename(newfname, oldfname);
    chown(oldfname, (uid_t)0, (gid_t)0);
    chmod(oldfname, S_IWUSR|S_IRUSR | S_IRGRP | S_IROTH);

  bail_out:

    if (!badlock) cm_mutex_unlock();
    if (newfname != NULL) free((void*)newfname);
    if (oldfname != NULL) free((void*)oldfname);

    return eflag;
}


/*
 *  tables.c
 *  (C)Copyright 2005-2008, RW Penney
 */
