/* help.c - core analysis suite
 *
 * Copyright (C) 1999, 2000, 2001, 2002 Mission Critical Linux, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * 11/09/99, 1.0    Initial Release
 * 11/12/99, 1.0-1  Bug fixes
 * 12/10/99, 1.1    Fixes, new commands, support for v1 SGI dumps
 * 01/18/00, 2.0    Initial gdb merger, support for Alpha
 * 02/01/00, 2.1    Bug fixes, new commands, options, support for v2 SGI dumps
 * 02/29/00, 2.2    Bug fixes, new commands, options
 * 04/11/00, 2.3    Bug fixes, new command, options, initial PowerPC framework
 * 04/12/00  ---    Transition to BitKeeper version control
 * 
 * BitKeeper ID: @(#)help.c 1.21
 *
 * 09/28/00  ---    Transition to CVS version control
 *
 * CVS: $Revision: 1.118 $ $Date: 2002/02/18 18:48:01 $
 */

#include "defs.h"

static void reshuffle_cmdlist(void);
static int sort_command_name(const void *, const void *);
static void display_help_screen(char *);
static void display_commands(void);
static void display_copying_info(void);
static void display_warranty_info(void);
static void display_output_info(void);
static void display_input_info(void);
static void display_README(void);
static char *gnu_public_license[];
static char *version_info[];
static char *output_info[];
static char *input_info[];
static char *README[];

/*
 *  Get an updated count of commands for subsequent help menu display,
 *  reshuffling the deck if this is the first time or if something's changed.
 */
void
help_init(void)
{
        struct command_table_entry *cp;
	struct extension_table *ext;

	for (pc->ncmds = 0, cp = &base_command_table[0]; cp->name; cp++) {
		if (!(cp->flags & HIDDEN_COMMAND))
                	pc->ncmds++;
	}

        for (ext = extension_table; ext; ext = ext->next) {
		for (cp = ext->command_table; cp->name; cp++)
			pc->ncmds++;
	}

        if (!pc->cmdlist) {
		pc->cmdlistsz = pc->ncmds;        
        	if ((pc->cmdlist = (char **)
                	malloc(sizeof(char *) * pc->cmdlistsz)) == NULL)
                        	error(FATAL,
                                    	"cannot malloc command list space\n");
	} else if (pc->ncmds > pc->cmdlistsz) {
		pc->cmdlistsz = pc->ncmds;
		if ((pc->cmdlist = (char **)realloc(pc->cmdlist,
                	sizeof(char *) * pc->cmdlistsz)) == NULL)
				error(FATAL, 
					"cannot realloc command list space\n");
	}

	reshuffle_cmdlist();
}

/*
 *  If the command list is modified during runtime, re-shuffle the list
 *  for proper help menu display.
 */
static void
reshuffle_cmdlist(void)
{
	int i, cnt;
        struct command_table_entry *cp;
	struct extension_table *ext;

	for (i = 0; i < pc->cmdlistsz; i++) 
		pc->cmdlist[i] = NULL;

        for (cnt = 0, cp = &base_command_table[0]; cp->name; cp++) {
		if (!(cp->flags & HIDDEN_COMMAND))
                	pc->cmdlist[cnt++] = cp->name;
	}

        for (ext = extension_table; ext; ext = ext->next) {
                for (cp = ext->command_table; cp->name; cp++)
                	pc->cmdlist[cnt++] = cp->name;
        }

	if (cnt > pc->cmdlistsz)
		error(FATAL, "help table malfunction!\n");

        qsort((void *)pc->cmdlist, (size_t)cnt,
                sizeof(char *), sort_command_name);
}


/*
 *  The help list is in alphabetical order, with exception of the "q" command,
 *  which has historically always been the last command in the list.
 */

static int
sort_command_name(const void *name1, const void *name2)
{
	char **s1, **s2;

	s1 = (char **)name1;
	s2 = (char **)name2;

	if (STREQ(*s1, "q"))  
		return 1;

	return strcmp(*s1, *s2);
}


/*
 *  Get help for a command, to dump an internal table, or the GNU public
 *  license copying/warranty information.
 */
void
cmd_help(void)
{
	int c;
	int oflag;

	oflag = 0;

        while ((c = getopt(argcnt, args, "efDdmngcaBbHhksvVoptTzLx")) != EOF) {
                switch(c)
                {
		case 'e':
			dump_extension_table(VERBOSE);
			return;

		case 'f':
			dump_filesys_table(VERBOSE);
			return;

		case 'D':
			dumpfile_memory(DUMPFILE_MEM_DUMP);
			return;

		case 'x':
			dump_text_value_cache(VERBOSE);
			return;

		case 'd':
			dump_dev_table();
			return;

		case 'm':
			dump_machdep_table();
			return;

		case 'g':
			dump_gdb_data();
			return;

		case 'n':
			dump_net_table();
			return;

		case 'a':
			dump_alias_data();
			return;

		case 'b':
			dump_shared_bufs();
			return;

		case 'B':
			dump_build_data();
			return;

		case 'c':
			dump_numargs_cache();
			return;

		case 'H':
			dump_hash_table(VERBOSE);
			return;

		case 'h':
			dump_hash_table(!VERBOSE);
 			return;

		case 'k':
			dump_kernel_table();
			return;

		case 's':
			dump_symbol_table();
			return;

		case 'V':
			dump_vm_table(VERBOSE);
			return;

		case 'v':
			dump_vm_table(!VERBOSE);
			return;

		case 'o':
			oflag = TRUE;
			break;

		case 'T':
			dump_task_table(VERBOSE);
			return;

		case 't':
			dump_task_table(!VERBOSE);
			return;

		case 'p':
			dump_program_context();
			return;

		case 'z':
			fprintf(fp, "help options:\n");
			fprintf(fp, " -e - extension table data\n");
			fprintf(fp, " -a - alias data\n");
			fprintf(fp, " -b - shared buffer data\n");
			fprintf(fp, " -B - build data\n");
			fprintf(fp, " -c - numargs cache\n");
			fprintf(fp, " -d - device table\n");
			fprintf(fp, " -D - dumpfile memory usage\n");
			fprintf(fp, " -f - filesys table\n");
			fprintf(fp, " -k - kernel_table\n");
			fprintf(fp, " -m - machdep_table\n");
			fprintf(fp, " -s - symbol table data\n");
			fprintf(fp, " -v - vm_table\n");
			fprintf(fp, " -V - vm_table (verbose)\n");
			fprintf(fp, " -o - offset_table and size_table\n");
			fprintf(fp, " -t - task_table\n");
			fprintf(fp, " -x - text cache\n");
			fprintf(fp, " -T - task_table plus context_array\n");
			fprintf(fp, " -p - program_context\n");
			fprintf(fp, " -h - hash_table data\n");
			fprintf(fp, " -H - hash_table data (verbose)\n");
			fprintf(fp, " -L - LKCD page cache environment\n");
			return;

		case 'L':
			dumpfile_memory(DUMPFILE_ENVIRONMENT);
			return;

                default:  
			argerrs++;
                        break;
                }
        }

        if (argerrs)
                cmd_usage(pc->curcmd, COMPLETE_HELP);

	if (!args[optind]) {
		if (oflag) 
			dump_offset_table(NULL);
		else 
			display_help_screen("");
		return;
	}

        do {
		if (oflag) 
			dump_offset_table(args[optind]);
		else	
        		cmd_usage(args[optind], COMPLETE_HELP);
		optind++;
        } while (args[optind]);
}

/*
 *  Format and display the help menu.
 */

static void
display_help_screen(char *indent)
{
        int i, j, rows;
	char **namep;

	help_init();

	fprintf(fp, "\n%s", indent);

        rows = (pc->ncmds + (HELP_COLUMNS-1)) / HELP_COLUMNS;

        for (i = 0; i < rows; i++) {
                namep = &pc->cmdlist[i];
                for (j = 0; j < HELP_COLUMNS; j++) {
                        fprintf(fp,"%-15s", *namep);
                        namep += rows;
                        if ((namep - pc->cmdlist) >= pc->ncmds)
                                break;
                }
                fprintf(fp,"\n%s", indent);
        }

        fprintf(fp, "\n%s%s version: %-6s   gdb version: %s\n", indent,
		pc->program_name, pc->program_version, pc->gdb_version); 
	fprintf(fp,
		"%sFor help on any command above, enter \"help <command>\".\n",
		indent);
	fprintf(fp, "%sFor help on input options, enter \"help input\".\n",
		indent);
	fprintf(fp, "%sFor help on output options, enter \"help output\".\n",
		indent);
	fprintf(fp, "%sFor the most recent version: "
		    "http://www.missioncriticallinux.com/download\n\n", indent);
}

/*
 *  Used for generating HTML pages, dump the commands in the order
 *  they would be seen on the help menu, i.e., from left-to-right, row-by-row.
 *  Line ends are signaled with a "BREAK" string.
 */
static void
display_commands(void)
{
        int i, j, rows;
	char **namep;

	help_init();
        rows = (pc->ncmds + (HELP_COLUMNS-1)) / HELP_COLUMNS;

        for (i = 0; i < rows; i++) {
                namep = &pc->cmdlist[i];
                for (j = 0; j < HELP_COLUMNS; j++) {
                        fprintf(fp,"%s\n", *namep);
                        namep += rows;
                        if ((namep - pc->cmdlist) >= pc->ncmds) {
                                fprintf(fp, "BREAK\n");
                                break;
                        }
                }
        }
}


/*
 *  Help data for a command must be formatted using the following template:
 
"command-name",
"command description line", 
"argument-usage line",
"description...",
"description...",
"description...",
NULL,
 
 *  The first line is concatenated with the second line, and will follow the
 *  help command's "NAME" header.
 *  The first and third lines will also be concatenated, and will follow the
 *  help command's "SYNOPSIS" header.  If the command has no arguments, enter
 *  a string consisting of a space, i.e., " ".
 *  The fourth and subsequent lines will follow the help command's "DESCRIPTION"
 *  header.
 *
 *  The program name can be referenced by using the %%s format.  The final
 *  entry in each command's help data string list must be a NULL.
 */


char *help_foreach[] = {
"foreach",
"display command data for multiple tasks in the system",
"[[pid | taskp | name | [kernel | user]] ...] command [flag] [argument]",
"  This command allows for a an examination of various kernel data associated",
"  with any, or all, active tasks in the system, without having to set the",
"  context to each targeted task.\n",
"      pid  perform the command(s) on this PID.",
"    taskp  perform the command(s) on task referenced by this hexadecimal", 
"           task_struct pointer.",
"     name  perform the command(s) on all commands with this name.  If the",
"           command name can be confused with a foreach command name, then",
"           precede the name string with a \"\\\".",
"     user  perform the command(s) on all user (non-kernel) threads.",
"   kernel  perform the command(s) on all kernel threads.",
"   active  perform the command(s) on all active threads.\n",
"  If none of the task-identifying arguments above are entered, the command",
"  will be performed on all tasks.\n",
"  command  select one or more of the following commands on the tasks",
"           selected, or on all tasks:\n",
"             bt  same as the \"bt\" command  (optional flags: -r -t -l -e -R)",
"             vm  same as the \"vm\" command  (optional flags: -p -v -m -R)",
"           task  same as the \"task\" command  (optional flag: -R)",
"          files  same as the \"files\" command  (optional flag: -R)",
"            net  same as the \"net\" command  (optional flags: -s -S -R)",
"            set  same as the \"set\" command",
"            sig  same as the \"sig\" command",
"           vtop  same as the \"vtop\" command  (optional flags: -c -u -k)\n",
"     flag  Pass this optional flag to the command selected.",
" argument  Pass this argument to the command selected.",
" ",
"  A header containing the PID, task address, cpu and command name will be",
"  pre-pended before the command output for each selected task.  Consult the",
"  help page of each of the command types above for details.",
"\nEXAMPLES",
"  Display the stack traces for all tasks:\n",
"    %s> foreach bt",
"    PID: 4752   TASK: c7680000  CPU: 1   COMMAND: \"xterm\"",
"     #0 [c7681edc] schedule at c01135f6",
"        (void)",
"     #1 [c7681f34] schedule_timeout at c01131ff",
"        (24)",
"     #2 [c7681f64] do_select at c0132838",
"        (5, c7681fa4, c7681fa0)",
"     #3 [c7681fbc] sys_select at c0132dad",
"        (5, 8070300, 8070380, 0, 0)",
"     #4 [bffffb0c] system_call at c0109944",
"        EAX: 0000008e  EBX: 00000005  ECX: 08070300  EDX: 08070380 ",
"        DS:  002b      ESI: 00000000  ES:  002b      EDI: 00000000 ",
"        SS:  002b      ESP: bffffadc  EBP: bffffb0c ",
"        CS:  0023      EIP: 402259ee  ERR: 0000008e  EFLAGS: 00000246 ",
"    ",
"    PID: 557    TASK: c5600000  CPU: 0   COMMAND: \"nfsd\"",
"     #0 [c5601f38] schedule at c01135f6",
"        (void)",
"     #1 [c5601f90] schedule_timeout at c01131ff",
"        (c5600000)",
"     #2 [c5601fb8] svc_recv at c805363a",
"        (c0096f40, c5602800, 7fffffff, 100, c65c9f1c)",
"     #3 [c5601fec] (nfsd module) at c806e303",
"        (c5602800, c5602800, c0096f40, 6c6e0002, 50)",
"     #4 [c65c9f24] kernel_thread at c010834f",
"        (0, 0, ext2_file_inode_operations)",
"    ",
"    PID: 824    TASK: c7c84000  CPU: 0   COMMAND: \"mingetty\"",
"    ...\n",
"  Display the task_struct structure for each \"bash\" command:\n",
"    %s> foreach bash task",
"    ...\n",
"  Display the open files for all tasks:\n",
"    %s> foreach files",
"    ...\n",
NULL
};

char *help_ascii[] = {
"ascii",
"translate a hexadecimal string to ASCII",
"value ...",
"  Translates 32-bit or 64-bit hexadecimal values to ASCII.  If no argument",
"  is entered, an ASCII chart is displayed.",
"\nEXAMPLES",
"  Translate the hexadecimal value of 0x62696c2f7273752f to ASCII:",
"\n    %s> ascii 62696c2f7273752f",
"    62696c2f7273752f: /usr/lib",
"\n  Display an ASCII chart:",
"\n    %s> ascii",
" ",  
"          0    1   2   3   4   5   6   7",
"        +-------------------------------",
"      0 | NUL DLE  SP  0   @   P   '   p",
"      1 | SOH DC1  !   1   A   Q   a   q",
"      2 | STX DC2  \"   2   B   R   b   r",
"      3 | ETX DC3  #   3   C   S   c   s",
"      4 | EOT DC4  $   4   D   T   d   t",
"      5 | ENQ NAK  \%   5   E   U   e   u",
"      6 | ACK SYN  &   6   F   V   f   v",
"      7 | BEL ETB  `   7   G   W   g   w",
"      8 |  BS CAN  (   8   H   X   h   x",
"      9 |  HT  EM  )   9   I   Y   i   y",
"      A |  LF SUB  *   :   J   Z   j   z",
"      B |  VT ESC  +   ;   K   [   k   {",
"      C |  FF  FS  ,   <   L   \\   l   |",
"      D |  CR  GS  _   =   M   ]   m   }",
"      E |  SO  RS  .   >   N   ^   n   ~",
"      F |  SI  US  /   ?   O   -   o  DEL",
NULL
};

char *help_quit[] = {
"quit",
"exit this session",
" ",
"  Bail out of the current %s session.",
"\nNOTE",
"  This command is equivalent to the \"exit\" command.",
NULL
};

char *help_exit[] = {
"exit",
"exit this session",
" ",
"  Bail out of the current %s session.",
"\nNOTE",
"  This command is equivalent to the \"q\" command.",
NULL            
};

char *help_help[] = { 
"help",
"get help",
"[command | all]",
"  When entered with no argument, a list of all currently available %s",
"  commands is listed.  If a name of a %s command is entered, a man-like",
"  page for the command is displayed.  If \"all\" is entered, help pages",
"  for all commands will be displayed.  If neither of the above is entered,",
"  the argument string will be passed on to the gdb help command.",
NULL               
};

char *help_set[] = {
"set",
"set a process context or internal %s variable",
"[pid | taskp | [-c cpu] | -p] | [%s_variable [setting]] ...",
"  This command either sets a new context, or gets the current context for",
"  display.  The context can be set by the use of:\n",
"      pid  a process PID.",
"    taskp  a hexadecimal task_struct pointer.",
"   -c cpu  sets the context to the active task on a cpu (dumpfiles only).",
"       -p  sets the context to the panic task, or back to the %s task on",
"           a live system.",
"",
"  If no argument is entered, the current context is displayed.  The context",
"  consists of the PID, the task pointer, the CPU, and task state.",
"  ",
"  This command may also be used to set internal %s variables.  If no value",
"  argument is entered, the current value of the %s variable is shown.  These",
"  are the %s variables, acceptable arguments, and purpose:\n",
"       scroll  on | off     controls output scrolling.",
"       silent  on | off     controls %s prompt (and scrolling if off).",
"        radix  10 | 16      sets output radix to 10 or 16.",
"      refresh  on | off     controls internal task list refresh.",
"         hash  on | off     controls internal list verification.",
"           vi  (no arg)     set editing mode to vi (from .%src file only).",
"        emacs  (no arg)     set editing mode to emacs (from .%src file only).",
"     namelist  filename     name of kernel (from .%src file only).",
"     dumpfile  filename     name of core dumpfile (from .%src file only).",
"      console  device-name  sets debug console device.",
"        debug  number       sets %s debug level.",
"         core  (no arg)     toggles core flag -- if on, drops core when the",
"                            next non-fatal error message is printed.\n",
"  Internal variables may be set in four manners:\n",
"    1. entering the set command in $HOME/.%src.",
"    2. entering the set command in .%src in the current directory.",
"    3. executing an input file containing the set command.",
"    4. during runtime with this command.\n",
"  During initialization, $HOME/.%src is read first, followed by the",
"  .%src file in the current directory.  Set commands in the .%src file",
"  in the current directory override those in $HOME/.%src.  Set commands ",
"  entered with this command or by runtime input file override those",
"  defined in either .%src file.  Multiple set command arguments or argument",
"  pairs may be entered in one command line.",
"\nEXAMPLES",
"  Set the current context to task c2fe8000:\n",
"    %s> set c2fe8000",
"         PID: 15917",
"     COMMAND: \"bash\"",
"        TASK: c2fe8000  ",
"         CPU: 0",
"       STATE: TASK_INTERRUPTIBLE\n",
"  Show the current context:\n",
"    %s> set",
"         PID: 1525",
"     COMMAND: \"bash\"",
"        TASK: c1ede000",
"         CPU: 0",
"       STATE: TASK_INTERRUPTIBLE\n",
"  Set the context back to the panicking task:\n",
"    %s> set -p",
"         PID: 698",
"     COMMAND: \"gen12\"",
"        TASK: f9d78000",
"         CPU: 2",
"       STATE: TASK_RUNNING (PANIC)\n",
"  Turn off output scrolling:\n",
"    %s> set scroll off",
"    scroll: off",
NULL               
};

char *help_p[] = {
"p",
"print the value of an expression",
"expression",
"  This command passes its arguments on to gdb \"print\" command for evaluation.",
"",
"    expression   The expression to be evaluated.\n",
"  The default output format is decimal, but that can be changed at any time",
"  with the two built-in aliases \"hex\" and \"dec\".  Alternatively, there",
"  are two other built-in aliases, \"px\" and \"pd\", which force the command",
"  output to be displayed in hexadecimal or decimal, without changing the",
"  default mode. ",
"\nEXAMPLES",
"  Print the contents of jiffies:\n",
"    %s> p jiffies",
"    jiffies = $6 = 166532620",
"    %s> px jiffies",
"    jiffies = $7 = 0x9ed174b",
"    %s> pd jiffies",
"    jiffies = $8 = 166533160",
" ",
"  Print the contents of the vm_area_struct \"init_mm\":\n", 
"    %s> p init_mm",
"    init_mm = $5 = {",
"      mmap = 0xc022d540, ",
"      mmap_avl = 0x0, ",
"      mmap_cache = 0x0, ",
"      pgd = 0xc0101000, ",
"      count = {",
"        counter = 0x6",
"      }, ",
"      map_count = 0x1, ",
"      mmap_sem = {",
"        count = {",
"          counter = 0x1",
"        }, ",
"        waking = 0x0, ",
"        wait = 0x0",
"      }, ",
"      context = 0x0, ",
"      start_code = 0xc0000000, ",
"      end_code = 0xc022b4c8, ",
"      start_data = 0x0, ",
"      end_data = 0xc0250388, ",
"      start_brk = 0x0, ",
"      brk = 0xc02928d8, ",
"      start_stack = 0x0, ",
"      arg_start = 0x0, ",
"      arg_end = 0x0, ",
"      env_start = 0x0, ",
"      env_end = 0x0, ",
"      rss = 0x0, ",
"      total_vm = 0x0, ",
"      locked_vm = 0x0, ",
"      def_flags = 0x0, ",
"      cpu_vm_mask = 0x0, ",
"      swap_cnt = 0x0, ",
"      swap_address = 0x0, ",
"      segments = 0x0",
"    }",
NULL               
};

char *help_ps[] = {
"ps",
"display process status information",
"[-k|-u][-s][-p|-c|-t] [pid | taskp | command] ...",
"  This command displays process status for selected, or all, processes" ,
"  in the system.  If no arguments are entered, the process data is",
"  is displayed for all processes.  Selected process identifiers can be",
"  entered in the following forms:\n",
"       pid  a process PID.",
"     taskp  a hexadecimal task_struct pointer.",
"   command  a command name.  If a command name is made up of letters that",
"            are all numerical values, precede the name string with a \"\\\".",
"        -k  select all kernel threads.",
"        -u  select all user tasks.",
" ",
"  The process identifier types may be mixed.  For each task, the following",
"  items are displayed:",
" ",
"    1. the process PID.",
"    2. the parent process PID.",
"    3. the CPU number that the task ran on last.",
"    4. the task_struct address or the kernel stack pointer of the process.",
"       (see -s option below)",
"    5. the task state.",
"    6. the percentage of physical memory being used by this task.",
"    7. the virtual address size of this task in kilobytes.",
"    8. the resident set size of this task in kilobytes.",
"    9. the command name.",
" ",
"  The default output shows the task_struct address of each process under a",
"  column titled \"TASK\".  This can be changed to show the kernel stack ",
"  pointer under a column titled \"KSTACKP\".",
" ",
"       -s  replace the TASK column with the KSTACKP column.",
" ",
"  On SMP machines, the active task on each CPU will be highlighted by an",
"  angle bracket (\">\") preceding its information.",
" ",
"  Alternatively, information regarding parent-child relationships, or",
"  per-task time usage data may be displayed:",
" ",
"       -p  display the parental hierarchy of selected, or all, tasks.",  
"       -c  display the children of selected, or all, tasks.",
"       -t  display the task run time, start time, and cumulative user",
"           and system times.",
"\nEXAMPLES",
"  Show the process status of all current tasks:\n",
"    %s> ps",
"       PID    PPID  CPU   TASK    ST  %MEM   VSZ   RSS  COMM",
"    >     0      0   3  c024c000  RU   0.0     0     0  [swapper]",
"    >     0      0   0  c0dce000  RU   0.0     0     0  [swapper]",
"          0      0   1  c0fa8000  RU   0.0     0     0  [swapper]",
"    >     0      0   2  c009a000  RU   0.0     0     0  [swapper]",
"          1      0   1  c0098000  IN   0.0  1096   476  init",
"          2      1   1  c0090000  IN   0.0     0     0  [kflushd]",
"          3      1   1  c000e000  IN   0.0     0     0  [kpiod]",
"          4      1   3  c000c000  IN   0.0     0     0  [kswapd]",
"          5      1   1  c0008000  IN   0.0     0     0  [mdrecoveryd]",
"        253      1   2  fbc4c000  IN   0.0  1088   376  portmap",
"        268      1   2  fbc82000  IN   0.1  1232   504  ypbind",
"        274    268   2  fa984000  IN   0.1  1260   556  ypbind",
"        321      1   1  fabf6000  IN   0.1  1264   608  syslogd",
"        332      1   1  fa9be000  RU   0.1  1364   736  klogd",
"        346      1   2  fae88000  IN   0.0  1112   472  atd",
"        360      1   2  faeb2000  IN   0.1  1284   592  crond",
"        378      1   2  fafd6000  IN   0.1  1236   560  inetd",
"        392      1   0  fb710000  IN   0.1  2264  1468  named",
"        406      1   3  fb768000  IN   0.1  1284   560  lpd",
"        423      1   1  fb8ac000  IN   0.1  1128   528  rpc.statd",
"        434      1   2  fb75a000  IN   0.0  1072   376  rpc.rquotad",
"        445      1   2  fb4a4000  IN   0.0  1132   456  rpc.mountd",
"        460      1   1  fa938000  IN   0.0     0     0  [nfsd]",
"        461      1   1  faa86000  IN   0.0     0     0  [nfsd]",
"        462      1   0  fac48000  IN   0.0     0     0  [nfsd]",
"        463      1   0  fb4ca000  IN   0.0     0     0  [nfsd]",
"        464      1   0  fb4c8000  IN   0.0     0     0  [nfsd]",
"        465      1   2  fba6e000  IN   0.0     0     0  [nfsd]",
"        466      1   1  fba6c000  IN   0.0     0     0  [nfsd]",
"        467      1   2  fac04000  IN   0.0     0     0  [nfsd]",
"        468    461   2  fa93a000  IN   0.0     0     0  [lockd]",
"        469    468   2  fa93e000  IN   0.0     0     0  [rpciod]",
"        486      1   0  fab54000  IN   0.1  1596   880  amd",
"        523      1   2  fa84e000  IN   0.1  1884  1128  sendmail",
"        538      1   0  fa82c000  IN   0.0  1112   416  gpm",
"        552      1   3  fa70a000  IN   0.1  2384  1220  httpd",
"        556    552   3  fa776000  IN   0.1  2572  1352  httpd",
"        557    552   2  faba4000  IN   0.1  2572  1352  httpd",
"        558    552   1  fa802000  IN   0.1  2572  1352  httpd",
"        559    552   3  fa6ee000  IN   0.1  2572  1352  httpd",
"        560    552   3  fa700000  IN   0.1  2572  1352  httpd",
"        561    552   0  fa6f0000  IN   0.1  2572  1352  httpd",
"        562    552   3  fa6ea000  IN   0.1  2572  1352  httpd",
"        563    552   0  fa67c000  IN   0.1  2572  1352  httpd",
"        564    552   3  fa674000  IN   0.1  2572  1352  httpd",
"        565    552   3  fa66a000  IN   0.1  2572  1352  httpd",
"        582      1   2  fa402000  IN   0.2  2968  1916  xfs",
"        633      1   2  fa1ec000  IN   0.2  5512  2248  innd",
"        636      1   3  fa088000  IN   0.1  2536   804  actived",
"        676      1   0  fa840000  IN   0.0  1060   384  mingetty",
"        677      1   1  fa590000  IN   0.0  1060   384  mingetty",
"        678      1   2  fa3b8000  IN   0.0  1060   384  mingetty",
"        679      1   0  fa5b8000  IN   0.0  1060   384  mingetty",
"        680      1   1  fa3a4000  IN   0.0  1060   384  mingetty",
"        681      1   2  fa30a000  IN   0.0  1060   384  mingetty",
"        683      1   3  fa5d8000  IN   0.0  1052   280  update",
"        686    378   1  fa3aa000  IN   0.1  2320  1136  in.rlogind",
"        687    686   2  f9e52000  IN   0.1  2136  1000  login",
"        688    687   0  f9dec000  IN   0.1  1732   976  bash",
"    >   700    688   1  f9d62000  RU   0.0  1048   256  gen12",
" ",
"  Display the parental hierarchy of the \"%s\" process on a live system:\n",
"    %s> ps -p 4249",
"    PID: 0      TASK: c0252000  CPU: 0   COMMAND: \"swapper\"",
"     PID: 1      TASK: c009a000  CPU: 1   COMMAND: \"init\"",
"      PID: 632    TASK: c73b6000  CPU: 1   COMMAND: \"prefdm\"",
"       PID: 637    TASK: c5a4a000  CPU: 1   COMMAND: \"prefdm\"",
"        PID: 649    TASK: c179a000  CPU: 0   COMMAND: \"kwm\"",
"         PID: 683    TASK: c1164000  CPU: 0   COMMAND: \"kfm\"",
"          PID: 1186   TASK: c165a000  CPU: 0   COMMAND: \"xterm\"",
"           PID: 1188   TASK: c705e000  CPU: 1   COMMAND: \"bash\"",
"            PID: 4249   TASK: c6b9a000  CPU: 0   COMMAND: \"crash\"",
" ",
"  Display all children of the \"kwm\" window manager:\n",
"    %s> ps -c kwm",
"      PID: 649    TASK: c179a000  CPU: 0   COMMAND: \"kwm\"",
"      PID: 682    TASK: c2d58000  CPU: 1   COMMAND: \"kwmsound\"",
"      PID: 683    TASK: c1164000  CPU: 1   COMMAND: \"kfm\"",
"      PID: 685    TASK: c053c000  CPU: 0   COMMAND: \"krootwm\"",
"      PID: 686    TASK: c13fa000  CPU: 0   COMMAND: \"kpanel\"",
"      PID: 687    TASK: c13f0000  CPU: 1   COMMAND: \"kbgndwm\"",
" ",
"  Show the time usage data for pid 18844:\n",
"    %s> ps -t 18844",
"    PID: 18844  TASK: c3012000  CPU: 0   COMMAND: \"bash\"",
"        RUN TIME: 00:31:23",
"      START TIME: 51390045",
"       USER TIME: 1",
"     SYSTEM TIME: 3",
" ",
"  Show the process status of PID 1, task f9dec000, and all nfsd tasks:\n",
"    %s> ps 1 f9dec000 nfsd",
"       PID    PPID  CPU   TASK    ST  %MEM   VSZ   RSS  COMM",
"          1      0   1  c0098000  IN   0.0  1096   476  init",
"        688    687   0  f9dec000  IN   0.1  1732   976  bash",
"        460      1   1  fa938000  IN   0.0     0     0  [nfsd]",
"        461      1   1  faa86000  IN   0.0     0     0  [nfsd]",
"        462      1   0  fac48000  IN   0.0     0     0  [nfsd]",
"        463      1   0  fb4ca000  IN   0.0     0     0  [nfsd]",
"        464      1   0  fb4c8000  IN   0.0     0     0  [nfsd]",
"        465      1   2  fba6e000  IN   0.0     0     0  [nfsd]",
"        466      1   1  fba6c000  IN   0.0     0     0  [nfsd]",
"        467      1   2  fac04000  IN   0.0     0     0  [nfsd]",
" ",
"  Show all kernel threads:\n",
"    %s> ps -k",
"       PID    PPID  CPU   TASK    ST  %MEM   VSZ   RSS  COMM",
"          0      0   1  c0fac000  RU   0.0     0     0  [swapper]",
"          0      0   0  c0252000  RU   0.0     0     0  [swapper]",
"          2      1   1  c0fa0000  IN   0.0     0     0  [kflushd]",
"          3      1   1  c03de000  IN   0.0     0     0  [kpiod]",
"          4      1   1  c03dc000  IN   0.0     0     0  [kswapd]",
"          5      1   0  c0092000  IN   0.0     0     0  [mdrecoveryd]",
"        336      1   0  c4a9a000  IN   0.0     0     0  [rpciod]",
"        337      1   0  c4830000  IN   0.0     0     0  [lockd]",
"        487      1   1  c4ba6000  IN   0.0     0     0  [nfsd]",
"        488      1   0  c18c6000  IN   0.0     0     0  [nfsd]",
"        489      1   0  c0cac000  IN   0.0     0     0  [nfsd]",
"        490      1   0  c056a000  IN   0.0     0     0  [nfsd]",
"        491      1   0  c0860000  IN   0.0     0     0  [nfsd]",
"        492      1   1  c0254000  IN   0.0     0     0  [nfsd]",
"        493      1   0  c0a86000  IN   0.0     0     0  [nfsd]",
"        494      1   0  c0968000  IN   0.0     0     0  [nfsd]",
" ",
"  Show the kernel stack pointer of each user task:\n",
"    %s> ps -us",
"       PID    PPID  CPU  KSTACKP  ST  %MEM   VSZ   RSS  COMM",
"          1      0   0  c009bedc  IN   0.0  1096    52  init",
"        239      1   0  c15e7ed8  IN   0.2  1332   224  pump",
"        280      1   1  c7cbdedc  IN   0.2  1092   208  portmap",
"        295      1   0  c7481edc  IN   0.0  1232     0  ypbind",
"        301    295   0  c7c7bf28  IN   0.1  1260   124  ypbind",
"        376      1   1  c5053f28  IN   0.0  1316    40  automount",
"        381      1   0  c34ddf28  IN   0.2  1316   224  automount",
"        391      1   1  c2777f28  IN   0.2  1316   224  automount",
"    ...",
NULL               
};

char *help_rd[] = {
"rd",
"read memory",
"[-dDsup][-8|-16|-32|-64][-o offs][-e addr] [address|symbol] [count]",
"  This command displays the contents of memory, with the output formatted",
"  in several different manners.  The starting address may be entered either",
"  symbolically or by address.  The default output size is the size of a long",
"  data type, and the default output format is hexadecimal.  When hexadecimal",
"  output is used, the output will be accompanied by an ASCII translation.\n",
"       -p  address argument is a physical address.",
"       -u  address argument is a user virtual address; only required on",
"           processors with common user and kernel virtual address spaces.",
"       -d  display output in signed decimal format (default is hexadecimal).",
"       -D  display output in unsigned decimal format (default is hexadecimal).",
"       -s  displays output symbolically when appropriate.",
#ifdef NOTDEF
"    -o       Shows offset value from the starting address.",
#endif
"       -8  display output in 8-bit values.",
"      -16  display output in 16-bit values.",
"      -32  display output in 32-bit values (default on 32-bit machines).",
"      -64  display output in 64-bit values (default on 64-bit machines).",
"  -o offs  offset the starting address by offs.",
"  -e addr  display memory until reaching specified ending hexadecimal address.",
"  address  starting hexadecimal address:",
"             1  the default presumes a kernel virtual address.",
"             2. -p specifies a physical address.",
"             3. -u specifies a user virtual address, but is only necessary on",
"              processors with common user and kernel virtual address spaces.",
"   symbol  symbol of starting address to read.",
"    count  number of memory locations to display (default is 1).",
"\nEXAMPLES",
"  Display the kernel_version string:\n",
"    %s> rd kernel_version 4 ",
"    c0226a6c:  2e322e32 35312d35 00000000 00000001   2.2.5-15........\n",
"  Display the same block of memory, with and without symbols:\n",
"    %s> rd c1157f00 52   ",
"    c1157f00:  c0131f7a 00000400 00000015 c013206e   z...........n ..",
"    c1157f10:  00000100 c3d4c140 00000100 00000246   ....@.......F...",
"    c1157f20:  019b2065 c2a5bb90 080ac618 c02a83d0   e ............*.",
"    c1157f30:  40000025 01a45067 c1156000 00000000   %..@gP...`......",
"    c1157f40:  c011b4f7 c1156000 c2a5bb90 080ac618   .....`..........",
"    c1157f50:  00000001 00000000 c1a45000 c19b2000   .........P... ..",
"    c1157f60:  c1157f84 0000003b c022c000 c1156000   ....;.....\"..`..",
"    c1157f70:  00000000 fffffe00 bffff6fc 0000002e   ................",
"    c1157f80:  c022c000 ffffffff c01178ba c1156000   ..\"......x...`..",
"    c1157f90:  00000000 080ac618 bffff6ac 00000001   ................",
"    c1157fa0:  c1156000 c1156000 c1157fb8 c1156000   .`...`.......`..",
"    c1157fb0:  c1157fb8 c1156000 c1156000 c115608c   .....`...`...`..",
"    c1157fc0:  c01096c8 ffffffff bffff6fc 00000002   ................\n",
"    %s> rd -s c1157f00 52",
"    c1157f00:  alloc_fd_array+0x1a 00000400 00000015 expand_fd_array+0x72 ",
"    c1157f10:  00000100 c3d4c140 00000100 00000246 ",
"    c1157f20:  019b2065 c2a5bb90 080ac618 c02a83d0 ",
"    c1157f30:  40000025 01a45067 c1156000 00000000 ",
"    c1157f40:  do_wp_page+0x17f c1156000 c2a5bb90 080ac618 ",
"    c1157f50:  00000001 00000000 c1a45000 c19b2000 ",
"    c1157f60:  c1157f84 0000003b init_task_union c1156000 ",
"    c1157f70:  00000000 fffffe00 bffff6fc 0000002e ",
"    c1157f80:  init_task_union ffffffff sys_wait4+0x2be c1156000 ",
"    c1157f90:  00000000 080ac618 bffff6ac 00000001 ",
"    c1157fa0:  c1156000 c1156000 c1157fb8 c1156000 ",
"    c1157fb0:  c1157fb8 c1156000 c1156000 c115608c ",
"    c1157fc0:  system_call+0x34 ffffffff bffff6fc 00000002\n",
"  Read jiffies in hexadecimal and decimal format:\n",
"    %s> rd jiffies",
"    c0213ae0:  0008cc3a                              :...\n",
"    %s> rd -d jiffies",
"    c0213ae0:        577376\n",
"  Access the same memory in different sizes:\n",
"    %s> rd -64 kernel_version",
"    c0226a6c:  35312d352e322e32                    2.2.5-15\n",
"    %s> rd -32 kernel_version 2",
"    c0226a6c:  2e322e32 35312d35                     2.2.5-15\n",
"    %s> rd -16 kernel_version 4",
"    c0226a6c:  2e32 2e32 2d35 3531                       2.2.5-15\n",
"    %s> rd -8 kernel_version 8 ",
"    c0226a6c:  32 2e 32 2e 35 2d 31 35                           2.2.5-15",
"\n  Read the range of memory from c009bf2c to c009bf60:\n",
"    %s> rd c009bf2c -e c009bf60",
"    c009bf2c:  c009bf64 c01328c3 c009bf64 c0132838   d....(..d...8(..",
"    c009bf3c:  0000002a 00000004 c57d77e8 00000104   *........w}.....",
"    c009bf4c:  0000000b c009a000 7fffffff 00000000   ................",
"    c009bf5c:  00000000                              ....",
NULL               
};

char *help_wr[] = {
"wr",
"write memory",
"[-u|-k|-p] [-8|-16|-32|-64] [address|symbol] value",
"  This command modifies the contents of memory.  The starting address may be",
"  entered either symbolically or by address.  The default modification size ",
"  is the size of a long data type.  Write permission must exist on the",
"  /dev/mem.  When writing to memory on a live system, this command should ",
"  obviously be used with great care.\n",
"       -u  address argument is a user virtual address.",
"       -k  address argument is a kernel virtual address.",
"       -p  address argument is a physical address.",
"       -8  write data in an 8-bit value.",
"      -16  write data in a 16-bit value.",
"      -32  write data in a 32-bit values (default on 32-bit machines).",
"      -64  write data in a 64-bit values (default on 64-bit machines).",
"  address  address to write.  The address is considered virtual unless the",
"           -p option is used.  If a virtual address is specified, the",
"           -u or -k options are necessary only if the address space cannot", 
"           be determined from the address value itself.  If a user virtual",  
"           address is specified, the address space of the current context",
"           implied.  The address must be expressed in hexadecimal format.",
"   symbol  symbol of starting address to write.",
"    value  the value of the data to write.",
"\nEXAMPLES",
"  Turn on a debug flag:\n",
"    %s> wr my_debug_flag 1",
NULL               
};

char *help_bt[] = {
"bt",
"backtrace",
"[-a | -r | -t | -l | -e | -g] [-R ref] [ -I ip ] [-S sp] [ pid | taskp] ... ",
"  Display a kernel stack backtrace.  If no arguments are given, the stack",
"  trace of the current context will be displayed.\n",
"       -a  displays the stack traces of the active task on each CPU.",
"           (only applicable to crash dumps)",
"       -r  display raw stack data, consisting of a memory dump of the two",
"           pages of memory containing the task_union structure.",
"       -t  display all text symbols found from the last known stack location",
"           to the top of the stack. (helpful if kernel was compiled with",
"           the -fomit-frame-pointer CFLAG)",
"       -l  show file and line number of each stack trace text location.",
"       -e  search the stack for possible kernel and user mode exception frames.",
"       -g  use gdb stack trace code. (alpha only)",
"   -R ref  display stack trace only if there is a reference to this symbol",
"           or text address.",
"    -I ip  use ip as the starting text location.",
"    -S sp  use sp as the starting stack frame address.",
"      pid  displays the stack trace(s) of this pid.",
"    taskp  displays the stack trace the the task referenced by this hexadecimal",
"           task_struct pointer.\n",
"  The arguments that are passed to each function are shown under each frame.",
"  The number of arguments shown are a \"best-guess\"; extra argument values",
"  are displayed when the number of arguments cannot be accurately determined.",
"  Arguments are not shown on all architectures.",
"\nEXAMPLES",
"  Display the stack trace of the active task when the kernel panicked:\n",
"    %s> bt -a",
"    PID: 286    TASK: c0b3a000  CPU: 0   COMMAND: \"in.rlogind\"",
"    #0 [c0b3be90] crash_save_current_state at c011aed0",
"       (c0b3a000)",
"    #1 [c0b3bea4] panic at c011367c",
"       (c02095a4)",
"    #2 [c0b3bee8] tulip_interrupt at c01bc820",
"       (9, eth0_dev, c0b3bf44)",
"    #3 [c0b3bf08] handle_IRQ_event at c010a551",
"       (9, c0b3bf44, c08e4190)",
"    #4 [c0b3bf2c] do_8259A_IRQ at c010a319",
"       (9, c0b3bf44)",
"    #5 [c0b3bf3c] do_IRQ at c010a653",
"       (c0e68280)",
"    #6 [c0b3bfbc] ret_from_intr at c0109634",
"       EAX: 00000000  EBX: c0e68280  ECX: 00000000  EDX: 00000004  EBP: c0b3bfbc",
"       DS:  0018      ESI: 00000004  ES:  0018      EDI: c0e68284 ",
"       CS:  0010      EIP: c012f803  ERR: ffffff09  EFLAGS: 00000246 ",
"    #7 [c0b3bfbc] sys_select at c012f803",
"       (4, bfffc9a0, 0, bfffc8a0)",
"    #8 [bfffd224] system_call at c0109598",
"       EAX: 0000008e  EBX: 00000004  ECX: bfffc9a0  EDX: 00000000 ",
"       DS:  002b      ESI: bfffc8a0  ES:  002b      EDI: 00000000 ",
"       SS:  002b      ESP: bfffc82c  EBP: bfffd224 ",
"       CS:  0023      EIP: 400d032e  ERR: 0000008e  EFLAGS: 00000246  ",
"\n  Display the stack traces of PID 340 and task c0c86000:\n",
"    %s> bt 340 c0c86000",
"    PID: 340    TASK: c1126000  CPU: 0   COMMAND: \"httpd\"",
"    #0 [c1127e78] schedule at c0110f23",
"       (void)",
"    #1 [c1127eb4] interruptible_sleep_on at c011116b",
"       (c0985670)",
"    #2 [c1127efc] posix_lock_file at c0130bb1",
"       (c0e6cf00, c1127f34, 1)",
"    #3 [c1127f70] fcntl_setlk at c01303d1",
"       (14, 7, 8077988)",
"    #4 [c1127fbc] sys_fcntl at c012e97b",
"       (14, 7, 8077988)",
"    #5 [bffffcbc] system_call at c0109598",
"       EAX: 00000037  EBX: 00000014  ECX: 00000007  EDX: 08077988 ",
"       DS:  002b      ESI: 00000003  ES:  002b      EDI: 000001ec ",
"       SS:  002b      ESP: bffffca4  EBP: bffffcbc ",
"       CS:  0023      EIP: 400d59a4  ERR: 00000037  EFLAGS: 00000292  ",
"    ",
"    PID: 216    TASK: c0c86000  CPU: 0   COMMAND: \"inetd\"",
"    #0 [c0c87eec] schedule at c0110f23",
"       (void)",
"    #1 [c0c87f34] schedule_timeout at c0110c6f",
"       (2d)",
"    #2 [c0c87f64] do_select at c012f2d8",
"       (e, c0c87fa4, c0c87fa0)",
"    #3 [c0c87fbc] sys_select at c012f83d",
"       (e, bffffc14, 0, 0, 0)",
"    #4 [bffffdb8] system_call at c0109598",
"       EAX: 0000008e  EBX: 0000000e  ECX: bffffc14  EDX: 00000000 ",
"       DS:  002b      ESI: 00000000  ES:  002b      EDI: 00000000 ",
"       SS:  002b      ESP: bffffbbc  EBP: bffffdb8 ",
"       CS:  0023      EIP: 4007e32e  ERR: 0000008e  EFLAGS: 00000202",
" ",
"  Dump the text symbols found in the current context's stack:\n",
"    %s> bt -t",
"    PID: 527    TASK: c06be000  CPU: 0   COMMAND: \"sendmail\"",
"         EIP: schedule+0x352",
"    c06be038: schedule_timeout+0x65",
"    c06be058: process_timeout",
"    c06be060: do_select+0xac",
"    c06be090: sys_select+0x405",
"    c06be0e8: system_call+0x34",
" ",
"  Search the current stack for possible exception frames:\n",
"    %s> bt -e",
"    PID: 286    TASK: c0b3a000  CPU: 0   COMMAND: \"in.rlogind\"",
"    ",
"     KERNEL-MODE EXCEPTION FRAME AT c0b3bf44:",
"       EAX: 00000000  EBX: c0e68280  ECX: 00000000  EDX: 00000004  EBP: c0b3bfbc",
"       DS:  0018      ESI: 00000004  ES:  0018      EDI: c0e68284 ",
"       CS:  0010      EIP: c012f803  ERR: ffffff09  EFLAGS: 00000246 ",
"    ",
"     USER-MODE EXCEPTION FRAME AT c0b3bfc4:",
"       EAX: 0000008e  EBX: 00000004  ECX: bfffc9a0  EDX: 00000000 ",
"       DS:  002b      ESI: bfffc8a0  ES:  002b      EDI: 00000000 ",
"       SS:  002b      ESP: bfffc82c  EBP: bfffd224 ",
"       CS:  0023      EIP: 400d032e  ERR: 0000008e  EFLAGS: 00000246 ",
NULL               
};

char *help_btop[] = {
"btop",
"bytes to page",
"address ...",
"  This command translates a hexadecimal address to its page number.",
"\nEXAMPLES",
"    %s> btop 512a000",
"    512a000: 512a",
NULL               
};

char *help_extend[] = {
"extend",
"extend the %s command set",  
"[shared-object ...] | [-u [shared-object ...]]",
"  This command dynamically loads or unloads %s extension shared object",
"  libraries:\n",
"    shared-object     load the specified shared object file; more than one",
"                      one object file may be entered.",
"    -u shared-object  unload the specified shared object file; if no file",
"                      arguments are specified, unload all objects.", 
"\n  If no arguments are entered, the current set of shared object files and ",
"  a list of their commands will be displayed.  The registered commands",
"  contained in each shared object file will appear automatically in the ",
"  \"help\" command screen.",
"\n  An example of a shared object prototype file, and how to compile it",
"  into a shared object, is appended below.",
"\nEXAMPLES",
"  Load two shared object files:\n",
"    %s> extend extlib1.so extlib2.so",
"    ./extlib1.so: shared object loaded",
"    ./extlib2.so: shared object loaded",
"\n  Display the current set of shared object files and their commands:\n",
"    %s> extend",
"    SHARED OBJECT  COMMANDS",
"    ./extlib1.so   echo util bin",
"    ./extlib2.so   smp show",
"\n  Unload one of the shared object files:\n",
"    %s> extend -u extlib1.so",
"    ./extlib1.so: shared object unloaded",
"\n  Unload all currently-loaded object files:\n",
"    %s> extend -u",
"    ./extlib2.so: shared object unloaded",
"\nCREATING A SHARED OBJECT",
"  The extend command loads shared object files using dlopen(3), which in",
"  turn calls the shared object's _init() function.  The shared object's _init()",
"  function should register its command set by calling register_extension(),",
"  passing it a pointer to an array of one or more structures of the",
"  following type:",
" ",
"    struct command_table_entry {",
"            char *name;",
"            cmd_func_t func;",
"            char **help_data,",
"            ulong flags;",
"    };",
" ",
"  Each command_table_entry structure contains the ASCII name of a command,",
"  the command's function address, a pointer to an array of help data strings,",
"  and a flags field.  The help_data field is optional; if it is non-NULL, it",
"  should point to an array of character strings used by the \"help\"",
"  command, and during command failures.  The flags field currently has one",
"  available bit setting, REFRESH_TASK_TABLE, which should be set if it is ",
"  preferable to reload the current set of running processes just prior to ",
"  executing the command (on a live system).  Terminate the array of ",
"  command_table_entry structures with an entry with a NULL command name.  ",
" ",
"  Below is an example shared object file consisting of just one command, ",
"  called \"echo\", which simply echoes back all arguments passed to it.",
"  Note the comments contained within it for further details.  To build it,",
"  cut and paste the following output into a file, and call it, for example,",
"  \"extlib.c\".  Then compile like so:",
" ",
"    gcc -nostartfiles -shared -rdynamic -o extlib.so extlib.c",
" ",
"  The extlib.so file may be dynamically linked into %s during runtime, or",
"  during initialization by putting \"extend extlib.so\" into a .%src file",
"  located in the current directory, or in the user's $HOME directory.",
"  ",
"---------------------------------- cut here ----------------------------------",
" ",
"#include \"defs.h\"    /* From the crash source top-level directory */",
"",
"void cmd_echo();     /* Declare the commands and their help data. */",
"char *help_echo[];",
"",
"static struct command_table_entry command_table[] = {",
"        \"echo\", cmd_echo, help_echo, 0,           /* One or more commands, */",
"        NULL,                                     /* terminated by NULL, */",
"};",
"",
"",
"_init() /* Register the command set. */",
"{ ",
"        register_extension(command_table);",
"}",
" ",
"/* ",
" *  The _fini() function is called if the shared object is unloaded. ",
" *  If desired, perform any cleanups here. ",
" */",
"_fini() { }",
"",
"",
"/* ",
" *  Arguments are passed to the command functions in the global args[argcnt]",
" *  array.  See getopt(3) for info on dash arguments.  Check out defs.h and",
" *  other %s commands for usage of the myriad of utility routines available",
" *  to accomplish what your task.",
" */",
"void",
"cmd_echo()",
"{",
"        int c;",
"",
"        while ((c = getopt(argcnt, args, \"\")) != EOF) {",
"                switch(c)",
"                {",
"                default:",
"                        argerrs++;",
"                        break;",
"                }",
"        }",
"",
"        if (argerrs)",
"                cmd_usage(pc->curcmd, SYNOPSIS);",
"",
"        while (args[optind]) ",
"                fprintf(fp, \"%%s \", args[optind++]);",
"",
"        fprintf(fp, \"\\n\");",
"}",
"",
"/* ",  
" *  The optional help data is simply an array of strings in a defined format.",
" *  For example, the \"help echo\" command will use the help_echo[] string",
" *  array below to create a help page that looks like this:",
" * ",
" *    NAME",
" *      echo - echoes back its arguments",
" *",
" *    SYNOPSIS",
" *      echo arg ...",
" *",
" *    DESCRIPTION",
" *      This command simply echoes back its arguments.",
" *",
" *    EXAMPLE",
" *      Echo back all command arguments:",
" *",
" *        crash> echo hello, world",
" *        hello, world",
" *",
" */",
" ",
"char *help_echo[] = {",
"        \"echo\",                        /* command name */",
"        \"echoes back its arguments\",   /* short description */",
"        \"arg ...\",                     /* argument synopsis, or \" \" if none */",
" ",
"        \"  This command simply echoes back its arguments.\",",
"        \"\\nEXAMPLE\",",
"        \"  Echo back all command arguments:\\n\",",
"        \"    %s> echo hello, world\",",
"        \"    hello, world\",",
"        NULL",
"};",
"",
NULL               
};

char *help_mach[] = {
"mach",
"machine specific data",    
"[-c]",
"  This command displays data specific to a machine type.\n",
"    -c  Display each cpu's cpuinfo structure (x86 and ia64 only).",
"        Display the hwrpb_struct, and each cpu's percpu_struct (alpha only).",
"\nEXAMPLES", 
"    %s> mach",
"           MACHINE TYPE: i686",
"            MEMORY SIZE: 128 MB",
"                   CPUS: 2",
"        PROCESSOR SPEED: 447 Mhz",
"                     HZ: 100",
"              PAGE SIZE: 4096",
"          L1 CACHE SIZE: 16",
"    KERNEL VIRTUAL BASE: c0000000",
"    KERNEL VMALLOC BASE: c8000000",
"      KERNEL STACK SIZE: 8192",
NULL               
};

char *help_timer[] = {
"timer",
"timer queue data",
" ",
"  This command displays the timer queue entries, both old- and new-style,",
"  in chronological order.  In the case of the old-style timers, the",
"  timer_table array index is shown; in the case of the new-style timers, ",
"  the timer_list address is shown.",
"\nEXAMPLES",
"    %s> timer",
"    JIFFIES",
"      68102",
"    EXPIRES  TIMER_LIST/TABLE  FUNCTION",
"      68346      c0241934      c01775d4  <tcp_sltimer_handler>",
"      68379      c0241204      c01696d8  <dev_do_watchdog>",
"      68523      c7fcdfc0      c0112d6c  <process_timeout>",
"      68718      c7fd8edc      c018719c  <irlmp_discovery_timer_expired>",
"      68723   timer_table[2]   c01c707c  <rs_timer>",
"      68742      c20c1f7c      c0112d6c  <process_timeout>",
"      68742      c20c1f7c      c0112d6c  <process_timeout>",
"      68742      c20c1f7c      c0112d6c  <process_timeout>",
"      68752      c7fd1fc4      c0112d6c  <process_timeout>",
"      68752      c7fd1fc4      c0112d6c  <process_timeout>",
"      68989      c0241d40      c0168060  <neigh_periodic_timer>",
"      69028      c2533f7c      c0112d6c  <process_timeout>",
"      69134      c22dd868      c0181948  <unix_destroy_timer>",
"      71574      c0241430      c0169ea4  <rt_check_expire>",
"      72179      c7fb1c48      c01cb9a0  <vortex_timer>",
"      73144      c1b17f10      c0112d6c  <process_timeout>",
"      73259      c17a5f10      c0112d6c  <process_timeout>",
"     112929      c203ff10      c0112d6c  <process_timeout>",
"     372010      c2323f7c      c0112d6c  <process_timeout>",
"     372138      c2191f10      c0112d6c  <process_timeout>",
"    8653052      c1f13f10      c0112d6c  <process_timeout>",
NULL               
};



char *help_runq[] = {
"runq",
"run queue",
" ",
"  This command displays the tasks on the run queue.",
"\nEXAMPLES",
"    %s> runq",
"    PID: 435    TASK: c322c000  CPU: 1   COMMAND: \"httpd\"",
"    PID: 253    TASK: c6580000  CPU: 0   COMMAND: \"portmap\"",
"    PID: 5655   TASK: c0a86000  CPU: 1   COMMAND: \"nfsd\"",
"    PID: 3      TASK: c0fa0000  CPU: 1   COMMAND: \"kflushd\"",
"    PID: 318    TASK: c6734000  CPU: 1   COMMAND: \"syslogd\"",
"    PID: 5657   TASK: c129e000  CPU: 1   COMMAND: \"nfsd\"",
"    PID: 5653   TASK: c0d26000  CPU: 1   COMMAND: \"nfsd\"",
"    PID: 329    TASK: c4138000  CPU: 1   COMMAND: \"klogd\"",
"    PID: 2      TASK: c03dc000  CPU: 0   COMMAND: \"kswapd\"",
NULL               
};


char *help_repeat[] = {
"repeat",
"repeat a command",
"[-seconds] command",
"  This command repeats a command indefinitely, optionally delaying a given",
"  number of seconds between each command execution.\n",
"    -seconds   The number of seconds to delay between command executions.",
"               This option must precede the command name to be executed.",
" ",
"  Command execution may be stopped with CTRL-C, or if scrolling is in effect,",
"  by entering \"q\".  This command is meant for use on a live system; it is",
"  hard to conceive of a reason to use it when debugging a crash dump.",
"\nEXAMPLES",
"  Display the value of jiffies once per second:\n",
"    %s> repeat -1 p jiffies",
"    jiffies = $1 = 155551079",
"    jiffies = $2 = 155551180",
"    jiffies = $3 = 155551281",
"    jiffies = $4 = 155551382",
"    jiffies = $5 = 155551483",
"    jiffies = $6 = 155551584",
"    jiffies = $7 = 155551685",
"    jiffies = $8 = 155551786",
"    jiffies = $9 = 155551887",
"    jiffies = $10 = 155551988",
"    jiffies = $11 = 155552089",
"    jiffies = $12 = 155552190",
"    jiffies = $13 = 155552291",
"    jiffies = $14 = 155552392",
"    jiffies = $15 = 155552493",
"    jiffies = $16 = 155552594",
"    jiffies = $17 = 155552695",
"    jiffies = $18 = 155552796",
"    ...",
NULL               
};


char *help_pte[] = {
"pte",
"translate a page table entry",
"contents ...",
"  This command translates the hexadecimal contents of a PTE into its physical",
"  page address and page bit settings.  If the PTE references a swap location,",
"  the swap device and offset are displayed.",
"\nEXAMPLES\n",
"    %s> pte d8e067",
"     PTE    PHYSICAL  FLAGS",
"    d8e067   d8e000   (PRESENT|RW|USER|ACCESSED|DIRTY)",
" ",
"    %s> pte 13f600",
"     PTE      SWAP     OFFSET",
"    13f600  /dev/hda2   5104",
NULL               
};


char *help_swap[] = {
"swap",
"swap device information",
" ",
"  This command displays information for each configured swap device.",
"\nEXAMPLES",
"    %s> swap",
"    FILENAME           TYPE         SIZE       USED   PCT  PRIORITY",
"    /dev/sda8        PARTITION     136516k    47896k  35%     -1",
NULL               
};



char *help_dev[] = {
"dev",
"device data",
"[-i | -p]",
"  If no argument is entered, this command dumps the contents of the chrdevs",
"  and blkdevs arrays.\n",
"    -i  display I/O port usage; on 2.4 kernels, also display I/O memory usage.",
"    -p  display PCI device data.",
"\nEXAMPLES",
"  Display the chrdevs and blkdevs array contents:\n",
"    %s> dev",
"    CHRDEV  NAME        OPERATIONS",
"       1    mem         c02c9580  <memory_fops>",
"       2    pty         c02c9280  <tty_fops>",
"       3    ttyp        c02c9280  <tty_fops>",
"       4    ttyS        c02c9280  <tty_fops>",
"       5    cua         c02c9280  <tty_fops>",
"       7    vcs         c02ca940  <vcs_fops>",
"      10    misc        c02c5480  <misc_fops>",
"     128    ptm         c02c9280  <tty_fops>",
"     136    pts         c02c9280  <tty_fops>",
"     162    raw         c02ca880  <raw_fops>",
"    ",
"    BLKDEV  NAME        OPERATIONS",
"       2    fd          c02c4fc0  <floppy_fops>",
"       8    sd          c02cec80  <sd_fops>",
"      22    ide1        c02c4274  <ide_fops>",
"      65    sd          c02cec80  <sd_fops>",
"      66    sd          c02cec80  <sd_fops>",

"\n  Display PCI data:\n",
"    %s> dev -p",
"    PCI_DEV  BU:SL.FN CLASS: VENDOR-DEVICE",
"    c00051c0 00:00.0  Host bridge: Intel 440BX - 82443BX Host",
"    c0005250 00:01.0  PCI bridge: Intel 440BX - 82443BX AGP",
"    c00052e0 00:07.0  ISA bridge: Intel 82371AB PIIX4 ISA",
"    c0005370 00:07.1  IDE interface: Intel 82371AB PIIX4 IDE",
"    c0005400 00:07.2  USB Controller: Intel 82371AB PIIX4 USB",
"    c0005490 00:07.3  Bridge: Intel 82371AB PIIX4 ACPI",
"    c0005520 00:11.0  Ethernet controller: 3Com 3C905B 100bTX",
"    c00055b0 00:13.0  PCI bridge: DEC DC21152",
"    c0005640 01:00.0  VGA compatible controller: NVidia [PCI_DEVICE 28]",
"    c00056d0 02:0a.0  SCSI storage controller: Adaptec AIC-7890/1",
"    c0005760 02:0e.0  SCSI storage controller: Adaptec AIC-7880U",

"\n  Display I/O port and I/O memory usage:\n",
"    %s> dev -i",
"    RESOURCE    RANGE    NAME",
"    c03036d4  0000-ffff  PCI IO",
"    c0302594  0000-001f  dma1",
"    c03025b0  0020-003f  pic1",
"    c03025cc  0040-005f  timer",
"    c03025e8  0060-006f  keyboard",
"    c0302604  0080-008f  dma page reg",
"    c0302620  00a0-00bf  pic2",
"    c030263c  00c0-00df  dma2",
"    c0302658  00f0-00ff  fpu",
"    c122ff20  0170-0177  ide1",
"    c122f240  0213-0213  isapnp read",
"    c122ff40  02f8-02ff  serial(auto)",
"    c122ff00  0376-0376  ide1",
"    c03186e8  03c0-03df  vga+",
"    c122ff60  03f8-03ff  serial(auto)",
"    c123851c  0800-083f  Intel Corporation 82371AB PIIX4 ACPI",
"    c1238538  0840-085f  Intel Corporation 82371AB PIIX4 ACPI",
"    c122f220  0a79-0a79  isapnp write",
"    c122f200  0cf8-0cff  PCI conf1",
"    c1238858  dc00-dc7f  3Com Corporation 3c905B 100BaseTX [Cyclone]",
"    c122fc00  dc00-dc7f  00:11.0",
"    c12380c8  dce0-dcff  Intel Corporation 82371AB PIIX4 USB",
"    c1238d1c  e000-efff  PCI Bus #02",
"    c1237858  e800-e8ff  Adaptec AIC-7880U",
"    c1237458  ec00-ecff  Adaptec AHA-2940U2/W / 7890",
"    c1239cc8  ffa0-ffaf  Intel Corporation 82371AB PIIX4 IDE",
"    ",
"    RESOURCE        RANGE        NAME",
"    c03036f0  00000000-ffffffff  PCI mem",
"    c0004000  00000000-0009ffff  System RAM",
"    c03026ac  000a0000-000bffff  Video RAM area",
"    c03026fc  000c0000-000c7fff  Video ROM",
"    c0302718  000c9800-000cdfff  Extension ROM",
"    c0302734  000ce000-000ce7ff  Extension ROM",
"    c0302750  000ce800-000cffff  Extension ROM",
"    c03026e0  000f0000-000fffff  System ROM",
"    c0004040  00100000-07ffdfff  System RAM",
"    c0302674  00100000-0028682b  Kernel code",
"    c0302690  0028682c-0031c63f  Kernel data",
"    c0004060  07ffe000-07ffffff  reserved",
"    c1239058  ec000000-efffffff  Intel Corporation 440BX/ZX - 82443BX/ZX Host",
"                                 bridge",
"    c1238d54  f1000000-f1ffffff  PCI Bus #02",
"    c1239554  f2000000-f5ffffff  PCI Bus #01",
"    c1237074  f4000000-f5ffffff  nVidia Corporation Riva TnT2 [NV5]",
"    c1238d38  fa000000-fbffffff  PCI Bus #02",
"    c1237874  faffe000-faffefff  Adaptec AIC-7880U",
"    c127ec40  faffe000-faffefff  aic7xxx",
"    c1237474  fafff000-faffffff  Adaptec AHA-2940U2/W / 7890",
"    c127eec0  fafff000-faffffff  aic7xxx",
"    c1239538  fc000000-fdffffff  PCI Bus #01",
"    c1237058  fc000000-fcffffff  nVidia Corporation Riva TnT2 [NV5]",
"    c1238874  fe000000-fe00007f  3Com Corporation 3c905B 100BaseTX [Cyclone]",
"    c0004080  fec00000-fec0ffff  reserved",
"    c00040a0  fee00000-fee0ffff  reserved",
"    c00040c0  ffe00000-ffffffff  reserved",

NULL               
};



char *help_search[] = {
"seach",
"search memory",
"[-s start | -k | -u] [-e end | -l length] [-m mask] value ...",
"  This command searches a range of user or kernel memory space for given value.",
"  If no end nor length value is entered, the search stops at the end of",
"  user or kernel address space, whichever is appropriate.",
" ",
"  An optional mask value may be entered to mask off \"don't care\" bits.\n",
"   -s start  Start the search at this hexadecimal user or kernel virtual ",
"             address, or kernel symbol.",
"         -k  If no start address is specified, start the search at the base",
"             of kernel address space.  If a start address is specified, then",
"             this option specifies that the start address is a kernel virtual",
"             address. (only required on processors with overlapping",
"             kernel/user virtual address spaces)",
"         -u  If no start address is specified, start the search at the base",
"             of the current context's user address space.  If a start address",
"             is specified, then this option specifies that the start address",
"             is a user virtual address. (only required on processors with ",
"             overlapping kernel/user virtual address spaces)",
"     -e end  Stop the search at this hexadecimal user or kernel virtual ",
"             address, or kernel symbol.",
"  -l length  Length in bytes of address range to search.",
"    -m mask  Ignore the bits that are set in the hexadecimal mask value.",
"      value  Search for this hexadecimal value.",
" ",
"  If the -s start option is not used, then either -u or -k must be used in",
"  order to determine whether to search user or kernel memory.  The starting ",
"  address must be long-word aligned.  Address ranges that start in user space",
"  and end in kernel space are not accepted.",
"\nEXAMPLES",
"  Search the current context's address space for all instances of 0xdeadbeef:",
"\n    %s> search -u deadbeef",
"    bffff81c: deadbeef",
"\n  Search all kernel memory above the kernel text space for all instances",
"  of 0xabcd occuring in the lower 16-bits of each 32-bit word: \n",
"    %s> search -s _etext -m ffff0000 abcd",
"    c071481c: abcd",
"    c0c2b0fc: 804abcd",
"    c0cf5e74: 7489abcd",
"    c17c0b44: c012abcd",
"    c1dac730: 3dbeabcd",
"    c226d0e8: ffffabcd",
"    c23ed5dc: abcd",
"    c3022544: 3dbeabcd",
"    c3069b58: 3dbeabcd",
"    c3e86e84: aabcd",
"    c3e88ed0: aabcd",
"    c3e8ee5c: aabcd",
"    c3e9df50: aabcd",
"    c3e9e930: aabcd",
"    c440a778: 804abcd",
"    c486eb44: 3dbeabcd",
"    c578f0fc: 804abcd",
"    c6394f90: 8ababcd",
"    c65219f0: 3abcd",
"    c661399c: abcd",
"    c68514ac: 8abcd",
"    c7e036bc: 3dbeabcd",
"    c7e12568: 5abcd",
"    c7e1256c: 5abcd",
"\n  Search the user address space of the current context for all instances",
"  of 0xdeadbeef:\n",
"    %s> search -u deadbeef",
"    81aba5c: deadbeef",
"    81abaa8: deadbeef",
"    bfffc698: deadbeef",
"    bffff390: deadbeef",
"\n  Search the 4K page at c532c000 for all instances of 0xffffffff:\n",
"    %s> search -s c532c000 -l 4096 ffffffff",
"    c532c33c: ffffffff",
"    c532c3fc: ffffffff",
"\n  Search the static kernel data area for all instances of c2d400eb:\n",
"    %s> search -s _etext -e _edata c2d400eb",
"    c022b550: c2d400eb",
"    c022b590: c2d400eb",
"    c022b670: c2d400eb",
"    c022b6e0: c2d400eb",
"    c022b7b0: c2d400eb",
"    c022b7e0: c2d400eb",
"    c022b8b0: c2d400eb",
NULL               
};


char *help_irq[] = {
"irq",
"IRQ data",
"[-d | -b | [index ...]]",
"  This command collaborates the data in an irq_desc_t, along with its",
"  associated hw_interrupt_type and irqaction structure data, into a",
"  consolidated per-IRQ display.  Alternatively, the intel interrupt",
"  descriptor table may be dumped, or bottom half data may be displayed.",
"  If no index value argument(s) nor any options are entered, the IRQ",
"  data for all IRQs will be displayed.\n",
"    index   a valid IRQ index.",
"       -d   dump the intel interrupt descriptor table.",
"       -b   dump bottom half data.",
"\nEXAMPLES",
"  Display the relevant data for IRQ 18:\n", 
"    %s> irq 18",
"        IRQ: 18",
"     STATUS: 0 ",
"    HANDLER: c02301e0  <ioapic_level_irq_type>",
"             typename: c01f9e0c  \"IO-APIC-level\"",
"              startup: c0110234  <unmask_IO_APIC_irq>",
"             shutdown: c01101cc  <mask_IO_APIC_irq>",
"               handle: c0110518  <do_level_ioapic_IRQ>",
"               enable: c0110234  <unmask_IO_APIC_irq>",
"              disable: c01101cc  <mask_IO_APIC_irq>",
"     ACTION: c009c6b0",
"              handler: c01ce818  <do_aic7xxx_isr>",
"                flags: 4000000  (SA_SHIRQ)",
"                 mask: 0",
"                 name: c0217780  \"aic7xxx\"",
"               dev_id: c0090078",
"                 next: c009c770",
"     ACTION: c009c770",
"              handler: c01ce818  <do_aic7xxx_isr>",
"                flags: 4000000  (SA_SHIRQ)",
"                 mask: 0",
"                 name: c0217780  \"aic7xxx\"",
"               dev_id: c0091078",
"                 next: 0",
"      DEPTH: 0\n",
"  Display the intel interrupt descriptor table entries:\n",
"    %s> irq -d",
"      [0] divide_error",
"      [1] debug",
"      [2] nmi",
"      [3] int3",
"      [4] overflow",
"      [5] bounds",
"      [6] invalid_op",
"      [7] device_not_available",
"      [8] double_fault",
"      [9] coprocessor_segment_overrun",
"     [10] invalid_TSS",
"     [11] segment_not_present",
"     [12] stack_segment",
"     [13] general_protection",
"     [14] page_fault",
"     [15] spurious_interrupt_bug",
"     [16] coprocessor_error",
"     [17] alignment_check",
"     [18] ignore_int",
"     [19] ignore_int",
"     [20] ignore_int",
"     [21] ignore_int",
"    ...\n",
"    [250] IRQ0xda_interrupt",
"    [251] IRQ0xdb_interrupt",
"    [252] IRQ0xdc_interrupt",
"    [253] IRQ0xdd_interrupt",
"    [254] IRQ0xde_interrupt",
"    [255] spurious_interrupt\n",
"  Display the bottom half data:\n",
"    %s> irq -b",
"    BH_MASK   BH_ACTIVE",
"    00000f17  00000000",
"    ",
"    BH_BASE  FUNCTION",
"       0     c0114148  <timer_bh>",
"       1     c019b4f4  <console_bh>",
"       2     c0113c1c  <tqueue_bh>",
"       4     c01abbbc  <do_serial_bh>",
"       8     c0153098  <net_bh>",
"       9     c01b3230  <scsi_bottom_half_handler>",
"      10     c0113c88  <immediate_bh>",
"      11     c01a8c80  <kbd_bh>",
NULL               
};


char *help_cpu[] = {
"cpu",
"set context to the active task on a cpu",
"[cpu]",
"  This command sets the current context to the active task on a cpu.  If no",
"  argument is given, the current context is displayed.\n",
"    cpu   a valid cpu number\n",
"  This command is not allowable on live systems.",
"\nEXAMPLES",
"    %s> cpu 1",
"         PID: 286",
"     COMMAND: \"in.rlogind\"",
"        TASK: c0b3a000 ",
"         CPU: 1 ",
"       STATE: TASK_RUNNING (ACTIVE)",
NULL               
};


char *help_sys[] = {
"sys",
"system data",
"[-c [name|number]] ",
"  This command displays system-specific data.  If no arguments are entered,\n"
"  the same system data shown during %s invocation is shown.\n",
"    -c [name|number]  If no name or number argument is entered, dump all",
"                      sys_call_table entries.  If a name string is entered,",
"                      search the table for all entries containing the string.",
"                      If a number is entered, the table entry associated with",
"                      that number is displayed.  If the current output radix",
"                      has been set to 16, the system call numbers will be ",
"                      displayed in hexadecimal.",
"    -panic            Panic a live system.  Requires write permission to", 
"                      /dev/mem.  Results in the %s context causing an",
"                      \"Attempted to kill the idle task!\" panic.  (The dump",
"                      will indicate that the %s context has a PID of 0).",

"\nEXAMPLES",
"  Display essential system information:\n",
"    %s> sys",
"          KERNEL: vmlinux.4",
"        DUMPFILE: lcore.cr.4",
"            CPUS: 4",
"            DATE: Mon Oct 11 18:48:55 1999",
"          UPTIME: 10 days, 14:14:39",
"    LOAD AVERAGE: 0.74, 0.23, 0.08",
"           TASKS: 77",
"        NODENAME: test.mclinux.com",
"         RELEASE: 2.2.5-15smp",
"         VERSION: #24 SMP Mon Oct 11 17:41:40 CDT 1999",
"         MACHINE: i686  (500 MHz)",
"          MEMORY: 1 GB",
"\n  Dump the system call table:\n",
"    %s> sys -c",
"    NUM  SYSTEM CALL                FILE AND LINE NUMBER",
"      0  sys_ni_syscall             ../kernel/sys.c: 48",
"      1  sys_exit                   ../kernel/exit.c: 404",
"      2  sys_fork                   ../arch/i386/kernel/process.c: 771",
"      3  sys_read                   ../fs/read_write.c: 117",
"      4  sys_write                  ../fs/read_write.c: 146",
"      5  sys_open                   ../fs/open.c: 754",
"      6  sys_close                  ../fs/open.c: 839",
"      7  sys_waitpid                ../kernel/exit.c: 503",
"      8  sys_creat                  ../fs/open.c: 789",
"      9  sys_link                   ../fs/namei.c: 1213",
"     10  sys_unlink                 ../fs/namei.c: 1074",
"     11  sys_execve                 ../arch/i386/kernel/process.c: 806",
"    ...",
"\n  Find the system call number of the select system call:\n",
"    %s> sys -c select",
"    NUM  SYSTEM CALL                FILE AND LINE NUMBER",
"     65  sys_select                 ../fs/select.c: 259",
" ",
"    If the current output radix has been set to 16, the system call numbers",
"    will be displayed in hexadecimal.",
NULL               
};

char *help_log[] = {
"log",
"dump system message buffer",
"[-c]",
"  This command dumps the kernel log_buf contents in chronological order.",
"  ",
"    -m  Display the message log level preceding each message.",
"\nEXAMPLES",
"  Dump the kernel message buffer:\n",
"    %s> log",
"    Linux version 2.2.5-15smp (root@mclinux1) (gcc version egcs-2.91.66 19990",
"    314/Linux (egcs-1.1.2 release)) #1 SMP Thu Aug 26 11:04:37 EDT 1999",
"    Intel MultiProcessor Specification v1.4",
"        Virtual Wire compatibility mode.",
"    OEM ID: DELL     Product ID: WS 410       APIC at: 0xFEE00000",
"    Processor #0 Pentium(tm) Pro APIC version 17",
"    Processor #1 Pentium(tm) Pro APIC version 17",
"    I/O APIC #2 Version 17 at 0xFEC00000.",
"    Processors: 2",
"    mapped APIC to ffffe000 (fee00000)",
"    mapped IOAPIC to ffffd000 (fec00000)",
"    Detected 447696347 Hz processor.",
"    Console: colour VGA+ 80x25",
"    Calibrating delay loop... 445.64 BogoMIPS",
"    ...",
"      8K byte-wide RAM 5:3 Rx:Tx split, autoselect/Autonegotiate interface.",
"      MII transceiver found at address 24, status 782d.",
"      Enabling bus-master transmits and whole-frame receives.",
"    Installing knfsd (copyright (C) 1996 okir@monad.swb.de).",
"    nfsd_init: initialized fhcache, entries=256",
"    ...",
" ",
"  Do the same thing, but also show the log level preceding each message:\n",
"    %s> log -m",
"    <4>Linux version 2.2.5-15smp (root@mclinux1) (gcc version egcs-2.91.66 19990",
"    314/Linux (egcs-1.1.2 release)) #1 SMP Thu Aug 26 11:04:37 EDT 1999",
"    <4>Intel MultiProcessor Specification v1.4",
"    <4>    Virtual Wire compatibility mode.",
"    <4>OEM ID: DELL     Product ID: WS 410       APIC at: 0xFEE00000",
"    <4>Processor #0 Pentium(tm) Pro APIC version 17",
"    <4>Processor #1 Pentium(tm) Pro APIC version 17",
"    <4>I/O APIC #2 Version 17 at 0xFEC00000.",
"    <4>Processors: 2",
"    <4>mapped APIC to ffffe000 (fee00000)",
"    <4>mapped IOAPIC to ffffd000 (fec00000)",
"    <4>Detected 447696347 Hz processor.",
"    <4>Console: colour VGA+ 80x25",
"    <4>Calibrating delay loop... 445.64 BogoMIPS",
"    ...",
"    <6>  8K byte-wide RAM 5:3 Rx:Tx split, autoselect/Autonegotiate interface.",
"    <6>  MII transceiver found at address 24, status 782d.",
"    <6>  Enabling bus-master transmits and whole-frame receives.",
"    <6>Installing knfsd (copyright (C) 1996 okir@monad.swb.de).",
"    <7>nfsd_init: initialized fhcache, entries=256",
"    ... ",
NULL               
};

char *help_whatis[] = {
"whatis",
"search symbol table for data or type information",
"[struct | union | typedef | symbol] ",
"  This command displays the definition of structures, unions, typedefs or",
"  text/data symbols.\n",
"    struct  a structure name. The output is the same as if the \"struct\"",
"            command was used.",
"     union  a union name. The output is the same as if the \"union\" command",
"            was used.", 
"   typedef  a typedef name. If the typedef translates to a structure or union",
"            the output is the same as if the \"struct\" or \"union\" command",
"            was used. If the typedef is a primitive datatype, the one-line",
"            declaration is displayed.",
"    symbol  a kernel symbol.  ",
"\nEXAMPLES",
"   Display the definition of a linux_binfmt structure:\n", 
"    %s> whatis linux_binfmt",
"    struct linux_binfmt {",
"      struct linux_binfmt  *next;",
"      struct module  *module;",
"      int (*load_binary) ();",
"      int (*load_shlib) ();",
"      int (*core_dump) ();",
"    };",
"    ",
"  Since a kmem_bufctl_t is typedef'd to be a kmem_bufctl_s structure, the",
"  output of the following two commands is identical:\n",
"    %s> whatis kmem_bufctl_s",
"    struct kmem_bufctl_s {",
"      union {",
"        struct kmem_bufctl_s  *buf_nextp;",
"        kmem_slab_t *buf_slabp;",
"        void *buf_objp;",
"      } u;",
"    };",
"    ",
"    %s> whatis kmem_bufctl_t",
"    struct kmem_bufctl_s {",
"      union {",
"        struct kmem_bufctl_s *buf_nextp;",
"        kmem_slab_t *buf_slabp;",
"        void *buf_objp;",
"      } u;",
"    };",
"    SIZE: 4  (0x4)",
"    ",
"  Display the type data of sys_read() and jiffies text and data symbols:\n", 
"    %s> whatis sys_read",
"    ssize_t sys_read(unsigned int, char *, size_t);",
" ",
"    %s> whatis jiffies",
"    long unsigned int jiffies;\n",
"  Display definition of a kdev_t typedef:\n",
"    %s> whatis kdev_t",
"    typedef short unsigned int kdev_t;",
"    SIZE: 2  (0x2)",
NULL               
};


char *help_mount[] = {
"mount",
"mounted filesystem data",
"[-f] [-i] [vfsmount | superblock | devname | dirname | inode]",
"  This command displays basic information about the currently-mounted",
"  filesystems.  The per-filesystem dirty inode list or list of open",
"  files for the filesystem may also be displayed.\n",
"     -f  dump dentries and inodes for open files in each filesystem.",
"     -i  dump all dirty inodes associated with each filesystem.\n",
"  Filesystems may be selected in the following forms:\n",
"    vfsmount  hexadecimal address of filesystem vfsmount structure.",
"  superblock  hexadecimal address of filesystem super_block structure.",
"     devname  device name of filesystem.",
"     dirname  directory where filesystem is mounted.",
"       inode  hexadecimal address of an open inode of a filesystem.",
"\nEXAMPLES",
"  Display mounted filesystem data:\n",
"    %s> mount",
"    VFSMOUNT SUPERBLK TYPE   DEVNAME         DIRNAME",
"    c0089ea0 c0088a00 ext2   /dev/root       /    ",
"    c0089cf0 c0088c00 proc   /proc           /proc",
"    c0089e10 c0088800 ext2   /dev/sda5       /boot",
"    c0089d80 c0088600 ext2   /dev/sda6       /usr",
"    c0089f30 c0088400 devpts none            /dev/pts",
"    c3f4b010 c0088200 ext2   /dev/sda1       /home",
"    c6bf3d10 c0088000 nfs    home:/home1     /home1",
"    c49b90a0 c43a2a00 nfs    home:/usr/local /usr/local ",
" ",
"  Display the open files associated with each mounted filesystem:\n",
"    %s> mount -f",
"    VFSMOUNT SUPERBLK TYPE   DEVNAME         DIRNAME",
"    c7fb2b80 c7fb3200 ext2   /dev/root       /",
"    OPEN FILES:",
"     DENTRY    INODE    TYPE  PATH",
"    c6d02200  c6d0f7a0  REG   usr/X11R6/lib/libX11.so.6.1",
"    c6d02100  c6d0f9e0  REG   usr/X11R6/lib/libXext.so.6.3",
"    c6d02000  c6d0fc20  REG   usr/X11R6/lib/libICE.so.6.3",
"    c6d02680  c6d0f320  REG   usr/X11R6/bin/xfs",
"    c7106580  c70c5440  CHR   dev/psaux",
"    ...",
" ",
"  Display the dirty inodes associated with each mounted filesystem:\n",
"    %s> mount -i",
"    VFSMOUNT SUPERBLK TYPE   DEVNAME         DIRNAME",
"    c0089ea0 c0088a00 ext2   /dev/root       /            ",
"    DIRTY INODES",
"    c7ad4008",
"    c2233438",
"    c72c4008",
"    c7d6b548",
"    c3af1a98",
"    c7d6b768",
"    c3c4e228",
"    ...",
" ",
"  Display the mounted filesystem containing inode c5000aa8:\n",
"    %s> mount c5000aa8",
"    VFSMOUNT SUPERBLK TYPE   DEVNAME         DIRNAME",
"    c0089f30 c0088600 ext2   /dev/sda6       /usr ",               
" ",
NULL               
};


char *help_alias[] = {
"alias",
"command aliases",
"[alias] [command string]",
"  This command creates an alias for a given command string.  If no arguments",
"  are entered, the current list of aliases are displayed.  If one argument is",
"  entered, the command string for that alias, if any, is displayed.\n",
"           alias  the single word to be used as an alias",
"  command string  the word(s) that will be substituted for the alias\n",
"  Aliases may be created in four manners:\n",
"    1. entering the alias in $HOME/.%src.",
"    2. entering the alias in .%src in the current directory.",
"    3. executing an input file containing the alias command.",
"    4. during runtime with this command.\n",
"  During initialization, $HOME/.%src is read first, followed by the",
"  .%src file in the current directory.  Aliases in the .%src file",
"  in the current directory override those in $HOME/.%src.  Aliases ",
"  entered with this command or by runtime input file override those",
"  defined in either .%src file.  Aliases may be deleted by entering an",
"  empty string for the second argument.  If redirection characters are to",
"  be part of the command string, the command string must be enclosed by",
"  quotation marks.\n",
"  Note that there are a number of helpful built-in aliases -- see the ",
"  first example below.",
"\nEXAMPLES",
"  Display the currently-defined aliases, which in this example, only",
"  consist of the built-in aliases:\n",
"    %s> alias",
"    ORIGIN   ALIAS    COMMAND",
"    builtin  man      help ",
"    builtin  ?        help ",
"    builtin  quit     q",
"    builtin  sf       set scroll off", 
"    builtin  sn       set scroll on", 
"    builtin  hex      set radix 16",
"    builtin  dec      set radix 10",
"    builtin  g        gdb",
"    builtin  px       p -x",
"    builtin  pd       p -d",
"    builtin  for      foreach", 
"    builtin  size     *",
"    builtin  dmesg    log",
" ",
"  Create a new alias to be added to the list:\n",
"    %s> alias kp kmem -p",
"    ORIGIN   ALIAS    COMMAND",
"    runtime  kp       kmem -p\n",
"  Create an alias with redirection characters:\n",
"    %s> alias ksd \"kmem -p | grep slab | grep DMA\"",
"    ORIGIN   ALIAS    COMMAND",
"    runtime  ksd      kmem -p | grep slab | grep DMA\n",
"  Remove an alias:\n",
"    %s> alias kp \"\"",
"    alias deleted: kp",
NULL               
};

char *help_pointer[] = {
"*",
"pointer-to short-cut",
"(struct or union command arguments)",
"  This command is a short-cut command that replaces the requirement to enter",
"  \"struct\" or \"union\" command names.  For details on the arguments to",
"  those commands, enter \"help struct\" or \"help union\".",
"\nEXAMPLES",
"  Dump the page structure at address c02943c0:",
" ",
"    %s> *page c02943c0",
"    struct page {",
"      next = 0xc0fae740, ",
"      prev = 0xc0018fb0, ",
"      inode = 0x0, ",
"      offset = 0x3f000, ",
"      next_hash = 0xc02d6310, ",
"      count = {",
"        counter = 0x1",
"      }, ",
"      flags = 0x310, ",
"      wait = 0xc02943d8, ",
"      pprev_hash = 0x0, ",
"      buffers = 0x0",
"    }",

NULL               
};


char *help_vtop[] = {
"vtop",
"virtual to physical",
"[-c [pid | taskp]] [-u|-k] address ...",
"  This command translates a user or kernel virtual address to its physical",
"  address.  Also displayed is the PTE translation, the vm_area_struct data",
"  for user virtual addresses, the mem_map page data associated with the",
"  physical page, and the swap location or file location if the page is",
"  not mapped.  The -u and -k options specify that the address is a user",
"  or kernel virtual address; -u and -k are not necessary on processors whose",
"  virtual addresses self-define themselves as user or kernel.  User addresses",
"  are translated with respect to the current context unless the -c option",
"  is used.  Kernel virtual addresses are translated using the swapper_pg_dir",
"  as the base page directory unless the -c option is used.",
" ",
"   -u                 The address is a user virtual address; only required",
"                      on processors with overlapping user and kernel virtual",
"                      address spaces.",
"   -k                 The address is a kernel virtual address; only required",
"                      on processors with overlapping user and kernel virtual",
"                      address spaces.",
"   -c [pid | taskp]   Translate the virtual address from the page directory",
"                      of the specified PID or hexadecimal task_struct pointer.",
"                      However, if this command is invoked from \"foreach vtop\",",
"                      the pid or taskp argument should NOT be entered; the",
"                      address will be translated using the page directory of",
"                      each task specified by \"foreach\".",
"   address            A hexadecimal user or kernel virtual address.",
"\nEXAMPLES",
"  Translate user virtual address 80b4000:\n",
"    %s> vtop 80b4000",
"    VIRTUAL   PHYSICAL",
"    80b4000   660f000",
"    ",
"    PAGE DIRECTORY: c37f0000",
"      PGD: c37f0080 => e0d067",
"      PMD: c37f0080 => e0d067",
"      PTE: c0e0d2d0 => 660f067",
"     PAGE: 660f000",
" ",
"      PTE    PHYSICAL  FLAGS",
"    660f067   660f000  (PRESENT|RW|USER|ACCESSED|DIRTY)",
"    ",
"      VMA      START      END      FLAGS  FILE",
"    c773daa0   80b4000   810c000    77",
"    ",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c0393258   660f000         0     17000  1  uptodate",
"    ",
"  Translate kernel virtual address c806e000, first using swapper_pg_dir",
"  as the page directory base, and secondly, using the page table base",
"  of PID 1359:\n",
"    %s> vtop c806e000",
"    VIRTUAL   PHYSICAL",
"    c806e000  2216000",
"    ",
"    PAGE DIRECTORY: c0101000",
"      PGD: c0101c80 => 94063",
"      PMD: c0101c80 => 94063",
"      PTE: c00941b8 => 2216063",
"     PAGE: 2216000",
" ",
"      PTE    PHYSICAL  FLAGS",
"    2216063   2216000  (PRESENT|RW|ACCESSED|DIRTY)",
"    ",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c02e9370   2216000         0         0  1  ",
"    ",
"    %s> vtop -c 1359 c806e000",
"    VIRTUAL   PHYSICAL",
"    c806e000  2216000",
"    ",
"    PAGE DIRECTORY: c5caf000",
"      PGD: c5cafc80 => 94063",
"      PMD: c5cafc80 => 94063",
"      PTE: c00941b8 => 2216063",
"     PAGE: 2216000",
"    ",
"      PTE    PHYSICAL  FLAGS",
"    2216063   2216000  (PRESENT|RW|ACCESSED|DIRTY)",
" ",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c02e9370   2216000         0         0  1  ",
"    ",
"  Determine swap location of user virtual address 40104000:\n",
"    %s> vtop 40104000",
"    VIRTUAL   PHYSICAL",
"    40104000  (not mapped)",
"    ",
"    PAGE DIRECTORY: c40d8000",
"      PGD: c40d8400 => 6bbe067",
"      PMD: c40d8400 => 6bbe067",
"      PTE: c6bbe410 => 58bc00  ",
" ",
"     PTE      SWAP     OFFSET",
"    58bc00  /dev/sda8   22716",
"    ",
"      VMA      START      END     FLAGS  FILE",
"    c7200ae0  40104000  40b08000    73   ",
" ",
"    SWAP: /dev/sda8  OFFSET: 22716",
NULL               
};


char *help_vm[] = { 
"vm",
"virtual memory",
"[-p | -v | -m | [-R reference] | [-f vm_flags]] [pid | taskp] ... ",
"  This command displays basic virtual memory information of a context,",
"  consisting of a pointer to its mm_struct and page dirctory, its RSS and ",
"  total virtual memory size; and a list of pointers to each vm_area_struct,",
"  its starting and ending address, vm_flags value, and file pathname.  If no",
"  arguments are entered, the current context is used.  Additionally, the -p ",
"  option translates each virtual page of each VM area to its physical address.",
"  The -R option, typically invoked from \"foreach vm\", searches for references",
"  to a supplied number, address, or filename argument, and prints only the",
"  essential information leading up to and including the reference.  ",
"  Alternatively, the -m or -v options may be used to dump the task's mm_struct",
"  or all of its vm_area_structs respectively.  The -p, -v, -m, -R and -f",
"  options are all mutually exclusive.\n",
"            -p  translate each virtual page to its physical address, or if", 
"                the page is not mapped, its swap device and offset, or",
"                filename and offset.",
"  -R reference  search for references to this number or filename.",
"            -m  dump the mm_struct assocated with the task.",
"            -v  dump all of the vm_area_structs associated with the task.",
"   -f vm_flags  translate the bits of a FLAGS (vm_flags) value.", 
"           pid  a process PID.",
"         taskp  a hexadecimal task_struct pointer.",
"\nEXAMPLES",
"  Display the virtual memory data of the current context:\n",
"    %s> vm",
"    PID: 30986  TASK: c0440000  CPU: 0   COMMAND: \"bash\"",
"       MM       PGD       RSS    TOTAL_VM",
"    c303fe20  c4789000    88k      1728k",
"      VMA      START      END     FLAGS  FILE",
"    c0d1f540   8048000   80ad000  1875   /bin/bash",
"    c0d1f400   80ad000   80b3000  1873   /bin/bash",
"    c0d1f880   80b3000   80ec000    77",
"    c0d1f0c0  40000000  40012000   875   /lib/ld-2.1.1.so",
"    c0d1f700  40012000  40013000   873   /lib/ld-2.1.1.so",
"    c0d1fe00  40013000  40014000    77",
"    c0d1f580  40014000  40016000    73",
"    c0d1f280  4001a000  4004b000    75   /usr/lib/libncurses.so.4.2",
"    c0d1f100  4004b000  40054000    73   /usr/lib/libncurses.so.4.2",
"    c0d1f600  40054000  40057000    73",
"    c0d1f9c0  40057000  40059000    75   /lib/libdl-2.1.1.so",
"    c0d1f800  40059000  4005a000    73   /lib/libdl-2.1.1.so",
"    c0d1fd00  4005a000  40140000    75   /lib/libc-2.1.1.so",
"    c0d1fe40  40140000  40145000    73   /lib/libc-2.1.1.so",
"    c0d1f780  40145000  40148000    73",
"    c0d1f140  40148000  40150000    75   /lib/libnss_files-2.1.1.so",
"    c0d1fa80  40150000  40151000    73   /lib/libnss_files-2.1.1.so",
"    c0d1fb00  40151000  4015a000    75   /lib/libnss_nisplus-2.1.1.so",
"    c5f754e0  4015a000  4015b000    73   /lib/libnss_nisplus-2.1.1.so",
"    c0d1fec0  4015b000  4016d000    75   /lib/libnsl-2.1.1.so",
"    c5f75460  4016d000  4016e000    73   /lib/libnsl-2.1.1.so",
"    c5f75420  4016e000  40170000    73",
"    c5f753e0  40170000  40178000    75   /lib/libnss_nis-2.1.1.so",
"    c5f753a0  40178000  40179000    73   /lib/libnss_nis-2.1.1.so",
"    c0d1f240  bfffc000  c0000000   177\n",
"  Display the virtual memory data along with page translations for PID 386:",
"\n    %s> vm -p 386",
"    PID: 386    TASK: c11cc000  CPU: 0   COMMAND: \"atd\"",
"       MM       PGD       RSS    TOTAL_VM",
"    c7e30560  c10e5000    104k     1112k",
"      VMA      START      END     FLAGS  FILE",
"    c0fbe6a0   8048000   804b000  1875   /usr/sbin/atd",
"     VIRTUAL  PHYSICAL",
"     8048000  20e1000",
"     8049000  17c6000",
"     804a000  1f6f000",
"      VMA      START      END     FLAGS  FILE",
"    c61e0ba0   804b000   804d000  1873   /usr/sbin/atd",
"     VIRTUAL  PHYSICAL",
"     804b000  254d000",
"     804c000  6a9c000",
"      VMA      START      END     FLAGS  FILE",
"    c61e04e0   804d000   8050000    77   ",
"     VIRTUAL  PHYSICAL",
"     804d000  219d000",
"     804e000  2617000",
"     804f000  SWAP: /dev/sda8  OFFSET: 24225",
"      VMA      START      END     FLAGS  FILE",
"    c61e0720  40000000  40012000   875   /lib/ld-2.1.1.so",
"     VIRTUAL  PHYSICAL",
"    40000000  FILE: /lib/ld-2.1.1.so  OFFSET: 0",
"    40001000  FILE: /lib/ld-2.1.1.so  OFFSET: 1000",
"    40002000  FILE: /lib/ld-2.1.1.so  OFFSET: 2000",
"    40003000  FILE: /lib/ld-2.1.1.so  OFFSET: 3000",
"    40004000  FILE: /lib/ld-2.1.1.so  OFFSET: 4000",
"    40005000  FILE: /lib/ld-2.1.1.so  OFFSET: 5000",
"    ...",
" ",
"  Although the -R option is typically invoked from \"foreach vm\", it can be",
"  executed directly.  This example displays all VM areas with vm_flags of 75:\n",
"    %s> vm -R 75",
"    PID: 694    TASK: c0c76000  CPU: 1   COMMAND: \"crash\"",
"       MM       PGD      RSS    TOTAL_VM",
"    c6c43110  c0fe9000  8932k    10720k ",
"      VMA       START      END   FLAGS  FILE",
"    c322c0d0  40019000  4004a000    75  /usr/lib/libncurses.so.4.2",
"    c67537c0  40056000  40071000    75  /lib/libm-2.1.1.so",
"    c6753d00  40072000  40074000    75  /lib/libdl-2.1.1.so",
"    c6753540  40075000  40081000    75  /usr/lib/libz.so.1.1.3",
"    c6753740  40085000  4016b000    75  /lib/libc-2.1.1.so",
" ",
"  One reason to use -R directly is to pare down the output associated with",
"  the -p option on a task with a huge address space.  This example displays",
"  the page data associated with virtual address 40121000:\n",
"    %s> vm -R 40121000",
"    PID: 694    TASK: c0c76000  CPU: 0   COMMAND: \"crash\"",
"       MM       PGD      RSS    TOTAL_VM",
"    c6c43110  c0fe9000  8928k    10720k ",
"      VMA       START      END   FLAGS  FILE",
"    c6753740  40085000  4016b000    75  /lib/libc-2.1.1.so",
"    VIRTUAL   PHYSICAL",
"    40121000  FILE: /lib/libc-2.1.1.so  OFFSET: 9c000",
" ",
"  Display the mm_struct for PID 4777:",
"\n    %s> vm -m 4777",
"    PID: 4777   TASK: c0896000  CPU: 0   COMMAND: \"bash\"",
"    struct mm_struct {",
"      mmap = 0xc6caa1c0, ",
"      mmap_avl = 0x0, ",
"      mmap_cache = 0xc6caabc0, ",
"      pgd = 0xc100a000, ",
"      count = {",
"        counter = 0x1",
"      }, ",
"      map_count = 0x14, ",
"      mmap_sem = {",
"        count = {",
"          counter = 0x1",
"        }, ",
"        waking = 0x0, ",
"        wait = 0x0",
"      }, ",
"      context = 0x0, ",
"      start_code = 0x8048000, ",
"      end_code = 0x809c6f7, ",
"      start_data = 0x0, ",
"      end_data = 0x80a2090, ",
"      start_brk = 0x80a5420, ",
"      brk = 0x80b9000, ",
"      start_stack = 0xbffff9d0, ",
"      arg_start = 0xbffffad1, ",
"      arg_end = 0xbffffad7, ",
"      env_start = 0xbffffad7, ",
"      env_end = 0xbffffff2, ",
"      rss = 0xf6, ",
"      total_vm = 0x1a3, ",
"      locked_vm = 0x0, ",
"      def_flags = 0x0, ",
"      cpu_vm_mask = 0x0, ",
"      swap_cnt = 0x23d,",
"      swap_address = 0x0, ",
"      segments = 0x0",
"    }",
" ",
"  Display all of the vm_area_structs for task c47d4000:",
"\n    %s> vm -v c47d4000",
"    PID: 4971   TASK: c47d4000  CPU: 1   COMMAND: \"login\"",
"    struct vm_area_struct {",
"      vm_mm = 0xc4b0d200, ",
"      vm_start = 0x8048000, ",
"      vm_end = 0x804d000, ",
"      vm_next = 0xc3e3abd0, ",
"      vm_page_prot = {",
"        pgprot = 0x25",
"      }, ",
"      vm_flags = 0x1875, ",
"      vm_avl_height = 0x1, ",
"      vm_avl_left = 0x0, ",
"      vm_avl_right = 0x0, ",
"      vm_next_share = 0x0, ",
"      vm_pprev_share = 0xc3e3abf0, ",
"      vm_ops = 0xc02392a0, ",
"      vm_offset = 0x0, ",
"      vm_file = 0xc1e23660, ",
"      vm_pte = 0x0",
"    }",
"    struct vm_area_struct {",
"      vm_mm = 0xc4b0d200, ",
"      vm_start = 0x804d000, ",
"      vm_end = 0x804e000, ",
"      vm_next = 0xc3e3a010, ",
"      vm_page_prot = {",
"        pgprot = 0x25",
"      }, ",
"      vm_flags = 0x1873, ",
"      vm_avl_height = 0x2, ",
"      vm_avl_left = 0xc3e3a810, ",
"      vm_avl_right = 0xc3e3a010, ",
"      vm_next_share = 0xc3e3a810, ",
"      vm_pprev_share = 0xc3699c14",
"      ...",
" ",
"  Translate a FLAGS value:\n",
"    %s> vm -f 3875",
"    3875: (READ|EXEC|MAYREAD|MAYWRITE|MAYEXEC|DENYWRITE|EXECUTABLE|LOCKED)",
NULL               
};

char *help_task[] = {
"task",
"task_struct contents",
"[-R member[,member]] [pid | taskp] ...",
"  This command dumps a formatted display of the contents of a task_struct.",
"  Multiple task or PID numbers may be entered; if no arguments are entered,",
"  the task_struct of the current context is displayed.  The -R option,",
"  typically invoked indirectly via \"foreach task\", pares the output down",
"  to one or more structure members.",
" ",
"        pid  a process PID.",
"      taskp  a hexadecimal task_struct pointer.",
"  -R member  a comma-separated list of one or more task_struct members.",  
"\nEXAMPLES",
"  Dump the task_struct structure of the current context:\n",
"    %s> task",
"    PID: 18138  TASK: c7d12000  CPU: 1   COMMAND: \"crash\"",
"    struct task_struct {",
"      state = 0, ",
"      flags = 0, ",
"      sigpending = 0, ",
"      addr_limit = {",
"        seg = 3221225472",
"      }, ",
"      exec_domain = 0xc02386c0, ",
"      need_resched = 0, ",
"      counter = 2, ",
"      priority = 20, ",
"      avg_slice = 0, ",
"      has_cpu = 1, ",
"      processor = 1, ",
"      last_processor = 0, ",
"      lock_depth = 0, ",
"      next_task = 0xc7808000, ",
"      prev_task = 0xc6d1c000, ",
"      next_run = 0xc0252000, ",
"      prev_run = 0xc0252000, ",
"      binfmt = 0xc023abd0, ",
"      exit_code = 0, ",
"      exit_signal = 17, ",
"         .",
"         .",
"         .",
"      files = 0xc09a7d60, ",
"      mm = 0xc753fb50, ",
"      sigmask_lock = {",
"        lock = 0",
"      }, ",
"      sig = 0xc4745800, ",
"      signal = {",
"        sig = {65536, 0}",
"      }, ",
"      blocked = {",
"        sig = {65536, 0}",
"      }, ",
"      sigqueue = 0x0, ",
"      sigqueue_tail = 0xc7d124ac, ",
"      sas_ss_sp = 0, ",
"      sas_ss_size = 0",
"    }",
" ",
"  Display the ngroups and groups task_struct members for PID 2958:\n",
"    %s> task -R ngroups,groups 2958",
"    PID: 2958   TASK: c6718000  CPU: 0   COMMAND: \"bash\"",
"      ngroups = 6, ",
"      groups = {504, 8, 9, 1000, 1007, 1006, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,",
"        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},",
" ",
"  NOTE: When this command is invoked directly (i.e., not from \"foreach\"), it",
"  is not necessary to include the \"-R\" before the task_struct member name(s).",

NULL               
};

char *help_sig[] = {
"sig",
"task signal handling",
"[[-l] | [-s sigset]] | [pid | taskp] ...",
"  This command displays signal-handling data of one or more tasks.  Multiple",
"  task or PID numbers may be entered; if no arguments are entered, the signal",
"  handling data of the current context will be displayed.  The default display",
"  shows:",
" ",
"    1.  Whether the task has an unblocked signal pending.",
"    2.  The contents of the \"signal\" and \"blocked\" sigset_t structures",
"        from the task_struct, both of which are represented as a 64-bit ",
"        hexadecimal value.", 
"    3.  A formatted dump of the \"sig\" signal_struct structure referenced by",
"        the task_struct.  For each defined signal, it shows the sigaction",
"        structure address, the signal handler, the signal sigset_t mask ",
"        (also expressed as a 64-bit hexadecimal value), and the flags.",
"    4.  For each queued signal, if any, its signal number and associated",
"        siginfo structure address.",
" ",
"  The -l option lists the signal numbers and their name(s).  The -s option",
"  translates a 64-bit hexadecimal value representing the contents of a",
"  sigset_t structure into the signal names whose bits are set.",
" ",        
"        pid  a process PID.",
"      taskp  a hexadecimal task_struct pointer.",
"         -l  displays the defined signal numbers and names.",
"  -s sigset  translates a 64-bit hexadecimal value representing a sigset_t",
"             into a list of signal names associated with the bits set.",
"\nEXAMPLES",
"  Dump the signal-handling data of PID 614:\n",
"    %s> sig 614",
"    PID: 614    TASK: c6f26000  CPU: 1   COMMAND: \"httpd\"",
"    SIGPENDING: no",
"        SIGNAL: 0000000000000000",
"       BLOCKED: 0000000000000000",
"    SIGNAL_STRUCT: c1913800  COUNT: 1",
"     SIG SIGACTION  HANDLER       MASK       FLAGS   ",
"     [1]  c1913804  8057c98 0000000000000201 0 ",
"     [2]  c1913818  8057c8c 0000000000000000 0 ",
"     [3]  c191382c  SIG_DFL 0000000000000000 0 ",
"     [4]  c1913840  8057bd8 0000000000000000 80000000 (SA_RESETHAND)",
"     [5]  c1913854  SIG_DFL 0000000000000000 0 ",
"     [6]  c1913868  8057bd8 0000000000000000 80000000 (SA_RESETHAND)",
"     [7]  c191387c  8057bd8 0000000000000000 80000000 (SA_RESETHAND)",
"     [8]  c1913890  SIG_DFL 0000000000000000 0 ",
"     [9]  c19138a4  SIG_DFL 0000000000000000 0 ",
"    [10]  c19138b8  8057c98 0000000000000201 0 ",
"    [11]  c19138cc  8057bd8 0000000000000000 80000000 (SA_RESETHAND)",
"    [12]  c19138e0  SIG_DFL 0000000000000000 0 ",
"    [13]  c19138f4  SIG_IGN 0000000000000000 0 ",
"    [14]  c1913908  SIG_DFL 0000000000000000 0 ",
"    [15]  c191391c  8057c8c 0000000000000000 0 ",
"    [16]  c1913930  SIG_DFL 0000000000000000 0 ",
"    [17]  c1913944  SIG_DFL 0000000000000000 0 ",
"    [18]  c1913958  SIG_DFL 0000000000000000 0 ",
"    [19]  c191396c  SIG_DFL 0000000000000000 0 ",
"    [20]  c1913980  SIG_DFL 0000000000000000 0 ",
"    [21]  c1913994  SIG_DFL 0000000000000000 0 ",
"    [22]  c19139a8  SIG_DFL 0000000000000000 0 ",
"    [23]  c19139bc  SIG_DFL 0000000000000000 0 ",
"    [24]  c19139d0  SIG_DFL 0000000000000000 0 ",
"    [25]  c19139e4  SIG_DFL 0000000000000000 0 ",
"    [26]  c19139f8  SIG_DFL 0000000000000000 0 ",
"    [27]  c1913a0c  SIG_DFL 0000000000000000 0 ",
"    [28]  c1913a20  SIG_DFL 0000000000000000 0 ",
"    [29]  c1913a34  SIG_DFL 0000000000000000 0 ",
"    [30]  c1913a48  SIG_DFL 0000000000000000 0 ",
"    [31]  c1913a5c  SIG_DFL 0000000000000000 0 ",
"    SIGQUEUE: (empty)",
"    ",
"  Translate the sigset_t mask value, cut-and-pasted from the signal handling",
"  data from signals 1 and 10 above:",
" ",
"    %s> sig -s 0000000000000201",
"    SIGHUP SIGUSR1",
" ",
"  List the signal numbers and their names:",
" ",
"    %s> sig -l",
"     [1] SIGHUP",
"     [2] SIGINT",
"     [3] SIGQUIT",
"     [4] SIGILL",
"     [5] SIGTRAP",
"     [6] SIGABRT/SIGIOT",
"     [7] SIGBUS",
"     [8] SIGFPE",
"     [9] SIGKILL",
"    [10] SIGUSR1",
"    [11] SIGSEGV",
"    [12] SIGUSR2",
"    [13] SIGPIPE",
"    [14] SIGALRM",
"    [15] SIGTERM",
"    [16] SIGSTKFLT",
"    [17] SIGCHLD/SIGCLD",
"    [18] SIGCONT",
"    [19] SIGSTOP",
"    [20] SIGTSTP",
"    [21] SIGTTIN",
"    [22] SIGTTOU",
"    [23] SIGURG",
"    [24] SIGXCPU",
"    [25] SIGXFSZ",
"    [26] SIGVTALRM",
"    [27] SIGPROF",
"    [28] SIGWINCH",
"    [29] SIGIO/SIGPOLL",
"    [30] SIGPWR",
"    [31] SIGSYS",

NULL               
};

char *help_struct[] = {
"struct",
"structure contents",
"struct_name[.member] [[-o] | [[-r] [address | symbol] [count]]]",
"  This command displays either a structure definition, or a formatted display",
"  of the contents of a structure at a specified address.  When no address is",
"  specified, the structure definition is shown along with the structure size.",
"  A structure member may be appended to the structure name in order to limit",
"  the scope of the data displayed to that particular member; when no address",
"  is specified, the member's offset and definition are shown.\n",
"    struct_name  name of a C-code structure used by the kernel.",
"        .member  name of a structure member.",
"             -o  show member offsets when displaying structure definitions.",
"             -r  raw dump of structure data.",
"        address  hexadecimal address of a structure.",  
"         symbol  symbolic reference to the address of a structure.",
"          count  count of structures to dump from an array of structures;",
"                 if used, this must be the last argument entered.\n",
"  Structure data, sizes, and member offsets are shown in the current output",
"  radix.",
"\nEXAMPLES",
"  Display the vm_area_struct at address c1e44f10:\n",
"    %s> struct vm_area_struct c1e44f10",
"    struct vm_area_struct {",
"      vm_mm = 0xc2857750,",
"      vm_start = 0x8048000, ",
"      vm_end = 0x80a5000, ",
"      vm_next = 0xc1e44a10,",
"      vm_page_prot = {",
"        pgprot = 0x25      ",
"      },",
"      vm_flags = 0x1875,",
"      vm_avl_height = 0x2,   ",
"      vm_avl_left = 0xc30fe200,",
"      vm_avl_right = 0xc30fed00,",
"      vm_next_share = 0x0,       ",
"      vm_pprev_share = 0xc1e44a30,",
"      vm_ops = 0xc0215ca0,",
"      vm_offset = 0x0,       ",
"      vm_file = 0xc0bfdc70,",
"      vm_pte = 0   ",
"    }\n",
"  Display the definition and size of a vm_area_struct structure.  This first",
"  example below displays just the structure and size.  The second example",
"  uses the -o option to also display member offsets.  Both examples were",
"  run with the output radix set to 10 (decimal):\n",
"    %s> struct vm_area_struct",
"    struct vm_area_struct {",
"        struct mm_struct *vm_mm;",
"        long unsigned int vm_start;",
"        long unsigned int vm_end;",
"        struct vm_area_struct *vm_next;",
"        pgprot_t vm_page_prot;",
"        short unsigned int vm_flags;",
"        short int vm_avl_height;",
"        struct vm_area_struct *vm_avl_left;",
"        struct vm_area_struct *vm_avl_right;",
"        struct vm_area_struct *vm_next_share;",
"        struct vm_area_struct **vm_pprev_share;",
"        struct vm_operations_struct *vm_ops;",
"        long unsigned int vm_offset;",
"        struct file *vm_file;",
"        long unsigned int vm_pte;",
"    }",
"    SIZE: 56\n",
"    %s> struct vm_area_struct -o",
"    struct vm_area_struct {",
"       [0] struct mm_struct *vm_mm;",
"       [4] long unsigned int vm_start;",
"       [8] long unsigned int vm_end;",
"      [12] struct vm_area_struct *vm_next;",
"      [16] pgprot_t vm_page_prot;",
"      [20] short unsigned int vm_flags;",
"      [22] short int vm_avl_height;",
"      [24] struct vm_area_struct *vm_avl_left;",
"      [28] struct vm_area_struct *vm_avl_right;",
"      [32] struct vm_area_struct *vm_next_share;",
"      [36] struct vm_area_struct **vm_pprev_share;",
"      [40] struct vm_operations_struct *vm_ops;",
"      [44] long unsigned int vm_offset;",
"      [48] struct file *vm_file;",
"      [52] long unsigned int vm_pte;",
"    }",
"    SIZE: 56\n",
"  Display the pgd member of the mm_struct at address c2857bd0:\n",
"    %s> struct mm_struct.pgd c2857bd0",
"      pgd = 0xc168c000,\n",
"  Display the definition and offset of the pgd member of an mm_struct:\n",
"    %s> struct mm_struct.pgd",
"    struct mm_struct {",
"       [12] pgd_t *pgd;",
"    }\n",
"  Display the array of tcp_sl_timer structures declared by tcp_slt_array[]:\n",
"    %s> struct tcp_sl_timer tcp_slt_array 4",
"    struct tcp_sl_timer {",
"      count = {",
"        counter = 0x0       ",
"      },",
"      period = 0x32,      ",
"      last = 0x1419e4,  ",
"      handler = 0xc0164854  <tcp_syn_recv_timer>",
"    }",
"    struct tcp_sl_timer {",
"      count = {",
"        counter = 0x2       ",
"      },",
"      period = 0x753,     ",
"      last = 0x14a6df,  ",
"      handler = 0xc01645b0  <tcp_keepalive>",
"    }",
"    struct tcp_sl_timer {",
"      count = {",
"        counter = 0x0       ",
"      },",
"      period = 0x2ee,     ",
"      last = 0x143134,  ",
"      handler = 0xc016447c  <tcp_twkill>",
"    }",
"    struct tcp_sl_timer {",
"      count = {",
"        counter = 0x0       ",
"      },",
"      period = 0x64,      ",
"      last = 0x143198,  ",
"      handler = 0xc0164404  <tcp_bucketgc>",
"    }",
"\nNOTE",
"  If the structure name does not conflict with any %s command name, the",
"  \"struct\" command may be dropped.  Accordingly, the examples above could",
"  also have been accomplished like so:\n",
"    %s> vm_area_struct c1e44f10",
"    %s> vm_area_struct",
"    %s> vm_area_struct -o",
"    %s> mm_struct.pgd c2857bd0",
"    %s> mm_struct.pgd",
"    %s> tcp_sl_timer tcp_slt_array 4\n",
"  Lastly, the short-cut \"*\" pointer-to command may also be used to negate",
"  the need to enter the \"struct\" command name (enter \"help *\" for details).",
NULL               
};


char *help_union[] = {
"union",
"union contents",
"union_name[.member] [[-o] | [[-r] [address | symbol] [count]]]",
"  This command displays either a union definition, or a formatted display",
"  of the contents of a union at a specified address.  When no address is",
"  specified, the union definition is shown along with the union size.",
"  A union member may be appended to the structure name in order to limit",
"  the scope of the data displayed to that particular member; when no address",
"  is specified, the member's offset (always 0) and definition are shown.\n",
"    struct_name  name of a C-code union used by the kernel.",
"        .member  name of a union member.",  
"             -o  show member offsets when displaying union definitions.",
"                 (always 0)",
"             -r  raw dump of union data.",
"        address  hexadecimal address of a union.",  
"         symbol  symbolic reference to the address of a union.",
"          count  count of unions to dump from an array of unions; if used,",
"                 this must be the last argument entered.\n",
"  Union data, sizes, and member offsets are shown in the current output radix.",
"\nEXAMPLES",
"\n  Display the bdflush_param union definition, and then an instance of it:\n",
"    %s> union bdflush_param",
"    union bdflush_param {",
"        struct {",
"            int nfract;",
"            int ndirty;",
"            int nrefill;",
"            int nref_dirt;",
"            int dummy1;",
"            int age_buffer;",
"            int age_super;",
"            int dummy2;",
"            int dummy3;",
"        } b_un;",
"        unsigned int data[9];",
"    }",
"    ",
"    SIZE: 36  (0x24)",
" ",
"    %s> union bdflush_param bdf_prm",
"    union bdflush_param {",
"      b_un = {",
"        nfract = 40, ",
"        ndirty = 500, ",
"        nrefill = 64, ",
"        nref_dirt = 256, ",
"        dummy1 = 15, ",
"        age_buffer = 3000, ",
"        age_super = 500, ",
"        dummy2 = 1884, ",
"        dummy3 = 2",
"      }, ",
"      data = {40, 500, 64, 256, 15, 3000, 500, 1884, 2}",
"    }",
"\nNOTE",
"  If the union name does not conflict with any %s command name, the",
"  \"union\" command may be dropped.  Accordingly, the examples above could",
"  also have been accomplished like so:\n",
"    %s> bdflush_param",
"    %s> bdflush_param bdf_prm",
" ",
"  Lastly, the short-cut \"*\" (pointer-to) command may also be used to negate",
"  the need to enter the \"union\" command name (enter \"help *\" for details).",
NULL               
};

char *help_ptov[] = {
"ptov",
"physical to virtual",
"address ...",
"  This command translates a hexadecimal physical address into a kernel ",
"  virtual address.",
"\nEXAMPLES",
"  Translate physical address 56e000 into a kernel virtual address:\n",
"    %s> ptov 56e000",
"    VIRTUAL   PHYSICAL",
"    c056e000  56e000",
NULL               
};

char *help_mod[] = {
"mod",
"module information and loading of symbols and debugging data",
"[ -s module [objfile] | -d module | -S | -D | -r ] ",
"  With no arguments, this command displays basic information of the currently",
"  installed modules, consisting of the module address, name, size, and the",
"  object file name, if known.\n",
"  The arguments are concerned with with the loading or deleting of symbolic",
"  and debugging data from a module's object file:\n",

"  -s module [objfile]  Loads symbolic and debugging data from the object file",
"                       for the module specified.  If no objectfile argument is",
"                       appended, a search will be made for an object file",
"                       consisting of the module name with a .o suffix,",
"                       starting at the /lib/modules/<release> directory on",
"                       the host system.  If an objfile argument is appended,",
"                       that file will be used.",
"            -d module  Deletes the symbolic and debugging data of the module",
"                       specified.",
"                   -S  Load symbolic and debugging data from the object file",
"                       for all loaded modules.  For each module, a search",
"                       will be made for an object file consisting of the",
"                       module name with a .o suffix, starting at the",
"                       /lib/modules/<release> directory of the host system.",
"                   -D  Deletes the symbolic and debugging data of all modules.",
"                   -r  Reinitialize module data. All currently-loaded symbolic",
"                       and debugging data will be deleted, and the installed",
"                       module list will be updated (live system only).",
" ",
"  After symbolic and debugging data have been loaded, backtraces and text",
"  disassembly will be displayed appropriately.  Depending upon the processor",
"  architecture, data may also printed symbolically with the \"p\" command;",
"  at a minimum, the \"rd\" command may be used with module data symbols.",
" ",
"  If %s can recognize that the set of modules has changed while running a",
"  session on a live kernel, the module data will be reinitialized the next",
"  time this command is run; the -r option forces the reinitialization.",
"\nEXAMPLES",
"  Display the currently-installed modules:\n",
"    %s> mod",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8019000  soundcore    2788  (not loaded)",
"    c801b000  soundlow      336  (not loaded)",
"    c801d000  sound       59864  (not loaded)",
"    c802d000  ad1848      15728  (not loaded)",
"    c8032000  uart401      6000  (not loaded)",
"    c8035000  cs4232       2472  (not loaded)",
"    c8043000  opl3        11048  (not loaded)",
"    c8047000  3c59x       18152  (not loaded)",
"    c804d000  sunrpc      53796  (not loaded)",
"    c805c000  lockd       31528  (not loaded)",
"    c8065000  nfsd       151896  (not loaded)",
"    c8092000  nfs         29752  (not loaded)",
" ",
"  Load the symbolic and debugging data of all modules:\n",
"    %s> mod -S",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8019000  soundcore    2788  /lib/modules/2.2.5-15/misc/soundcore.o",
"    c801b000  soundlow      336  /lib/modules/2.2.5-15/misc/soundlow.o",
"    c801d000  sound       59864  /lib/modules/2.2.5-15/misc/sound.o",
"    c802d000  ad1848      15728  /lib/modules/2.2.5-15/misc/ad1848.o",
"    c8032000  uart401      6000  /lib/modules/2.2.5-15/misc/uart401.o",
"    c8035000  cs4232       2472  /lib/modules/2.2.5-15/misc/cs4232.o",
"    c8043000  opl3        11048  /lib/modules/2.2.5-15/misc/opl3.o",
"    c8047000  3c59x       18152  /lib/modules/2.2.5-15/net/3c59x.o",
"    c804d000  sunrpc      53796  /lib/modules/2.2.5-15/misc/sunrpc.o",
"    c805c000  lockd       31528  /lib/modules/2.2.5-15/fs/lockd.o",
"    c8065000  nfsd       151896  /lib/modules/2.2.5-15/fs/nfsd.o",
"    c8092000  nfs         29752  /lib/modules/2.2.5-15/fs/nfs.o",
"    ",
"  Load the symbolic and debugging data of the soundcore module from its",
"  known location:",
" ",
"    %s> mod -s soundcore",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8019000  soundcore    2788  /lib/modules/2.2.5-15/misc/soundcore.o",
"    ",
"  Delete the current symbolic and debugging data of the soundcore module, ",
"  and then re-load it from a specified object file:",
" ",
"    %s> mod -d soundcore",
"    %s> mod -s soundcore /tmp/soundcore.o",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8019000  soundcore    2788  /tmp/soundcore.o",
" ",
"  After installing a new kernel module on a live system, reinitialize the",
"  installed module list:\n",
"    %s> !insmod mdacon",
"    %s> mod",
"    mod: NOTE: modules have changed on this system -- reinitializing",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8019000  soundcore    2788  (not loaded)",
"    c801b000  soundlow      336  (not loaded)",
"    c801d000  sound       59864  (not loaded)",
"    c802d000  ad1848      15728  (not loaded)",
"    c8032000  uart401      6000  (not loaded)",
"    c8035000  cs4232       2472  (not loaded)",
"    c8043000  opl3        11048  (not loaded)",
"    c8047000  3c59x       18152  (not loaded)",
"    c804d000  sunrpc      53796  (not loaded)",
"    c805c000  lockd       31528  (not loaded)",
"    c8065000  nfs         29752  (not loaded)",
"    c806e000  autofs       9316  (not loaded)",
"    c8072000  nfsd       151896  (not loaded)",
"    c80a1000  mdacon       3556  (not loaded)",
NULL               
};


char *help__list[] = {
"list",
"linked list",
"[[-o] offset] [-e end] [-s struct[.member]] [-H] start",
"  This command dumps the contents of a linked list.  The entries in a linked",
"  are typically data structures that are tied together in one of two formats:",
" ",
"  1. A starting address points to a data structure; that structure contains",
"     a member that is a pointer to the next structure, and so on.  The list",
"     typically ends when a \"next\" pointer value contains one of the",
"     following:\n",
"       a. a NULL pointer.",
"       b. a pointer to the start address.",
"       c. a pointer to the first item pointed to by the start address.",
"       d. a pointer to its containing structure.",
"  ",
"  2. Many Linux lists are linked via embedded list_head structures contained ",
"     within the data structures in the list.  The linked list is headed by an",
"     external LIST_HEAD, which is simply a list_head structure initialized to",
"     point to itself, signifying that the list is empty:",
" ",
"       struct list_head {",
"           struct list_head *next, *prev;",
"       };",
" ",
"       #define LIST_HEAD_INIT(name) { &(name), &(name) }"
" ",
"       #define LIST_HEAD(name) struct list_head name = LIST_HEAD_INIT(name)",
" ",
"     In the case of list_head-type lists, the \"next\" pointer is the address",
"     of the embedded list_head structure in the next structure, and not the",
"     address of the structure itself.  The list typically ends when the",
"     list_head's next pointer points back to the LIST_HEAD address.",
" ",
"  This command can handle both types of linked list; in both cases the list",
"  of addresses that are dumped are the addresses of the data structures",
"  themselves.",
" ",
"  The arguments are as follows:\n",
"  [-o] offset  The offset within the structure to the \"next\" pointer",
"               (default is 0).  If non-zero, the offset may be entered",
"               in either of two manners:\n",
"               1. In \"structure.member\" format; the \"-o\" is not necessary.", 
"               2. A number of bytes; the \"-o\" is only necessary on processors",
"                  where the offset value could be misconstrued as a kernel",
"                  virtual address.\n",
"       -e end  If the list ends in a manner unlike the typical manners that",
"               are described above, an explicit ending address value may be",
"               entered.",
"    -s struct  For each address in list, format and print as this type of",
"               structure; use the \"struct.member\" format in order to display", 
"               a particular member of the structure.",
" ",
"  The meaning of the \"start\" argument, which can be expressed either",
"  symbolically or in hexadecimal format, depends upon whether the -H option",
"  is pre-pended or not:",
" ",
"      start  The address of the first structure in the list.",
"   -H start  The address of the LIST_HEAD structure, typically expressed",
"             symbolically.",
"\nEXAMPLES",
"  Note that each task_struct is linked to its parent's task_struct via the",
"  p_pptr member:",
" ",
"    %s> struct task_struct.p_pptr",
"    struct task_struct {",
"       [136] struct task_struct *p_pptr;",
"    }",
" ",    
"  That being the case, given a task_struct pointer of c169a000, show its ",
"  parental hierarchy back to the \"init_task\" (the \"swapper\" task):\n",
"    %s> list task_struct.p_pptr c169a000",
"    c169a000",
"    c0440000",
"    c50d0000",
"    c0562000",
"    c0d28000",
"    c7894000",
"    c6a98000",
"    c009a000",
"    c0252000\n",
"  Given that the \"task_struct.p_pptr\" offset is 136 bytes, the same",
"  result could be accomplished like so:\n",
"    %s> list 136 c169a000",
"    c169a000",
"    c0440000",
"    c50d0000",
"    c0562000",
"    c0d28000",
"    c7894000",
"    c6a98000",
"    c009a000",
"    c0252000",
" ",
"  The list of currently-registered file system types are headed up by a",
"  struct file_system_type pointer named \"file_systems\", and linked by",
"  the \"next\" field in each file_system_type structure.  The following",
"  sequence displays the address and name of each registered file system type:",
" ",
"    %s> p file_systems",
"    file_systems = $2 = (struct file_system_type *) 0xc02ebea0",
"    %s> list file_system_type.next -s file_system_type.name 0xc02ebea0",
"    c02ebea0",
"      name = 0xc0280372 \"proc\", ",
"    c02fd4a0",
"      name = 0xc02bf348 \"sockfs\", ",
"    c02eb544",
"      name = 0xc027c25a \"tmpfs\", ",
"    c02eb52c",
"      name = 0xc027c256 \"shm\", ",
"    c02ebbe0",
"      name = 0xc027e054 \"pipefs\", ",
"    c02ec9c0",
"      name = 0xc0283c13 \"ext2\", ",
"    c02ecaa8",
"      name = 0xc0284567 \"iso9660\", ",
"    c02ecc08",
"      name = 0xc0284cf5 \"nfs\", ",
"    c02edc60",
"      name = 0xc028d832 \"autofs\", ",
"    c02edfa0",
"      name = 0xc028e1e0 \"devpts\"",
" ",
"  The system run queue is a linked list headed up by the \"runqueue_head\",",
"  which is defined like so:",
" ",
"    static LIST_HEAD(runqueue_head);",
" ",
"  The run queue linking is done with the \"run_list\" member of the task_struct:",
" ",
"    %s> struct task_struct.run_list",
"    struct task_struct {",
"        [60] struct list_head run_list;",
"    }",
" ",
"  Therefore, to view the list of task_struct addresses in the run queue,",
"  either of the following commands will work:\n",
"    %s> list task_struct.run_list -H runqueue_head",
"    f79ac000",
"    f7254000",
"    f7004000",
"    %s> list 60 -H runqueue_head",
"    f79ac000",
"    f7254000",
"    f7004000",
" ",
"  Lastly, the vfsmount structures of the mounted filesystems are linked by",
"  the LIST_HEAD \"vfsmntlist\", which uses the mnt_list list_head of each ",
"  vfsmount structure in the list.  To dump each vfsmount structure in the list,",
"  append the -s option:\n",
"    %s> list -H vfsmntlist vfsmount.mnt_list -s vfsmount",
"    c3fc9e60",
"    struct vfsmount {",
"      mnt_hash = {",
"        next = 0xc3fc9e60, ",
"        prev = 0xc3fc9e60",
"      }, ",
"      mnt_parent = 0xc3fc9e60, ",
"      mnt_mountpoint = 0xc3fc5dc0, ",
"      mnt_root = 0xc3fc5dc0, ",
"      mnt_instances = {",
"        next = 0xc3f60a74, ",
"        prev = 0xc3f60a74",
"      }, ",
"      mnt_sb = 0xc3f60a00, ",
"      mnt_mounts = {",
"        next = 0xf7445e08, ",
"        prev = 0xf7445f88",
"      }, ",
"      mnt_child = {",
"        next = 0xc3fc9e88, ",
"        prev = 0xc3fc9e88",
"      }, ",
"      mnt_count = {",
"        counter = 209",
"      }, ",
"      mnt_flags = 0, ",
"      mnt_devname = 0xc8465b20 \"/dev/root\", ",
"      mnt_list = {",
"        next = 0xf7445f9c, ",
"        prev = 0xc02eb828",
"      }, ",
"      mnt_owner = 0",
"    }",
"    f7445f60",
"    struct vfsmount {",
"    ...",
NULL               
};


char *help_ptob[] = {
"ptob",
"page to bytes",
"page_number ...",
"  This command translates a page frame number to its byte value.",
"\nEXAMPLES",
"    %s> ptob 512a",
"    512a: 512a000",
NULL               
};

char *help_gdb[] = {
"gdb",
"gdb command",
"command ...",
"  This command passes its arguments directly to gdb for processing.",
"  This is typically not necessary, but where ambiguities between %s and",
"  gdb command names exist, this will force the command to be executed by gdb.",
"\nEXAMPLES",
"    %s> gdb help",
"    List of classes of commands:",
"    ",
"    aliases -- Aliases of other commands",
"    breakpoints -- Making program stop at certain points",
"    data -- Examining data",
"    files -- Specifying and examining files",
"    internals -- Maintenance commands",
"    obscure -- Obscure features",
"    running -- Running the program",
"    stack -- Examining the stack",
"    status -- Status inquiries",
"    support -- Support facilities",
"    tracepoints -- Tracing of program execution without stopping the program",
"    user-defined -- User-defined commands",
"    ",
"    Type \"help\" followed by a class name for a list of commands in that class.",
"    Type \"help\" followed by command name for full documentation.",
"    Command name abbreviations are allowed if unambiguous.",
NULL               
};


char *help_kmem[] = {
"kmem",
"kernel memory",
"[-f|-F|-p|-c|-C|-i|-s|-S|-v|-n] [-[l|L][a|i]] [slab-name] [address]",
"  This command displays information about the use of kernel memory.\n",
"        -f  displays the contents of the system free memory headers.",
"            also verifies that the page count equals nr_free_pages.",
"        -F  same as -f, but also dumps all pages linked to that header.",
"        -p  displays basic information about each page in the system ",
"            mem_map[] array.",
"        -c  walks through the page_hash_table and verifies page_cache_size.",
"        -C  same as -c, but also dumps all pages in the page_hash_table.",
"        -i  displays general memory usage information",
"        -s  displays basic kmalloc() slab data.",
"        -S  displays all kmalloc() slab data, including all slab objects,",
"            and whether each object is in use or is free.",
"        -v  displays the vmlist entries.",
"        -n  display NUMA data (if supported).",
"       -la  walks through the active_list and verifies nr_active_pages.",
"       -li  walks through the inactive_list and verifies nr_inactive_pages.",
"       -La  same as -la, but also dumps each page in the active_list.",
"       -Li  same as -li, but also dumps each page in the inactive_list.",
" slab-name  when used with -s or -S, limits the command to only the slab",
"            of name \"slab-cache\".",
"   address  when used without any flag, the address can be a kernel virtual,",
"            or physical address; a search is made through the symbol table,",
"            the kmalloc() slab subsystem, the free list, the page_hash_table,",
"            the vmalloc() vmlist subsystem, and the mem_map array. If found",
"            in any of those areas, the information will be dumped in the",
"            same manner as if the flags were used.",
"   address  when used with -s or -S, searches the kmalloc() slab subsystem",
"            for the slab containing of this virtual address, showing whether",
"            it is in use or free.",
"   address  when used with -f, the address can be either a page pointer,",
"            a physical address, or a kernel virtual address; the free_area",
"            header containing the page (if any) is displayed.",
"   address  when used with -p, the address can be either a page pointer, a",
"            physical address, or a kernel virtual address; its basic mem_map",
"            page information is displayed.",
"   address  when used with -c, the address must be a page pointer address;",
"            the page_hash_table entry containing the page is displayed.",
"   address  when used with -l, the address must be a page pointer address;",
"            the page address is displayed if it is contained with the list.",
"   address  when used with -v, the address can be a mapped kernel virtual",
"            address or physical address; the vmlist containing the address",
"            is displayed.\n",
"  All address arguments above must be expressed in hexadecimal format.",
"\nEXAMPLES",
"  Display memory usage information:\n",
"    %s> kmem -i",
"                  PAGES     TOTAL        PERCENTAGE",
"     TOTAL MEM   127777   523374592         ----",
"          FREE      928     3801088    0% of TOTAL MEM",
"          USED   126849   519573504   99% of TOTAL MEM",
"        SHARED    37981   155570176   29% of TOTAL MEM",
"       BUFFERS      369     1511424    0% of TOTAL MEM",
"        CACHED    37856   155058176   29% of TOTAL MEM",
"          SLAB    76138   311861248   59% of TOTAL MEM",
"    ",
"    TOTAL HIGH        0           0    0% of TOTAL MEM",
"     FREE HIGH        0           0    0% of TOTAL HIGH",
"     TOTAL LOW   127777   523374592  100% of TOTAL MEM",
"      FREE LOW      928     3801088    0% of TOTAL LOW",
"    ",
"    TOTAL SWAP    51198   209707008         ----",
"     SWAP USED        0           0    0% of TOTAL SWAP",
"     SWAP FREE    51198   209707008  100% of TOTAL SWAP",
" ",
"  Display and verify free memory data:\n",
"    %s> kmem -f",
"    NODE",
"      0",
"    ZONE  NAME        SIZE    FREE  MEM_MAP   START_PADDR  START_MAPNR",
"      0   DMA         4096    3372  c4000040       0            0     ",
"    AREA  SIZE  FREE_AREA_STRUCT  BLOCKS  PAGES",
"      0     4k      c02eb004           2      2",
"      1     8k      c02eb010           3      6",
"      2    16k      c02eb01c           5     20",
"      3    32k      c02eb028           4     32",
"      4    64k      c02eb034           5     80",
"      5   128k      c02eb040           3     96",
"      6   256k      c02eb04c           3    192",
"      7   512k      c02eb058           1    128",
"      8  1024k      c02eb064           1    256",
"      9  2048k      c02eb070           5   2560",
"    ZONE  NAME        SIZE    FREE  MEM_MAP   START_PADDR  START_MAPNR",
"      1   Normal    225280  202269  c4044040    1000000        4096   ",
"    AREA  SIZE  FREE_AREA_STRUCT  BLOCKS  PAGES",
"      0     4k      c02eb0b8           1      1",
"      1     8k      c02eb0c4           2      4",
"      2    16k      c02eb0d0           0      0",
"      3    32k      c02eb0dc           1      8",
"      4    64k      c02eb0e8           1     16",
"      5   128k      c02eb0f4           0      0",
"      6   256k      c02eb100           0      0",
"      7   512k      c02eb10c           0      0",
"      8  1024k      c02eb118           0      0",
"      9  2048k      c02eb124         395 202240",
"    ZONE  NAME        SIZE    FREE  MEM_MAP   START_PADDR  START_MAPNR",
"      2   HighMem   819200  748686  c4ee0040    38000000      229376  ",
"    AREA  SIZE  FREE_AREA_STRUCT  BLOCKS  PAGES",
"      0     4k      c02eb16c          10     10",
"      1     8k      c02eb178           2      4",
"      2    16k      c02eb184           0      0",
"      3    32k      c02eb190           2     16",
"      4    64k      c02eb19c           1     16",
"      5   128k      c02eb1a8           1     32",
"      6   256k      c02eb1b4           1     64",
"      7   512k      c02eb1c0           0      0",
"      8  1024k      c02eb1cc           0      0",
"      9  2048k      c02eb1d8        1462 748544",
"    ",
"    nr_free_pages: 954327  (verified)",
" ",
"  Dump all the base addresses of each free memory area from above:\n",
"    %s> kmem -F",
"    NODE",
"      0",
"    ZONE  NAME        SIZE    FREE  MEM_MAP   START_PADDR  START_MAPNR",
"      0   DMA         4096    3372  c4000040       0            0     ",
"    AREA  SIZE  FREE_AREA_STRUCT",
"      0     4k      c02eb004      ",
"    c400ded8",
"    c4042528",
"    AREA  SIZE  FREE_AREA_STRUCT",
"      1     8k      c02eb010      ",
"    c400de50",
"    c400cee8",
"    c40424a0",
"    AREA  SIZE  FREE_AREA_STRUCT",
"      2    16k      c02eb01c      ",
"    c400dd40",
"    c400cf70",
"    c40425b0",
"    c400f7d0",
"    c40028a0",
"    AREA  SIZE  FREE_AREA_STRUCT",
"      3    32k      c02eb028      ",
"    c4042280",
"    c400f8e0",
"    c4002680",
"    c4000260",
"    AREA  SIZE  FREE_AREA_STRUCT",
"      4    64k      c02eb034      ",
"    c400d080",
"    c4041e40",
"    ...",
" ",
"  Dump the mem_map[] array:\n",
"    %s> kmem -p",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c0294000         0         0         0  0  DMA,reserved",
"    c0294028      1000         0         0  0  DMA,reserved",
"    c0294050      2000         0         0  0  DMA,reserved",
"    c0294078      3000         0    2cf000  1  DMA,uptodate",
"    c02940a0      4000         0         0  1  DMA,slab",
"    c02940c8      5000         0         0  1  DMA,slab",
"    c02940f0      6000         0         0  1  DMA,slab",
"    c0294118      7000         0         0  1  DMA,slab",
"    c0294140      8000         0         0  1  DMA,slab",
"    c0294168      9000         0         0  0  DMA,slab",
"    c0294190      a000         0         0  0  DMA,slab",
"    c02941b8      b000         0         0  0  DMA,slab",
"    c02941e0      c000         0         0  1  DMA,slab",
"    c0294208      d000         0         0  0  DMA,slab",
"    c0294230      e000         0         0  1  DMA,slab",
"    c0294258      f000         0         0  0  DMA,slab",
"    c0294280     10000         0     11000  1  DMA,uptodate",
"    c02942a8     11000         0     10000  1  DMA,uptodate",
"    c02942d0     12000         0     1e000  1  DMA,uptodate",
"    c02942f8     13000         0    153000  1  DMA,uptodate",
"    c0294320     14000  c3243650     4b000  1  DMA,referenced,uptodate",
"    c0294348     15000  c3243650     4a000  1  DMA,referenced,uptodate",
"    ...",
"    ",
"  Use the commands above with a page pointer or a physical address argument:\n",
"    %s> kmem -f c40425b0",
"    NODE ",
"      0 ",
"    ZONE  NAME        SIZE    FREE  MEM_MAP   START_PADDR  START_MAPNR",
"      0   DMA         4096    3372  c4000040       0            0     ",
"    AREA  SIZE  FREE_AREA_STRUCT ",
"      2    16k      c02eb01c      ",
"    c40425b0  (c40425b0 is 1st of 4 pages) ",
" ",
"    %s> kmem -p c035de00",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c035de00   50c0000         0    129000  0  uptodate\n",
"    %s> kmem -p 50c0000",
"      PAGE    PHYSICAL   INODE     OFFSET  CNT FLAGS",
"    c035de00   50c0000         0    129000  0  uptodate\n",
"  Display the vmlist entry data:\n",
"    %s> kmem -v",
"    VM_STRUCT      ADDRESS RANGE       SIZE",
"     c009c560   c8000000 - c8002000    8192",
"     c009c620   c8002000 - c8004000    8192",
"     c009cda0   c8004000 - c8016000   73728",
"     c009cd70   c8016000 - c8019000   12288",
"     c009cf80   c8019000 - c801b000    8192",
"     c009cfb0   c801b000 - c801d000    8192",
"     c009cef0   c801d000 - c802d000   65536",
"     c3afd060   c802d000 - c8032000   20480",
"     c3afd090   c8032000 - c8035000   12288",
"     c3afd0c0   c8035000 - c8037000    8192",
"     c3afd150   c8037000 - c8039000    8192",
"     c3afd180   c8039000 - c803b000    8192",
"     c3afd210   c803b000 - c803d000    8192",
"     c3afd2a0   c803d000 - c8040000   12288",
"     c3afd2d0   c8040000 - c8043000   12288",
"     c3afd300   c8043000 - c8047000   16384",
"     c3afddb0   c8047000 - c804d000   24576",
"     c2f8a320   c804d000 - c805c000   61440",
"     c2f8a380   c805c000 - c8065000   36864",
"     c2f8a3b0   c8065000 - c806e000   36864",
"     c2f8aa70   c806e000 - c8095000  159744",
"     c2f8ab60   c8095000 - c8097000    8192",
"     c2f519e0   c8097000 - c8099000    8192",
" ",
"  Determine (and verify) the page cache size:\n",
"    %s> kmem -c",
"    page_cache_size: 18431 (verified)",
" ",
"  Dump all pages in the page_hash_table:\n",
"    %s> kmem -C",
"    page_hash_table[0]",
"    c0325b40",
"    c03a0598",
"    c03b4070",
"    c0364c28",
"    c0357690",
"    c02ef338",
"    c02d7c60",
"    c02c11e0",
"    c02a3d70",
"    page_hash_table[1]",
"    c0394ce8",
"    c03c4218",
"    c03b4048",
"    c0364c00",
"    c0357668",
"    c02d6e50",
"    c02d7dc8",
"    c02c0cb8",
"    c02db630",
"    c02ebad0",
"    page_hash_table[2]",
"    c037e808",
"    c034e248",
"    c03b4020",
"    c02ec868",
"    c03baa60",
"    ...",
"    page_hash_table[2047]",
"    c033a798",
"    c0390b48",
"    c03b4098",
"    c0364890",
"    c03576b8",
"    c02d2c38",
"    c02d7c88",
"    c02de5d8",
"    ",
"    page_cache_size: 18437 (verified)",
"    ",
"  Find the page_hash_table entry containing page c03576b8:\n",
"    %s> kmem -c c03576b8",
"    page_hash_table[2047]",
"    c03576b8",
" ",
"  Display kmalloc() slab data:\n",
"    %s> kmem -s",
"    CACHE     NAME                OBJSIZE  ALLOCATED     TOTAL  SLABS  SSIZE",
"    c02eadc0 kmem_cache               232         58        68      4     4k",
"    f79c2888 ip_vs_conn               128          0         0      0     4k",
"    f79c2970 tcp_tw_bucket             96          0         0      0     4k",
"    f79c2a58 tcp_bind_bucket           32         12       565      5     4k",
"    f79c2b40 tcp_open_request          64          0        59      1     4k",
"    f79c2c28 inet_peer_cache           64          1        59      1     4k",
"    f79c2d10 ip_fib_hash               32         11       339      3     4k",
"    f79c2df8 ip_dst_cache             160          8       120      5     4k",
"    f79c2ee0 arp_cache                128          1        30      1     4k",
"    c8402970 blkdev_requests           96      30208     37800    945     4k",
"    c8402a58 nfs_read_data            384          0         0      0     4k",
"    c8402b40 nfs_write_data           384          0         0      0     4k",
"    c8402c28 nfs_page                  96          0         0      0     4k",
"    c8402d10 dnotify cache             20          0         0      0     4k",
"    c8402df8 file lock cache           92          3       336      8     4k",
"    c8402ee0 fasync cache              16          0         0      0     4k",
"    c84027a0 uid_cache                 32          3       339      3     4k",
"    c84026b8 skbuff_head_cache        160        320       624     26     4k",
"    c84025d0 sock                     832         32       180     20     8k",
"    c84024e8 sigqueue                 132          0       203      7     4k",
"    c8402400 cdev_cache                64         19       472      8     4k",
"    c8402318 bdev_cache                64          8       236      4     4k",
"    c8402230 mnt_cache                 96         11       120      3     4k",
"    c8402148 inode_cache              480        817       848    106     4k",
"    c8402060 dentry_cache             128       1352      1470     49     4k",
"    c8403ee0 filp                      96        244       440     11     4k",
"    c8403df8 names_cache             4096          0        12     12     4k",
"    c8403d10 buffer_head               96      14936     16000    400     4k",
"    c8403c28 mm_struct                128         25       240      8     4k",
"    c8403b40 vm_area_struct            64        393      1298     22     4k",
"    c8403a58 fs_cache                  64         30       472      8     4k",
"    c8403970 files_cache              416         30       135     15     4k",
"    c8403888 signal_act              1312         32        99     33     4k",
"    c84037a0 size-131072(DMA)      131072          0         0      0   128k",
"    c84036b8 size-131072           131072          1         1      1   128k",
"    c84035d0 size-65536(DMA)        65536          0         0      0    64k",
"    c84034e8 size-65536             65536          0         0      0    64k",
"    c8403400 size-32768(DMA)        32768          0         0      0    32k",
"    c8403318 size-32768             32768          0         1      1    32k",
"    c8403230 size-16384(DMA)        16384          0         0      0    16k",
"    c8403148 size-16384             16384          0         0      0    16k",
"    c8403060 size-8192(DMA)          8192          0         0      0     8k",
"    c8401ee0 size-8192               8192          1         2      2     8k",
"    c8401df8 size-4096(DMA)          4096          0         0      0     4k",
"    c8401d10 size-4096               4096         30        30     30     4k",
"    c8401c28 size-2048(DMA)          2048          0         0      0     4k",
"    c8401b40 size-2048               2048         37       132     66     4k",
"    c8401a58 size-1024(DMA)          1024          0         0      0     4k",
"    c8401970 size-1024               1024        301       328     82     4k",
"    c8401888 size-512(DMA)            512          0         0      0     4k",
"    c84017a0 size-512                 512        141       168     21     4k",
"    c84016b8 size-256(DMA)            256          0         0      0     4k",
"    c84015d0 size-256                 256         80       435     29     4k",
"    c84014e8 size-128(DMA)            128          0         0      0     4k",
"    c8401400 size-128                 128        508       840     28     4k",
"    c8401318 size-64(DMA)              64          0         0      0     4k",
"    c8401230 size-64                   64        978      1357     23     4k",
"    c8401148 size-32(DMA)              32          0         0      0     4k",
"    c8401060 size-32                   32       1244      1808     16     4k",

" ",
"  Display all slab data in the \"arp_cache\" cache:\n",
"    %s> kmem -S arp_cache",
"    CACHE     NAME                OBJSIZE  ALLOCATED     TOTAL  SLABS  SSIZE",
"    f79c2ee0 arp_cache                128          1        30      1     4k",
"    SLAB      MEMORY    TOTAL  ALLOCATED  FREE",
"    f729d000  f729d0a0     30          1    29",
"    FREE / [ALLOCATED]",
"       f729d0a0  (cpu 7 cache)",
"       f729d120  (cpu 7 cache)",
"       f729d1a0  (cpu 7 cache)",
"       f729d220  (cpu 7 cache)",
"       f729d2a0  (cpu 7 cache)",
"       f729d320  (cpu 7 cache)",
"       f729d3a0  (cpu 7 cache)",
"       f729d420  (cpu 7 cache)",
"       f729d4a0  (cpu 7 cache)",
"       f729d520  (cpu 7 cache)",
"       f729d5a0  (cpu 7 cache)",
"       f729d620  (cpu 7 cache)",
"       f729d6a0  (cpu 7 cache)",
"       f729d720  (cpu 7 cache)",
"       f729d7a0  (cpu 7 cache)",
"       f729d820  (cpu 7 cache)",
"       f729d8a0  (cpu 7 cache)",
"       f729d920  (cpu 7 cache)",
"       f729d9a0  (cpu 7 cache)",
"       f729da20  (cpu 7 cache)",
"       f729daa0  (cpu 7 cache)",
"       f729db20  (cpu 7 cache)",
"       f729dba0  (cpu 7 cache)",
"       f729dc20  (cpu 7 cache)",
"       f729dca0  (cpu 7 cache)",
"       f729dd20  (cpu 7 cache)",
"       f729dda0  (cpu 7 cache)",
"       f729de20  (cpu 7 cache)",
"       f729dea0  (cpu 3 cache)",
"      [f729df20]",
" ",
"  Search the kmalloc() slab subsystem for address c3fbdb60:\n",
"    %s> kmem -s c3fbdb60",
"    CACHE     NAME                OBJSIZE  ALLOCATED     TOTAL  SLABS  SSIZE",
"    c8402970 blkdev_requests           96      30208     37800    945     4k",
"    SLAB      MEMORY    TOTAL  ALLOCATED  FREE",
"    c3fbd020  c3fbd0e0     40         40     0",
"    FREE / [ALLOCATED]",
"      [c3fbdb60]",
" ",
"  Make a generic search (no flags) for the same address c3fbdb60:\n",
"    %s> kmem c3fbdb60 ",
"    CACHE     NAME                OBJSIZE  ALLOCATED     TOTAL  SLABS  SSIZE",
"    c8402970 blkdev_requests           96      30208     37800    945     4k",
"    SLAB      MEMORY    TOTAL  ALLOCATED  FREE",
"    c3fbd020  c3fbd0e0     40         40     0 ",
"    FREE / [ALLOCATED]",
"      [c3fbdb60]",
" ",
"      PAGE     PHYSICAL   MAPPING    INDEX CNT FLAGS",
"    c410ee74    3fbd000         0         0  1 slab",
" ",
"  Verify the active and inactive page lists:\n",
"    %s> kmem -la -li",
"    nr_active_pages: 1893 (verified)",
"    nr_inactive_pages: 2491 (verified)",

"\n  Display NUMA data (if supported):\n",
"    %s> kmem  -n",
"    NODE    SIZE    PGLIST_DATA   BOOTMEM_DATA",
"      0    32766      c02dff40      c033e14c",
"    ",
"    MEM_MAP   START_PADDR  START_MAPNR",
"    c100000c       0            0",
"    ",
"    ZONE  NAME        SIZE    FREE   MEM_MAP  START_PADDR  START_MAPNR",
"      0   DMA         4096      88  c100000c            0            0",
"      1   Normal     28670     607  c104c00c      1000000         4096",
"      2   HighMem        0       0         0            0            0",

NULL               
};

char *help_dis[] = {
"dis",
"disassemble",
"[-r][-l] [address | symbol | (expression)] [count]",
"  This command disassembles source code instructions starting (or ending) at",
"  a text address that may be expressed by value, symbol or expression:\n",
"            -r  (reverse) displays all instructions from the start of the ",
"                routine up to and including the designated address.",
"            -l  displays source code line number data in addition to the ",
"                disassembly output.", 
"       address  starting hexadecimal text address.",
"        symbol  symbol of starting text address.",
"  (expression)  expression evaluating to a starting text address.",
"         count  the number of instructions to be disassembled (default is 1).",
"                If no count argument is entered, and the starting address",
"                is entered as a text symbol, then the whole routine will be",
"                disassembled.  The count argument is ignored when used with",
"                the -r option.",
"\nEXAMPLES",
"  Disassemble the sys_signal() routine without, and then with, line numbers:\n",
"    %s> dis sys_signal",
"    0xc0112c88 <sys_signal>:        push   \%ebp",
"    0xc0112c89 <sys_signal+1>:      mov    \%esp,\%ebp",
"    0xc0112c8b <sys_signal+3>:      sub    $0x28,\%esp",
"    0xc0112c8e <sys_signal+6>:      mov    0xc(\%ebp),\%eax",
"    0xc0112c91 <sys_signal+9>:      mov    \%eax,0xffffffec(\%ebp)",
"    0xc0112c94 <sys_signal+12>:     movl   $0xc0000000,0xfffffff0(\%ebp)",
"    0xc0112c9b <sys_signal+19>:     lea    0xffffffd8(\%ebp),\%eax",
"    0xc0112c9e <sys_signal+22>:     push   \%eax",
"    0xc0112c9f <sys_signal+23>:     lea    0xffffffec(\%ebp),\%eax",
"    0xc0112ca2 <sys_signal+26>:     push   \%eax",
"    0xc0112ca3 <sys_signal+27>:     pushl  0x8(\%ebp)",
"    0xc0112ca6 <sys_signal+30>:     call   0xc01124b8 <do_sigaction>",
"    0xc0112cab <sys_signal+35>:     test   \%eax,\%eax",
"    0xc0112cad <sys_signal+37>:     jne    0xc0112cb2 <sys_signal+42>",
"    0xc0112caf <sys_signal+39>:     mov    0xffffffd8(\%ebp),\%eax",
"    0xc0112cb2 <sys_signal+42>:     leave",
"    0xc0112cb3 <sys_signal+43>:     ret",
" ",
"    %s> dis -l sys_signal",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1074",
"    0xc0112c88 <sys_signal>:        push   \%ebp",
"    0xc0112c89 <sys_signal+1>:      mov    \%esp,\%ebp",
"    0xc0112c8b <sys_signal+3>:      sub    $0x28,\%esp",
"    0xc0112c8e <sys_signal+6>:      mov    0xc(\%ebp),\%eax",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1078",
"    0xc0112c91 <sys_signal+9>:      mov    \%eax,0xffffffec(\%ebp)",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1079",
"    0xc0112c94 <sys_signal+12>:     movl   $0xc0000000,0xfffffff0(\%ebp)",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1081",
"    0xc0112c9b <sys_signal+19>:     lea    0xffffffd8(\%ebp),\%eax",
"    0xc0112c9e <sys_signal+22>:     push   \%eax",
"    0xc0112c9f <sys_signal+23>:     lea    0xffffffec(\%ebp),\%eax",
"    0xc0112ca2 <sys_signal+26>:     push   \%eax",
"    0xc0112ca3 <sys_signal+27>:     pushl  0x8(\%ebp)",
"    0xc0112ca6 <sys_signal+30>:     call   0xc01124b8 <do_sigaction>",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1083",
"    0xc0112cab <sys_signal+35>:     test   \%eax,\%eax",
"    0xc0112cad <sys_signal+37>:     jne    0xc0112cb2 <sys_signal+42>",
"    0xc0112caf <sys_signal+39>:     mov    0xffffffd8(\%ebp),\%eax",
"    /usr/src/linux-2.2.5/kernel/signal.c: 1084",
"    0xc0112cb2 <sys_signal+42>:     leave",
"    0xc0112cb3 <sys_signal+43>:     ret",
" ",
"  Given a return address expression of \"do_no_page+65\", find out the ",
"  function that do_no_page() calls by using the reverse flag:\n",
"    %s> dis -r (do_no_page+65)",
"    0xc011ea68 <do_no_page>:        push   \%ebp",
"    0xc011ea69 <do_no_page+1>:      mov    \%esp,\%ebp",
"    0xc011ea6b <do_no_page+3>:      push   \%edi",
"    0xc011ea6c <do_no_page+4>:      push   \%esi",
"    0xc011ea6d <do_no_page+5>:      push   \%ebx",
"    0xc011ea6e <do_no_page+6>:      mov    0xc(\%ebp),\%ebx",
"    0xc011ea71 <do_no_page+9>:      mov    0x10(\%ebp),\%edx",
"    0xc011ea74 <do_no_page+12>:     mov    0x14(\%ebp),\%edi",
"    0xc011ea77 <do_no_page+15>:     mov    0x28(\%ebx),\%eax",
"    0xc011ea7a <do_no_page+18>:     test   \%eax,\%eax",
"    0xc011ea7c <do_no_page+20>:     je     0xc011ea85 <do_no_page+29>",
"    0xc011ea7e <do_no_page+22>:     mov    0x18(\%eax),\%ecx",
"    0xc011ea81 <do_no_page+25>:     test   \%ecx,\%ecx",
"    0xc011ea83 <do_no_page+27>:     jne    0xc011eab0 <do_no_page+72>",
"    0xc011ea85 <do_no_page+29>:     mov    $0xffffe000,\%eax",
"    0xc011ea8a <do_no_page+34>:     and    \%esp,\%eax",
"    0xc011ea8c <do_no_page+36>:     decl   0x30(\%eax)",
"    0xc011ea8f <do_no_page+39>:     jns    0xc011ea9a <do_no_page+50>",
"    0xc011ea91 <do_no_page+41>:     lock btrl $0x0,0xc022fb60",
"    0xc011ea9a <do_no_page+50>:     push   \%edi",
"    0xc011ea9b <do_no_page+51>:     mov    0x18(\%ebp),\%esi",
"    0xc011ea9e <do_no_page+54>:     push   \%esi",
"    0xc011ea9f <do_no_page+55>:     push   \%ebx",
"    0xc011eaa0 <do_no_page+56>:     mov    0x8(\%ebp),\%esi",
"    0xc011eaa3 <do_no_page+59>:     push   \%esi",
"    0xc011eaa4 <do_no_page+60>:     call   0xc011e9e4 <do_anonymous_page>",
"    0xc011eaa9 <do_no_page+65>:     jmp    0xc011eb47 <do_no_page+223>",
"    ",
NULL               
};


char *help_eval[] = {
"eval",
"evaluate",
"[-b][-l] (expression) | value",
"  This command evaluates an expression or numeric value, and displays its",
"  result in hexadecimal, decimal, octal and binary. If the resultant value",
"  is an integral number of gigabytes, megabytes, or kilobytes, a short-hand",
"  translation of the number will also be shown next to the hexadecimal",
"  value.  If the most significant bit is set, the decimal display will show",
"  both unsigned and signed (negative) values.  Expressions must of the format", 
"  (x operator y), where \"x\" and \"y\" may be either numeric values or",
"  symbols.  The list of operators are:\n",
"                     +  -  &  |  ^  *  \%  /  <<  >>\n",
"  Enclosing the expression within parentheses is optional except when the",
"  \"|\", \"<<\" or \">>\" operators are used.  The single \"value\" argument may",
"  be a number or symbol.  Number arguments must be hexadecimal or decimal.",
"  A leading \"0x\" identifies a number as hexadecimal, but is not required",
"  when obvious.  Numbers may be followed by the letters \"k\" or \"K\", \"m\"",
"  or \"M\", and \"g\" or \"G\", which multiplies the value by a factor of 1024,",
"  1 megabyte or 1 gigabyte, respectively.  Numeric arguments may be preceded",
"  by the one's complement operator ~.",
" ",
"    -b  Indicate which bit positions in the resultant value are set.",
"    -l  Numeric arguments are presumed to be 64-bit values, and the result",
"        will be expressed as a 64-bit value. (ignored on 64-bit processors)",
" ",
" The -b and -l options must precede the expression or value arguments.",
"\nEXAMPLES",
"   %s> eval 128m",
"   hexadecimal: 8000000  (128MB)",
"       decimal: 134217728  ",
"         octal: 1000000000",
"        binary: 00001000000000000000000000000000",
"    ",
"   %s> eval 128 * 1m",
"   hexadecimal: 8000000  (128MB)",
"       decimal: 134217728  ",
"         octal: 1000000000",
"        binary: 00001000000000000000000000000000",
"    ",
"   %s> eval (1 << 27)",
"   hexadecimal: 8000000  (128MB)",
"       decimal: 134217728  ",
"         octal: 1000000000",
"        binary: 00001000000000000000000000000000",
"   ",
"   %s> eval -b 41dc065",
"   hexadecimal: 41dc065",
"       decimal: 69058661  ",
"         octal: 407340145",
"        binary: 00000100000111011100000001100101",
"      bits set: 26 20 19 18 16 15 14 6 5 2 0 ",
" ",
"   %s> eval -lb 64g",
"   hexadecimal: 1000000000  (64GB)",
"       decimal: 68719476736",
"         octal: 1000000000000",
"        binary: 0000000000000000000000000001000000000000000000000000000000000000",
"      bits set: 36",
NULL               
};


char *help_sym[] = {
"sym",
"translate a symbol to its virtual address, or vice-versa ",
"[-l] | [-M] | [-m module] | [-p|-n] | [-q string] | [symbol | vaddr]",
"  This command translates a symbol to its virtual address, or a static ",
"  kernel virtual address to its symbol -- or to a symbol-plus-offset value,",
"  if appropriate.  Additionally, the symbol type is shown in parentheses, ",
"  and if the symbol is a known text value, the file and line number are shown.\n",
"              -l  dumps all symbols and their values.",
"              -M  dumps the current set of module symbols.",
"       -m module  dumps the current set of symbols for a specified module.",
"              -p  display the target symbol and the previous symbol.",
"              -n  display the target symbol and the next symbol.",
"       -q string  searches for all symbols containing \"string\".",
"          symbol  a kernel text or data symbol.",
"           vaddr  a kernel virtual address.",
"\nEXAMPLES",
"  Translate data symbol jiffies to its value, and vice-versa:\n",
"    %s> sym jiffies",
"    c0213ae0 (D) jiffies\n", 
"    %s> sym c0213ae0",
"    c0213ae0 (D) jiffies\n", 
"  Translate a text address to its symbolic value and source file:\n",
"    %s> sym c0109944",
"    c0109944 (T) system_call+0x34  ../linux-2.2.5/arch/i386/kernel/signal.c: 723\n",
"  Dump the whole symbol table:\n",
"    %s> sym -l",
"    c0100000 (T) _stext",
"    c0100000 (A) _text",
"    c0100000 (t) startup_32",
"    c0100000 (T) stext",
"    c01000a4 (t) checkCPUtype",
"    c0100139 (t) is486",
"    c0100148 (t) is386",
"    c01001b1 (t) L6",
"    c01001b3 (t) ready",
"    c01001b4 (t) check_x87",
"    c01001da (t) setup_idt",
"    c01001f7 (t) rp_sidt",
"    c0100204 (T) stack_start",
"    c010020c (t) int_msg",
"    c0100220 (t) ignore_int",
"    c0100242 (t) idt_descr",
"    c0100244 (T) idt",
"    c010024a (t) gdt_descr",
"    c010024c (T) gdt",
"    c0101000 (T) swapper_pg_dir",
"    c0102000 (T) pg0",
"    c0103000 (T) empty_bad_page",
"    c0104000 (T) empty_bad_page_table",
"    c0105000 (T) empty_zero_page",
"    ...\n",
"  Find all symbols containing the string \"pipe\":\n",
"    %s> sym -q pipe",
"    c010ec60 (T) sys_pipe  ",
"    c012f660 (t) pipe_read  ",
"    c012f7b8 (t) pipe_write  ",
"    c012f9c0 (t) pipe_lseek  ",
"    c012f9d0 (t) bad_pipe_r  ",
"    c012f9dc (t) bad_pipe_w  ",
"    c012f9e8 (t) pipe_ioctl  ",
"    c012fa18 (t) pipe_poll  ",
"    c012fb00 (t) pipe_release  ",
"    c012fb48 (t) pipe_read_release  ",
"    c012fb5c (t) pipe_write_release  ",
"    c012fb70 (t) pipe_rdwr_release  ",
"    c012fba0 (t) pipe_read_open  ",
"    c012fbb0 (t) pipe_write_open  ",
"    c012fbc0 (t) pipe_rdwr_open  ",
"    c012fbec (t) get_pipe_inode  ",
"    c012fcc4 (T) do_pipe  ",
"    c023a920 (D) read_pipe_fops  ",
"    c023a960 (D) write_pipe_fops  ",
"    c023a9a0 (D) rdwr_pipe_fops  ",
"    c023a9e0 (D) pipe_inode_operations  ",

"\n  Dump the symbols of the uart401 module, both before, and then after,",
"  the complete set of symbols are loaded with the \"mod -s\" command:\n",
"    %s> sym -m uart401",
"    c8032000 MODULE START: uart401",
"    c8032138 (?) uart401intr  ",
"    c803235c (?) attach_uart401  ",
"    c8032638 (?) probe_uart401  ",
"    c80326d4 (?) unload_uart401  ",
"    c8033770 MODULE END: uart401",
"    %s> mod -s uart401",
"     MODULE   NAME         SIZE  OBJECT FILE",
"    c8032000  uart401      6000  /lib/modules/2.2.14/misc/uart401.o",
"    %s> sym -m uart401",
"    c8032000 MODULE START: uart401",
"    c8032050 (t) my_notifier_call  ",
"    c8032084 (t) uart401_status  ",
"    c8032098 (t) uart401_cmd  ",
"    c80320a8 (t) uart401_read  ",
"    c80320bc (t) uart401_write  ",
"    c80320cc (t) uart401_input_loop  ",
"    c8032138 (T) uart401intr  ",
"    c8032168 (t) uart401_open  ",
"    c80321c8 (t) uart401_close  ",
"    c80321f4 (t) uart401_out  ",
"    c80322ac (t) uart401_start_read  ",
"    c80322b4 (t) uart401_end_read  ",
"    c80322bc (t) uart401_kick  ",
"    c80322c4 (t) uart401_buffer_status  ",
"    c80322cc (t) enter_uart_mode  ",
"    c803235c (T) attach_uart401  ",
"    c803259c (t) reset_uart401  ",
"    c8032638 (T) probe_uart401  ",
"    c80326d4 (T) unload_uart401  ",
"    c8032760 (T) init_module",
"    c80327cc (T) cleanup_module  ",
"    c8032b00 (d) sound_notifier  ",
"    c8032b0c (d) detected_devc  ",
"    c8032b20 (d) std_synth_info  ",
"    c8032bc0 (d) std_midi_synth  ",
"    c8033600 (d) uart401_operations  ",
"    c80336c4 (D) io  ",
"    c80336c8 (D) irq  ",
"    c80336e0 (b) hw_info.508  ",
"    c8033770 MODULE END: uart401",

"\n  Display the value of jiffies, along with the next and previous symbols:\n",
"    %s> sym -np jiffies",
"    c023027c (D) prof_shift",  
"    c0230280 (D) jiffies ", 
"    c02302a0 (D) task",
NULL               
};

char *help_files[] = {
"files",
"open files",
"[-l | -d dentry] | [-R reference] [pid | taskp] ... ",
"  This command displays information about open files of a context.",
"  It prints the context's current root directory and current working", 
"  directory, and then for each open file descriptor it prints a pointer",
"  to its file struct, a pointer to its dentry struct, a pointer to the",
"  inode, the file type, and the pathname.  If no arguments are entered,",
"  the current context is used.  The -R option, typically invoked from",
"  \"foreach files\", searches for references to a supplied number, address,",
"  or filename argument, and prints only the essential information leading",
"  up to and including the reference.  The -l and -d options are not context",
"  specific, and only show the data requested.\n",
"            -l  display files open by lockd server for client locks.",
"     -d dentry  given a hexadecimal dentry address, display its inode,",
"                super block, file type, and full pathname.",
"  -R reference  search for references to this file descriptor number,",
"                filename, or dentry, inode, or file structure address.",
"           pid  a process PID.",
"         taskp  a hexadecimal task_struct pointer.",
"\nEXAMPLES",
"  Display the open files of the current context:\n",
"    %s> files",
"    PID: 720    TASK: c67f2000  CPU: 1   COMMAND: \"innd\"",
"    ROOT: /    CWD: /var/spool/news/articles",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      0  c6b9c740  c7cc45a0  c7c939e0  CHR   /dev/null",
"      1  c6b9c800  c537bb20  c54d0000  REG   /var/log/news/news",
"      2  c6df9600  c537b420  c5c36360  REG   /var/log/news/errlog",
"      3  c74182c0  c6ede260  c6da3d40  PIPE",
"      4  c6df9720  c696c620  c69398c0  SOCK",
"      5  c6b9cc20  c68e7000  c6938d80  SOCK",
"      6  c6b9c920  c7cc45a0  c7c939e0  CHR   /dev/null",
"      7  c6b9c680  c58fa5c0  c58a1200  REG   /var/lib/news/history",
"      8  c6df9f00  c6ede760  c6da3200  PIPE",
"      9  c6b9c6e0  c58fa140  c5929560  REG   /var/lib/news/history.dir",
"     10  c7fa9320  c7fab160  c7fafd40  CHR   /dev/console",
"     11  c6b9c7a0  c58fa5c0  c58a1200  REG   /var/lib/news/history",
"     12  c377ec60  c58fa5c0  c58a1200  REG   /var/lib/news/history",
"     13  c4528aa0  c58fa6c0  c52fbb00  REG   /var/lib/news/history.pag",
"     14  c6df9420  c68e7700  c6938360  SOCK",
"     15  c6df9360  c68e7780  c6938120  SOCK",
"     16  c6b9c0e0  c68e7800  c6772000  SOCK",
"     17  c6b9c200  c6b5f9c0  c6b5cea0  REG   /var/lib/news/active",
"     21  c6b9c080  c6ede760  c6da3200  PIPE", 
" ",
"  Display the files opened by the \"crond\" daemon, which is PID 462:\n",
"  %s> files 462",
"    PID: 462    TASK: f7220000  CPU: 2   COMMAND: \"crond\"",
"    ROOT: /    CWD: /var/spool",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      0  f7534ae0  f7538de0  f7518dc0  CHR   /dev/console",
"      1  f7368f80  f72c7a40  f72f27e0  FIFO  pipe:/[1456]",
"      2  f74f3c80  f72c79c0  f72f2600  FIFO  pipe:/[1457]",
"      3  f7368b60  f72a5be0  f74300c0  REG   /var/run/crond.pid",
"      4   f7534360  f73408c0  f72c2840  REG   /var/log/cron",
"      7  f7368ce0  f72c7940  f72f2420  FIFO  pipe:/[1458]",
"      8  f7295de0  f72c7940  f72f2420  FIFO  pipe:/[1458]",
"     21  f74f36e0  f747cdc0  f747e840  CHR   /dev/null",
" ",
"  The -R option is typically invoked from \"foreach files\".  This example",
"  shows all tasks that have \"/dev/pts/4\" open:\n",
"    %s> foreach files -R pts/4",
"    PID: 18633  TASK: c310a000  CPU: 0   COMMAND: \"crash\"",
"    ROOT: /    CWD: /home/CVS_pool/crash ",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      0  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"      1  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"      2  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"    ",
"    PID: 18664  TASK: c2392000  CPU: 1   COMMAND: \"less\"",
"    ROOT: /    CWD: /home/CVS_pool/crash ",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      1  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"      2  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"    ",
"    PID: 23162  TASK: c5088000  CPU: 1   COMMAND: \"bash\"",
"    ROOT: /    CWD: /home/CVS_pool/crash ",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      0  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"      1  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"      2  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"    255  c1412850  c2cb96d0  c2cad430  CHR   /dev/pts/4",
"    ",
"    PID: 23159  TASK: c10fc000  CPU: 1   COMMAND: \"xterm\"",
"    ROOT: /    CWD: /homes/anderson/ ",
"     FD    FILE     DENTRY    INODE    TYPE  PATH",
"      5  c1560da0  c2cb96d0  c2cad430  CHR   /dev/pts/4",
" ",
"  Display information about the dentry at address f745fd60:\n",
"    %s> files -d f745fd60",
"     DENTRY    INODE    SUPERBLK  TYPE  PATH",
"     f745fd60  f7284640  f73a3e00  REG   /var/spool/lpd/lpd.lock",
NULL               
};

char *help_fuser[] = {
"fuser",
"file users",
"[pathname | inode]",
"  This command displays the tasks using specified files or sockets.",
"  Tasks will be listed that reference the file as the current working", 
"  directory, root directory, an open file descriptor, or that mmap the",
"  file.  If the file is held open in the kernel by the lockd server on",
"  behalf of a client discretionary file lock, the client hostname is",
"  listed.\n",
"    pathname  the full pathname of the file.",
"    inode     the hexadecimal inode address for the file.",
"\nEXAMPLES",
"  Display the tasks using file /usr/lib/libkfm.so.2.0.0\n",
"    %s> fuser /usr/lib/libkfm.so.2.0.0",
"     PID    TASK    COMM            USAGE",
"     779  c5e82000  \"kwm\"           mmap",
"     808  c5a8e000  \"krootwm\"       mmap",
"     806  c5b42000  \"kfm\"           mmap",
"     809  c5dde000  \"kpanel\"        mmap",     
NULL               
};

char *help_net[] = {
"net",
"network command",
"[-a] [[-s | -S] [-R ref] [pid | taskp]] [-n addr]",
"  Display various network related data:\n",
"      -a  display the ARP cache.",
"      -s  display open network socket/sock addresses, their family and type,",
"          and their source and destination addresses and ports.",
"      -S  displays open network socket/sock addresses followed by a dump",
"          of both structures.",
"  -n addr translates an IP address expressed as a decimal or hexadecimal ",
"          value into a standard numbers-and-dots notation.",
"  -R ref  socket or sock address, or file descriptor.",
"     pid  a process PID.",
"   taskp  a hexadecimal task_struct pointer.\n",
"  If no arguments are entered, the list of network devices, names and IP",
"  addresses are displayed.  The -R option, typically invoked from \"foreach net\",",
"  and in conjunction with the -s or -S options, searches for references to a",
"  socket address, sock address, or a file descriptor; if found, only the",
"  referenced fd/socket/sock data will be displayed.",
"\nEXAMPLES",
"  Display the network device list:",
"\n    %s> net", 
"     DEVICE   NAME   IP ADDRESS(ES)",
"    c0249f20  lo     127.0.0.1",
"    c7fe6d80  eth0   10.1.8.20", 
" ",
"  Dump the ARP cache:\n",
"    %s> net -a",
"    IP ADDRESS      HW TYPE    HW ADDRESS         DEVICE  STATE",
"    0.0.0.0         UNKNOWN    00 00 00 00 00 00  lo      40 (NOARP)",
"    192.168.1.1     ETHER      00:50:54:fe:ef:23  eth0    04 (STALE)",
"    192.168.1.10    ETHER      00:90:27:9c:6c:79  eth0    02 (REACHABLE)",
"    192.168.1.118   ETHER      00:c0:4f:60:00:e2  eth0    02 (REACHABLE)",
"   ",
"  Display the sockets for PID 2517, using both -s and -S output formats:\n",
"    %s> net -s 2517",
"    PID: 2517   TASK: c1598000  CPU: 1   COMMAND: \"rlogin\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     3  c57375dc  c1ff1850  INET:STREAM      10.1.8.20:1023      10.1.16.62:513",
"    ",
"    %s> net -S 2517",
"    PID: 2517   TASK: c1598000  CPU: 1   COMMAND: \"rlogin\"",
"    FD   SOCKET     SOCK",
"     3  c57375dc  c1ff1850",
"    ",
"    struct socket {",
"      state = SS_CONNECTED,",
"      flags = 131072,",
"      ops = 0xc023f820,",
"      inode = 0xc5737540,",
"      fasync_list = 0x0,",
"      file = 0xc58892b0,",
"      sk = 0xc1ff1850,",
"      wait = 0xc14d9ed4,",
"      type = 1,",
"      passcred = 0 '\\000',",
"      tli = 0 '\\000'",
"    }",
"    struct sock {",
"      sklist_next = 0xc1ff12f0,",
"      sklist_prev = 0xc216bc00,",
"      bind_next = 0x0,",
"      bind_pprev = 0xc0918448,",
"      daddr = 1041236234,",
"      rcv_saddr = 336068874,",
"      dport = 258,",
"      num = 1023,",
"      bound_dev_if = 0,",
"      next = 0x0,",
"      pprev = 0xc0286dd4,",
"      state = 1 '\\001',",
"      zapped = 0 '\\000',",
"      sport = 65283,",
"      family = 2,",
"      reuse = 0 '\\000',",
"      ...",
" "
"  Translate the rcv_saddr from above into dotted-decimal notation:\n",
"    %s> net -n 1041236234",
"    10.1.16.62",
" ",
"  From \"foreach\", find all tasks with references to socket c08ea3cc:\n",
"    %s> foreach net -s -R c08ea3cc",
"    PID: 2184   TASK: c7026000  CPU: 1   COMMAND: \"klines.kss\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2200   TASK: c670a000  CPU: 1   COMMAND: \"kpanel\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2201   TASK: c648a000  CPU: 1   COMMAND: \"kbgndwm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 19294  TASK: c250a000  CPU: 0   COMMAND: \"prefdm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2194   TASK: c62dc000  CPU: 1   COMMAND: \"kaudioserver\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2195   TASK: c6684000  CPU: 1   COMMAND: \"maudio\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2196   TASK: c6b58000  CPU: 1   COMMAND: \"kwmsound\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2197   TASK: c6696000  CPU: 0   COMMAND: \"kfm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2199   TASK: c65ec000  CPU: 0   COMMAND: \"krootwm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 694    TASK: c1942000  CPU: 0   COMMAND: \"prefdm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 698    TASK: c6a2c000  CPU: 1   COMMAND: \"X\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
"    PID: 2159   TASK: c4a5a000  CPU: 1   COMMAND: \"kwm\"",
"    FD   SOCKET     SOCK    FAMILY:TYPE         SOURCE:PORT     DESTINATION:PORT",
"     5  c08ea3cc  c50d3c80  INET:STREAM        0.0.0.0:1026         0.0.0.0:0",
"    ",
NULL               
};


char *help_waitq[] = {
"waitq",
"list tasks queued on a wait queue",
" [ symbol ] | [ struct.member struct_addr ] | [ address ]",
"  This command walks the wait queue list displaying the tasks which ",
"  are blocked on the specified wait queue.  The command differentiates",
"  between the old- and new-style wait queue structures used by the kernel.",
"  It can be invoked with the following argument types:",
" ",
"                     symbol  a global symbol of a wait queue.",
"  struct.member struct_addr  a structure name and wait queue member combination",
"                             followed by the structure's hexadecimal address.",
"                    address  a hexadecimal wait queue pointer.",
"\nEXAMPLES\n",
"  Find out if any tasks are blocked on the \"buffer_wait\" wait queue:\n",
"    %s> waitq buffer_wait",
"    wait queue \"buffer_wait\" (c02927f0) is empty",
" ",
"  See who is blocked on the \"wait_chldexit\" queue of task c5496000:\n",
"    %s> waitq task_struct.wait_chldexit c5496000",
"    PID: 30879  TASK: c5496000  CPU: 0   COMMAND: \"bash\"",
" ",
"  Display the task list waiting on a known task queue:\n",
"    %s> waitq c3534098",
"    PID: 13691  TASK: c3534000  CPU: 1   COMMAND: \"bash\"",
NULL
};


/*
 *  Find out what the help request is looking for, accepting aliases as well,
 *  and print the relevant strings from the appropriate help table.
 */
void
cmd_usage(char *cmd, int helpflag)
{
	int i;
        int found;
	char **p;
	struct command_table_entry *cp;
	char buf[BUFSIZE];
	struct alias_data *ad;

	if (STREQ(cmd, "copying")) {
		display_copying_info();
		return;
	}

	if (STREQ(cmd, "warranty")) {
		display_warranty_info();
		return;
	}

	if (STREQ(cmd, "input")) {
		display_input_info();
		return;
	}

        if (STREQ(cmd, "output")) {
                display_output_info();
                return;
        }

	if (STREQ(cmd, "all")) {
		display_input_info();
                display_output_info();
		help_init();
		for (i = 0; i < pc->ncmds; i++)
			cmd_usage(pc->cmdlist[i], COMPLETE_HELP);
		display_warranty_info();
		display_copying_info();
		return;
	}

	if (STREQ(cmd, "commands")) {
		display_commands();
		return;
	}

	if (STREQ(cmd, "README")) {
		display_README();
		return;
	}

	found = FALSE;
retry:
	if ((cp = get_command_table_entry(cmd))) {
		if ((p = cp->help_data))
			found = TRUE;
	}

       /*
	*  Check for alias names or gdb commands.
	*/
	if (!found) {
		if ((ad = is_alias(cmd))) {
			cmd = ad->args[0];
			goto retry;
		}

		if (helpflag == SYNOPSIS) { 
                	fprintf(fp,
                         "No usage data for the \"%s\" command is available.\n",
				cmd);
			RESTART();
		}

		if (STREQ(pc->curcmd, "help")) {
			if (cp)
                		fprintf(fp,
                          "No help data for the \"%s\" command is available.\n",
					cmd);
			else if (!gdb_pass_through(concat_args(buf, 0, FALSE), 
				NULL, GNU_RETURN_ON_ERROR))
				fprintf(fp, 
				    "No help data for \"%s\" is available.\n",
                                	cmd);
		}
		return;
        }

	p++;

        if (helpflag == SYNOPSIS) {
                p++;
                fprintf(fp, "Usage: %s ", cmd);
		fprintf(fp, *p, pc->program_name, pc->program_name);
		fprintf(fp, "\nEnter \"help %s\" for details.\n", cmd);
                RESTART();
        }

        fprintf(fp, "\nNAME\n  %s - ", cmd);
        fprintf(fp, *p, pc->program_name);

        fprintf(fp, "\n\nSYNOPSIS\n");
        p++;
        fprintf(fp, "  %s ", cmd);
        fprintf(fp, *p, pc->program_name);

        fprintf(fp,"\n\nDESCRIPTION\n");
        p++;
        do {
		if (strstr(*p, "%") && !strstr(*p, "%s")) 
			print_verbatim(fp, *p);
		else
                	fprintf(fp, *p, pc->program_name, pc->program_name);
		fprintf(fp, "\n");
                p++;
        } while (*p);

        fprintf(fp, "\n");
}


static
char *input_info[] = {

"Interactive %s commands are gathered using the GNU readline library,",
"which implements a command line history mechanism, and a command line editing",
"mode.\n", 

"The command line history consists of a numbered list of previously run ",
"commands, which can be viewed by entering \"h\" at any time.  A previously", 
"run command can be re-executed in a number of manners:",
" ",       
"  1. To re-execute the last command, enter \"r\" alone, or \"!!\".",
"  2. Enter \"r\" followed by the number identifying the desired command.",
"  3. Enter \"r\" followed by a uniquely-identifying set of characters from",
"     the beginning of the desired command string.",
"  4. Recycle back through the command history list by hitting the up-arrow",
"     key until the desired command is re-displayed, and then hit <ENTER>.",
"     If you go too far back, hit the down-arrow key.",
" ",
"The command line editing mode can be set to emulate either vi or emacs.",
"The mode can be set in the following manners, listed in increasing order of",
"precedence:",
" ",
"  1. The setting of the EDITOR environment variable.",
"  2. An entry in either in a .%src file, which can be located either",
"     in the current directory or in your home directory.  The entry must",
"     be of the form \"set vi\" or \"set emacs\".",
"  3. By use of the %s \"-e\" command line option, as in \"-e vi\" or \"-e emacs\".",
" ",
"To edit a previously entered command:",
" ",
"  1. Recycle back through the command history list until the desired command",
"     is re-displayed.",
"  2. Edit the line using vi or emacs editing commands, as appropriate. ",
"  3. Hit <ENTER>.",
" ",
"It should be noted that command line re-cycling may be accomplished by using",
"the CTRL-p and CTRL-n keys instead of the up- and down-arrow keys; in vi mode",
"you can enter <ESC>, then \"k\" to cycle back, or \"j\" to cycle forward.",
" ",
"Lastly, a set of %s commands may be entered into a regular file that can",
"used as input, again using standard command line syntax:\n",
"  %s> < inputfile\n",
"An input file may be also be run from the %s command line using the -i ",
"option:\n",
"  $ %s -i inputfile",
"",
"Lastly, if a command is entered that is not recognized, it is checked against",
"the kernel's list of variables, structure, union or typedef names, and if ",
"found, the command is passed to p, struct, union or whatis.  That being the ",
"case, as long as a kernel variable/structure/union name is different than any",
"of the current commands, the appropriate command above will be executed.  If",
"not, the command will be passed on to the built-in gdb command for execution.",
NULL
};


/*
 *  Display information concerning command input options: history,
 *  command recall, command-line editing, and input files.
 */
static void
display_input_info(void)
{
	int i;

        for (i = 0; input_info[i]; i++) {
                fprintf(fp, input_info[i],  
                        pc->program_name, pc->program_name);
                fprintf(fp, "\n");
	}
}

static
char *output_info[] = {

"\nBy default, %s command output is piped to \"/usr/bin/less -E -X\" along",
"with a prompt line.  This behavior can be turned off in two ways:\n",
"  1. Enter \"set scroll off\" during runtime.",
"  2. Enter \"set scroll off\" in a .%src file, which can be located either",
"      in the current directory or in your home directory.\n",
"Command output may be piped to an external command using standard command",
"line pipe syntax.  For example:\n", 
"  %s> log | grep MHz\n",
"Command output may be redirected to a file using standard command line syntax.",
"For example:\n",
"  %s> foreach bt > bt.all\n",
"Use double brackets to append the output to a pre-existing file:\n",
"  %s> ps >> crash.data\n",
"The default output radix for gdb output and certain %s commands is decimal.",
"This can be changed to hexadecimal by using the built-in alias \"hex\".",
"It can be reverted back to decimal by using the built-in alias \"dec\".",
" ",
NULL
};

/*
 *  Display information concerning command output options.
 */
static void
display_output_info(void)
{
        int i;

        for (i = 0; output_info[i]; i++) {
                fprintf(fp, output_info[i], 
			pc->program_name, pc->program_name);
                fprintf(fp, "\n");
	}
}


/*
 *  Display the program name, version, and the GPL-required stuff for
 *  interactive programs.
 */
void
display_version(void)
{
	int i;

	if (pc->flags & SILENT)
		return;

        fprintf(fp, "\n%s %s\n", pc->program_name, pc->program_version);

        for (i = 0; version_info[i]; i++) 
                fprintf(fp, "%s\n", version_info[i]);
}

static 
char *version_info[] = {

"Copyright (C) 1999, 2000, 2001, 2002  Mission Critical Linux, Inc.",
"This program is free software, covered by the GNU General Public License,",
"and you are welcome to change it and/or distribute copies of it under",
"certain conditions.  Enter \"help copying\" to see the conditions.",
"This program has absolutely no warranty.  Enter \"help warranty\" for details.",
" ",
NULL
};

/*
 *  "help copying" output
 */
static void
display_copying_info(void)
{
	int i;

	for (i = 0; !strstr(gnu_public_license[i], "NO WARRANTY"); i++)
		fprintf(fp, "%s\n", gnu_public_license[i]);
}

/*
 *  "help warranty" output
 */
static void
display_warranty_info(void)
{
        int i;

        for (i = 0; !strstr(gnu_public_license[i], "NO WARRANTY"); i++)
                ;

	do {
		fprintf(fp, "%s\n", gnu_public_license[i]);
	} while (!strstr(gnu_public_license[i++], "END OF TERMS"));
}

static
char *gnu_public_license[] = {
"",
"		    GNU GENERAL PUBLIC LICENSE",
"		       Version 2, June 1991",
"",
" Copyright (C) 1989, 1991 Free Software Foundation, Inc.",
"                       59 Temple Place, Suite 330, Boston, MA  02111-1307  USA",
" Everyone is permitted to copy and distribute verbatim copies",
" of this license document, but changing it is not allowed.",
"",
"			    Preamble",
"",
"  The licenses for most software are designed to take away your",
"freedom to share and change it.  By contrast, the GNU General Public",
"License is intended to guarantee your freedom to share and change free",
"software--to make sure the software is free for all its users.  This",
"General Public License applies to most of the Free Software",
"Foundation's software and to any other program whose authors commit to",
"using it.  (Some other Free Software Foundation software is covered by",
"the GNU Library General Public License instead.)  You can apply it to",
"your programs, too.",
"",
"  When we speak of free software, we are referring to freedom, not",
"price.  Our General Public Licenses are designed to make sure that you",
"have the freedom to distribute copies of free software (and charge for",
"this service if you wish), that you receive source code or can get it",
"if you want it, that you can change the software or use pieces of it",
"in new free programs; and that you know you can do these things.",
"",
"  To protect your rights, we need to make restrictions that forbid",
"anyone to deny you these rights or to ask you to surrender the rights.",
"These restrictions translate to certain responsibilities for you if you",
"distribute copies of the software, or if you modify it.",
"",
"  For example, if you distribute copies of such a program, whether",
"gratis or for a fee, you must give the recipients all the rights that",
"you have.  You must make sure that they, too, receive or can get the",
"source code.  And you must show them these terms so they know their",
"rights.",
"",
"  We protect your rights with two steps: (1) copyright the software, and",
"(2) offer you this license which gives you legal permission to copy,",
"distribute and/or modify the software.",
"",
"  Also, for each author's protection and ours, we want to make certain",
"that everyone understands that there is no warranty for this free",
"software.  If the software is modified by someone else and passed on, we",
"want its recipients to know that what they have is not the original, so",
"that any problems introduced by others will not reflect on the original",
"authors' reputations.",
"",
"  Finally, any free program is threatened constantly by software",
"patents.  We wish to avoid the danger that redistributors of a free",
"program will individually obtain patent licenses, in effect making the",
"program proprietary.  To prevent this, we have made it clear that any",
"patent must be licensed for everyone's free use or not licensed at all.",
"",
"  The precise terms and conditions for copying, distribution and",
"modification follow.",
"",
"		    GNU GENERAL PUBLIC LICENSE",
"   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION",
"",
"  0. This License applies to any program or other work which contains",
"a notice placed by the copyright holder saying it may be distributed",
"under the terms of this General Public License.  The \"Program\", below,",
"refers to any such program or work, and a \"work based on the Program\"",
"means either the Program or any derivative work under copyright law:",
"that is to say, a work containing the Program or a portion of it,",
"either verbatim or with modifications and/or translated into another",
"language.  (Hereinafter, translation is included without limitation in",
"the term \"modification\".)  Each licensee is addressed as \"you\".",
"",
"Activities other than copying, distribution and modification are not",
"covered by this License; they are outside its scope.  The act of",
"running the Program is not restricted, and the output from the Program",
"is covered only if its contents constitute a work based on the",
"Program (independent of having been made by running the Program).",
"Whether that is true depends on what the Program does.",
"",
"  1. You may copy and distribute verbatim copies of the Program's",
"source code as you receive it, in any medium, provided that you",
"conspicuously and appropriately publish on each copy an appropriate",
"copyright notice and disclaimer of warranty; keep intact all the",
"notices that refer to this License and to the absence of any warranty;",
"and give any other recipients of the Program a copy of this License",
"along with the Program.",
"",
"You may charge a fee for the physical act of transferring a copy, and",
"you may at your option offer warranty protection in exchange for a fee.",
"",
"  2. You may modify your copy or copies of the Program or any portion",
"of it, thus forming a work based on the Program, and copy and",
"distribute such modifications or work under the terms of Section 1",
"above, provided that you also meet all of these conditions:",
"",
"    a) You must cause the modified files to carry prominent notices",
"    stating that you changed the files and the date of any change.",
"",
"    b) You must cause any work that you distribute or publish, that in",
"    whole or in part contains or is derived from the Program or any",
"    part thereof, to be licensed as a whole at no charge to all third",
"    parties under the terms of this License.",
"",
"    c) If the modified program normally reads commands interactively",
"    when run, you must cause it, when started running for such",
"    interactive use in the most ordinary way, to print or display an",
"    announcement including an appropriate copyright notice and a",
"    notice that there is no warranty (or else, saying that you provide",
"    a warranty) and that users may redistribute the program under",
"    these conditions, and telling the user how to view a copy of this",
"    License.  (Exception: if the Program itself is interactive but",
"    does not normally print such an announcement, your work based on",
"    the Program is not required to print an announcement.)",
"",
"These requirements apply to the modified work as a whole.  If",
"identifiable sections of that work are not derived from the Program,",
"and can be reasonably considered independent and separate works in",
"themselves, then this License, and its terms, do not apply to those",
"sections when you distribute them as separate works.  But when you",
"distribute the same sections as part of a whole which is a work based",
"on the Program, the distribution of the whole must be on the terms of",
"this License, whose permissions for other licensees extend to the",
"entire whole, and thus to each and every part regardless of who wrote it.",
"",
"Thus, it is not the intent of this section to claim rights or contest",
"your rights to work written entirely by you; rather, the intent is to",
"exercise the right to control the distribution of derivative or",
"collective works based on the Program.",
"",
"In addition, mere aggregation of another work not based on the Program",
"with the Program (or with a work based on the Program) on a volume of",
"a storage or distribution medium does not bring the other work under",
"the scope of this License.",
"",
"  3. You may copy and distribute the Program (or a work based on it,",
"under Section 2) in object code or executable form under the terms of",
"Sections 1 and 2 above provided that you also do one of the following:",
"",
"    a) Accompany it with the complete corresponding machine-readable",
"    source code, which must be distributed under the terms of Sections",
"    1 and 2 above on a medium customarily used for software interchange; or,",
"",
"    b) Accompany it with a written offer, valid for at least three",
"    years, to give any third party, for a charge no more than your",
"    cost of physically performing source distribution, a complete",
"    machine-readable copy of the corresponding source code, to be",
"    distributed under the terms of Sections 1 and 2 above on a medium",
"    customarily used for software interchange; or,",
"",
"    c) Accompany it with the information you received as to the offer",
"    to distribute corresponding source code.  (This alternative is",
"    allowed only for noncommercial distribution and only if you",
"    received the program in object code or executable form with such",
"    an offer, in accord with Subsection b above.)",
"",
"The source code for a work means the preferred form of the work for",
"making modifications to it.  For an executable work, complete source",
"code means all the source code for all modules it contains, plus any",
"associated interface definition files, plus the scripts used to",
"control compilation and installation of the executable.  However, as a",
"special exception, the source code distributed need not include",
"anything that is normally distributed (in either source or binary",
"form) with the major components (compiler, kernel, and so on) of the",
"operating system on which the executable runs, unless that component",
"itself accompanies the executable.",
"",
"If distribution of executable or object code is made by offering",
"access to copy from a designated place, then offering equivalent",
"access to copy the source code from the same place counts as",
"distribution of the source code, even though third parties are not",
"compelled to copy the source along with the object code.",
"",
"  4. You may not copy, modify, sublicense, or distribute the Program",
"except as expressly provided under this License.  Any attempt",
"otherwise to copy, modify, sublicense or distribute the Program is",
"void, and will automatically terminate your rights under this License.",
"However, parties who have received copies, or rights, from you under",
"this License will not have their licenses terminated so long as such",
"parties remain in full compliance.",
"",
"  5. You are not required to accept this License, since you have not",
"signed it.  However, nothing else grants you permission to modify or",
"distribute the Program or its derivative works.  These actions are",
"prohibited by law if you do not accept this License.  Therefore, by",
"modifying or distributing the Program (or any work based on the",
"Program), you indicate your acceptance of this License to do so, and",
"all its terms and conditions for copying, distributing or modifying",
"the Program or works based on it.",
"",
"  6. Each time you redistribute the Program (or any work based on the",
"Program), the recipient automatically receives a license from the",
"original licensor to copy, distribute or modify the Program subject to",
"these terms and conditions.  You may not impose any further",
"restrictions on the recipients' exercise of the rights granted herein.",
"You are not responsible for enforcing compliance by third parties to",
"this License.",
"",
"  7. If, as a consequence of a court judgment or allegation of patent",
"infringement or for any other reason (not limited to patent issues),",
"conditions are imposed on you (whether by court order, agreement or",
"otherwise) that contradict the conditions of this License, they do not",
"excuse you from the conditions of this License.  If you cannot",
"distribute so as to satisfy simultaneously your obligations under this",
"License and any other pertinent obligations, then as a consequence you",
"may not distribute the Program at all.  For example, if a patent",
"license would not permit royalty-free redistribution of the Program by",
"all those who receive copies directly or indirectly through you, then",
"the only way you could satisfy both it and this License would be to",
"refrain entirely from distribution of the Program.",
"",
"If any portion of this section is held invalid or unenforceable under",
"any particular circumstance, the balance of the section is intended to",
"apply and the section as a whole is intended to apply in other",
"circumstances.",
"",
"It is not the purpose of this section to induce you to infringe any",
"patents or other property right claims or to contest validity of any",
"such claims; this section has the sole purpose of protecting the",
"integrity of the free software distribution system, which is",
"implemented by public license practices.  Many people have made",
"generous contributions to the wide range of software distributed",
"through that system in reliance on consistent application of that",
"system; it is up to the author/donor to decide if he or she is willing",
"to distribute software through any other system and a licensee cannot",
"impose that choice.",
"",
"This section is intended to make thoroughly clear what is believed to",
"be a consequence of the rest of this License.",
"",
"  8. If the distribution and/or use of the Program is restricted in",
"certain countries either by patents or by copyrighted interfaces, the",
"original copyright holder who places the Program under this License",
"may add an explicit geographical distribution limitation excluding",
"those countries, so that distribution is permitted only in or among",
"countries not thus excluded.  In such case, this License incorporates",
"the limitation as if written in the body of this License.",
"",
"  9. The Free Software Foundation may publish revised and/or new versions",
"of the General Public License from time to time.  Such new versions will",
"be similar in spirit to the present version, but may differ in detail to",
"address new problems or concerns.",
"",
"Each version is given a distinguishing version number.  If the Program",
"specifies a version number of this License which applies to it and \"any",
"later version\", you have the option of following the terms and conditions",
"either of that version or of any later version published by the Free",
"Software Foundation.  If the Program does not specify a version number of",
"this License, you may choose any version ever published by the Free Software",
"Foundation.",
"",
"  10. If you wish to incorporate parts of the Program into other free",
"programs whose distribution conditions are different, write to the author",
"to ask for permission.  For software which is copyrighted by the Free",
"Software Foundation, write to the Free Software Foundation; we sometimes",
"make exceptions for this.  Our decision will be guided by the two goals",
"of preserving the free status of all derivatives of our free software and",
"of promoting the sharing and reuse of software generally.",
"",
"\n			    NO WARRANTY",
"",
"  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY",
"FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN",
"OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES",
"PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED",
"OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF",
"MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS",
"TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE",
"PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,",
"REPAIR OR CORRECTION.",
"",
"  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING",
"WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR",
"REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,",
"INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING",
"OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED",
"TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY",
"YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER",
"PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE",
"POSSIBILITY OF SUCH DAMAGES.",
"",
"		     END OF TERMS AND CONDITIONS\n",
};

#define README_CRASH_VERSION      "README_CRASH_VERSION"
#define README_GNU_GDB_VERSION    "README_GNU_GDB_VERSION"
#define README_DATE               "README_DATE"
#define README_GPL_INFO           "README_GPL_INFO"
#define README_HELP_MENU          "README_HELP_MENU"
#define README_ENTER_DIRECTORY    "README_ENTER_DIRECTORY"

static void
display_README(void)
{
        int i, j;
    	time_t time_now;           

        for (i = 0; README[i]; i++) {
		if (STREQ(README[i], README_CRASH_VERSION)) {
			fprintf(fp, "    crash %s\n", pc->program_version);
		} else if (STREQ(README[i], README_GNU_GDB_VERSION)) {
			fprintf(fp, "    GNU gdb %s\n", pc->gdb_version);
		} else if (STREQ(README[i], README_DATE)) {
    			time(&time_now);               
        		fprintf(fp, "            DATE: %s\n", 
                		strip_linefeeds(ctime(&time_now)));
		} else if (STREQ(README[i], README_HELP_MENU)) {
			display_help_screen("    ");
		} else if (STREQ(README[i], README_GPL_INFO)) {
        		for (j = 0; version_info[j]; j++)
                		fprintf(fp, "    %s\n", version_info[j]);
                } else if (STREQ(README[i], README_ENTER_DIRECTORY)) {
                        fprintf(fp, 
 "  To build this utility, simply uncompress the tar file, enter the crash%s\n",
                                pc->program_version);
		} else 
			fprintf(fp, "%s\n", README[i]);
        }
}

static 
char *README[] = {
"",
"",
"                         CORE ANALYSIS SUITE",
"",
"  The core analysis suite is a self-contained tool that can be used to",
"  investigate either live systems, kernel core dumps created from the",
"  Kernel Core Dump patch offered by Mission Critical Linux, or kernel",
"  core dumps created by the LKCD patch.",
"",
"  o  The tool is loosely based on the SVR4 crash command, but has been",
"     completely integrated with gdb in order to be able to display ",
"     formatted kernel data structures, disassemble source code, etc.",
"     ",
"  o  The current set of available commands consist of common kernel core",
"     analysis tools such as a context-specific stack traces, source code",
"     disassembly, kernel variable displays, memory display, dumps of ",
"     linked-lists, etc.  In addition, any gdb command may be entered,",
"     which in turn will be passed onto the gdb module for execution.  ",
"",
"  o  There are several commands that delve deeper into specific kernel",
"     subsystems, which also serve as templates for kernel developers",
"     to create new commands for analysis of a specific area of interest.",
"     Adding a new command is a simple affair, and a quick recompile",
"     adds it to the command menu.",
"",
"  o  The intent is to make the tool independent of Linux version dependencies,",
"     building in recognition of major kernel code changes so as to adapt to ",
"     new kernel versions, while maintaining backwards compatibility.",
"",
"  These are the current prerequisites: ",
"",
"  o  At this point, Intel-, Alpha-, PowerPC-, and IA64-based kernels are ",
"     supported.  The IA64 version of crash has only been tested on live",
"     systems.  Other architectures will be addressed in the future.",
"",
"  o  One size fits all -- the utility can be run on any Linux kernel version ",
"     from 2.2.5-15 through 2.4.*.",
"",
"  o  In order to contain debugging data, the top-level kernel Makefile's CFLAGS",
"     definition must be modified, and the kernel rebuilt.  In 2.2 kernels,",
"     change the following line:",
"",
"        CFLAGS = -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer",
"",
"     to:",
"",
"        CFLAGS = -Wall -Wstrict-prototypes -O2 -g",
"",
"     In 2.4 kernels, change the following line:",
"",
"        CFLAGS := $(CPPFLAGS) -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer -fno-strict-aliasing",
"",
"     to:",
"",
"        CFLAGS := $(CPPFLAGS) -Wall -Wstrict-prototypes -O2 -g -fno-strict-aliasing",
"",
"     The deletion of -fomit-frame-pointer is done for dependable stack ",
"     traces.",
"",
"  o  After the kernel is re-compiled, the uncompressed \"vmlinux\" kernel",
"     that is created in the top-level kernel build directory must be saved.",
"",
README_ENTER_DIRECTORY,
"  subdirectory, and type \"make\".  The initial build will take several minutes ",
"  because the gdb module must be configured and and built.  ",
"",
"  If the tool is run against a crash dumpfile, two arguments are required, the",
"  uncompressed kernel name and the core dumpfile name.  ",
"",
"  If run on a live system, only the kernel name is required, because /dev/mem ",
"  will be used as the \"dumpfile\".  If the kernel file is stored in /boot, /,",
"  /boot/efi, or in any /usr/src subdirectory, then no command line arguments",
"  are required -- the first kernel found that matches /proc/version will be",
"  used as the namelist.",
"  ",
"  For example, if the command name \"crash\" is kept intact in the Makefile,",
"  invoking it on a live system would look like this:",
"",
"    $ crash",
"    ",
README_CRASH_VERSION,
README_GPL_INFO,
README_GNU_GDB_VERSION,
"    Copyright 2000 Free Software Foundation, Inc.",
"    GDB is free software, covered by the GNU General Public License, and you are",
"    welcome to change it and/or distribute copies of it under certain conditions",
"    Type \"show copying\" to see the conditions.",
"    There is absolutely no warranty for GDB.  Type \"show warranty\" for details.",
"    This GDB was configured as \"i686-pc-linux-gnu\"..",
"     ",
"          KERNEL: /boot/vmlinux",
"        DUMPFILE: /dev/mem (live system)",
"            CPUS: 1",
README_DATE,
"          UPTIME: 10 days, 22:55:18",
"    LOAD AVERAGE: 0.08, 0.03, 0.01",
"           TASKS: 42",
"        NODENAME: ha2.mclinux.com",
"         RELEASE: 2.4.0-test10",
"         VERSION: #11 SMP Thu Nov 4 15:09:25 EST 2000",
"         MACHINE: i686  (447 MHz)",
"	  MEMORY: 128 MB",
"             PID: 3621                                  ",
"         COMMAND: \"crash\"",
"            TASK: c463c000  ",
"             CPU: 0",
"           STATE: TASK_RUNNING (ACTIVE)",
"",
"    crash> help",
README_HELP_MENU,
"    crash> ",
" ",
"  When run on a dumpfile, both the kernel namelist and dumpfile ",
"  must be entered on the command line, as in:",
"",
"    $ crash vmlinux.17 lcore.cr.17",
"",
README_CRASH_VERSION,
README_GPL_INFO,
README_GNU_GDB_VERSION,
"    Copyright 2000 Free Software Foundation, Inc.",
"    GDB is free software, covered by the GNU General Public License, and you are",
"    welcome to change it and/or distribute copies of it under certain conditions",
"    Type \"show copying\" to see the conditions.",
"    There is absolutely no warranty for GDB.  Type \"show warranty\" for details.",
"    This GDB was configured as \"i686-pc-linux-gnu\"...",
"    ",
"          KERNEL: vmlinux.17                        ",
"        DUMPFILE: lcore.cr.17",
"            CPUS: 1",
"            DATE: Wed Nov 10 19:54:47 1999",
"          UPTIME: 00:02:02",
"    LOAD AVERAGE: 0.82, 0.34, 0.12",
"           TASKS: 43",
"        NODENAME: ha3.mclinux.com",
"         RELEASE: 2.2.5-15",
"         VERSION: #39 Wed Nov 10 17:43:16 CST 1999",
"         MACHINE: i686  (501 MHz)",
"          MEMORY: 96 MB",
"           PANIC: \"tulip_interrupt\"",
"             PID: 286",
"         COMMAND: \"in.rlogind\"",
"            TASK: c0b3a000  ",
"             CPU: 0",
"           STATE: TASK_RUNNING (PANIC)",
"    ",
"    crash>",
"",
"  When run on an LKCD core dump, the command line would be like so:",
"",
"    $ crash vmlinux vmdump.0",
"",
"  The tool's environment is context-specific.  On a live system, the default",
"  context is the command itself; on a dump the default context will be the",
"  task that panicked.  The most commonly-used commands are:",
"",
"    set     - set a new task context by pid, task address, or cpu.",
"    bt      - backtrace of the current context, or as specified with arguments.",
"    p       - print the contents of a kernel variable.",
"    rd      - read memory, which may be either kernel virtual, user virtual, or",
"              physical.",
"    ps      - simple process listing.",
"    log     - dump the kernel log_buf.",
"    struct  - print the contents of a structure at a specified address.",
"    foreach - execute a command on all tasks, or those specified, in the system.",
" ",
"  Detailed help concerning the use of each of the commands in the menu above ",
"  may be displayed by entering \"help command\", where \"command\" is one of those ",
"  listed above.  Rather than getting bogged down in details here, simply",
"  run the help command on each of the commands above.  Note that many commands",
"  have multiple options so as to avoid the proliferation of command names.",
"",
"  Command output may be piped to external commands or redirected to files.",
"  Enter \"help output\" for details.",
"",
"  The command line history mechanism allows for command-line recall and ",
"  command-line editing.  Input files containing a set of crash commands may ",
"  be substituted for command-line input.  Enter \"help input\" for details.",
"",
"  Note that a .crashrc file (or .<your-command-name>rc if the name has been ",
"  changed), may contain any number of \"set\" or \"alias\" commands -- see the",
"  help pages on those two commands for details.",
" ",
"  Lastly, if a command is entered that is not recognized, it is checked",
"  against the kernel's list of variables, structure, union or typedef names, ",
"  and if found, the command is passed to \"p\", \"struct\", \"union\" or \"whatis\".",
"  That being the case, as long as a kernel variable/structure/union name is ",
"  different than any of the current commands.",
"",
"  (1) A kernel variable can be dumped by simply entering its name:",
" ",
"      crash> init_mm",
"      init_mm = $2 = {",
"        mmap = 0xc022d540, ",
"        mmap_avl = 0x0, ",
"        mmap_cache = 0x0, ",
"        pgd = 0xc0101000, ",
"        count = {",
"          counter = 0x6",
"        }, ",
"        map_count = 0x1, ",
"        mmap_sem = {",
"          count = {",
"            counter = 0x1",
"          }, ",
"          waking = 0x0, ",
"          wait = 0x0",
"        }, ",
"        context = 0x0, ",
"        start_code = 0xc0000000, ",
"        end_code = 0xc022b4c8,",
"        end_data = c0250388,",
"        ...",
"      ",
"  (2) A structure or can be dumped simply by entering its name and address:  ",
"",
"      crash> vm_area_struct c5ba3910",
"      struct vm_area_struct {",
"        vm_mm = 0xc3ae3210, ",
"        vm_start = 0x821b000, ",
"        vm_end = 0x8692000, ",
"        vm_next = 0xc5ba3890, ",
"        vm_page_prot = {",
"          pgprot = 0x25",
"        }, ",
"        vm_flags = 0x77, ",
"        vm_avl_height = 0x4, ",
"        vm_avl_left = 0xc0499540, ",
"        vm_avl_right = 0xc0499f40, ",
"        vm_next_share = 0xc04993c0, ",
"        vm_pprev_share = 0xc0499060, ",
"        vm_ops = 0x0, ",
"        vm_offset = 0x0, ",
"        vm_file = 0x0, ",
"        vm_pte = 0x0",
"      }",
"",
"",
"  The crash utility has been designed to facilitate the task of adding new ",
"  commands.  New commands may be permanently compiled into the crash executable,",
"  or dynamically added during runtime using shared object files.",
" ",
"  To permanently add a new command to the crash executable's menu:",
"",
"    1. For a command named \"xxx\", put a reference to cmd_xxx() in defs.h.",
"   ",
"    2. Add cmd_xxx into the base_command_table[] array in global_data.c. ",
"",
"    3. Write cmd_xxx(), putting it in one of the appropriate files.  Look at ",
"       the other commands for guidance on getting symbolic data, reading",
"       memory, displaying data, etc...",
"",
"    4. Recompile and run.",
"",
"  Note that while the initial compile of crash, which configures and compiles",
"  the gdb module, takes several minutes, subsequent re-compiles to do such",
"  things as add new commands or fix bugs just takes a few seconds.",
"",
"  Alternatively, you can create shared object library files consisting of",
"  crash command extensions, that can be dynamically linked into the crash",
"  executable during runtime or during initialization.  This will allow the",
"  the same shared object to be used with subsequent crash releases without",
"  having to re-merge the command's code into each new set of crash sources.",
"  The dynamically linked-in commands will automatically show up in the crash",
"  help menu.  For details, enter \"help extend\" during runtime, or enter",
"  \"crash -h extend\" from the shell command line.",
" ",
"  Complete documentation regarding this utility can be accessed by visiting:",
" ",
"             http://oss.missioncriticallinux.com/projects/crash",
" ",
"  From the page above you can join the Crash Mailing List, which we invite",
"  you to use for reporting bugs, making suggestions, asking questions, or",
"  sending any new commands you'd like to see permanently incorporated.",
"  Alternatively, you can send the same type of queries to:",
"",
"                    support@missioncriticallinux.com",
"",
"",
"",
"",
	0
};
