/* $Id: courieruserinfo.c,v 1.1 2005/10/15 22:20:30 astjean Exp $
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Copyright (c) Andrew St. Jean <andrew@arda.homeunix.net> 2003-2005
 *
 * */

#include	"courieruserinfo.h"

static const char *short_options = "s:AudanmqohV";

static struct option long_options[] = {
	{ "service", required_argument, NULL, 's' },
	{ "all", no_argument, NULL, 'A' },
	{ "uid", no_argument, NULL, 'u' },
	{ "homedir", no_argument, NULL, 'd' },
	{ "address", no_argument, NULL, 'a' },
	{ "name", no_argument, NULL, 'n' },
	{ "maildir", no_argument, NULL, 'm' },
	{ "quota", no_argument, NULL, 'q' },
	{ "options", no_argument, NULL, 'o' },
	{ "help", no_argument, NULL, 'h' },
	{ "version", no_argument, NULL, 'V' },
	{ "stderr", no_argument, NULL, OPT_STDERR },
	{ NULL, 0, NULL, 0 }
};

const char *usage =
"Usage: " PACKAGE " [Options] userid\n"
"\n"
"Retrieve information for a user using available courier\n"
"authentication modules\n"
"\n"
"Options:\n"
"  -s, --service=SERVICE\tspecify the service name to use\n"
"\t\t\tdefault value is 'login'\n"
"  -A, --all\t\tdisplay all user information (default)\n"
"  -u, --uid\t\tdisplay UID and GID\n"
"  -d, --homedir\t\tdisplay home directory\n"
"  -a, --address\t\tdisplay authenticated login ID\n"
"  -n, --name\t\tdisplay name\n"
"  -m, --maildir\t\tdisplay maildir directory\n"
"  -q, --quota\t\tdisplay quota\n"
"  -o, --options\t\tdisplay account options\n"
"  --stderr\t\tsend logging information to standard error\n"
"\t\t\tinstead of syslog\n"
"  -h, --help\t\tdisplay this help and exit\n"
"  -V, --version\t\tdisplay version information and exit\n";

#define MAX_LEN_USERNAME	65   /* max length for username */
#define MAX_LEN_SERVICE         129  /* max length of service name */

#define DISPLAY_ALL		0xffff
#define DISPLAY_UIDGID		0x01
#define DISPLAY_HOMEDIR		0x02
#define DISPLAY_ADDRESS		0x04
#define DISPLAY_NAME		0x08
#define DISPLAY_MAILDIR		0x10
#define DISPLAY_QUOTA		0x20
#define DISPLAY_OPTIONS		0x40

static int
callback_pre(struct authinfo *a, void *display_opts)
{
	int display;
	struct passwd *pw;

	/* I don't want to have to cast display_opts every
	 * time I use it so I'll make a local copy and
	 * cast it once. */
	display = *(int *)display_opts;

#define PTR(x) ((x) ? (x):"")
	
	if (display & DISPLAY_UIDGID){
		if (a->sysuserid != NULL) {
                        printf("uid=%d\n", *a->sysuserid);
                        printf("gid=%d\n", a->sysgroupid);
		} else if (a->sysusername) {
			pw = getpwnam(a->sysusername);
		        if (!pw){
		                perror("getpwnam");
				printf("uid=\n");
				printf("gid=\n");
		        } else {
				printf("uid=%d\n", pw->pw_uid);
				printf("gid=%d\n", pw->pw_gid);
			}
		} else {
			printf("uid=\n");
			printf("gid=\n");
		}
	}
	if (display & DISPLAY_HOMEDIR)
		printf("home=%s\n", PTR(a->homedir));
	if (display & DISPLAY_ADDRESS)
		printf("authaddr=%s\n", PTR(a->address));
	if (display & DISPLAY_NAME)
		printf("authfullname=%s\n", PTR(a->fullname));
	if (display & DISPLAY_MAILDIR)
		printf("maildir=%s\n", PTR(a->maildir));
	if (display & DISPLAY_QUOTA)
		printf("quota=%s\n", PTR(a->quota));
	if (display & DISPLAY_OPTIONS)
		printf("options=%s\n", PTR(a->options));

#undef PTR

	return(0);
}

size_t
validate_str(const char *data){
	return(strspn(data, SAFE_CHARS));
}

void
end(int exit_code){
	end_logging();
	exit(exit_code);
}

int opt_stderr = 0;

int
main(int argc, char **argv)
{
char		username[MAX_LEN_USERNAME];
char		service[MAX_LEN_SERVICE];
int		i, rc, display = 0;
size_t		n;

	*username = *service = 0;

	/* set the default value for service */
	strlcpy(service, "login", 6);

	init_logging();

	/* process the command line options */
	opterr = 0;
	while (1){
		int option_index = 0;
		int c = getopt_long(argc, argv, short_options, long_options, &option_index);

		if (c == -1) break;


		switch (c){
		case 's':
			n = validate_str(optarg);
			if (optarg[n]){
				logging(LOG_ERR, "illegal character in service name: %c", optarg[n]);
				end(2);
			}
			n = strlcpy(service, optarg, sizeof(service));
			if (n >= sizeof(service)){
				logging(LOG_ERR, "service name too long (max %d characters)", MAX_LEN_SERVICE -1 );
				end(2);
			}
			break;
		case 'u':
			display = display | DISPLAY_UIDGID;
			break;
		case 'd':
			display = display | DISPLAY_HOMEDIR;
			break;
		case 'a':
			display = display | DISPLAY_ADDRESS;
			break;
		case 'n':
			display = display | DISPLAY_NAME;
			break;
		case 'm':
			display = display | DISPLAY_MAILDIR;
			break;
		case 'q':
			display = display | DISPLAY_QUOTA;
			break;
		case 'o':
			display = display | DISPLAY_OPTIONS;
			break;
		case OPT_STDERR:
			opt_stderr = 1;
			break;
		case 'h':
			puts(usage);
			end(0);
		case 'V':
			puts(PACKAGE " " VERSION);
			end(0);
		case '?':
			puts("Invalid command-line, see --help");
			end(2);
		}
	}

	/* The -A option is a no-op. We take care of displaying
	 * all user information here. */
	if (display == 0)
		display = display | DISPLAY_ALL;

	/* get the username from the command line */
	if (optind < argc){
		n = validate_str(argv[optind]);
		if (argv[optind][n]){
			logging(LOG_ERR, "illegal character in username: %c", argv[optind][n]);
			end(2);
		}
		n = strlcpy(username, argv[optind], sizeof(username));
		if (n >= sizeof(username)){
			logging(LOG_ERR, "username too long (max %d characters)", MAX_LEN_USERNAME - 1);
			end(2);
		}
	} else {
		puts("Username required, see --help");
		end(2);
	}

	/* get information for the specified user */
	rc = auth_getuserinfo(service, username, callback_pre, &display);

	if (rc > 0){
		logging(LOG_ERR, "authentication failure");
		end(1);
	}
	if (rc < 0){
		logging(LOG_ERR, "user %s not found", username);
		end(1);
	}

	end(0);
	return (0);
}
