/***********************************************************************
 *               Copyright (C) 1995 Joe English
 *                   Freely redistributable
 ***********************************************************************
 *
 * intern.c,v 1.5 1998/11/10 00:06:56 jenglish Exp
 *
 * Author:	Joe English
 * Created: 	12Sep91, updated for CoST 6Jan1995
 * Description: String table lookup routines.
 *	char *t = intern(char *s) returns a permanent, unique copy of 's'.
 * 	If strcmp(s1,s2) == 0, then intern(s1) == intern(s2);
 *	lcintern(s) and ucintern(s) fold s to lower/uppercase first.
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "project.h"
#include "ctrie.h"
#include "strmgt.h"

/*
 * Private:
 */
#define ST_INCREMENT 8000
static CTrie strmap = 0;
static char *string_table = NULL;
static unsigned string_table_left = 0;

/* need(n) -- makes sure that at least N bytes are available
 * in the string table.
 */
static void need(int len)
{
    if (len > string_table_left) {	/* Grab another chunk: */
	string_table = (char *)malloc(ST_INCREMENT);		/* %%%check*/
	string_table_left = ST_INCREMENT;
    }

    return;
}

/* stash(len) -- input is in string_table ... string_table+len;
 * if in map, return stashed copy, otherwise stash new copy.
 */
static const char *stash(int len)
{
    CTrieNode p;
    char *ret;

    if (!strmap)
	strmap = ctrie_create();

    p = ctrie_lookup(strmap,string_table);

    if (ctrie_hasvalue(p))
    {
	ret=ctrie_getvalue(p);
	ASSERT(strcmp(ret,string_table) == 0, "map(string) != string");
    }
    else
    {
	ret = string_table;
	ctrie_setvalue(p,(void *)ret);
	string_table += len;
	string_table_left -= len;
    }
    return ret;
}

/*
 * Public interface:
 */
const char *intern(const char *s)
{
    int len = strlen(s) + 1;
    need(len);
    strcpy(string_table, s);
    return stash(len);
}

const char *ucintern(const char *s)
{
    int len = strlen(s)+1;
    char *t;

    need(len);
    t = string_table;
    while (*s)
	*t++ = toupper(*s++);
    *t++ = '\0';

    return stash(len);
}

const char *lcintern(const char *s)
{
    int len = strlen(s)+1;
    char *t;

    need(len);
    t = string_table;
    while (*s)
	*t++ = tolower(*s++);
    *t++ = '\0';

    return stash(len);
}

