;;;
;;; GLSPIN-GLX.LISP
;;;
;;; Translation of "spin.c" by Brian Paul
;;;
;;; Note: this example shows how to use XCreateWindow and glXChooseVisual
;;; to select a double-buffered visual.
;;;

(eval-when (:compile-toplevel :load-toplevel)
  (require :xlib-gl)
  (require :gl))

(defpackage :glspin-glx
  (:use :cl :gl :xlib-gl)
  (:export :main))

(in-package :glspin-glx)

(defvar Xrot)
(defvar Xstep)
(defvar Yrot)
(defvar Ystep)
(defvar Zrot)
(defvar Zstep)
(defvar Step)
(defvar Scale)
(defvar *Object*)
(defvar *animate?*)

(defvar *display* nil)
(defvar *window*)
(defvar *context*)

(defvar *debug* nil)

(defun make-object ()
  (when *debug* (format t "MAKE-OBJECT~%"))
  (let ((list (glgenlists 1)))
    (glnewlist list GL_COMPILE)
    ;;
    (glBegin GL_LINE_LOOP)
    (glVertex3f 1.0 0.5 -0.4)
    (glVertex3f 1.0 -0.5 -0.4)
    (glVertex3f -1.0 -0.5 -0.4)
    (glVertex3f -1.0 0.5 -0.4)
    (glEnd)
    ;;
    (glBegin GL_LINE_LOOP)
    (glVertex3f 1.0 0.5 0.4)
    (glVertex3f 1.0 -0.5 0.4)
    (glVertex3f -1.0 -0.5 0.4)
    (glVertex3f -1.0 0.5 0.4)
    (glEnd)
    ;;
    (glBegin GL_LINES)
    (glVertex3f 1.0 0.5 -0.4)   (glVertex3f 1.0 0.5 0.4)
    (glVertex3f 1.0 -0.5 -0.4)  (glVertex3f 1.0 -0.5 0.4)
    (glVertex3f -1.0 -0.5 -0.4) (glVertex3f -1.0 -0.5 0.4)
    (glVertex3f -1.0 0.5 -0.4)  (glVertex3f -1.0 0.5 0.4)
    (glEnd)
    ;;
    (glEndList)
    ;;
    list))

(defun reshape (width height)
  (when *debug* (format t "RESHAPE~%"))
  (glViewport 0 0 width height)
  (glMatrixMode GL_PROJECTION)
  (glLoadIdentity)
  (glFrustum -1d0 1d0 -1d0 1d0 5d0 15d0)
  (glMatrixMode GL_MODELVIEW))

(defun animate ()
  (when *debug* (format t "ANIMATE~%"))
  ;;
  (setf xrot (+ xrot xstep))
  (setf yrot (+ yrot ystep))
  (setf zrot (+ zrot zstep))
  ;;
  (cond
   ((>= xrot 360.0)
    (setf xrot 0)
    (setf xstep 0)
    (setf ystep step))
   ((>= yrot 360.0)
    (setf yrot 0)
    (setf ystep 0)
    (setf zstep step))
   ((>= zrot 360.0)
    (setf zrot 0)
    (setf zstep 0)
    (setf xstep step)))
  (draw))

(defun draw ()
  (when *debug* (format t "DRAW~%"))
  (glClear GL_COLOR_BUFFER_BIT)
  (glPushMatrix)
  (glTranslatef 0.0 0.0 -10.0)
  (glScalef Scale Scale Scale)
  (cond
   ((> xstep 0)
    (glrotatef (coerce xrot 'single-float) 1.0 0.0 0.0))
   ((> ystep 0)
    (glrotatef (coerce yrot 'single-float) 0.0 1.0 0.0))
   ((> zstep 0)
    (glrotatef (coerce zrot 'single-float) 0.0 0.0 1.0)))
  ;;
  (glcalllist *object*)
  ;;
  (glpopmatrix)
  ;;
  (when *debug*
    (format t "glXSwapBuffers, display ~A, window ~A~%" *display* *window*))
  (glxswapbuffers *display* *window*)
  (when *debug* (format t "Exiting DRAW~%"))
  )


(defun event-loop (display)
  (let ((done? nil)
	(debug t)
	(event (make-xevent)))
    ;;
    ;; Main event loop
    (loop
      ;;
      ;; If we are animating, calc and redraw each frame until an event occurs
      (when *animate?*
	(when debug (format t "Animate...~%"))
	(loop
	  (animate)
	  (when (> (xeventsqueued display queuedafterflush) 0)
	    (return))))
      ;;
      ;; Handle events.  If we are not animating, we wait here for event.
      (when debug (format t "Waiting for event..."))
      (xnextevent display event)
      (let ((event-type (xanyevent-type event)))
	(when debug (format t "Event:~a~%" event-type))
	(cond
	 ;;
	 ;; Expose
	 ((eq event-type expose)
	  ;;
	  ;; Gobble all other expose events
	  (loop
	    (when (zerop (xeventsqueued display queuedalready))
	    ;;(when (null-ptr (xeventsqueued display queuedalready))
	      (return))
	    (xnextevent display event)
	    (let ((event-type (xanyevent-type event)))
	      (unless (eq event-type expose)
		(xputbackevent display event)
		(return)))
	    (when debug (format t "Gobble event:~a~%" event-type)))
	  (draw))
	 ;;
	 ;; Resize
	 ((eq event-type configurenotify)
	  (reshape (xconfigureevent-width event)
		   (xconfigureevent-height event)))
	 ((eq event-type buttonpress)
	  (let ((button (xbuttonevent-button event)))
	    (when debug (format t "Button:~a~%" button))
	    (cond ((eq button button1)
		   (setf *animate?* (not *animate?*)))
		  ((eq button button3)
		   (setf done? t)))))))
      ;;
      (when done? (return)))
    (free-xevent event)))

(defun create-gl-window (display width height)
  ;; Create a double buffered, RGBA window
  ;; Warning ... this code will probably only work for True Color visual.
  (let* ((screen (XDefaultScreen display))
	 (root (XRootWindow display screen))
	 ;; Setup a byte-array of integers, terminated by "None"
	 ;; CMU does not use "C compatible fixnums", needs to use
	 ;; (signed-byte 32) instead
	 (attrib (make-array 9
			     :element-type
			     #+(or cmu sbcl) '(signed-byte 32)
                             #-(or cmu sbcl) 'fixnum
			     :initial-contents
			     (list GLX_RGBA
                                   GLX_RED_SIZE 1
				   GLX_GREEN_SIZE 1
                                   GLX_BLUE_SIZE 1
				   GLX_DOUBLEBUFFER
                                   None
                                   )))
	 (visinfo (glXChooseVisual display screen attrib)))
    ;;(when (null-pointer visinfo)
    (when (zerop visinfo)
      (error "CREATE-GL-WINDOW: Couldn't get an RGB, double-buffered visual"))
    (let ((attr (make-xsetwindowattributes)))
      (set-xsetwindowattributes-background_pixel! attr 0)
      (set-xsetwindowattributes-border_pixel! attr 0)
      (set-xsetwindowattributes-colormap!
       attr
       (XcreateColormap display root
			(XVisualInfo-visual visinfo) AllocNone))
      (set-xsetwindowattributes-event_mask!
       attr (+ StructureNotifyMask ExposureMask ButtonPressMask))
      (let* ((mask (+ CWBackPixel CWBorderPixel CWColormap CWEventMask))
	     (window (XCreateWindow display root 0 0 width height
				    0
				    (XVisualInfo-depth visinfo)
				    InputOutput
				    (XVisualInfo-visual visinfo)
				    mask attr))
	     (glXContext (setf *context* (glXCreateContext
					  display visinfo NULL 1))))
	(glXMakeCurrent display window glXContext)
	(XMapWindow display window)
	window))))

(defun main ()
  ;;
  (unless *display* (setf *display* (xopendisplay "")))
  (setf *window* (create-gl-window *display* 300 300))
  ;; Create object as display list
  (setf *object* (make-object))
  ;;
  (glcullface GL_BACK)
  ;;(glenable GL_CULL_FACE)
  (gldisable GL_DITHER)
  (glshademodel GL_FLAT)
  (glcolor3f 1.0 1.0 1.0)
  ;;
  ;; Initial state of animation.
  (setf scale 1.0)
  (setf xrot 0.0)  (setf yrot 0.0)  (setf zrot 0.0)
  (setf step 0.5)
  (setf xstep step)  (setf ystep 0.0)  (setf zstep 0.0)
  (setf *animate?* t)
  ;;
  (unwind-protect
       (event-loop *display*)
    ;;
    ;; Cleanup and exit
    (glxdestroycontext *display* *context*)
    (xdestroywindow *display* *window*)
    (let ((event (make-xevent)))
      (do ()
	  ((zerop (xpending *display*)))
	(xnextevent *display* event))
      (free-xevent event))
    ;;(xclosedisplay *display*)
    ))
