(in-package :contextl)

(defun parse-method-body (form body)
  (let* ((in-layerp (eq (car body) :in-layer))
         (layer-spec (if in-layerp (cadr body) 't)))
    (when (consp layer-spec)
      (unless (null (cddr layer-spec))
        (error "Incorrect :in-layer specification in ~S." form)))
    (loop with layer = (if (atom layer-spec)
                         layer-spec
                         (cadr layer-spec))
          with layer-arg = (if (atom layer-spec)
                             (gensym "LAYER-ARG-")
                             (car layer-spec))
          for tail on (if in-layerp (cddr body) body)
          until (listp (car tail))
          collect (car tail) into qualifiers
          finally
          (when (member :in-layer qualifiers)
            (error "Incorrect occurrence of :in-layer in ~S. Must occur before qualifiers." form))
          (return (values layer-arg layer qualifiers (car tail) (cdr tail))))))

(defun prepare-layer (layer)
  (if (symbolp layer)
    (defining-layer layer)
    layer))

(defun prepare-layered-method-body (form layer-arg body)
  (loop for (first . rest) on body
        count (stringp first) into nof-seen-strings
        while (or (stringp first)
                  (and (consp first)
                       (eq (car first) 'declare)))
        collect first into declarations
        finally
        (when (> nof-seen-strings 1)
          (error "Too many documentation strings in ~S." form))
        (return (nconc declarations
                       `((flet ((call-next-layered-method (&rest args)
                                  (declare (dynamic-extent args))
                                  (if args
                                    (apply #'call-next-method ,layer-arg args)
                                    (call-next-method))))
                           (declare (inline call-next-layered-method)
                                    #-lispworks
                                    (ignorable (function call-next-layered-method)))
                           ,first ,@rest))))))

(defmacro define-layered-function (name (&rest args) &body options)
  (let ((definer (get-layered-function-definer-name name)))
    (with-unique-names (layer-arg rest-arg)
      `(progn
         (defgeneric ,definer (,layer-arg ,@args)
           (:argument-precedence-order 
            ,@(let ((argument-precedence-order (assoc :argument-precedence-order options)))
                (if argument-precedence-order
                  (cdr argument-precedence-order)
                  (required-args args)))
            ,layer-arg)
           ,@(loop for option in (remove :argument-precedence-order options :key #'car)
                   if (eq (car option) :method)
                   collect (multiple-value-bind
                               (layer-arg layer qualifiers args method-body)
                               (parse-method-body option (cdr option))
                             `(:method ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
                               ,@(prepare-layered-method-body option layer-arg method-body)))
                   else collect option))
         (declaim (inline ,name))
         (defun ,name (&rest ,rest-arg)
           (declare #-openmcl (dynamic-extent ,rest-arg)
                    (optimize (speed 3) (debug 0) (safety 0)
                              (compilation-speed 0)))
           (apply #',definer (layer-context-prototype *active-context*) ,rest-arg))
         #',definer))))

(defmacro define-layered-method (&whole form name &body body)
  (multiple-value-bind
      (layer-arg layer qualifiers args method-body)
      (parse-method-body form body)
    `(defmethod ,(get-layered-function-definer-name name)
                ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
       ,@(prepare-layered-method-body form layer-arg method-body))))
