/*
 *    functions for the graphic output of the pixmap
 */

#include <string.h>

#include "ct1.h"
GdkFont *font, *smallfont, *symbfont, *ssymbfont, *slfont, *boldfont;

#ifndef MIN
#define MIN(a, b)        ((a) < (b) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a, b)        ((a) > (b) ? (a) : (b))
#endif

void
CreatePix ()
/* initializes fonts, linestyles and factors */
{
  float ztab[] = { 0.4, 0.6, 0.8, 1, 1.2 };
  int ltab[] = { 0, 0, 2, 2, 2 };
  int ftab[] = { 2, 3, 4, 5, 6 };

  size_factor = ztab[zoom_factor];
  head.pix_width = 1600;
  head.pix_height = 1600;
  Set_Line (ltab[zoom_factor]);

  if (!Load_Font (ftab[zoom_factor]))
    {
      fprintf (stderr, "chemtool: can't load any font\n");
      exit (1);
    }

}

void
FreePix ()
/* erases the drawing area */
{
  GdkRectangle update_rect;
  gdk_draw_rectangle (picture,
		      drawing_area->style->white_gc,
		      TRUE, 0, 0,
		      drawing_area->allocation.width,
		      drawing_area->allocation.height);

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = drawing_area->allocation.width;
  update_rect.height = drawing_area->allocation.height;
  gtk_widget_draw ((GtkWidget *) drawing_area, &update_rect);

}



void
CopyPlane ()
/* updates the display with the contents of the background pixmap */
{
  GdkRectangle update_rect;
  gdk_draw_pixmap (drawing_area->window,
		   drawing_area->style->
		   fg_gc[GTK_WIDGET_STATE (drawing_area)], picture, 0, 0, 0,
		   0, head.pix_width, head.pix_height);
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = head.pix_width;
  update_rect.height = head.pix_height;
}

void
Display_Mol ()
/* calls the individual bond and text drawing functions to display the
   picture */
{
  struct data *hpc;
  struct dc *hpc_c;
  struct spline *hpc_sp;
  int d;
  char *c;
  struct xy_co *coord;
  GdkRectangle update_rect;

  if (head.pix_width == 0 || head.pix_height == 0)
    return;
  gdk_draw_rectangle (picture,
		      drawing_area->style->white_gc,
		      TRUE, 0, 0, head.pix_width, head.pix_height);

  hpc = da_root.next;
  if (!hpc)
    fprintf (stderr,
	     "Help - somebody ate my atoms (hp->n=%d, da_root.next NULL)!!!\n",
	     hp->n);
  for (d = 0; d < hp->n; d++)
    {
      switch (hpc->bond)
	{
	case 0:
	  Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty, hpc->smarked + hpc->tmarked);	/*single */
	  break;
	case 1:
	  {			/*leftdouble */
	    coord = multi_bonds (hpc->x, hpc->y, hpc->tx, hpc->ty, 10);
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 2:
	  {			/*rightdouble */
	    coord = multi_bonds (hpc->tx, hpc->ty, hpc->x, hpc->y, 10);
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 5:
	  DrawWedge (hpc->x, hpc->y, hpc->tx, hpc->ty,
		     hpc->smarked + hpc->tmarked);
	  break;
	case 6:
	  DrawDashedWedge (hpc->x, hpc->y, hpc->tx, hpc->ty,
			   hpc->smarked + hpc->tmarked);
	  break;
	case 7:
	  DrawWiggly (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  break;
	case 8:
	  DrawArrow (hpc->x, hpc->y, hpc->tx, hpc->ty, 1,
		     hpc->smarked + hpc->tmarked);
	  break;
	case 9:
	  DrawArrow (hpc->x, hpc->y, hpc->tx, hpc->ty, 2,
		     hpc->smarked + hpc->tmarked);
	  break;
	case 10:
	  DrawWide (hpc->x, hpc->y, hpc->tx, hpc->ty,
		    hpc->smarked + hpc->tmarked);
	  break;
	case 11:
	  DrawCircle (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  break;
	case 12:
	  DrawDotted (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  break;
	case 13:
	  DrawAcross (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  break;
	case 4:
	  {			/*middouble */
	    coord = center_double_bond (hpc->x, hpc->y, hpc->tx, hpc->ty, 4);
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	    coord++;
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 3:
	  {			/*triple */
	    coord = multi_bonds (hpc->x, hpc->y, hpc->tx, hpc->ty, 10);
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	    coord = multi_bonds (hpc->tx, hpc->ty, hpc->x, hpc->y, 10);
	    Drawline (coord->x, coord->y, coord->tx, coord->ty,
		      hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 14:
	  {			/*left partial double */
	    coord = multi_bonds (hpc->x, hpc->y, hpc->tx, hpc->ty, 10);
	    DrawDashed (coord->x, coord->y, coord->tx, coord->ty,
			hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 15:
	  {			/*right partial double */
	    coord = multi_bonds (hpc->tx, hpc->ty, hpc->x, hpc->y, 10);
	    DrawDashed (coord->x, coord->y, coord->tx, coord->ty,
			hpc->smarked + hpc->tmarked);
	    Drawline (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  }
	  break;
	case 16:		/*zebra dashed single bond */
	  DrawStripe (hpc->x, hpc->y, hpc->tx, hpc->ty,
		      hpc->smarked + hpc->tmarked);
	  break;
	}
      hpc = hpc->next;
      if (!hpc)
	{
	  fprintf (stderr,
		   "Help - somebody ate my atoms (d=%d,hp->n=%d)!!!\n", d,
		   hp->n);
	  hp->n = d;
	  continue;
	}
    }

  hpc_c = dac_root.next;
  for (d = 0; d < hp->nc; d++)
    {
      c = hpc_c->c;
      Drawstring (hpc_c->x, hpc_c->y, c, hpc_c->direct, hpc_c->marked);
      hpc_c = hpc_c->next;
    }

  hpc_sp = sp_root.next;
  for (d = 0; d < hp->nsp; d++)
    {
      Drawspline (hpc_sp->x0, hpc_sp->y0, hpc_sp->x1, hpc_sp->y1,
		  hpc_sp->x2, hpc_sp->y2, hpc_sp->x3, hpc_sp->y3,
		  hpc_sp->type, hpc_sp->marked);
      hpc_sp = hpc_sp->next;
    }


  gdk_draw_pixmap (drawing_area->window,
		   drawing_area->style->
		   fg_gc[GTK_WIDGET_STATE (drawing_area)], picture, 0, 0, 0,
		   0, head.pix_width, head.pix_height);
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = head.pix_width;
  update_rect.height = head.pix_height;
  gtk_widget_draw ((GtkWidget *) drawing_area, &update_rect);
}

void
draw_preview_bonds (x, y, tx, ty, b)
/* calls the individual bond functions to create the preview image */
     int x, y, tx, ty, b;
{
  struct xy_co *coord;

  switch (b)
    {
    case 0:
      Drawline (x, y, tx, ty, 0);	/*single */
      break;
    case 1:
      {				/*leftdouble */
	coord = multi_bonds (x, y, tx, ty, 10);
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
      }
      break;
    case 2:
      {				/*rightdouble */
	coord = multi_bonds (tx, ty, x, y, 10);
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
      }
      break;
    case 5:
      DrawWedge (x, y, tx, ty, 0);
      break;
    case 6:
      DrawDashedWedge (x, y, tx, ty, 0);
      break;
    case 7:
      DrawWiggly (x, y, tx, ty, 0);
      break;
    case 8:
      DrawArrow (x, y, tx, ty, 1, 0);
      break;
    case 9:
      DrawArrow (x, y, tx, ty, 2, 0);
      break;
    case 10:
      DrawWide (x, y, tx, ty, 0);
      break;
    case 11:
      DrawCircle (x, y, tx, ty, 0);
      break;
    case 12:
      DrawDotted (x, y, tx, ty, 0);
      break;
    case 13:
      DrawAcross (x, y, tx, ty, 0);
      break;
    case 4:
      {				/*middouble */
	coord = center_double_bond (x, y, tx, ty, 4);
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
	coord++;
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
      }
      break;
    case 3:
      {				/*triple */
	coord = multi_bonds (x, y, tx, ty, 10);
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
	coord = multi_bonds (tx, ty, x, y, 10);
	Drawline (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
      }
      break;
    case 14:
      {				/*left partial double */
	coord = multi_bonds (x, y, tx, ty, 10);
	DrawDashed (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
      }
      break;
    case 15:
      {				/*right partial double */
	coord = multi_bonds (tx, ty, x, y, 10);
	DrawDashed (coord->x, coord->y, coord->tx, coord->ty, 0);
	Drawline (x, y, tx, ty, 0);
      }
      break;
    case 16:			/*zebra dashed single bond */
      DrawStripe (x, y, tx, ty, 0);
      break;
    }
}

void
Drawline (int x, int y, int tx, int ty, int active)
/* draws a single bond */
{
  if (!active || xbmflag)
    gdk_draw_line (picture, drawing_area->style->black_gc,
		   (gint) (x * size_factor), (gint) (y * size_factor),
		   (gint) (tx * size_factor), (gint) (ty * size_factor));
  else
    gdk_draw_line (picture, drawing_area->style->bg_gc[GTK_STATE_SELECTED],
		   (gint) (x * size_factor), (gint) (y * size_factor),
		   (gint) (tx * size_factor), (gint) (ty * size_factor));
}

void
DrawAcross (int x, int y, int tx, int ty, int active)
/* draws a single bond on a white background */
{
  gint npoints = 4;
  GdkPoint mypoints[4];
  struct xy_co *coord;
  GdkGC *mygc;

  mygc = drawing_area->style->white_gc;

  coord = center_double_bond (x, y, tx, ty, 6);
  mypoints[0].x = (coord->x + (coord->tx - coord->x) / 4) * size_factor;
  mypoints[0].y = (coord->y + (coord->ty - coord->y) / 4) * size_factor;
  mypoints[1].x = (coord->tx - (coord->tx - coord->x) / 4) * size_factor;
  mypoints[1].y = (coord->ty - (coord->ty - coord->y) / 4) * size_factor;
  coord++;
  mypoints[2].x = (coord->tx - (coord->tx - coord->x) / 4) * size_factor;
  mypoints[2].y = (coord->ty - (coord->ty - coord->y) / 4) * size_factor;
  mypoints[3].x = (coord->x + (coord->tx - coord->x) / 4) * size_factor;
  mypoints[3].y = (coord->y + (coord->ty - coord->y) / 4) * size_factor;
  gdk_draw_polygon (picture, mygc, TRUE, (GdkPoint *) mypoints,
		    (gint) npoints);
  if (!active || xbmflag)
    gdk_draw_line (picture, drawing_area->style->black_gc,
		   (gint) (x * size_factor), (gint) (y * size_factor),
		   (gint) (tx * size_factor), (gint) (ty * size_factor));
  else
    gdk_draw_line (picture, drawing_area->style->bg_gc[GTK_STATE_SELECTED],
		   (gint) (x * size_factor), (gint) (y * size_factor),
		   (gint) (tx * size_factor), (gint) (ty * size_factor));
}

void
DrawDashed (int x, int y, int tx, int ty, int active)
/* draws a dashed bond */
{
  GdkSegment mypoints[6];
  int xlen, ylen, i;
  gint nsegments = 6;

  xlen = tx - x;
  ylen = ty - y;

  for (i = 0; i < nsegments; i++)
    {
      mypoints[i].x1 = (x + 2 * i * 0.09 * xlen) * size_factor;
      mypoints[i].y1 = (y + 2 * i * 0.09 * ylen) * size_factor;
      mypoints[i].x2 = (x + (2 * i + 1) * 0.09 * xlen) * size_factor;
      mypoints[i].y2 = (y + (2 * i + 1) * 0.09 * ylen) * size_factor;
    }
  if (!active || xbmflag)
    gdk_draw_segments (picture, drawing_area->style->black_gc,
		       (GdkSegment *) mypoints, (gint) nsegments);
  else
    gdk_draw_segments (picture,
		       drawing_area->style->bg_gc[GTK_STATE_SELECTED],
		       (GdkSegment *) mypoints, (gint) nsegments);
}

void
DrawDotted (int x, int y, int tx, int ty, int active)
/* draws a dotted bond */
{
  GdkPoint mypoints[10];
  int xlen, ylen, i;
  gint npoints = 10;

  xlen = tx - x;
  ylen = ty - y;

  for (i = 0; i < npoints; i++)
    {
      mypoints[i].x = (x + i * 0.1 * xlen) * size_factor;
      mypoints[i].y = (y + i * 0.1 * ylen) * size_factor;
    }
  if (!active || xbmflag)
    gdk_draw_points (picture, drawing_area->style->black_gc,
		     (GdkPoint *) mypoints, (gint) npoints);
  else
    gdk_draw_points (picture, drawing_area->style->bg_gc[GTK_STATE_SELECTED],
		     (GdkPoint *) mypoints, (gint) npoints);
}

void
DrawWide (int x, int y, int tx, int ty, int active)
/* draws a wide single bond */
{
  gint npoints = 4;
  GdkPoint mypoints[4];
  struct xy_co *coord;
  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active || xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  coord = center_double_bond (x, y, tx, ty, 6);
  mypoints[0].x = coord->x * size_factor;
  mypoints[0].y = coord->y * size_factor;
  mypoints[1].x = coord->tx * size_factor;
  mypoints[1].y = coord->ty * size_factor;
  coord++;
  mypoints[2].x = coord->tx * size_factor;
  mypoints[2].y = coord->ty * size_factor;
  mypoints[3].x = coord->x * size_factor;
  mypoints[3].y = coord->y * size_factor;
  gdk_draw_polygon (picture, mygc, TRUE, (GdkPoint *) mypoints,
		    (gint) npoints);

}

void
DrawStripe (int x, int y, int tx, int ty, int active)
/* draws a striped single bond */
{
  int i;
  GdkPoint mypoints[4];
  struct xy_co *coord;
  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active || xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  coord = center_double_bond (x, y, tx, ty, 4);
  mypoints[0].x = coord->x * size_factor;
  mypoints[0].y = coord->y * size_factor;
  mypoints[1].x = coord->tx * size_factor;
  mypoints[1].y = coord->ty * size_factor;
  coord++;
  mypoints[2].x = coord->x * size_factor;
  mypoints[2].y = coord->y * size_factor;
  mypoints[3].x = coord->tx * size_factor;
  mypoints[3].y = coord->ty * size_factor;
  for (i = 0; i < 10; i++)
    {
      gdk_draw_line (picture, mygc,
		     mypoints[0].x + i * (mypoints[1].x - mypoints[0].x) / 10,
		     mypoints[0].y + i * (mypoints[1].y - mypoints[0].y) / 10,
		     mypoints[2].x + i * (mypoints[3].x - mypoints[2].x) / 10,
		     mypoints[2].y + i * (mypoints[3].y -
					  mypoints[2].y) / 10);
    }
}


void
DrawCircle (int x, int y, int tx, int ty, int active)
/* draws a circle of radius (tx/ty)-(x/y) around (x/y) */
{
  GdkGC *mygc;
  int topleftx, toplefty;
  int radius;


  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  radius = calc_vector (abs (tx - x), abs (ty - y)) * size_factor;

  topleftx = x * size_factor - radius;
  toplefty = y * size_factor - radius;
  gdk_draw_arc (picture, mygc, FALSE, topleftx, toplefty, 2 * radius,
		2 * radius, 0, 360 * 64);
  if (drawmode == 3 && !xbmflag)
    {
      mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];
      gdk_draw_rectangle (picture, mygc, TRUE, x * size_factor - 1,
			  y * size_factor - 1, 3, 3);
      gdk_draw_rectangle (picture, mygc, TRUE, tx * size_factor - 1,
			  ty * size_factor - 1, 3, 3);
    }
}


void
DrawWiggly (int x, int y, int tx, int ty, int active)
/* draws a wavy line */
{
  gint ax, ay, width, height, angle1, angle2;
  int narcs, veclen, boxlen;
  int i, xlen, ylen, ang;
  int boxcorx, boxcory;

  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];


  narcs = 5;

  xlen = tx - x;
  ylen = ty - y;

  veclen = calc_vector (abs (tx - x), abs (ty - y));
  ang =
    180. +
    asin (copysign ((float) ylen / (float) veclen, xlen * -ylen)) * 180. /
    3.14159;

  boxlen = 0.2 * veclen;
  boxcorx = (1. - (float) xlen / (float) veclen) * boxlen / 2;
  boxcory = (1. - (float) ylen / (float) veclen) * boxlen / 2;

  for (i = 0; i < narcs; ++i)
    {
      ax = (x + 0.2 * i * xlen - boxcorx) * size_factor;
      ay = (y + 0.2 * i * ylen - boxcory) * size_factor;
      width = boxlen * size_factor;
      height = boxlen * size_factor;
      angle1 = ang * 64;
      angle2 = -180 * 64 * pow (-1, i);
      if (x > tx)
	angle2 *= -1;
      gdk_draw_arc (picture, mygc, FALSE, ax, ay, width, height, angle1,
		    angle2);
    }

}

void
DrawArrow (int x, int y, int tx, int ty, int head, int active)
/* draws an arrow with either full or one-sided arrowhead */
{
  int xlen, ylen, xbase, ybase;
  gint npoints;
  float veclen, headfact, scalefact;
  GdkPoint mypoints[3];
  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  xlen = tx - x;
  ylen = ty - y;
  veclen = sqrt (xlen * xlen + ylen * ylen);
  scalefact=64./veclen; /* keep arrowhead size constant (64=std length)*/ 
/*  headfact = 1. - 20. / veclen;*/
  headfact = 0.8;
  xbase = x + headfact * xlen*scalefact;
  ybase = y + headfact * ylen*scalefact;

  xbase = tx - 0.2 *xlen*scalefact;
  ybase = ty - 0.2 *ylen*scalefact;

  mypoints[0].x = tx * size_factor;
  mypoints[0].y = ty * size_factor;

if (select_char (tx,ty,1) != NULL){
      xbase = x+headfact*xlen-12*xlen/veclen;
      ybase = y+headfact*ylen-12*ylen/veclen;
      mypoints[0].x = (tx-12*xlen/veclen)*size_factor;
      mypoints[0].y = (ty-12*ylen/veclen)*size_factor;
}
  mypoints[1].x = (xbase + 0.1 * ylen) * size_factor;
  mypoints[1].y = (ybase - 0.1 * xlen) * size_factor;
  mypoints[1].x = (xbase + 0.1 * ylen*scalefact) * size_factor;
  mypoints[1].y = (ybase - 0.1 * xlen*scalefact) * size_factor;

  if (head == 1)
    {
      mypoints[2].x = xbase * size_factor;	/*on baseline */
      mypoints[2].y = ybase * size_factor;
    }
  else
    {

      mypoints[2].x = (xbase - 0.1 * ylen) * size_factor;
      mypoints[2].y = (ybase + 0.1 * xlen) * size_factor;

      mypoints[2].x = (xbase - 0.1 * ylen*scalefact) * size_factor;
      mypoints[2].y = (ybase + 0.1 * xlen*scalefact) * size_factor;

    }

  npoints = 3;
  gdk_draw_line (picture, mygc, x * size_factor, y * size_factor,
		 tx * size_factor, ty * size_factor);
  gdk_draw_polygon (picture, mygc, TRUE, (GdkPoint *) mypoints,
		    (gint) npoints);
}

void
DrawDashedWedge (int x, int y, int tx, int ty, int active)
/* draws the dashed wedge for a bond behind the image plane */
{
  int i, xlen, ylen;
  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  xlen = tx - x;
  ylen = ty - y;


  for (i = 0; i < 10; i++)
    gdk_draw_line (picture, mygc,
		   (x + 0.1 * i * xlen - 0.01 * ylen * i) * size_factor,
		   (y + 0.1 * i * ylen + 0.01 * xlen * i) * size_factor,
		   (x + 0.1 * i * xlen + 0.01 * ylen * i) * size_factor,
		   (y + 0.1 * i * ylen - 0.01 * xlen * i) * size_factor);
}

void
DrawWedge (int x, int y, int tx, int ty, int active)
/* draws the wedge-shaped single bond 'pointing towards the viewer' */
{
  int xlen, ylen;
  gint npoints;
  GdkPoint mypoints[3];
  GdkGC *mygc;
  struct data *hpc;
  struct xy_co *coord;
  int d;
  int area;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  xlen = tx - x;
  ylen = ty - y;

  mypoints[0].x = x * size_factor;
  mypoints[0].y = y * size_factor;
  mypoints[1].x = (tx - 0.1 * ylen) * size_factor;
  mypoints[1].y = (ty + 0.1 * xlen) * size_factor;
  mypoints[2].x = (tx + 0.1 * ylen) * size_factor;
  mypoints[2].y = (ty - 0.1 * xlen) * size_factor;

  hpc = da_root.next;
  for (d = 0; d < hp->n; d++)
    {
      if (hpc->bond == 10)
	{
	  if ((fabs (hpc->x - tx) < 3 && fabs (hpc->y - ty) < 3)
	      || (fabs (hpc->tx - tx) < 3 && fabs (hpc->ty - ty) < 3))
	    {
	      coord =
		center_double_bond (hpc->x, hpc->y, hpc->tx, hpc->ty, 6);

	      if (fabs (hpc->x - tx) < 3 && fabs (hpc->y - ty) < 3)
		{

		  mypoints[1].x = coord->x * size_factor;
		  mypoints[1].y = coord->y * size_factor;
		  coord++;
		  mypoints[2].x = coord->x * size_factor;
		  mypoints[2].y = coord->y * size_factor;
		}
	      else
		{
		  mypoints[1].x = coord->tx * size_factor;
		  mypoints[1].y = coord->ty * size_factor;
		  coord++;
		  mypoints[2].x = coord->tx * size_factor;
		  mypoints[2].y = coord->ty * size_factor;
		}
	      area =
		0.5 * fabs (mypoints[0].x * (mypoints[1].y - mypoints[2].y) +
			    mypoints[1].x * (mypoints[2].y - mypoints[0].y) +
			    mypoints[2].x * (mypoints[0].y - mypoints[1].y));

	      if (fabs (area) < 76 * size_factor)
		{
		  mypoints[1].x = (tx - 0.05 * ylen) * size_factor;
		  mypoints[1].y = (ty + 0.05 * xlen) * size_factor;
		  mypoints[2].x = (tx + 0.05 * ylen) * size_factor;
		  mypoints[2].y = (ty - 0.05 * xlen) * size_factor;
		}
	    }
	}
      hpc = hpc->next;
    }
  npoints = 3;
  gdk_draw_polygon (picture, mygc, TRUE, (GdkPoint *) mypoints,
		    (gint) npoints);
}

void
Drawspline (int x0, int y0, int x1, int y1, int x2, int y2, int x3, int y3,
	    int type, int active)
/* draws a spline curve, optionally with full of half arrowhead  */
{
  int xlen, ylen, xbase, ybase;
  double px0, py0, px1, py1;
  int i;
  double t;
  gint npoints;
  GdkPoint mypoints[21];
  GdkGC *mygc;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  if (drawmode == 3 && !xbmflag)
    {
      gdk_draw_rectangle (picture, mygc, TRUE, x0 * size_factor - 1,
			  y0 * size_factor - 1, 3, 3);
      gdk_draw_rectangle (picture, mygc, TRUE, x1 * size_factor - 1,
			  y1 * size_factor - 1, 3, 3);
      gdk_draw_rectangle (picture, mygc, TRUE, x2 * size_factor - 1,
			  y2 * size_factor - 1, 3, 3);
      gdk_draw_rectangle (picture, mygc, TRUE, x3 * size_factor - 1,
			  y3 * size_factor - 1, 3, 3);
    }
  px0 = x0;
  py0 = y0;
  for (i = 0; i < 21; i++)
    {
      t = (float) i / 20.;
      px1 =
	t * t * t * (double) x3 + 3. * t * t * (1. - t) * (double) x2 +
	3. * t * (1. - t) * (1. - t) * (double) x1 + (1. - t) * (1. -
								 t) * (1. -
								       t) *
	px0;
      py1 =
	t * t * t * (double) y3 + 3. * t * t * (1. - t) * (double) y2 +
	3. * t * (1. - t) * (1. - t) * (double) y1 + (1. - t) * (1. -
								 t) * (1. -
								       t) *
	py0;
      mypoints[i].x = (gint) (px1 * size_factor);
      mypoints[i].y = (gint) (py1 * size_factor);
    }
  npoints = 21;
  if (type >= 0)
    gdk_draw_lines (picture, mygc, mypoints, npoints);

  if (type == -1)
    gdk_draw_polygon (picture, mygc, TRUE, mypoints, npoints);
  if (type == -2)
    gdk_draw_points (picture, mygc, mypoints, npoints);

  if (type <= 0)
    return;


  xlen = mypoints[20].x - mypoints[16].x;
  ylen = mypoints[20].y - mypoints[16].y;

  if (xlen != 0)
    xlen = (int) copysign (10. * size_factor, xlen);
  if (ylen != 0)
    ylen = (int) copysign (10. * size_factor, ylen);
  xbase = mypoints[16].x;
  ybase = mypoints[16].y;

  mypoints[0].x = mypoints[20].x;	/* arrow tip at the end of the last */
  mypoints[0].y = mypoints[20].y;	/* line segment */

  /* calculate both possible positions for arrow side */
  mypoints[1].x = (xbase + 0.5 * ylen);
  mypoints[1].y = (ybase - 0.5 * xlen);

  if (type == 1)
    {

      mypoints[2].x = (xbase - 0.5 * ylen);
      mypoints[2].y = (ybase + 0.5 * xlen);

      /* and choose the one facing outward (farther from the starting point) */
      px0 =
	(mypoints[1].x - x0 * size_factor) * (mypoints[1].x -
					      x0 * size_factor) +
	(mypoints[1].y - y0 * size_factor) * (mypoints[1].y -
					      y0 * size_factor);
      px1 =
	(mypoints[2].x - x0 * size_factor) * (mypoints[2].x -
					      x0 * size_factor) +
	(mypoints[2].y - y0 * size_factor) * (mypoints[2].y -
					      y0 * size_factor);

      if (px0 < px1)
	{
	  mypoints[1].x = mypoints[2].x;
	  mypoints[1].y = mypoints[2].y;
	}

      mypoints[2].x = xbase;	/*on baseline */
      mypoints[2].y = ybase;
    }
  else
    {
      mypoints[2].x = (xbase - 0.5 * ylen);	/* on opposite side of line */
      mypoints[2].y = (ybase + 0.5 * xlen);
    }

  npoints = 3;
  gdk_draw_polygon (picture, mygc, TRUE, (GdkPoint *) mypoints,
		    (gint) npoints);
}


void
Drawstring (int x, int y, char *c, int direct, int active)
/* draws a label with sub- and superscripting, symbols and text justification */
{
  int ha, tw, a;
  int nch, textl, chl, chr;
  char *l, *hc, hl[2];
  int d, hx, n;
  int shiftall = 0;
  int variance[MAXCL];
  char text[MAXCL];
  char symbol[MAXCL];
  char slanted[MAXCL];
  char bold[MAXCL];
  GdkGC *mygc;
  int save_x, sub_lastx, sup_lastx;

  mygc = drawing_area->style->black_gc;
  if (active && !xbmflag)
    mygc = drawing_area->style->bg_gc[GTK_STATE_SELECTED];

  x = x * size_factor;
  hx = x;
  y = y * size_factor;

  a = gdk_char_height (font, 'X');
  ha = gdk_char_height (font, 'x') / 2;
  n = strlen (c);

  hc = c;
  for (d = 0; d < n; d++)
    {
      symbol[d] = FALSE;
      slanted[d] = FALSE;
      bold[d]=FALSE;
      if (shiftall != 0)
	{
	  if (*hc == '}')
	    {
	      shiftall = 0;
	      if (*(hc + 1))
		hc++;
	      --n;
	    }
	  variance[d] = shiftall;
	}
      if (shiftall == 0)
	{
	  if ((variance[d] = Extra_char (*hc)) != 0)
	    {
	      if (*(hc + 1))
		{
		  hc++;
		  --n;
		  if (*hc == '{')
		    {
		      shiftall = variance[d];
		      hc++;
		      --n;
		    }
		}
	      else
		{
		  variance[d] = 0;
		}
	    }
	}
      text[d] = *hc++;
      if (text[d] == '@' && *hc)
	{
	  symbol[d] = TRUE;
	  text[d] = *hc++;
	  --n;
	}

      else if (text[d] == '|' && *hc)
	{
	  slanted[d] = TRUE;
	  text[d] = *hc++;
	  --n;
	}

      else if (text[d] == '#' && *hc)
	{
	  bold[d] = TRUE;
	  text[d] = *hc++;
	  --n;
	}

      else if (text[d] == '\\' && *hc)
	{
	  text[d] = *hc++;
	  --n;
	}
    }

  nch = n;
  if (nch > 0)
    {
      chl = 0;
      chr = 0;
      x = save_x = sub_lastx = sup_lastx = 0;
      for (d = 0; d < nch; d++)
	{
	  if (variance[d])
	    {
	      tw = gdk_char_width (smallfont, text[d]) + 2;
	      if (symbol[d])
		tw = gdk_char_width (ssymbfont, text[d]);

	      if (variance[d] < 0)
		{
		  if (save_x < x)
		    x = sub_lastx;
		  x = sub_lastx = x + tw;
		}
	      else
		{
		  if (save_x < x)
		    x = sup_lastx;
		  x = sup_lastx = x + tw;
		}
	    }
	  else /* not sub- or superscripted */
	    {
	      tw = 1 + gdk_char_width (font, text[d]);
	      if (symbol[d])
		tw = gdk_char_width (symbfont, text[d]);
	      if (bold[d])
	      	tw = gdk_char_width (boldfont, text[d]);
	      if (slanted[d])
		tw = 1 + gdk_char_width (slfont, text[d]);

	      if (save_x < x)
		{
		  if (sub_lastx < sup_lastx)
		    x = sup_lastx;
		  else
		    x = sub_lastx;
		}
	      x = save_x = sub_lastx = sup_lastx = x + tw;
	      chr = tw;
	    }
	  if (chl == 0)
	    chl = tw;
	}
      textl = save_x;
      switch (direct)
	{
	case 0:
	  hx = hx - chl / 2;
	  break;
	case -1:
	  hx = hx - (textl - 1) / 2;
	  break;
	case -2:
	  hx = hx + chr / 2 - (textl - 1);
	  break;
	}
    }
  nch = n;

  x = hx;
  save_x = sub_lastx = sup_lastx = x;
  for (d = 0; d < n; d++)
    {
      if (variance[d])
	{
	  tw = gdk_char_width (smallfont, text[d]) + 2;
	  if (symbol[d])
	    tw = gdk_char_width (ssymbfont, text[d]);
	  if (variance[d] < 0)
	    {
	      if (save_x < x)
		x = sub_lastx;
	      sub_lastx = x + tw;
	    }
	  else
	    {
	      if (save_x < x)
		x = sup_lastx;
	      sup_lastx = x + tw;
	    }
	}
      else
	{ /* not sub- or superscripted */
	  tw = 1 + gdk_char_width (font, text[d]);
	  if (symbol[d])
	    tw = gdk_char_width (symbfont, text[d]);
	  if (bold[d])
	    tw = gdk_char_width (boldfont, text[d]);
	  if (slanted[d])
	    tw = 1 + gdk_char_width (slfont, text[d]);
	  if (save_x < x)
	    {
	      if (sub_lastx < sup_lastx)
		x = sup_lastx;
	      else
		x = sub_lastx;
	    }
	  save_x = sub_lastx = sup_lastx = x + tw;
	}

/* erase a circular area slightly larger than the current character to
   create some space around the label */
      gdk_draw_arc (picture, drawing_area->style->white_gc, 1,
		    x - 5 * size_factor,
		    y - a + ha - 5 * size_factor + variance[d],
		    tw + 8 * size_factor, a + 10 * size_factor, 0, 360 * 64);
/*      gdk_draw_rect (picture, drawing_area->style->bg_gc[GTK_STATE_SELECTED], 1,
		    x - 7 * size_factor,
		    y - a + ha - 5 * size_factor + variance[d],
		    tw + 12 * size_factor, a + 12 * size_factor);*/
      x = x + tw;
    }

  n = nch;
  x = hx;
  save_x = sub_lastx = sup_lastx = x;
  for (d = 0; d < n; d++)
    {
      hl[0] = text[d];
      hl[1] = 0;
      l = &hl[0];

      if (variance[d])
	{
	  tw = gdk_char_width (smallfont, text[d]) + 2;
	  if (symbol[d])
	    tw = gdk_char_width (ssymbfont, text[d]);

	  if (variance[d] < 0)
	    {
	      if (save_x < x)
		x = sub_lastx;
	      sub_lastx = x + tw;
	    }
	  else
	    {
	      if (save_x < x)
		x = sup_lastx;
	      sup_lastx = x + tw;
	    }

	  if (symbol[d])
	    {
	      gdk_draw_text (picture, ssymbfont,
			     drawing_area->style->white_gc, x,
			     y + ha + variance[d], l, strlen (l));
	      gdk_draw_text (picture, ssymbfont, mygc, x,
			     y + ha + variance[d], l, strlen (l));
	    }
	  else
	    {
	      gdk_draw_text (picture, smallfont,
			     drawing_area->style->white_gc, x,
			     y + ha + variance[d], l, strlen (l));
	      gdk_draw_text (picture, smallfont, mygc, x,
			     y + ha + variance[d], l, strlen (l));
	    }
	  x = x + tw;
	}
      else /* not sub- or superscripted */
	{
	  tw = 1 + gdk_char_width (font, text[d]);
	  if (symbol[d])
	    tw = gdk_char_width (symbfont, text[d]);
	  if (bold[d])
	    tw = gdk_char_width (boldfont, text[d]);
	  if (slanted[d])
	    tw = 1 + gdk_char_width (slfont, text[d]);

	  if (save_x < x)
	    {
	      if (sub_lastx < sup_lastx)
		x = sup_lastx;
	      else
		x = sub_lastx;
	    }

	  if (symbol[d])
	    {
	      gdk_draw_text (picture, symbfont, drawing_area->style->white_gc,
			     x, y + ha, l, strlen (l));
	      gdk_draw_text (picture, symbfont, mygc, x, y + ha, l,
			     strlen (l));
	    }
	  else if (bold[d])
	    {
	      gdk_draw_text (picture, boldfont, drawing_area->style->white_gc,
			     x, y + ha, l, strlen (l));
	      gdk_draw_text (picture, boldfont, mygc, x, y + ha, l, strlen (l));
	    }
	  else if (slanted[d])
	    {
	      gdk_draw_text (picture, slfont, drawing_area->style->white_gc,
			     x, y + ha, l, strlen (l));
	      gdk_draw_text (picture, slfont, mygc, x, y + ha, l, strlen (l));
	    }
	  else
	    {
	      gdk_draw_text (picture, font, drawing_area->style->white_gc,
			     x, y + ha, l, strlen (l));
	      gdk_draw_text (picture, font, mygc, x, y + ha, l, strlen (l));
	    }
	  x = x + tw;
	  save_x = sub_lastx = sup_lastx = x;
	}
    }
}

int
Extra_char (char c)
/* process control characters for sub- and superscripting */
{
  int ha = gdk_char_height (font, 'x') / 2;
  if (c == '_')
    return (ha);
  if (c == '^')
    return (-ha);
  return (0);
}

int
Load_Font (int fnt)
/* load the normal, subscript and symbol fonts for all zoom scales */
{
/* NB: entries 0 and 1 are used for preview only, the default font is 4(!) */
  gchar *symtype[] = { "*-symbol-*-*-8-*", "*-symbol-*-*-10-*",
    "*-symbol-*-*-12-*", "*-symbol-*-*-14-*", "*-symbol-*-*-18-*",
    "*-symbol-*-*-18-*", "*-symbol-*-*-24-*"
  };

  gchar *ssymtype[] = { "*-symbol-*-*-8-*", "*-symbol-*-*-8-*",
    "*-symbol-*-*-10-*", "*-symbol-*-*-12-*", "*-symbol-*-*-14-*",
    "*-symbol-*-*-18-*", "*-symbol-*-*-24-*"
  };

  gchar *fntype[] =
    { "*-helvetica-medium-r-normal--8-*", "*-helvetica-medium-r-normal--10-*",
    "*-helvetica-medium-r-normal--12-*", "*-helvetica-medium-r-normal--14-*",
    "*-helvetica-medium-r-normal--18-*",
    "*-helvetica-medium-r-normal--18-*", "*-helvetica-medium-r-normal--24-*"
  };

  gchar *fallback[] = { "5x7", "6x10", "7x13", "9x15", "8x16",
    "10x20", "12x24"
  };

  gchar *sfntype[] =
    { "*-helvetica-medium-r-normal--8-*", "*-helvetica-medium-r-normal--8-*",
    "*-helvetica-medium-r-normal--10-*", "*-helvetica-medium-r-normal--12-*",
    "*-helvetica-bold-r-normal--14-*",
    "*-helvetica-bold-r-normal--14-*", "*-helvetica-medium-r-normal--24-*"
  };
/*  {"5x7", "5x7", "6x10", "8x13", "7x14",
   "8x13bold", "10x20"};*/

  gchar *sltype[] =
    { "*-helvetica-medium-o-normal--8-*", "*-helvetica-medium-o-normal--10-*",
    "*-helvetica-medium-o-normal--12-*", "*-helvetica-medium-o-normal--14-*",
    "*-helvetica-medium-o-normal--18-*",
    "*-helvetica-medium-o-normal--18-*", "*-helvetica-medium-o-normal--24-*"
  };

  gchar *btype[] =
    { "*-helvetica-bold-r-normal--8-*", "*-helvetica-bold-r-normal--10-*",
    "*-helvetica-bold-r-normal--12-*", "*-helvetica-bold-r-normal--14-*",
    "*-helvetica-bold-r-normal--18-*",
    "*-helvetica-bold-r-normal--18-*", "*-helvetica-bold-r-normal--24-*"
  };

  if (fnt > 6)
    return (0);

  if (font)
    gdk_font_unref (font);
  if (symbfont)
    gdk_font_unref (symbfont);
  if (smallfont)
    gdk_font_unref (smallfont);
  if (ssymbfont)
    gdk_font_unref (ssymbfont);
  if (slfont)
    gdk_font_unref (slfont);
  if (boldfont)
    gdk_font_unref (boldfont);

  if ((font = gdk_font_load (fntype[fnt])) == NULL)
    {
      fprintf (stderr, "failed to load font %s !!!\n", fntype[fnt]);
      if ((font = gdk_font_load (fallback[fnt])) == NULL)
	return (0);
    }
  if ((symbfont = gdk_font_load (symtype[fnt])) == NULL)
    {
      fprintf (stderr, "Failed to load symbol font, using standard font\n");
      symbfont = font;
    }
  if ((ssymbfont = gdk_font_load (ssymtype[fnt])) == NULL)
    ssymbfont = font;
  if ((smallfont = gdk_font_load (sfntype[fnt])) == NULL)
    smallfont = font;
  if ((slfont = gdk_font_load (sltype[fnt])) == NULL)
    slfont = font;
  if ((boldfont = gdk_font_load (btype[fnt])) == NULL)
    boldfont = font;
  return (1);
}

void
Set_Line (int lsty)
/* define standard line attributes for bond drawing */
{
  gdk_gc_set_line_attributes (drawing_area->style->white_gc, lsty, 0,
			      GDK_CAP_BUTT, GDK_JOIN_MITER);
  gdk_gc_set_line_attributes (drawing_area->style->black_gc, lsty, 0,
			      GDK_CAP_BUTT, GDK_JOIN_MITER);
}
