/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * Gradient editor module copyight (C) 1996-1997 Federico Mena Quintero
 * federico@nuclecu.unam.mx
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURIGHTE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * 21.03.2000 - Stefan Ondrejicka
 * I have removed unnecessary parts and adjusted for use with ChBg
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "config.h"
#include "gimpgradient.h"

#define BOUNDS(a,x,y)  ((a < x) ? x : ((a > y) ? y : a))

/***** Magic numbers *****/

#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */

#define EPSILON 1e-10

/* Gradient functions */

static gradient_t *grad_new_gradient(void);

/* Segment functions */

static grad_segment_t *seg_new_segment(void);
static void            seg_free_segment(grad_segment_t *seg);
static void            seg_free_segments(grad_segment_t *seg);

static grad_segment_t *seg_get_segment_at(gradient_t *grad, double pos);

/* Calculation functions */

static double calc_linear_factor(double middle, double pos);
static double calc_curved_factor(double middle, double pos);
static double calc_sine_factor(double middle, double pos);
static double calc_sphere_increasing_factor(double middle, double pos);
static double calc_sphere_decreasing_factor(double middle, double pos);

static void calc_rgb_to_hsv(double *r, double *g, double *b);
static void calc_hsv_to_rgb(double *h, double *s, double *v);

static void
grad_get_color_at_orig(gradient_t *curr_gradient, double pos, double *r, double *g, double *b, double *a)
{
	double          factor;
	grad_segment_t *seg;
	double          seg_len, middle;
	double          h0, s0, v0;
	double          h1, s1, v1;

	/* if there is no gradient return a totally transparent black */
	if (curr_gradient == NULL) 
	  {
	    r = 0; g = 0; b = 0; a = 0;
	    return;
	  }

	if (pos < 0.0)
		pos = 0.0;
	else if (pos > 1.0)
		pos = 1.0;

	seg = seg_get_segment_at(curr_gradient, pos);

	if (!seg)
	{
		*r = 0.0;
		*g = 0.0;
		*b = 0.0;
		return;
	}

	seg_len = seg->right - seg->left;

	if (seg_len < EPSILON) {
		middle = 0.5;
		pos    = 0.5;
	} else {
		middle = (seg->middle - seg->left) / seg_len;
		pos    = (pos - seg->left) / seg_len;
	} /* else */

	switch (seg->type) {
		case GRAD_LINEAR:
			factor = calc_linear_factor(middle, pos);
			break;

		case GRAD_CURVED:
			factor = calc_curved_factor(middle, pos);
			break;

		case GRAD_SINE:
			factor = calc_sine_factor(middle, pos);
			break;

		case GRAD_SPHERE_INCREASING:
			factor = calc_sphere_increasing_factor(middle, pos);
			break;

		case GRAD_SPHERE_DECREASING:
			factor = calc_sphere_decreasing_factor(middle, pos);
			break;

		default:
			factor = 0.0; /* Shut up -Wall */
			break;
	} /* switch */

	/* Calculate color components */

	*a = seg->a0 + (seg->a1 - seg->a0) * factor;

	if (seg->color == GRAD_RGB) {
		*r = seg->r0 + (seg->r1 - seg->r0) * factor;
		*g = seg->g0 + (seg->g1 - seg->g0) * factor;
		*b = seg->b0 + (seg->b1 - seg->b0) * factor;
	} else {
		h0 = seg->r0;
		s0 = seg->g0;
		v0 = seg->b0;

		h1 = seg->r1;
		s1 = seg->g1;
		v1 = seg->b1;

		calc_rgb_to_hsv(&h0, &s0, &v0);
		calc_rgb_to_hsv(&h1, &s1, &v1);

		s0 = s0 + (s1 - s0) * factor;
		v0 = v0 + (v1 - v0) * factor;

		switch (seg->color) {
			case GRAD_HSV_CCW:
				if (h0 < h1)
					h0 = h0 + (h1 - h0) * factor;
				else {
					h0 = h0 + (1.0 - (h0 - h1)) * factor;
					if (h0 > 1.0)
						h0 -= 1.0;
				} /* else */

				break;

			case GRAD_HSV_CW:
				if (h1 < h0)
					h0 = h0 - (h0 - h1) * factor;
				else {
					h0 = h0 - (1.0 - (h1 - h0)) * factor;
					if (h0 < 0.0)
						h0 += 1.0;
				} /* else */

				break;

			default:
				break;
		} /* switch */

		*r = h0;
		*g = s0;
		*b = v0;

		calc_hsv_to_rgb(r, g, b);
	} /* else */
} /* grad_get_color_at */

void
grad_get_color_at(gradient_t *curr_gradient, double pos, double *r, double *g, double *b)
{
	double a;

	grad_get_color_at_orig(curr_gradient, pos, r, g, b, &a);

	*r = a * (*r);
	*g = a * (*g);
	*b = a * (*b);
}

/***** Gradient functions *****/

/*****/

void
grad_free_gradient(gradient_t *grad)
{
	g_assert(grad != NULL);

	if (grad->segments)
		seg_free_segments(grad->segments);

	if (grad->filename)
		g_free(grad->filename);

	g_free(grad);
} /* grad_free_gradient */

/*****/

static gradient_t *
grad_new_gradient(void)
{
        gradient_t *grad;

        grad = g_malloc(sizeof(gradient_t));

        grad->segments     = NULL;
        grad->filename     = NULL;

        return grad;
} /* grad_new_gradient */

/*****/

gradient_t *
grad_get_random()
{
	int n;
	GSList *ptr;

	n = g_slist_length(cfg.grads);

	n = chbg_rand()%n;

	for (ptr = cfg.grads; ptr && n; n--, ptr = ptr->next);

	if (ptr)
		return (gradient_t *)ptr->data;

	return (gradient_t *)cfg.grads->data;
}

/*****/


static int
grad_cmp(gradient_t *g1, gradient_t *g2)
{
	return strcmp(g1->filename, g2->filename);
}

GSList *
grad_load_gradients(char *dirname)
{
	DIR *dir;
	struct dirent *dent;
	char next_dir[PATH_MAX];
	gradient_t *gr;
	GSList *rv = NULL;

	if (!(dir = opendir(dirname)))
	{
		perror(dirname);
		return rv;
	}

	while((dent = readdir(dir)))
	{
		if (!strcmp(dent->d_name , ".")) continue;
		if (!strcmp(dent->d_name , "..")) continue;             
		sprintf(next_dir , "%s/%s" , dirname , dent->d_name);

		gr = grad_load_gradient(next_dir);

		if (gr)
			rv = g_slist_append(rv, gr);
	}

	closedir(dir);

	rv = g_slist_sort(rv, (GCompareFunc)grad_cmp);

	return rv;
}

/*****/

gradient_t *
grad_load_gradient(char *filename)
{
	FILE           *file;
	gradient_t     *grad;
	grad_segment_t *seg, *prev;
	int             num_segments;
	int             i;
	int             type, color;
	char            line[1024];

	g_assert(filename != NULL);

	file = fopen(filename, "r");
	if (!file)
		return NULL;

	fgets(line, 1024, file);
	if (strcmp(line, "GIMP Gradient\n") != 0)
	{
		fclose(file);
		return NULL;
	}

	grad = grad_new_gradient();

	grad->filename = g_strdup(filename);

	fgets(line, 1024, file);
	num_segments = atoi(line);

	if (num_segments < 1) {
		g_message ("grad_load_gradient(): invalid number of segments in \"%s\"", filename);
		g_free(grad);
		fclose(file);
		return NULL;
	} /* if */

	prev = NULL;

	for (i = 0; i < num_segments; i++) {
		seg = seg_new_segment();
		seg->prev = prev;

		if (prev)
			prev->next = seg;
		else
			grad->segments = seg;

		fgets(line, 1024, file);

		if (sscanf(line, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%d%d",
			   &(seg->left), &(seg->middle), &(seg->right),
			   &(seg->r0), &(seg->g0), &(seg->b0), &(seg->a0),
			   &(seg->r1), &(seg->g1), &(seg->b1), &(seg->a1),
			   &type, &color) != 13) {
			g_message ("grad_load_gradient(): badly formatted "
				   "gradient segment %d in \"%s\" --- bad things may "
				   "happen soon", i, filename);
		} else {
			seg->type  = (grad_type_t) type;
			seg->color = (grad_color_t) color;
		} /* else */

		prev = seg;
	} /* for */

	fclose(file);

	return grad;
} /* grad_load_gradient */


GdkPixmap *grad_render(grad, h)
gradient_t *grad;
int h;
{
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkGC *gc;
	guchar *lnbuf,*p;
	guchar r,g,b;
	double gr,gg,gb;
	int w = 100,x,y;

	window = GDK_ROOT_PARENT();

	gc = gdk_gc_new(window);
	pixmap = gdk_pixmap_new(window, w, h, 
			gdk_visual_get_best_depth());

	lnbuf = g_malloc(3 * w);

	for (x = 0, p = lnbuf; x < w; x++, p += 3)
	{
		grad_get_color_at(grad, ((double)x)/(double)w,
			&gr, &gg, &gb);
		r = (guchar) (255.0 * gr);
		g = (guchar) (255.0 * gg);
		b = (guchar) (255.0 * gb);
		p[0] = r;
		p[1] = g;
		p[2] = b;
	}

	for (y = 0; y < h; y++)
		gdk_draw_rgb_image(pixmap , gc , 0 , y , w , 1 , 
			GDK_RGB_DITHER_NORMAL, lnbuf, 0);

	g_free(lnbuf);

	gdk_gc_unref(gc);

	return pixmap;
}

/***** Segment functions *****/

/*****/

static grad_segment_t *
seg_new_segment(void)
{
	grad_segment_t *seg;

	seg = g_malloc(sizeof(grad_segment_t));

	seg->left   = 0.0;
	seg->middle = 0.5;
	seg->right  = 1.0;

	seg->r0 = seg->g0 = seg->b0 = 0.0;
	seg->r1 = seg->g1 = seg->b1 = seg->a0 = seg->a1 = 1.0;

	seg->type  = GRAD_LINEAR;
	seg->color = GRAD_RGB;

	seg->prev = seg->next = NULL;

	return seg;
} /* seg_new_segment */

/*****/

static void
seg_free_segment(grad_segment_t *seg)
{
	g_assert(seg != NULL);

	g_free(seg);
} /* seg_free_segment */

/*****/

static void
seg_free_segments(grad_segment_t *seg)
{
	grad_segment_t *tmp;

	g_assert(seg != NULL);

	while (seg) {
		tmp = seg->next;
		seg_free_segment(seg);
		seg = tmp;
	} /* while */
} /* seg_free_segments */

/*****/

static grad_segment_t *
seg_get_segment_at(gradient_t *grad, double pos)
{
	grad_segment_t *seg = grad->segments;

	g_assert(grad != NULL);

	pos = BOUNDS(pos, 0.0, 1.0); /* to handle FP imprecision at the edges of the gradient */

	while (seg)
		if (pos >= seg->left) {
			if (pos <= seg->right) {
				return seg;
			} else
				seg = seg->next;
		} else
			seg = seg->prev;

	/* Oops: we should have found a segment, but we didn't */

	return NULL; /* To shut up -Wall */
} /* seg_get_segment_at */

/***** Calculation functions *****/

/*****/

static double
calc_linear_factor(double middle, double pos)
{
	if (pos <= middle) {
		if (middle < EPSILON)
			return 0.0;
		else
			return 0.5 * pos / middle;
	} else {
		pos -= middle;
		middle = 1.0 - middle;

		if (middle < EPSILON)
			return 1.0;
		else
			return 0.5 + 0.5 * pos / middle;
	} /* else */
} /* calc_linear_factor */

/*****/

static double
calc_curved_factor(double middle, double pos)
{
	if (middle < EPSILON)
		middle = EPSILON;

	return pow(pos, log(0.5) / log(middle));
} /* calc_curved_factor */

/*****/

static double
calc_sine_factor(double middle, double pos)
{
	pos = calc_linear_factor(middle, pos);

	return (sin((-M_PI / 2.0) + M_PI * pos) + 1.0) / 2.0;
} /* calc_sine_factor */

/*****/

static double
calc_sphere_increasing_factor(double middle, double pos)
{
	pos = calc_linear_factor(middle, pos) - 1.0;

	return sqrt(1.0 - pos * pos); /* Works for convex increasing and concave decreasing */
} /* calc_sphere_increasing_factor */

/*****/

static double
calc_sphere_decreasing_factor(double middle, double pos)
{
	pos = calc_linear_factor(middle, pos);

	return 1.0 - sqrt(1.0 - pos * pos); /* Works for convex decreasing and concave increasing */
} /* calc_sphere_decreasing_factor */

/*****/

static void
calc_rgb_to_hsv(double *r, double *g, double *b)
{
	double red, green, blue;
	double h, s, v;
	double min, max;
	double delta;

	red   = *r;
	green = *g;
	blue  = *b;

	h = 0.0; /* Shut up -Wall */

	if (red > green) {
		if (red > blue)
			max = red;
		else
			max = blue;

		if (green < blue)
			min = green;
		else
			min = blue;
	} else {
		if (green > blue)
			max = green;
		else
			max = blue;

		if (red < blue)
			min = red;
		else
			min = blue;
	} /* else */

	v = max;

	if (max != 0.0)
		s = (max - min) / max;
	else
		s = 0.0;

	if (s == 0.0)
		h = 0.0;
	else {
		delta = max - min;

		if (red == max)
			h = (green - blue) / delta;
		else if (green == max)
			h = 2 + (blue - red) / delta;
		else if (blue == max)
			h = 4 + (red - green) / delta;

		h /= 6.0;

		if (h < 0.0)
			h += 1.0;
		else if (h > 1.0)
			h -= 1.0;
	} /* else */

	*r = h;
	*g = s;
	*b = v;
} /* calc_rgb_to_hsv */

/*****/

static void
calc_hsv_to_rgb(double *h, double *s, double *v)
{
	double hue, saturation, value;
	double f, p, q, t;

	if (*s == 0.0) {
		*h = *v;
		*s = *v;
		*v = *v; /* heh */
	} else {
		hue        = *h * 6.0;
		saturation = *s;
		value      = *v;

		if (hue == 6.0)
			hue = 0.0;

		f = hue - (int) hue;
		p = value * (1.0 - saturation);
		q = value * (1.0 - saturation * f);
		t = value * (1.0 - saturation * (1.0 - f));

		switch ((int) hue) {
			case 0:
				*h = value;
				*s = t;
				*v = p;
				break;

			case 1:
				*h = q;
				*s = value;
				*v = p;
				break;

			case 2:
				*h = p;
				*s = value;
				*v = t;
				break;

			case 3:
				*h = p;
				*s = q;
				*v = value;
				break;

			case 4:
				*h = t;
				*s = p;
				*v = value;
				break;

			case 5:
				*h = value;
				*s = p;
				*v = q;
				break;
		} /* switch */
	} /* else */
} /* calc_hsv_to_rgb */

