/* undo.c
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "defines.h"
#include "font.h"

static struct UNDO_BUFFER {
    unsigned char *glyph;   /* history of glyph changes */
    unsigned char *x;       /* x-coordinate history */
    unsigned char *y;       /* y-coordinate history */
    short lost_orig;        /* original glyph has already shifted to nowhere */
    short pos;              /* current position in undo buffer */
    short size;             /* current maximum size of undo buffer */
} *buffers;

static int num_of_chars, height, is_file_modified;

/*------------------------------------------------------------------*/

int init_undo(struct CFE_FONT *font)
{
    num_of_chars = font->num_of_chars;
    height = font->height;

    buffers = (struct UNDO_BUFFER*)
        calloc(num_of_chars, sizeof(struct UNDO_BUFFER));
    
    if (buffers)
        return 1;
    else
        return 0;
}

/*------------------------------------------------------------------*/

void free_undo()
{
    int c;
    struct UNDO_BUFFER *buf;

    for (c = 0; c < num_of_chars; c++) {
        buf = &buffers[c];
        free(buf->glyph);
        free(buf->x);
        free(buf->y);
        buf->lost_orig = 0;
        buf->pos = 0;
        buf->size = 0;
    }
    free(buffers);
}

/*------------------------------------------------------------------*/

int get_undo(struct CFE_GLYPH *glyph)
{
    struct UNDO_BUFFER *buf = &buffers[glyph->num];

    buf->pos--;
    
    if (buf->size == 0) {
        buf->pos = 0;
        return 0;
    }
    
    if (buf->pos < (buf->size - 10)) {
        buf->size -= 10;
        buf->glyph = realloc(buf->glyph, buf->size * height);
        buf->x = realloc(buf->x, buf->size);
        buf->y = realloc(buf->y, buf->size);
        if (buf->size == 0) {
            buf->pos = 0;
            return 0;
        }
    }

    is_file_modified = 1;

    memcpy(glyph->data, buf->glyph + buf->pos * height, height);
    glyph->x = buf->x[buf->pos];
    glyph->y = buf->y[buf->pos];
    return 1;
}

/*------------------------------------------------------------------*/

void put_undo(struct CFE_GLYPH *glyph)
{
    struct UNDO_BUFFER *buf = &buffers[glyph->num];
    
    if (buf->pos == buf->size) {
        buf->size += 10;
        if (buf->size > NUM_OF_UNDO) {
            buf->size = NUM_OF_UNDO;
            buf->pos = (NUM_OF_UNDO - 10);
            memmove(buf->glyph, buf->glyph + height * 10,
                    (NUM_OF_UNDO - 10) * height);
            memmove(buf->x, buf->x + 10, NUM_OF_UNDO - 10);
            memmove(buf->y, buf->y + 10, NUM_OF_UNDO - 10);
            buf->lost_orig = 1;
        } else {
            buf->glyph = realloc(buf->glyph, buf->size * height);
            buf->x = realloc(buf->x, buf->size);
            buf->y = realloc(buf->y, buf->size);
            if (buf->y == 0) {
                perror(PACKAGE);
                exit(1);
            }
        }
    }
    
    is_file_modified = 1;

    memcpy(buf->glyph + buf->pos * height, glyph->data, height);
    buf->x[buf->pos] = glyph->x;
    buf->y[buf->pos] = glyph->y;
    buf->pos++;
}

/*------------------------------------------------------------------*/

int file_modified()
{
    return is_file_modified;
}

/*------------------------------------------------------------------*/

void clear_file_modified()
{
    is_file_modified = 0;
}

/*------------------------------------------------------------------*/

int glyph_modified(int glyph_num)
{
    struct UNDO_BUFFER *buf = &buffers[glyph_num];

    if (buf->pos != 0 || buf->lost_orig != 0)
        return 1;
    else
        return 0;
}


/* EOF */
