/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
 * \file cdw_thread.c
 * \brief Code creating threads and processes used in communication
 *        with external tools
 *
 * Main function in this file is cdw_thread_run_command(): its main tasks are:
 * \li creating pairs of sockets for communication with child process
 *     (cdw_thread_run_command() calls socketpair())
 * \li creating child process using fork()
 * \li replacing child process with external tool process using exec() call
 *     (this happens in child process)
 * \li creating two threads (with thread functions) that communicate with
 *     child process (this happens in parent process)
 * \li waiting for thread functions to return
 *
 * cdw_thread_run_command() hides threads and exec() from rest of project code.
 *
 * Thread functions are read_child_stdout() and read_child_stderr() (and
 * calculate_digest.read_and_write() - see below). They are executed in parent
 * process, reading data from stdout and stderr of child process, using
 * stdout_pipe[PARENTS_END] and stderr_pipe[PARENTS_END] file descriptors.
 * Child process is an external tool that creates iso image, burns data to
 * optical disc or does some other useful things. Purpose of those two
 * thread functions is to intercept output of child process: either data
 * printed by child process to its stdout or stderr. This output (charactr
 * strings) is placed by read_child_stdout() and read_child_stderr() in two
 * buffers: stdout_pipe_buffer[] and stderr_pipe_buffer[].
 *
 * Information read from pipes is then processed by parent process using
 * regexp functions called in thread functions.See pipe_regexp.c and
 * other *_pipe_regexp.c files for details on processing data
 * in *_pipe_buffer[] with regexp functions.
 *
 * File defines two generic thread functions that can be used for almost
 * any external command: read_child_stdout() and read_child_stderr().
 * There is also one special thread function: calculate_digest.read_and_write():
 * its task is to _write_ to stdin of child process, and it calls functions
 * related to one specific tool (md5sum). Description of read_child_*()
 * thread functions can be applied to this function if you replace 'read
 * stdout/stderr of child process' with 'write to stdin of child process'.
 */

#define _BSD_SOURCE /* usleep(), setenv() */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>

/* wait() */
#include <sys/types.h>
#include <sys/wait.h>

#include "gettext.h"
#include "cdw_logging.h"
#include "cdw_regex_dispatch.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_utils.h"
#include "cdw_digest.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"

/* choosing which descriptor in table (first or second) should belong
   to parent process and which belongs to child process seems to be
   purely arbitrary, but once set - it has to be used consistently;
   perhaps network applications using sockets already use industry-wide
   convention for this, where parent would be server and child would be
   client. I don't know nothing about such convention */
#define PARENTS_END 0
#define CHILDS_END 1

static const int cdw_stdout = 1;
static const int cdw_stderr = 2;


/* these are pairs of desciptors; their names seem to suggest that this
   is how _child_ process sees them: these are desciptors that will be
   connected to _child_ stdin, _child_ stdout and _child_ stderr;
   parent process will be looking at them and saying, "so I have listen
   on stdout_pipe[PARENTS_END] if I want to read stdout data of child
   process" */
int stdin_pipe[2];
int stdout_pipe[2];
int stderr_pipe[2];

cdw_task_t *thread_task;

/* time captured at the beginning of process */
time_t time0;

/* buffers for data read/written through pair of UNIX sockets */
char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];


/* this is THREAD.c, so avoid conflicts */
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;


/* two generic thread functions */
static void *read_child_stdout(void *dummy);
static void *read_child_stderr(void *dummy);
static void  cdw_thread_wait(cdw_task_t *thread_task);
static void  cdw_thread_set_child_env(void);

#define CDW_THREAD_EVAL_READ(m_char_buf, m_i, m_pipe_buf, m_new_lines, m_f) \
	cdw_assert (m_f == cdw_stdout || m_f == cdw_stderr, "ERROR: incorrect file %d\n", m_f);	\
	if (m_char_buf == 0x00) {					\
		;							\
	} else if (m_char_buf == '\n'					\
		   || m_char_buf == 0x0D				\
		   || m_char_buf == '\b' /* ^H */			\
		   || m_i == PIPE_BUFFER_SIZE) {			\
									\
		if (m_i != 0) { /* non-empty line */			\
			m_pipe_buf[m_i] = '\0';				\
			cdw_logging_write("%s\n", m_pipe_buf);		\
			/* weird thing with 2 "ifs", but otherwise gcc issues some warning */ \
			if (m_f == cdw_stdout) {			\
				cdw_regex_stdout_execute(thread_task);	\
			}						\
			if (m_f == cdw_stderr) {			\
				cdw_regex_stderr_execute(thread_task);	\
			}						\
			m_i = 0;					\
			m_new_lines = 0;				\
		} else { /* empty line */				\
			if (m_new_lines == 0) {				\
				m_pipe_buf[m_i] = '\0';			\
				cdw_logging_write("%s\n", m_pipe_buf);	\
				m_new_lines++;				\
			} else {					\
				; /* do nothing - avoid printing multiple empty lines */ \
			}						\
		}							\
	} else {							\
		m_pipe_buf[m_i] = m_char_buf;				\
		m_i++;							\
	}




/**
   \brief Read stdout output of child process

   Function runs in thread of parent process, reading stdout
   output of child process;

   original developer's comment:
   "I see CDrecorder version 0.1 source, and I write this code after this!!!
   THX to Sharad Mittal;"
*/
void *read_child_stdout(__attribute__((unused)) void *dummy)
{
	cdw_sdm ("INFO: thread %s() awaits for data\n", __func__);

	/* initialize regex variables */
	cdw_regex_stdout_prepare(thread_task);

	ssize_t r = 1;
	while (r > 0) {
		/* memset(stdout_pipe_buffer, '\0', sizeof(stdout_pipe_buffer)); */

		int i = 0;
		int newlines = 0;

		/* get some data from pipe */
		while (r > 0) {
			char buf = '\0';
			r = read(stdout_pipe[PARENTS_END], &buf, 1);
			int e = errno;
			pthread_mutex_lock(&mutex);

			if (r == 0) { /* EOF */
				; /* don't break inside mutex */
			} else if (r == -1) {
				cdw_vdm ("ERROR: read() returns -1, error = \"%s\"\n", strerror(e));
				/* don't break inside mutex */
			} else {
				CDW_THREAD_EVAL_READ(buf, i, stdout_pipe_buffer, newlines, cdw_stdout);
			}
			pthread_mutex_unlock(&mutex);
		}
	}

        cdw_regex_stdout_destroy(thread_task);

	return NULL;
}







/*
 * see comments before read_child_stdout()
 */
void *read_child_stderr(__attribute__((unused)) void *dummy)
{
	cdw_sdm ("INFO: thread %s() awaits for data\n", __func__);

	/* initialize regexp variables */
	cdw_regex_stderr_prepare(thread_task);

	ssize_t er = 1;
	while (er > 0) {
		/* memset(stderr_pipe_buffer, '\0', sizeof(stderr_pipe_buffer)); */

		int ei = 0;
		int enewlines = 0;

		/* get some data from pipe */
		while (er > 0) {
			char ebuf = '\0';
			er = read(stderr_pipe[PARENTS_END], &ebuf, 1);
			int e = errno;
			pthread_mutex_lock(&mutex);

			if (er == 0) { /* EOF */
				; /* don't break inside mutex */
			} else if (er == -1) {
				cdw_vdm ("ERROR: read() returns -1, error = \"%s\"\n", strerror(e));
				/* don't break inside mutex */
			} else {
				CDW_THREAD_EVAL_READ(ebuf, ei, stderr_pipe_buffer, enewlines, cdw_stderr);
			}
			pthread_mutex_unlock(&mutex);
		}
	}

	cdw_regex_stderr_destroy(thread_task);

	return NULL;
}





/**
   \brief Run external command

   Create pairs of sockets to new process (using socketpair()) and launch
   external program/tool (using provided 'command' and call to exec()) as
   new child process. The process will inherit created sockets.

   Function makes 'task' available as 'thread_task' so that thread functions
   (working in parent code) and regexp functions can set task->tool_status.

   TODO: make the function return values of type cdw_rv_t, for consistency.

   \param command - name of program (tool) that you want to run + it's arguments
   \param task - variable describing current task

   \return 0 on success
   \return -1 on failure
*/
int cdw_thread_run_command(const char *command, cdw_task_t *task)
{
	thread_task = task;
	cdw_sdm ("INFO: non-masked tool_status = 0x%04x\n", task->tool_status);

	//cdw_logging_write_separator();

	time0 = time(NULL);
	/* 2TRANS: this is time stamp (current time) printed to log file;
	   %d is an integer representing UNIX time */
	cdw_logging_write(_("Time stamp: %d\n"), (int) time0);
	/* 2TRANS: this is message printed in log file; this is current
	   time (in readable form, like "Wed Jun 30 21:49:08 1993");
	   %s is a string with time in readable, probably localized, form */
	cdw_logging_write(_("Time: %s\n"), ctime(&time0));
	/* 2TRANS: this is message printed in log file; "current command"
	   is a call to external command that will be performed */
	cdw_logging_write(_("Full command issued to shell: \"%s\"\n"), command);
	/* 2TRANS: this is message printed in log file; 'external tools' are
	   programs like cdrecord or mkisofs */
	cdw_logging_write(_("Output from external tool(s):\n\n"));

	if ((socketpair(AF_UNIX, SOCK_STREAM, 0, stdin_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stdout_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stderr_pipe) == 0)) {

		int fork_result;

		/* I'm using blocking read in thread code to minimize
		   idle looping, hence no O_NONBLOCK here */
		fcntl(stdin_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stdin_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fcntl(stdout_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stdout_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fcntl(stderr_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fork_result = fork();

		if (fork_result == -1) {
			/* 2TRANS: this is string displayed in log file when
			   something went very badly and cdw cannot create child
			   process - this is serious error */
			cdw_logging_write(_("Error: fork failure\n"));
			return -1;
		} else if (fork_result == 0) {
			/* we are child process; child process still should/will
			   write to stdout and stderr, but the stdout and
			   stderr file descriptors shouldn't be connected to
			   console any longer - they should be connected to
			   pipes; the pipes are inherited from parent process,
			   and are supposed to be used to communicate with
			   parent process, so parent process can know what
			   child process is doing;

			   child process inherits pipes, which includes
			   parent's ends of the pipes; they are useless in
			   child process, so it will be safe to close them
			   in child process; parent will keep them open to
			   read data from them - the data sent by child
			   process; */

			/* replace initial stdin/stdout/stderr file descriptors
			   of child process with ends of pipes */
			dup2(stdin_pipe[CHILDS_END], 0); /* WARNING: closing '0' may have strange influence on mkisofs */
			dup2(stdout_pipe[CHILDS_END], 1);
			dup2(stderr_pipe[CHILDS_END], 2);

			/* close unused instances of descriptors from pipe */
			close(stdin_pipe[CHILDS_END]);
			close(stdout_pipe[CHILDS_END]);
			close(stderr_pipe[CHILDS_END]);

			/* close parent's ends of pipes */
			close(stdin_pipe[PARENTS_END]);
			close(stdout_pipe[PARENTS_END]);
			close(stderr_pipe[PARENTS_END]);

			/* fcntl(stdin_pipe[CHILDS_END], F_SETFL, O_ASYNC); */
			/* fcntl(stdin_pipe[PARENTS_END], F_SETFL, O_ASYNC); */

			/* at this point child process still has descriptors
			   '0', '1' and '2' (stdin, stdout and stderr) open,
			   but they aren't connected to terminal (keyboard
			   and screen) anymore: they are now connected to
			   pipes, so any data printed by child application
			   to stdout or stderr (e.g. using
			   fprintf(stderr, ...) ) will be sent via pipe to
			   parent. Parent app will be able to read child
			   app data sent via pipe, as long as parent keeps
			   his ends of stdout and stderr pipe open. */

			/* below a copy of cdw turns into e.g. mkisofs */

			cdw_thread_set_child_env();
			int exec_ret = execl("/bin/sh", "sh", "-c", command, NULL);

			/* exec*() functions don't return */

			/* 2TRANS: this is message printed to log file when
			   something went very badly and  exec() function
			   returned after a call; %d is value returned by
			   exec() */
			cdw_logging_write(_("Error: exec() returned with value %d\n"), exec_ret);
			return -1;
		} else { /* we are still in parent process */

			/* now close childs ends of pipes: they are to be
			   operated by child process only, we shouldn't
			   read/write to them so let's close them and
			   don't bother with them anymore */
			close(stdin_pipe[CHILDS_END]);
			close(stderr_pipe[CHILDS_END]);
			close(stdout_pipe[CHILDS_END]);

			/* we will need two threads:
			   - one that reads stderr data of child thread
			     (read_child_stderr)
			   - one that reads stdout data of child thread
			     (read_child_stdout)

			   there is one special case when we create md5sum
			   process as child process: in such situation we need
			   to write data to process' stdin, and we will need
			   separate thread for this task; since md5sum does
			   not produce any data on stderr (I think) we can
			   disregard any data that would show on stderr, and
			   we don't have to create thread with read_child_stderr
			   function; instead create thread with function that
			   will write data to md5sum's stdin; */
			int rv;
			pthread_t thread_1, thread_2;

			cdw_sdm ("INFO: creating threads\n");

			if (thread_task->id == CDW_TASK_CALCULATE_DIGEST) {
				/* read_and_write has been configured to be a pointer to
				   a special thread function from cdw_calculate_digest module;
				   the function performs reading of data from data source
				   (this can be optical disc or regular file, which doesn't matter
				   at this point) and writing the data to stdin of digest tool process */
				thread_task->calculate_digest.target_fd = stdin_pipe[PARENTS_END];
				rv = pthread_create(&thread_1, NULL,
						    thread_task->calculate_digest.read_and_write,
						    (void *) thread_task);
				cdw_sdm ("INFO: created thread with read/write function from cdw_calculate_digest module\n");
			} else {
				/* "regular" thread with function for reading child's stderr */
				rv = pthread_create(&thread_1, NULL, &read_child_stderr, NULL);
			}

			if (rv != 0) {
				cdw_vdm ("ERROR: failed to create thread 1\n");
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread 1 creation error"));
			} else {
				cdw_sdm ("INFO: created thread with function read_child_stderr\n");
			}

			rv = pthread_create(&thread_2, NULL, &read_child_stdout, NULL);
			if (rv != 0) {
				cdw_vdm ("ERROR: failed to create thread 2\n");
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread 2 creation error"));
			} else {
				cdw_sdm ("INFO: created thread read_child_stdout\n");
			}


			/* First let the child process exit, then wait
			   for threads reading from stdout/stderr of
			   the exited child process. This order seems
			   to be better: the functions return on
			   reading EOF, which in turn (if memory
			   serves me well) is pushed through the pipe
			   on child process exiting. So first wait for
			   the child (wait()), and then wait for
			   threads (pthread_join()). */


			/* Perhaps for some external tool we won't be
			   able to parse its stdout/stderr output. Let's at
			   least have exit code of the tool. */
			cdw_thread_wait(thread_task);


			/* Threads have been created above. When
			   thread functions end we can execute
			   pthread_join()s */
			pthread_join(thread_2, NULL);
			pthread_join(thread_1, NULL);


			/* This file descriptor most probably has been
			   already closed by read/write function (otherwise
			   child process would still wait for data on open
			   stdin, so it would still be active, so thread would
			   be still active and could not have been joined).;

			   But just to be sure: close it again;
			   closing file equals marking end of input (eof) for
			   external processes (like md5sum) waiting for more data */
			close(stdin_pipe[PARENTS_END]);

			close(stderr_pipe[PARENTS_END]);
			close(stdout_pipe[PARENTS_END]);
		}
	} else { /* creating socket pairs failed */
		/* 2TRANS: this is a message printed to log file; "socket" is an UNIX socket */
		cdw_logging_write(_("Error: creating socket pairs failed\n"));
		return -1;
	}
	cdw_sdm ("INFO: tool status at exit = \"%s\"\n", thread_task->tool_status.ok ? "ok" : "error");
	return 0;
}





void cdw_thread_wait(cdw_task_t *task)
{
	wait(&(task->tool_status.child_exit_status_raw));
	if (WIFEXITED (task->tool_status.child_exit_status_raw)) {
		cdw_vdm ("INFO: child process called exit(%d)\n", WEXITSTATUS(task->tool_status.child_exit_status_raw));
	} else {
		cdw_vdm ("ERROR: child process terminated by signal %d\n", WTERMSIG (task->tool_status.child_exit_status_raw));
	}

	task->tool_status.child_exit_status = WEXITSTATUS(task->tool_status.child_exit_status_raw);

	return;
}





/* this function modifies child process' environment, which is a "throw-away"
   environment - no need to remember initial state of environment in order
   to restore it later */
void cdw_thread_set_child_env(void)
{
	/* since this function is called in child process, after modifying
	   stderr, debug messages (cdw_vdm()) will be printed to cdw.log file,
	   not to stderr; keep that in mind when debugging */
	int rv = setenv("LC_ALL", "C", 1);
	if (rv != 0) {
		int e = errno;
		cdw_vdm ("ERROR: setenv(\"LC_ALL\", \"C\", 1) returns !0, error = \"%s\"\n", strerror(e));
		/* don't return, try to execute rest of code anyway */
	}


	if (thread_task->id == CDW_TASK_BURN_FROM_FILES
	    && thread_task->burn.tool.id == CDW_TOOL_GROWISOFS) {

		if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)) {
			const char *fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_MKISOFS);
			cdw_assert (fullpath != (char *) NULL, "ERROR: ext tools module returned NULL pointer\n");
			int rv1 = setenv("MKISOFS", fullpath, 1);
			/* growisofs from Debain repository uses GENISOIMAGE instead of MKISOFS :) */
			int rv2 = setenv("GENISOIMAGE", fullpath, 1);
			if (rv1 != 0 || rv2 != 0) {
				int e = errno;
				cdw_vdm ("ERROR: setenv(\"MKISOFS\"|\"GROWISOFS\" ...) returns !0, error = \"%s\"\n", strerror(e));
			} else {
#ifndef NDEBUG
				const char *m = getenv("MKISOFS");
				cdw_assert (!strcmp(m, fullpath),
					    "ERROR: failed to set MKISOFS env variable to fullpath, env = \"%s\", fullpath = \"%s\"\n",
					    m, fullpath);
				cdw_vdm ("INFO: MKISOFS env variable set to \"%s\" = \"%s\"\n", m, fullpath);
				m = getenv("GENISOIMAGE");
				cdw_assert (!strcmp(m, fullpath),
					    "ERROR: failed to set GENISOIMAGE env variable to fullpath, env = \"%s\", fullpath = \"%s\"\n",
					    m, fullpath);
				cdw_vdm ("INFO: GENISOIMAGE env variable set to \"%s\" = \"%s\"\n", m, fullpath);
#endif
			}
		} else {
			cdw_vdm ("ERROR: burning files with growisofs, but no mkisofs available\n");
		}
	} else {
#ifndef NDEBUG
		cdw_vdm ("INFO: no conditions to set mkisofs path\n");
		if (thread_task->id != CDW_TASK_BURN_FROM_FILES) {
			cdw_vdm ("INFO: not burning from files\n");
		}
		if (thread_task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_vdm ("INFO: not burning with growisofs\n");
		}
#endif
	}

	return;
}





cdw_rv_t cdw_thread_send_key_to_child_process(int key)
{
	unsigned char buf = (unsigned char) key;
	cdw_vdm ("INFO: sending key %d / \"%s\" to child process\n", key, cdw_ncurses_key_label(key));
	ssize_t w = write(stdin_pipe[PARENTS_END], &buf, 1);
	if (w != 1) {
		cdw_vdm ("ERROR: write returns %zd\n", w);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}
