/**
 * @file amigapar.c
 * Interface to accessing the Amiga parallel port
 * @author Marko Mkel (msmakela@nic.funet.fi)
 * @author Olaf Seibert (rhialto@mbfys.kun.nl)
 */

/*
 * Copyright  1994-1997 Marko Mkel and Olaf Seibert
 * Copyright  2001,2002 Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef COMM_AMIGA
# include "amigapar.h"
# include <exec/exec.h>
# include <resources/misc.h>
# include <clib/exec_protos.h>
# include <clib/misc_protos.h>
# include <dos/dos.h>

# include <signal.h>
# include <stdio.h>

/** base address of the miscellanous library */
struct Library* MiscBase = 0;
/** flag: which parts of the parallel port have been allocated? */
static unsigned long allocated = 0;

/** Reserve the parallel port
 * @param ctl	flag: reserve the control lines as well
 * @return	0 on success, nonzero on failure
 */
int
alloc_port (int ctl)
{
  /** name of the reserving application */
  static const unsigned char myname[] = "cbmlink";
  /** name of the misc.resource */
  static const unsigned char misc_resource[] = "misc.resource";

  if (!(MiscBase = OpenResource (misc_resource))) {
    fputs ("OpenResource: cannot open misc.resource\n", stderr);
    return 1;
  }

  if (allocated & (1 << MR_PARALLELPORT));
  else {
    register const unsigned char* user =
      AllocMiscResource (MR_PARALLELPORT, myname);
    if (user) {
      fputs ("Printer data lines are used by ", stderr);
      fputs ((const char*) user, stderr);
      fputs (".\n", stderr);
      return 2;
    }
    allocated |= 1 << MR_PARALLELPORT;
  }
  if (allocated & (1 << MR_PARALLELBITS));
  else {
    register const unsigned char* user =
      AllocMiscResource (MR_PARALLELBITS, myname);
    if (user) {
      fputs ("Printer control lines are used by ", stderr);
      fputs ((const char*) user, stderr);
      fputs (".\n", stderr);
      return 2;
    }
    allocated |= 1 << MR_PARALLELBITS;
  }

  return 0;
}

/** Give back the parallel port
 * @return	0 on success, nonzero on failure
 */
int
free_port (void)
{
  if (allocated & (1 << MR_PARALLELBITS)) {
    /* turn POUT back into input */
    CTL_DDR &= ~POUT;
    FreeMiscResource (MR_PARALLELBITS);
  }
  if (allocated & (1 << MR_PARALLELPORT)) {
    /* set all data lines to output high */
    DATA = 0xFF;
    DATA_DDR = 0xFF;
    FreeMiscResource (MR_PARALLELPORT);
  }
  allocated = 0;
  return 0;
}

/** Check whether the transfer has been interrupted */
void
check_break (void)
{
  if (SetSignal (0L, SIGBREAKF_CTRL_C | SIGBREAKF_CTRL_D) &
      (SIGBREAKF_CTRL_C | SIGBREAKF_CTRL_D))
    raise (SIGINT);
}

#endif /* COMM_AMIGA */
