(* $Id: rpc_transport.mli 182 2004-05-25 16:49:11Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

(* This module encapsulates the transport mechanism for both client and
 * server programs.
 * A "transporter" can be used for both sending and receiving, even at the
 * same time. You cannot abort transporting; to do so you have to shut down
 * the underlying socket and open a new one.
 *
 * Note: The transporter never opens or closes file descriptors by itself.
 * Whoever want to use the transporter, has to do this job.
 *
 * TO SEND SOMETHING:
 * - "create" a transporter (or re-use an old transporter which has completed
 *   its last sending cycle)
 * - "put" a value into the transporter. No I/O happens.
 * - wait until the socket is ready and "send_part". Repeat this until the
 *   whole message is sent. Note that "send_part" blocks until at least 1
 *   octet can be sent, and then sends as much as can be immediately
 *   transported without further blocks. "send_part" is designed to be used
 *   in a Unix.select loop.
 *
 * TO RECEIVE SOMETHING:
 * - "create" a transporter (see above). You can share a transporter for
 *   sending and receiving messages.
 * - wait until the socket has got input OR until "is_buffer_empty" returns
 *   false, and "receive_part". Repeat this
 *   until the whole message has been received. Note that "receive_part"
 *   blocks until at least 1 octet has been received and then reads all
 *   currently available octets of the message without further blocks.
 *   (It reads (under rarely seldom circumstances) even more octets. These
 *   go into an internal buffer.)
 *   "receive_part" is designed to be used in a Unix.select loop.
 * - "get" the message
 *
 * There are send_sync and receive_sync functions for synchronously handled
 * RPC calls. These functions simply loop until all is sent or all is
 * received.
 *)

open Rpc
open Rpc_packer

type t     (* a transporter *)

(* General: *)

val create :             (* create transporter from file descriptor *)
      Unix.file_descr ->
      protocol ->        (* package semantics or stream semantics? *)
      mode ->            (* server socket or single bidirectional connection? *)
      t

  (* Stream-oriented connections:
   *
   * protocol = Tcp:  several messages can be sent over a bidirectional
   *                  persisting stream
   * protocol = Udp:  exchange of datagrams. The length of the datagrams
   *                  is limited to 8000 octets.
   * mode = Socket:   this means that the peer's internet address/port
   *                  can be found out
   * mode = BiPipe:   no internet address available
   *
   * - The file descriptor may be a TCP server socket that has accepted a
   *   connection. In this case use protocol=Tcp and mode=Socket.
   * - The file descriptor may be a stream-oriented Unix domain socket.
   *   In this case use protocol=Tcp and mode=BiPipe.
   * - The file descriptor may be a bidirectional pipe (socketpair).
   *   In this case use protocol=Tcp and mode=BiPipe.
   * - The file descriptor may be a UDP internet socket. In this case
   *   use protocol=Udp and mode=Socket.
   * - Not supported: packet exchange over Unix domain sockets
   *)

val descriptor :          (* get the descriptor *)
      t ->
      Unix.file_descr



(* Sending: *)

val set_receiver : t -> Unix.sockaddr -> unit
  (* If protocol=Udp and mode=Socket, set the address of the receiving
   * peer manually.
   * Raises an exception if used with a different combination of 'protocol'
   * and 'mode'.
   *)

val put     :             (* put a value into the transporter *)
      t ->
      packed_value ->
      unit
  (* associate a value to be sent with the transporter *)

  (* Change in rpc-0.4: The string contained in the packed value may be
   * modified (the RM is set).
   *)


val send_part :           (* send as much as immediately possible *)
      t ->
      bool
  (* start or continue sending the message. If the file descriptor is
   * ready to send, the function does not block. Returns 'true' if the
   * whole message has been sent.
   *)

val is_sending_complete :
      t ->
      bool
  (* The return value of the last 'send_part'. Returns 'false' after
   * a new value has been put into the transporter (i.e. before 'send_part'
   * is invoked). In the (theoretical) case that the message to be sent
   * is empty, 'true' is returned immediately.
   *)

val clean_output :
      t ->
      unit
  (* Delete the message you have put into the transporter. This is useful
   * to save memory if you close or abort a connection (normally, the next
   * 'put' replaces the old reference by a new one; you do not need to
   * 'clean_output' in this case)
   * Note: after 'clean_output' has been performed, 'is_sending_complete'
   * returns 'true'.
   *)

val send_sync :
      t ->
      packed_value ->
      unit
  (* Puts the given value into the transporter and sends it as a whole.
   * This is a convenience function for simple applications.
   * At the same time, it is a simple example how to use the other
   * functions:
   *
   * let send_sync t v =
   *   put t v;
   *   while not (is_sending_complete t) do
   *     send_part t
   *   done;
   *   clean_output t
   *)


(* Receiving: *)

val receive_part :        (* receive as much as immediately possible *)
      t ->
      bool
  (* Receives the next part of the message that is coming in. Receiving
   * data is more complicated than sending because it is possible that
   * more than one message arrives with the next internet packet. To
   * solve this, there is an internal buffer that it used to separate
   * messages. This guarantees that every call of 'receive_part' extracts
   * at most one message.
   * The return value is 'true' if the next message could be extracted
   * from the input stream. It is 'false' if there was neither a previously
   * received message in the internal buffer nor the next message has been
   * completely received.
   * The function does not block if there is a at least partial message in the
   * internal buffer or if there is currently data arriving at the file
   * descriptor.
   *)

val get_sender : t -> Unix.sockaddr
  (* returns the peer's address if protocol = Udp and mode = Socket. *)

val is_message_complete : (* is the message complete? *)
      t ->
      bool
  (* Returns 'true' if the last 'receive_part' extracted a message that
   * can be obtained using 'get'. Returns 'false' before the first invocation
   * of 'receive_part'.
   * Note: Empty messages are ok and handled as any other message.
   *)

val is_message_incomplete : (* is the message incomplete? *)
      t ->
      bool
  (* Returns 'true' if the beginning of a message has been extracted, but
   * the message is not yet complete. Returns 'false' before the first
   * invocation of 'receive_part'.
   * [is_message_incomplete] implies [is_buffer_empty].
   *)

val no_message :
      t ->
      bool
  (* Returns 'true' if there is neither a complete nor an incomplete
   * message. This can only happen before the first invocation of
   * [receive_part], or after [clean_input].
   * [no_message] implies [is_buffer_empty].
   *)

(* Note that at any time exactly one of the three functions
 * [is_message_complete], [is_message_incomplete], and [no_message]
 * will return 'true', and that the other two functions will return
 * 'false'.
 *)

val is_buffer_empty :
      t ->
      bool
  (* Returns 'true' if the next invocation of 'receive_part' would try to
   * read from the file descriptor. Returns 'false' if there is something
   * in the internal buffer that can be processed.
   *)


val at_eof :
      t ->
      bool
  (* Returns 'true' if the end of the stream has been reached.
   * Note: If protocol=Udp, 'at_eof' is never 'true'.
   *)

val peek_length :
      t ->
	int
  (* Returns the knowledge about the length of the message currently being
   * received.
   * This means exactly:
   * - [is_message_complete] is true: [peek_length] returns the exact length
   * - [is_message_incomplete] is true: [peek_length] returns a lower
   *   limit of the actual length
   * - [no_message] is true: the function fails.
   *
   * Note that [peek_length] returns max_int if the message is announced to
   * be longer.
   *
   * Because of the structure of the messages, the total length is not known
   * in advance. The record markers are used to estimate the length.
   *)


val get :
      t ->
      packed_value
  (* returns the previously received message.
   * This function must be called immediately after the message has become
   * complete.
   * If the message has been dropped, "" will be returned.
   *)

val drop :
      t ->
      unit
  (* drops the message currently being received.
   * works if [is_message_complete] or if [is_message_incomplete].
   *)

val clean_input :
      t ->
      unit
  (* deletes the last received message. This can only be done if the
   * message is complete or the stream is at EOF
   *)

val receive_sync :
      t ->
      packed_value
  (* receive message, return it, and clean up. Raises End_of_file if
   * EOF has been detected.
   *)

val verbose : bool -> unit
  (* set whether you want debug messages or not *)
  (* Caution! This function is not protected against concurrent invocations
   * from several threads.
   *)
