/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/PhysicalModel.h>

#include "LoadsSimulation.h"
#include "LoadsSimulationDriver.h"
#include "AnimationMotorAddon.h"
#include "LoadsManager.h"
#include "PMManagerDC.h"

// camiTK
#include <Application.h>
#include <MainWindow.h>
#include <InteractiveViewer.h>
using namespace camitk;

// qt
#include <QToolBar>
#include <QSettings>
#include <QTime>
#include <QTextStream>
#include <QMessageBox>
#include <QFileDialog>
#include <QLibrary>

// icons
#include "player_end.xpm"
#include "player_pause.xpm"
#include "player_play.xpm"
#include "player_rew.xpm"
#include "player_record.xpm"

//--------------- Constructor ---------------------------------
LoadsSimulation::LoadsSimulation(LoadsManager * myLoadsManager,  QWidget* parent) : QDialog(parent) {

  ui.setupUi(this);
  myLM = myLoadsManager;
  myMotor = NULL;
  simulationToolBar = NULL;

  // online recording
  video = false;
  output = false;
  imageId = 0;

  // create the load simulator driver and connect it
  simDriver = new LoadsSimulationDriver(myLM, NULL);
  simDriver->setTime(0.0);
  connect(simDriver, SIGNAL(doOneStep()), this, SLOT(doOneStep()));

  // get the normal bg color
  bgColor = ui.tLineEdit->palette().color(QPalette::Base);

  simDriver->resetTMaxToDefault();

  // create the tool bar actions
  rewindToolbar = new QAction(QIcon(player_rew_xpm), "rewind", this);
  connect(rewindToolbar, SIGNAL(triggered()), this, SLOT(rewind()));
  playToolbar = new QAction(QIcon(player_play_xpm), "simulate", this);
  connect(playToolbar, SIGNAL(triggered()), this, SLOT(simulate()));
  playOneStepToolbar = new QAction(QIcon(player_end_xpm), "step", this);
  connect(playOneStepToolbar, SIGNAL(triggered()), this, SLOT(simulateOneStep()));
  pauseToolbar = new QAction(QIcon(player_pause_xpm), "pause", this);
  connect(pauseToolbar, SIGNAL(triggered()), this, SLOT(pause()));
  tLineEditToolbar = new QLineEdit("time", this);
  tLineEditToolbar->setReadOnly(true);
  tLineEditToolbar->setText("0.0");
  tLineEditToolbar->setMaximumWidth(110);
  
  // create a new tool bar
  simulationToolBar = Application::getMainWindow()->addToolBar("Simulation toolbar");
  simulationToolBar->setFixedWidth(250);
  simulationToolBar->addAction(rewindToolbar);
  simulationToolBar->addAction(playToolbar);
  simulationToolBar->addAction(playOneStepToolbar);
  simulationToolBar->addAction(pauseToolbar);
  simulationToolBar->addWidget(tLineEditToolbar);
  
  // fix libpng problem: reload pixmap
  ui.playPushButton->setIcon(QPixmap(player_play_xpm));
  ui.pausePushButton->setIcon(QPixmap(player_pause_xpm));
  ui.playOneStepPushButton->setIcon(QPixmap(player_end_xpm));
  ui.rewindPushButton->setIcon(QPixmap(player_rew_xpm));
  ui.recordPushButton->setIcon(QPixmap(player_record_xpm));

  // get default (settings) values
  QSettings & settings = Application::getSettings();
  settings.beginGroup("physicalModelComponent");
  QString tmax(settings.value("animationMotorAddOn/tMax").toString());

  if (tmax != "") {
    ui.tMaxLineEdit->setText(tmax);
    simDriver->setTMax(tmax.toDouble());
  }
  else
    ui.tMaxLineEdit->setText(QString::number(simDriver->getTMax()));

  QString dt(settings.value("animationMotorAddOn/dt").toString());

  if (dt != "") {
    ui.dtLineEdit->setText(dt);
    simDriver->setDt(dt.toDouble());
  }
  else
    ui.dtLineEdit->setText(QString::number(simDriver->getDt()));

  QString refreshdt(settings.value("animationMotorAddOn/refreshDt").toString());

  if (refreshdt != "") {
    ui.refreshDtLineEdit->setText(refreshdt);
    simDriver->setRefreshDt(refreshdt.toDouble());
  }
  else
    ui.refreshDtLineEdit->setText(QString::number(simDriver->getRefreshDt()));

  settings.endGroup();

  // update the display
  updateDisplay(true);

  // reset the bg color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tLineEdit->setPalette(palette); // bgColor
  ui.tMaxLineEdit->setPalette(palette); // bgColor
  ui.dtLineEdit->setPalette(palette); // bgColor
  ui.refreshDtLineEdit->setPalette(palette); // bgColor

  // set the focus to tLineEdit
  ui.dtLineEdit->setFocus();

  // reset elapsed
  elapsed = 0.0;

  // connect ui
  connect(ui.tMaxLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(tMaxModified()));
  connect(ui.tMaxLineEdit, SIGNAL(returnPressed()), this, SLOT(tMaxChanged()));
  connect(ui.dtLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(dtModified()));
  connect(ui.dtLineEdit, SIGNAL(returnPressed()), this, SLOT(dtChanged()));
  connect(ui.rewindPushButton, SIGNAL(clicked()), this, SLOT(rewind()));
  connect(ui.pausePushButton, SIGNAL(clicked()), this, SLOT(pause()));
  connect(ui.refreshDtLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(refreshDtModified()));
  connect(ui.refreshDtLineEdit, SIGNAL(returnPressed()), this, SLOT(refreshDtChanged()));
  connect(ui.playOneStepPushButton, SIGNAL(clicked()), this, SLOT(simulateOneStep()));
  connect(ui.addonLocationPushButton, SIGNAL(clicked()), this, SLOT(addAnimationMotorAddon()));
  connect(ui.playPushButton, SIGNAL(clicked()), this, SLOT(simulate()));
  connect(ui.outputPushButton, SIGNAL(clicked()), this, SLOT(chooseOutputDir()));
  connect(ui.recordPushButton, SIGNAL(toggled(bool)), this, SLOT(videoToggled(bool)));

  iterationCount = 0.0;
}

//--------------- Destructor ---------------------------------
LoadsSimulation::~LoadsSimulation() {
  if (simDriver)
    delete simDriver;

  simDriver = NULL;

  if (myMotor)
    delete myMotor;

  myMotor = NULL;

  if (simulationToolBar)
    delete simulationToolBar;

  simulationToolBar = NULL;

  video = false;

  output = false;
}

//--------------- getTime ---------------------------------
double LoadsSimulation::getTime() const {
  if (simDriver)
    return simDriver->getTime();
  else
    return 0.0;
}


//--------------- init ---------------------------------
void LoadsSimulation::init() {

  // try to find an add-on
  if (myLM->getAnimationMotorAddonLocation() != "") {
    // a location is already there...
    loadAddon(myLM->getAnimationMotorAddonLocation());
  }
  else {
    // get default (settings) values
    QSettings & settings = Application::getSettings();
    settings.beginGroup("physicalModelComponent");
    // get the add-on if found in the settings
    QString addonFile(settings.value("animationMotorAddOn/location").toString());
    settings.endGroup();
    
    if (addonFile != "")
      loadAddon(addonFile);
  }

  // set things available or not
  enableButtons(myMotor != NULL);

  // update the display
  updateDisplay(true);
}

//--------------- rewind -----------------------
void LoadsSimulation::rewind() {
  pause();
  simDriver->setLoads(NULL);
  simDriver->setTime(0.0);
  myMotor->restart();
  updateDisplay(true);
  Application::getMainWindow()->getProgressBar()->setValue(0);
  imageId = 0;
  ui.videoInfoLabel->setText("");

  // reset elapsed
  elapsed = 0.0;
  iterationCount = 0.0;
  ui.simulFreq->setText("0.0");
}

//--------------- pause -----------------------
void LoadsSimulation::pause() {
  // stop the timer
  simDriver->stopTimer();

  // force display the movements and time display
  double refreshDt = simDriver->getRefreshDt();
  simDriver->setRefreshDt(-1);
  updateDisplay(true);
  simDriver->setRefreshDt(refreshDt);
}

//--------------- doOneStep -----------------------
bool LoadsSimulation::doOneStep() {
  // As this method is called by a timer, it could happen that the motor
  // has not finished one step of computation when the timer starts
  // the method again. This results in a strange state.
  // waitingForFinish act as a flag to say "wait, wait, we are
  // already computing a step, we can't do many at the same time,
  // take your chance next time".
  // That allows the timer to be regulated by the motor computation time.
  static bool waitingToFinish = false;

  if (waitingToFinish) {
    simDriver->slower();
    return false; // bye bye, we are too busy at the moment, we haven't done anything
  }

  waitingToFinish = true;

  // if t is ok, play it

  if (simDriver->getTime() <= simDriver->getTMax()) {
    // update cursor
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    // show the progress
    Application::getMainWindow()->getProgressBar()->setValue((int)(simDriver->getTime() *100.0 / simDriver->getTMax()));

    // keep what is the atom display state (just in case)
    LoadsManager::AtomDataDisplayType previousDisplayType = myLM->getAtomDataDisplay();

    // start timer
    QTime chrono;
    chrono.start();

    // do simulation steps until the next refresh time is reached
    Loads * newLoads = myMotor->doMove(simDriver->getDt(), simDriver->getNextRefreshTime());
    elapsed += chrono.elapsed() / 1000.0; // elapsed is in seconds
    iterationCount += simDriver->getRefreshDt() / simDriver->getDt();
    double freq = iterationCount / elapsed;
    ui.simulFreq->setText(QString::number(freq, 'f', 2));

    simDriver->setLoads(newLoads);

    // restore the atom display state
    if (previousDisplayType != LoadsManager::ADD_ON)
      myLM->setAtomDataDisplay(previousDisplayType);

    // update time and 3D display the movements and time display
    updateDisplay(false);

    // record video if needed

    if (video) {
      // make the video
      QString fn, ff;
      double time = myMotor->getTime();
      QTextStream fnTS(&fn);
      fnTS << outputDir << "/" << myLM->getPMManagerDC()->getPhysicalModel()->getName().c_str();
      // set the equivalent of sprintf(%08.3f",time)
      fnTS.setRealNumberNotation(QTextStream::FixedNotation);
      fnTS.setRealNumberPrecision(3);
      fnTS.setFieldWidth(8);
      fnTS.setPadChar('0');
      fnTS << time;
      fnTS << ".bmp";
      // ff is just the file name w/o the directory TODO use QFileInfo
      ff = fn.right(fn.lastIndexOf("/"));
      InteractiveViewer::get3DViewer()->screenshot(fn);
      ui.videoInfoLabel->setText(ff + " saved in " + outputDir);
      qApp->processEvents();
      // next one
      imageId++;
    }

    // update cursor
    QApplication::restoreOverrideCursor();
  }
  else {
    if (simDriver->isTimerActive()) {
      pause();
      Application::getMainWindow()->getProgressBar()->setValue(100);
    }

    // recording video: tell the user what happened
    if (imageId > 0) {
      QMessageBox::information(NULL, "Video Images Saved",
                               QString("Took %1").arg(imageId) + " screenshots\nAll saved in " +
                               outputDir);
    }

  }

  // release the flag, so that next time this method is called,
  // a new step could be computed
  waitingToFinish = false;

  return true; // bye bye, we have done one step
}

//--------------- simulate -----------------------
void LoadsSimulation::simulate() {
  simDriver->maxSpeed();

  if (!simDriver->isTimerActive()) {
    // timer wasn't active, action to be done = launch the timer
    simDriver->startTimer();
  } else {
    // timer is active, action to be done = play one step
    doOneStep();
  }
}

//--------------- simulateOneStep -----------------------
void LoadsSimulation::simulateOneStep() {
  // force one step to be done
  while (!doOneStep())
    ;
}

//---------------  updateTime -----------------------
void LoadsSimulation::updateTime(bool getMotorTime) {
  double t;

  if (getMotorTime && myMotor != NULL) {
    // get the animation motor add-on time
    t = myMotor->getTime();
    // force the simDriver time using t
    simDriver->setTime(t);
  }
  else {
    // get the simulation driver time
    t = simDriver->getTime();
  }

  // refresh the time display
  // tLineEdit
  ui.tLineEdit->setText(QString::number(t));

  tLineEditToolbar->setText(QString::number(t));

  ui.tLineEdit->update();
  tLineEditToolbar->update();
}

//---------------  updateDisplay -----------------------
void LoadsSimulation::updateDisplay(bool force) {
  // update the 3D display
  simDriver->updateDisplay(force);
  // update time display
  updateTime();
}

//---------------  dtChanged -----------------------
void LoadsSimulation::dtChanged() {
  simDriver->setDt(ui.dtLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.dtLineEdit->setPalette(palette); // bgColor
  // update the settings
  QSettings & settings = Application::getSettings();
  settings.beginGroup("physicalModelComponent");
  settings.setValue("animationMotorAddOn/dt", ui.dtLineEdit->text());
  settings.endGroup();  
}

//--------------- dtModified -----------------------
void LoadsSimulation::dtModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.dtLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//---------------  tMaxChanged -----------------------
void LoadsSimulation::tMaxChanged() {
  simDriver->setTMax(ui.tMaxLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tMaxLineEdit->setPalette(palette); // bgColor
  // update the settings
  QSettings & settings = Application::getSettings();
  settings.beginGroup("physicalModelComponent");
  settings.setValue("animationMotorAddOn/tMax", ui.tMaxLineEdit->text());
  settings.endGroup();
}

//--------------- tMaxModified  -----------------------
void LoadsSimulation::tMaxModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.tMaxLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//---------------  refreshDtChanged -----------------------
void LoadsSimulation::refreshDtChanged() {
  simDriver->setRefreshDt(ui.refreshDtLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.refreshDtLineEdit->setPalette(palette); // bgColor
  // update the settings
  QSettings & settings = Application::getSettings();
  settings.beginGroup("physicalModelComponent");
  settings.setValue("animationMotorAddOn/refreshDt", ui.refreshDtLineEdit->text());
  settings.endGroup();
}

//--------------- refreshDtModified -----------------------
void LoadsSimulation::refreshDtModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.refreshDtLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//--------------- addAnimationMotorAddon -----------------------
void LoadsSimulation::addAnimationMotorAddon() {
 QSettings & settings = Application::getSettings();
 settings.beginGroup("physicalModelComponent");
 QString lastOpened = QFileInfo(settings.value("animationMotorAddOn/location").toString()).absoluteDir().absolutePath();
  settings.endGroup();
  QString addonFile = QFileDialog::getOpenFileName(NULL, "Open an Animation Motor Add-on", lastOpened, "(*.so *.dll *.dylib)");

  if (!addonFile.isNull()) {
    loadAddon(addonFile);
  }
}

//--------------- loadAddon -----------------------
void LoadsSimulation::loadAddon(QString addonFile) {
  PtrToAnimationMotorAddon motorAddonFunction;

  addonFilename = addonFile;

  // try to load the add-on: first, look for the name
  motorAddonFunction = NULL;
  motorAddonFunction = (PtrToAnimationMotorAddon) QLibrary::resolve(addonFile, "getAnimationMotorAddon");

  // if the function is found, get the motor!

  if (motorAddonFunction) {
    // remove all tabs but the first one
    QWidget * motorWidget = ui.simulationTabs->widget(2);

    while (motorWidget != NULL) {
      ui.simulationTabs->removeTab(ui.simulationTabs->indexOf(motorWidget));
      delete motorWidget;
      motorWidget = ui.simulationTabs->widget(2);
    }

    // delete the previous motor
    if (myMotor != NULL) {
      delete myMotor;
      myMotor = NULL;
    }

    myMotor = motorAddonFunction(myLM);

    // set the name in the loads
    myLM->setAnimationMotorAddonLocation(addonFile);

    // update the settings
    QSettings & settings = Application::getSettings();
    settings.beginGroup("physicalModelComponent");
    settings.setValue("animationMotorAddOn/location", addonFile);
    settings.endGroup();
    
    // update the display
    ui.addonLocationLineEdit->setText(addonFile);
    QString ss(myMotor->getDescription());
    ui.descriptionLabel->setText(ss);

    // get the add-on widget and add it if non null

    if (myMotor != NULL) {
      motorWidget = myMotor->getMotorWidget();

      if (motorWidget) {
        addTab(motorWidget);
      }

      adjustSize();
    }

    // the action buttons
    enableButtons(myMotor != NULL);
  }
  else {
    QMessageBox::warning(NULL, "Animation Motor Add-on",
                         QString("File ") + addonFile + QString("\nis not a valid animation motor add-on."));
    show();
  }
}

//--------------- reject -----------------------
void LoadsSimulation::reject() {
  simDriver->stopTimer();
  Application::getMainWindow()->getProgressBar()->setValue(0);
//    uic_LoadsSimulation::reject();
}

//--------------- enableButtons -----------------------
void LoadsSimulation::enableButtons(bool enable) {
  // window buttons
  ui.playPushButton->setEnabled(enable);
  ui.rewindPushButton->setEnabled(enable);
  ui.pausePushButton->setEnabled(enable);
  ui.playOneStepPushButton->setEnabled(enable);
  ui.outputPushButton->setEnabled(enable);

  // toolbar buttons
  tLineEditToolbar->setEnabled(enable);
  rewindToolbar->setEnabled(enable);
  playToolbar->setEnabled(enable);
  playOneStepToolbar->setEnabled(enable);
  pauseToolbar->setEnabled(enable);
}

//--------------- addTab -----------------------
void LoadsSimulation::addTab(QWidget *w) {
  ui.simulationTabs->addTab(w, w->objectName());
}

//--------------- chooseOutputDir -----------------------
void LoadsSimulation::chooseOutputDir() {
  //QFileDialog* fd = new QFileDialog( this, "Output directory for screenshots", TRUE );
  //fd->setMode(QFileDialog::Directory);
  QString dir = QFileDialog::getExistingDirectory(this, tr("Output Directory For Dcreenshots"), ui.outputLineEdit->text());

  if (!dir.isNull()) {
    outputDir = dir;
    ui.outputLineEdit->setText(outputDir);
    output = true;
  }
}

//--------------- videoToggled -----------------------
void LoadsSimulation::videoToggled(bool state) {
  video = state;

  if (video) {
    if (output) {
      // get the value
      outputDir = ui.outputLineEdit->text();
    }
    else {
      // tell the user what happened
      QMessageBox::warning(NULL, "Missing output directory!",
                           QString("Could not record: choose an output directory first!"));
      ui.recordPushButton->setChecked(false);
    }
  }
}


