/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "PMManagerDC.h"
#include "StructuralComponentDC.h"
#include "StructuralComponentDCProperties.h"
#include "StructuralComponentDCPopup.h"
#include "AtomDC.h"
#include "CellDC.h"

#include <Log.h>
#include <Geometry.h>
#include <InteractiveViewer.h>
using namespace camitk;

#include <vtkUnstructuredGrid.h>
#include <vtkPolyData.h>
#include <vtkPoints.h>
#include <vtkPointSet.h>
#include <vtkTransformFilter.h>
#include <vtkPolyVertex.h>
#include <vtkDoubleArray.h>
#include <vtkCellArray.h>
#include <vtkIdTypeArray.h>
#include <vtkSmartPointer.h>

#include <pml/StructuralComponent.h>
#include <pml/Atom.h>
#include <pml/Cell.h>

// -------------------- default constructor  --------------------
StructuralComponentDC::StructuralComponentDC ( camitk::Component *parentDC, PMManagerDC * pmManagerDC, StructuralComponent *sc, bool delayRepresentationInit ) : ComponentDC ( parentDC, pmManagerDC, sc ) {

    myPopupMenu = NULL;
    myProp = NULL;
    pointData = NULL;

    camitk::Component::setName ( sc->getName().c_str() );

    // optimization for std::vector : keep these scalar updated
    nrOfParts = nrOfAtoms = 0;

    // Check if this is a list of atoms or of cells
    if ( sc->getNumberOfStructures() > 0 ) {
        if ( sc->composedBy() == StructuralComponent::ATOMS ) {
            Atom * a;
            AtomDC * adc;
            // update the counter
            nrOfAtoms = sc->getNumberOfStructures();

            // create the sub items (=create and register the DC)
            for ( unsigned int i = 0; i < nrOfAtoms; i++ ) {
                a = ( Atom * ) sc->getStructure ( i );

                // check if a dc already exist for this atom
                adc = myPMManagerDC->getDC ( a );

                if ( !adc ) {
                    // create a new atomdc (that will register itself)
                    adc = new AtomDC ( this, myPMManagerDC, a );
                }

                // force child addition
                attachChild ( adc );
            }
        }

        if ( sc->composedBy() == StructuralComponent::CELLS ) {
            Cell *c;
            CellDC *cdc;

            // create the Cells sub items
            for ( unsigned int i = 0; i < sc->getNumberOfStructures(); i++ ) {
                c = ( Cell * ) sc->getStructure ( i );

                // check if a dc already exist for this cell
                cdc = myPMManagerDC->getDC ( c );

                if ( !cdc ) {
                    // create a new atomdc (that will register itself)
                    cdc = new CellDC ( this, myPMManagerDC, c );
                }

                // force child addition
                attachChild ( cdc );

                // update the counter
                nrOfAtoms += c->getNumberOfStructures();
            }
        }
    }

    // associate this DC with this SC in the map
    myPMManagerDC->addStructuralComponentDCPair ( std::ComponentDCPair ( sc, this ) );

    // if the component should be seen by default
    if ( !delayRepresentationInit && getSC()->isVisible ( sc->getMode() ) ) {
        initRepresentation();
        // add it in the InteractiveViewer
        setVisibility ( InteractiveViewer::get3DViewer(), true );
    }

    myPMManagerDC->progressOneStep();

}

// --------------- destructor -------------------
StructuralComponentDC::~StructuralComponentDC() {
    destroyPointData();
    delete myPopupMenu;
    myPopupMenu = NULL;
    delete myProp;
    myProp = NULL;
}

// -------------------- getPropertyObject --------------------
QObject * StructuralComponentDC::getPropertyObject() {
    if ( !myProp ) {
        myProp = new StructuralComponentDCProperties ( this );
    }

    return myProp;
}

// --------------- updateAtoms -------------------
void StructuralComponentDC::updateAtoms() {
    if ( getSC()->composedBy() == StructuralComponent::CELLS ) {
        CellDC *cdc;
        unsigned int atomIndex;
        AtomDC *adc;

        nrOfParts = 0;
        std::vector<AtomDC *>::iterator it;
        // build the vector of all the DCs, without two identical DCs
        atomIndex = 0;
        atomDCs.clear();
        atomDCs.reserve ( nrOfAtoms ); // optimize the vector

        foreach ( camitk::Component *dc, childrenComponent ) {
            cdc = dynamic_cast<CellDC *> ( dc );
            nrOfParts += 1 + cdc->getChildren().size();

            foreach ( camitk::Component *dc2, cdc->getChildren() ) {
                adc = dynamic_cast<AtomDC *> ( dc2 );

                // if the atom here is not already there, add <a,atomIndex> in the map and adc in the vector
                it = std::find ( atomDCs.begin(), atomDCs.end(), adc );

                if ( it == atomDCs.end() ) {
                    adc->registerIndexInSCDC ( std::IndexInParentItemPair ( this, atomIndex++ ) );
                    atomDCs.push_back ( adc );
                }
            }
        }

        // update the nr of atoms
        nrOfAtoms = atomDCs.size();
    }
    else   // ComposedBy ATOMS
        if (getSC()->composedBy() == StructuralComponent::ATOMS) {
            // For structural component NOT featuring CELLS, i.e. only ATOMS
            unsigned int atomIndex = 0;
            AtomDC *adc;

            // loop over the atoms of the SC
            foreach ( camitk::Component *dc, childrenComponent ) {
                // add an atom index each time a vtkPoint representing an atom is updated
                // for instance when position changes during mechanical deformation
                // NOTE : without index, when running MMLDisplay::updateDisplay()
                // there is no way to get the correspondance between a 3D vtkPoint and the atom in the
                // physical model (the method 'getOrderNumberInSCDC' called in 'this->UpdatePosition()'
                // systematically returns -1 for SC only composed of ATOMS.
                adc = dynamic_cast<AtomDC *> ( dc );
          /*
                // if the atom here is not already there, add <a,atomIndex> in the map and adc in the vector
                std::vector<AtomDC *>::iterator it = std::find ( atomDCs.begin(), atomDCs.end(), adc );

                if ( it == atomDCs.end() ) {
                    */
                    adc->registerIndexInSCDC ( std::IndexInParentItemPair ( this, atomIndex++ ) );
                /*    atomDCs.push_back ( adc );
                }
                */

            }

            nrOfParts = nrOfAtoms = childrenComponent.size();
        }
        else // ComposedBy NOTHING
            nrOfParts = nrOfAtoms = childrenComponent.size();

}

// --------------- initRepresentation -------------------
void StructuralComponentDC::initRepresentation() {

    // check if it is a Structural Component containing a list of Cell or a list of Atom
    if ( getSC()->composedBy() == StructuralComponent::ATOMS ) {
        // create the vector and map
        updateAtoms();
        // create the geometry from the atom list
        myGeometry = atomsToGeometry();
    }
    else
        if ( getSC()->composedBy() == StructuralComponent::CELLS ) {
            // create the vector and map
            updateAtoms();
            // create the geometry from the cells
            myGeometry = cellsToGeometry();
        }

}


// --------------- cellsToGeometry -------------------
Geometry * StructuralComponentDC::cellsToGeometry() {
    Geometry *o;
    vtkSmartPointer<vtkUnstructuredGrid> uGrid;
    vtkSmartPointer<vtkPolyData> polyD;
    AtomDC *adc;
    Cell *c;
    CellDC *cdc;
    int atomOrderNumber;
    int *types;
    int *t;
    vtkSmartPointer<vtkCellArray> array;
    vtkSmartPointer<vtkIdTypeArray> idTypeArray;
    double pos[3];
    unsigned int i;
    unsigned int nrOfCellAtoms;
    unsigned int partId;
    unsigned int nrOfCells = childrenComponent.size();
    bool madeOfPolyLines = true;

    // create the vtkCellArray and other needed arrays
    types = new int [nrOfCells];
    t = types;
    array = vtkSmartPointer<vtkCellArray>::New();
    array->Allocate ( nrOfCells );
    idTypeArray = vtkSmartPointer<vtkIdTypeArray>::New();
    idTypeArray->SetNumberOfValues ( nrOfParts );

    // instanciate a new unstructured grid
    uGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();
    polyD = vtkSmartPointer<vtkPolyData>::New();

    partId = 0;

    foreach ( camitk::Component *dc, childrenComponent ) {
        // use the same order as the sub items, so that the order in the
        // sub item vector is the same as the order in the vtk cells
        cdc = dynamic_cast<CellDC *> ( dc );
        c = cdc->getCell();
        nrOfCellAtoms = c->getNumberOfStructures();

        // 1. build the array of point index (for all the cell's atom)
        // the order of the atoms should be the same as the SC::etAtoms() method
        idTypeArray->SetValue ( partId++, nrOfCellAtoms );

        foreach ( camitk::Component *dc2, cdc->getChildren() ) {
            // get the atom order number
            adc = dynamic_cast<AtomDC *> ( dc2 );
            atomOrderNumber = adc->getOrderNumberInSCDC ( this );
            // push the index in the VtkPoints of the j-th atom in the i-th cell
            idTypeArray->SetValue ( partId++, atomOrderNumber );
        }

        // 2. register the type for the i-th cell
        // get the vtk type

        switch ( c->getType() ) {
            case StructureProperties::TETRAHEDRON:
                *t++ = VTK_TETRA;
                break;
            case StructureProperties::HEXAHEDRON:
                *t++ = VTK_HEXAHEDRON;
                break;
            case StructureProperties::WEDGE:
                *t++ = VTK_WEDGE;
                break;
            case StructureProperties::PYRAMID:
                *t++ = VTK_PYRAMID;
                break;
            case StructureProperties::POLY_LINE:
                *t++ = VTK_POLY_LINE;
                break;
            case StructureProperties::LINE:
                *t++ = VTK_LINE;
                break;
            case StructureProperties::POLY_VERTEX:
                *t++ = VTK_POLY_VERTEX;
                break;
            case StructureProperties::TRIANGLE:
                *t++ = VTK_TRIANGLE;
                break;
            case StructureProperties::QUAD:
                *t++ = VTK_QUAD;
                break;
            default:
                *t++ = VTK_VERTEX;
                break;
        }

        madeOfPolyLines = madeOfPolyLines && ( ( c->getType() == StructureProperties::POLY_LINE ) || ( c->getType() == StructureProperties::LINE ) );
    }

    // create the vtkPoints array for the cells
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();

    thePoints->SetNumberOfPoints ( nrOfAtoms );

    // insert only the atoms that are in the vector (the one needed by the cells) (see updateAtoms() method)
    for ( i = 0; i < nrOfAtoms; i++ ) {
        atomDCs[i]->getAtom()->getPosition ( pos );
        thePoints->SetPoint ( i, pos[0], pos[1], pos[2] );
    }

    // insert everything everywhere (but at the right place!)
    array->SetCells ( nrOfCells, idTypeArray );

    if ( !madeOfPolyLines ) {
        uGrid->SetCells ( types, array );
        uGrid->SetPoints ( thePoints );
        // create the new geometry
        o = new Geometry ( this->getName(), uGrid, myPMManagerDC->toDCRenderingMode ( getSC()->getMode() ) );
    }
    else {
        // there are only polyline
        polyD->SetLines ( array );
        polyD->SetPoints ( thePoints );
        // create the new geometry
        o = new Geometry ( this->getName(), polyD, myPMManagerDC->toDCRenderingMode ( getSC()->getMode() ) );
    }

    // free up some memory
    delete [] types;

    // if the sc is not of the default color, change it
    o->setActorColor ( InterfaceGeometry::Surface, getSC()->getColor() );

    return o;
}

// --------------- atomsToGeometry -------------------
Geometry * StructuralComponentDC::atomsToGeometry() {
    Geometry *o;
    Atom *a;
    double pos[3];

    // create the vtk cell
    vtkSmartPointer<vtkPolyVertex> vtkCell = vtkSmartPointer<vtkPolyVertex>::New();
    vtkCell->GetPointIds()->SetNumberOfIds ( nrOfAtoms );

    // create the points for the cell
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();
    thePoints->SetNumberOfPoints ( nrOfAtoms );

    // NOTE : the index of each atom in the vtk structure is in fact its order number in the atoms structure (not its actual index)
    unsigned int i = 0;

    foreach ( camitk::Component *dc, childrenComponent ) {
        // use the same order as the sub items, so that the order in the
        // sub item vector is the same as the order in the vtk points
        a = dynamic_cast<AtomDC *> ( dc )->getAtom();
        a->getPosition ( pos );
        thePoints->SetPoint ( i, pos[0], pos[1], pos[2] );
        // build the id table
        vtkCell->GetPointIds()->SetId ( i, i );
        i++;
    }

    // instanciate a new unstructured grid
    vtkSmartPointer<vtkUnstructuredGrid> uGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();
    uGrid->Allocate ( 1, 1 );
    uGrid->InsertNextCell ( vtkCell->GetCellType(), vtkCell->GetPointIds() );
    uGrid->SetPoints ( thePoints );
    uGrid->Update();

    // create the new geometry (and give away the management of the uGrid...)
    o = new Geometry ( this->getName(), uGrid, myPMManagerDC->toDCRenderingMode ( getSC()->getMode() ) );

    return o;
}


// -------------------- getPopupMenu --------------------
QMenu * StructuralComponentDC::getPopupMenu ( QWidget* parent ) {
    if ( !myPopupMenu ) {
        myPopupMenu = new StructuralComponentDCPopup ( this, parent );
    }

    return myPopupMenu;
}

// -------------------- updatePosition --------------------
void StructuralComponentDC::updatePosition ( AtomDC *modifiedAtomDC, int atomOrderNumber ) {
    // update the 3D representation

    if ( myGeometry ) {
        // the order nr was not given, find it then !
        if ( atomOrderNumber == -1 ) {
            // I'm a cell or a SC called by my composing atoms
            if ( getSC()->composedBy() == StructuralComponent::ATOMS ) {
                // get the index of the modified  atom in the list
                atomOrderNumber = modifiedAtomDC->getOrderNumberInSCDC ( this );
            }
            else

                // I'm a SC whose cell has an atom that has changed
                if ( isInstanceOf ( "StructuralComponentDC" ) ) {
                    // should never happen normally as the AtomDC which is calling
                    // is able to give it order number in this (in 99% of the cases)
                    atomOrderNumber = modifiedAtomDC->getOrderNumberInSCDC ( this );
                }
        }

        // now, do it!
        if ( atomOrderNumber >= 0 ) {
            // get the new atom position
            double pos[3];
            modifiedAtomDC->getAtom()->getPosition ( pos );
            // set the new position in my geometry
            myGeometry->setPointPosition ( atomOrderNumber, pos[0], pos[1], pos[2] );
        }
    }

}


// -------------------- setPointSet --------------------
void StructuralComponentDC::setPointSet ( vtkSmartPointer<vtkPointSet> newPointSet ) {
    if ( !myGeometry )
        return;

    vtkSmartPointer<vtkPoints> vtkPts = newPointSet->GetPoints();

    // 1. tag this DC as modified
    getTopLevelComponent()->setModified();

    // 2. really update all the atom positions
    double pos[3];

    if ( getSC()->composedBy() == StructuralComponent::ATOMS ) {
        AtomDC *adc;

        // the atom's DC are the sub items
        for ( int i = 0; i < childrenComponent.size(); i++ ) {
            // get the new position
            vtkPts->GetPoint ( i, pos );
            // get the corresponding DC
            adc = dynamic_cast<AtomDC*> ( childrenComponent[i] );
            // set the new position for the atom (it will automatically update the DC, the atom, and all the cells, SC containing this atom)
            adc->setPosition ( pos[0], pos[1], pos[2] );
        }
    }
    else
        if ( getSC()->composedBy() == StructuralComponent::CELLS ) {
            // the atom's DC are in the vector
            for ( unsigned int i = 0; i < atomDCs.size(); i++ ) {
                // get the new position
                vtkPts->GetPoint ( i, pos );
                // set the new position for the atom (it will automatically update the DC, the atom, and all the cells, SC containing this atom)
                atomDCs[i]->setPosition ( pos[0], pos[1], pos[2] );
            }
        }

    // now call parent's method
    camitk::Component::setPointSet ( newPointSet );
}


//------------------------- pointPicked ------------------------
void StructuralComponentDC::pointPicked ( vtkIdType pointId, bool pickingIsSelecting  ) {
    if ( getSC()->composedBy() == StructuralComponent::ATOMS ) {
        //CAMITK_ERROR("StructuralComponentDC", "pointPicked", "Picking SC \"" << getName().toStdString() << "\": pointId #" << pointId << " in children DC named " << childrenComponent[pointId]->getName().toStdString());
        // the atomsToGeometry was used to create the Geometry...
        // ... thus the atoms are in the same order as the structures in the SC
        childrenComponent[pointId]->setSelected ( pickingIsSelecting );
    }
    else {
        if ( getSC()->composedBy() == StructuralComponent::CELLS) {
            // the pointId can be different than
            //CAMITK_ERROR("StructuralComponentDC", "pointPicked", "Picking SC \"" << getName().toStdString() << "\": pointId #" << pointId << " vs atomDCs size=" << atomDCs.size());
            if (pointId >= (vtkIdType) atomDCs.size()) {
                CAMITK_ERROR("StructuralComponentDC", "pointPicked", "Picking SC \"" << getName().toStdString() << "\": pointId #" << pointId << " is not a valid id (max=" << atomDCs.size() << ")");
            }
            else {
                //CAMITK_ERROR("StructuralComponentDC", "pointPicked", "Picking SC \"" << getName().toStdString() << "\": pointId #" << pointId << " in children DC named " << atomDCs[pointId]->getName().toStdString());
                // the cellsToGeometry method was used to create the Geometry...
                // ... thus the atomDC where placed in the vector
                atomDCs[pointId]->setSelected ( pickingIsSelecting );
            }
        }
    }
}

//------------------------ cellPicked ------------------------
void StructuralComponentDC::cellPicked ( vtkIdType cellId, bool pickingIsSelecting ) {
    if ( getSC()->composedBy() == StructuralComponent::CELLS ) {
        // the cellsToGeometry method was used to create the Geometry...
        // ... thus the cells are in the same order as the structures in the SC
        childrenComponent[cellId]->setSelected ( pickingIsSelecting );
    }

    // if the atomsToGeometry method was used to create the Geometry...
    // ... thus there are no cells! (the user should use point picking)
}

//------------------------ setSelected ------------------------
void StructuralComponentDC::setSelected ( const bool s, const bool recursive ) {
    // do not select sub atoms if this is not selected
    if ( recursive && !s && getSC()->composedBy() == StructuralComponent::ATOMS ) {
        camitk::Component::setSelected ( s, true );
    }
    else
        camitk::Component::setSelected ( s, false );
}

//-------------------------- setEnhancedModes --------------------------
void StructuralComponentDC::setEnhancedModes ( const EnhancedModes em ) {
    if ( getSC()->composedBy() != StructuralComponent::ATOMS )
        camitk::Component::setEnhancedModes ( em );
}

//-------------------------- setRenderingMode --------------------------
void StructuralComponentDC::setRenderingModes ( const RenderingModes m ) {
    camitk::Component::setRenderingModes ( m );
    getSC()->setMode ( myPMManagerDC->toPMRenderingMode ( m ) );
}

//---------------------------- setModeColor -------------------
void StructuralComponentDC::setActorColor ( const RenderingModes mode, const double r, const double g, const double b ) {
    camitk::Component::setActorColor ( mode, r, g, b );
    // get the current opacity for this mode
    double oldColor[4];
    getActorColor ( mode, oldColor );
    // set the color for of the SC
    getSC()->setColor ( r, g, b, oldColor[3] );
}

void StructuralComponentDC::setActorColor ( const RenderingModes mode, double newColor[4] ) {
    camitk::Component::setActorColor ( mode, newColor );
    getSC()->setColor ( newColor[0], newColor[1], newColor[2], newColor[3] );
}

//------------------------ setColor ---------------------------
void StructuralComponentDC::setColor ( const double r, const double g , const double b ) {
    camitk::Component::setColor ( r, g, b );
    getSC()->setColor ( r, g, b );
}

void StructuralComponentDC::setColor ( const double r, const double g, const double b, const double a ) {
    camitk::Component::setColor ( r, g, b, a );
    getSC()->setColor ( r, g, b, a );
}

//-------------------- composedBy --------------------
StructuralComponent::ComposedBy StructuralComponentDC::composedBy() {
    return getSC()->composedBy();
}

//-------------------- addChild --------------------
void StructuralComponentDC::addChild ( InterfaceNode *childNode ) {
    Structure *s = NULL;

    if ( dynamic_cast<AtomDC *> ( childNode ) )
        s = dynamic_cast<AtomDC *> ( childNode )->getAtom();
    else
        if ( dynamic_cast<CellDC *> ( childNode ) )
            s = dynamic_cast<CellDC *> ( childNode )->getCell();

    // check compatibility
    if ( s && getSC()->isCompatible ( s ) ) {
        // add with no check (that just has been done in the previous if statement)
        // but add only if not already there
        getSC()->addStructureIfNotIn ( s );
        // if it was possible to add it, then do the normal stuff
        camitk::Component::addChild ( childNode );
    }
}

//-------------------- removeChild -------------------
void StructuralComponentDC::removeChild ( InterfaceNode *childNode ) {
    Structure *s = NULL;

    if ( dynamic_cast<AtomDC *> ( childNode ) )
        s = dynamic_cast<AtomDC *> ( childNode )->getAtom();
    else
        if ( dynamic_cast<CellDC *> ( childNode ) )
            s = dynamic_cast<CellDC *> ( childNode )->getCell();

    if ( s ) {
        // update
        if ( dynamic_cast<AtomDC *> ( childNode ) ) {
            dynamic_cast<AtomDC *> ( childNode )->unregisterIndexInSCDC ( this );
        }
        else {
            int id = childrenComponent.indexOf ( dynamic_cast<camitk::Component *> ( childNode ) );

            if ( id != -1 )
                cellIdToBeRemoved.push_back ( id );
        }

        camitk::Component::removeChild ( childNode );

        getSC()->removeStructure ( s );
    }
}


//-------------------- structuralComponentToVtk -------------------
vtkSmartPointer<vtkUnstructuredGrid> StructuralComponentDC::structuralComponentToVtk ( StructuralComponent * sc, std::AtomVtkPointsIndexMap *atomVtkPointsIndexMap ) {

    unsigned int totalNumberOfCells = sc->getNumberOfCells();

    // create the vtkCellArray and other needed arrays
    int * types = new int [totalNumberOfCells];
    int * t = types;
    vtkSmartPointer<vtkCellArray> array = vtkSmartPointer<vtkCellArray>::New();
    array->Allocate ( totalNumberOfCells );
    vtkSmartPointer<vtkIdTypeArray> idTypeArray = vtkSmartPointer<vtkIdTypeArray>::New();

    // instanciate a new unstructured grid
    vtkSmartPointer<vtkUnstructuredGrid> uGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();

    unsigned int vtkPointsAtomIndex = 0; // index of the atom in the vtkPoints array

    std::AtomVtkPointsIndexMapIterator result; // result of a seach in the map
    Atom *a;
    Cell *c;
    int atomOrderNumber;
    unsigned int i;

    for ( i = 0; i < totalNumberOfCells; i++ ) {
        c = ( Cell * ) sc->getStructure ( i );

        // 1. build the array of point index (for all the cell's atom)
        // the order of the atoms should be the same as the SC::Atoms() method
        idTypeArray->InsertNextValue ( c->getNumberOfStructures() );

        for ( unsigned int j = 0; j < c->getNumberOfStructures(); j++ ) {
            a = ( Atom * ) c->getStructure ( j );

            // test if the atom here is not already in the list
            // In any case retrieve the index of a in the vtkPoints array
            result = atomVtkPointsIndexMap->find ( a );

            if ( result == atomVtkPointsIndexMap->end() )
                atomOrderNumber = -1;
            else
                atomOrderNumber = result->second;

            if ( atomOrderNumber == -1 ) {
                atomOrderNumber = vtkPointsAtomIndex;
                atomVtkPointsIndexMap->insert ( std::AtomVtkPointsIndexPair ( a, vtkPointsAtomIndex++ ) );
            }

            // push the index in the VtkPoints of the j-th atom in the i-th cell
            idTypeArray->InsertNextValue ( atomOrderNumber );
        }

        // 2. register the type for the i-th cell
        // get the vtk type
        switch ( c->getType() ) {
            case StructureProperties::TETRAHEDRON:
                *t++ = VTK_TETRA;
                break;
            case StructureProperties::HEXAHEDRON:
                *t++ = VTK_HEXAHEDRON;
                break;
            case StructureProperties::WEDGE:
                *t++ = VTK_WEDGE;
                break;
            case StructureProperties::PYRAMID:
                *t++ = VTK_PYRAMID;
                break;
            case StructureProperties::LINE:
                *t++ = VTK_LINE;
                break;
            case StructureProperties::POLY_LINE:
                *t++ = VTK_POLY_LINE;
                break;
            case StructureProperties::POLY_VERTEX:
                *t++ = VTK_POLY_VERTEX;
                break;
            case StructureProperties::TRIANGLE:
                *t++ = VTK_TRIANGLE;
                break;
            case StructureProperties::QUAD:
                *t++ = VTK_QUAD;
                break;
            default:
                *t++ = VTK_VERTEX;
                break;
        }
    }

    // create the vtkPoints array for the cells
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();

    // insert only the atoms that are in the map (the one needed by the cells) (thanks to the STL god!)
    std::AtomVtkPointsIndexMapIterator it;

    double pos[3];

    for ( it = atomVtkPointsIndexMap->begin(); it != atomVtkPointsIndexMap->end(); it++ ) {
        a = it->first;
        a->getPosition ( pos );
        thePoints->InsertPoint ( it->second, pos[0], pos[1], pos[2] );
    }

    // insert everything everywhere (but at the right place!)
    array->SetCells ( totalNumberOfCells, idTypeArray );
    uGrid->SetPoints ( thePoints );
    uGrid->SetCells ( types, array );
    delete [] types;

    return uGrid;
}

vtkSmartPointer<vtkUnstructuredGrid> StructuralComponentDC::structuralComponentToVtk ( StructuralComponent * sc ) {
    std::AtomVtkPointsIndexMap atomMap;
    return StructuralComponentDC::structuralComponentToVtk ( sc, &atomMap );
}


//------------------------ getPixmap ---------------------
#include "structuralcomponent_20x20.xpm"
QPixmap * StructuralComponentDC::myPixmap = NULL;
QPixmap StructuralComponentDC::getIcon() {
    if ( !myPixmap ) {
        myPixmap = new QPixmap ( structuralcomponent_20x20 );
    }

    return ( *myPixmap );
}

//-------------------- createPointData -------------------
void StructuralComponentDC::createPointData() {
    // check that we are really composed only by cells
    if ( !pointData && getSC()->composedBy() == StructuralComponent::CELLS ) {
        // create a new vtkDataArray
        pointData = vtkSmartPointer<vtkDoubleArray>::New();
        pointData->SetNumberOfTuples ( atomDCs.size() );
        setPointData ( pointData );

        // give the atom DCs the address of the corresponding point data
        for ( unsigned int i = 0; i < atomDCs.size(); i++ ) {
            atomDCs[i]->addPointData ( this, pointData->GetPointer ( i ) );
        }
    }
}

//-------------------- destroyPointData -------------------
void StructuralComponentDC::destroyPointData() {
    // delete the pointData
    if ( pointData ) {
        pointData = NULL;
        setPointData ( pointData );

        // clear the AtomDC's pointers
        for ( unsigned int i = 0; i < atomDCs.size(); i++ ) {
            atomDCs[i]->clearPointData();
        }
    }
}



