(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

open Report

(****************************)

type doc = int Report.report

let doc_of_html s = { Report.rep_eles = [Report.Leaf (fun () -> s)]; }

let doh = doc_of_html

let doc_of_report r = Obj.magic r

let report_of_doc d = d

let empty = doh ""

(****************************)

let seq docs = Th_seq.report docs

let concat sep docs =
  let rec iter acc = function
      [] -> acc
    | [d] -> d :: acc
    | d :: q -> iter (sep :: d :: acc) q
  in
  seq (iter [] (List.rev docs))

(****************************)

let tag t atts doc =
  let atts2 = List.map (fun (s1, s2) -> (s1, (fun () -> s2))) atts in
  let l = [
    Tag { tag = t ;
	  atts = atts2;
	  tag_subs = [Sub { sub_rep = (fun () -> doc) ; }] ;
	} 
  ] 
  in
  { rep_eles = l }

let link href ?target doc =
  let atts =
    ("href", fun () -> href) ::
    (match target with
      None -> []
    | Some t -> [("target",fun()->t)])
  in
  let l = [
    Tag { tag = "a" ;
	  atts = atts;
	  tag_subs = [Sub { sub_rep = (fun () -> doc); }] ;
	} 
  ] 
  in
  { rep_eles = l }

let br = doh "<br>"

let tr ?(atts=[]) docs = tag "tr" atts (seq docs)

let td ?w ?(atts=[]) doc = 
  tag "td" 
    (match w with 
      None -> atts
    | Some s -> ("width", s) :: atts)
    doc
let td_100 ?(atts=[]) doc = td ~w: "100%" ~atts doc

let td_top ?w ?(atts=[]) doc = td ?w ~atts: (("valign", "top")::atts) doc

let table ?w ?(atts=[]) rows = 
  tag "table"
    (match w with 
      None -> atts
    | Some s -> ("width", s) :: atts)
    (seq rows)

let table_100 ?(atts=[]) rows = table ~w: "100%" ~atts rows

let list_in_tag t ?(atts=[]) t2 ?(atts2=[]) l =
  tag t atts (seq (List.map (tag t2 atts2) l))

let ul_li l = list_in_tag "ul" "li" l
let ol_li l = list_in_tag "ol" "li" l

let ul ?(atts=[]) l = tag "ul" atts (seq l)
let ol ?(atts=[]) l = tag "ol" atts (seq l)

let h1 ?(atts=[]) doc = tag "h1" atts doc
let h2 ?(atts=[]) doc = tag "h2" atts doc
let h3 ?(atts=[]) doc = tag "h3" atts doc
let h4 ?(atts=[]) doc = tag "h4" atts doc
let h5 ?(atts=[]) doc = tag "h5" atts doc
let h6 ?(atts=[]) doc = tag "h6" atts doc

let tag_cl ?cl ?(atts=[]) t doc =
  tag t 
    (match cl with
      None -> atts
    | Some c -> ("class", c) :: atts)
    doc

let span ?cl ?(atts=[]) doc =
  tag_cl ?cl ~atts "span" doc 

let div ?cl ?(atts=[]) doc =
  tag_cl ?cl ~atts "div" doc 

let p doc = tag "p" [] doc

(****************************)

type css_names = Th_types.css_names = 
    {
      section_table : string ;
      section_title : string ;
      subsection_table : string ;
      subsection_title : string ;
      elements : string ;
      row : string array;
    } 

let css_names = Th_types.css_names

let set_css_names = Th_types.set_css_names


let css_code css =
  let buf = Buffer.create 256 in
  let f (s, s2) = Printf.bprintf buf "%s { %s }\n" s s2 in 
  List.iter f
    [
      "th."^css.section_title, 
      "background-color : #FFFFFF ; color : #006699 " ;

      "th."^css.subsection_title,
      ("background-color : #006699 ; color: #FFA34F; "^
       "font-size: 11px; font-weight : bold; height: 25px;") ;
      
      "table."^css.section_table, 
      "background-color : #FFFFFF;" ;

      "table."^css.subsection_table, 
      "background-color: #FFFFFF ; border: 2px #006699 solid;" ;
      
      "table."^css.elements, "background-color: #FFFFFF ;" ;

      "td."^css.row.(2),
      "vertical-align: top ; background-color: #EFEFEF; padding-left: 2 ; padding-right: 2 ;padding-bottom: 2; ";
      
      "td."^css.row.(0),
      "vertical-align: top ; background-color: #DEE3E7; padding-left: 2 ; padding-right: 2; padding-bottom: 2; ";

      "td."^css.row.(1),
      "vertical-align: top ; background-color: #D1D7DC; padding-left: 2 ; padding-right: 2; padding-bottom: 2; ";
    ] ;
  Buffer.contents buf




(****************************)

let frame_table ?(title="") ?(width="100%") rows =
  Th_frame_table.report title width (List.map report_of_doc rows)

let list_in_table ?(css=(css_names())) ?(title="") ?(sep=false) f_doc l =
  Th_list_in_table.row := 0;
  Th_list_in_table.report css title sep f_doc l

let double_list_in_table ?(css=(css_names())) ?(width="100%") ?(title="") l =
  Th_double_list_in_table.row := 0;
  Th_double_list_in_table.report css width title l

let vframeset frames = Th_frameset.report true frames
let hframeset frames = Th_frameset.report false frames

let tree l f_doc f_children =
  let rec iter ?(tab="") l =
    seq
      (List.map
	 (fun ele ->
	   seq
	     (
	      (doc_of_html tab) ::
	      (f_doc ele) ::
	      (doc_of_html "<br>\n") ::
	      (match f_children ele with
		[] -> []
	      | ch -> [iter ~tab: (tab^"&nbsp;&nbsp;&nbsp;&nbsp;") ch ]
	      )
	     )
	 )
	 l
      )
  in
  tag "table" [] 
    (tag "tr" []
       (tag "td" []
	  (iter l)
       )
    )

let tree_in_table ?css ?title l f_doc f_children =
  let max_depth = ref 1 in
  let rec iter n acc l =
    if !max_depth < n then max_depth := n;
    match l with
      [] -> acc
    | h :: q ->
	let acc2 = (n, f_doc h) :: acc in
	let acc3 = iter (n+1) acc2 (f_children h) in
	iter n acc3 q
  in
  let rows = List.rev (iter 0 [] l) in
  let rec build_tds acc n = 
    if n > 0 then
      build_tds (("20", doc_of_html "&nbsp;&nbsp;") :: acc) (n - 1)
    else
      List.rev acc
  in
  let colspan = string_of_int !max_depth in
  let f_row (n, doc) =
    let l = build_tds [] n in
    l @ [ "100%\" colspan=\""^colspan^"\"", doc ]    
  in
  list_in_table ?css ?title f_row rows

let page ?(style="") ?(more_head="") title body =
  Th_page_base.report style more_head title body

(**************************************************)

let escape_quotes s =
  let len = String.length s in
  let buf = Buffer.create len in
  for i = 0 to len - 1 do
    match s.[i] with
      '"' -> Buffer.add_string buf "&#34;"
    | c -> Buffer.add_char buf c
  done;
  Buffer.contents buf

let escape_entities s =
  let len = String.length s in
  let buf = Buffer.create len in
  for i = 0 to len - 1 do
    match s.[i] with
      '<' -> Buffer.add_string buf "&lt;"
    | '>' -> Buffer.add_string buf "&gt;"
    | '&' -> Buffer.add_string buf "&amp;"
    | c -> Buffer.add_char buf c
  done;
  Buffer.contents buf

let select name choices v =
  tag "select" ["name", name]
    (seq
       (List.map
          (fun (va, lab) -> 
            doc_of_html
              (Printf.sprintf "<option %s value=\"%s\">%s"
                 (if va = v then "selected" else "") 
                 (escape_quotes va) lab
              )
          )
          choices
       )
    )
(**************************************************)

type form_method = Get | Post
let string_of_form_method = function
    Get -> "get"
  | Post -> "post"

let form ?(met=Post) action body =
  tag "form" [("method", string_of_form_method met) ; ("action", action)]
    body

type input_type = 
    Text | Checkbox | Radio | Password | Submit | Reset | Hidden

let string_of_input_type = function
    Text -> "text"
  | Checkbox -> "checkbox"
  | Radio -> "radio"
  | Password -> "password"
  | Submit -> "submit"
  | Reset -> "reset"
  | Hidden -> "hidden"

let input typ ?size ?(checked=false) ?value varname =
  tag "input"
    (
     [ ("type", string_of_input_type typ) ; 
       ("name", varname) ;
     ] @
     (
      match checked with
	true -> ["checked", ""]
      |	false -> []
     ) @
     (
      match value with
	None -> []
      | Some v -> ["value", escape_quotes (escape_entities v)]
     ) @
     (
      match size with
	None -> []
      | Some v -> ["size", escape_quotes (escape_entities v)]
     )
    )
    empty

let submit_button label = input Submit ~value: label ""
let reset_button label =  input Reset ~value: label ""

(*************************************************)

let compute fmt doc =
  Report.compute ~html: true fmt
    (report_of_doc doc)

let compute_file file doc =
  Report.compute_file ~html: true file (report_of_doc doc)

let html_of_doc doc =
  let buf = Buffer.create 256 in
  let fmt = Format.formatter_of_buffer buf in
  compute fmt doc;
  Format.pp_print_flush fmt ();
  Buffer.contents buf


