(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** A HTML pretty-printer of OCaml code *)

open Odoc_info
module Naming = Odoc_html.Naming
open Odoc_info.Value
open Odoc_info.Module

let whole_page = ref true

let input_file_as_string nom =
  try
    let chanin = open_in_bin nom in
    let len = 1024 in
    let s = String.create len in
    let buf = Buffer.create len in
    let rec iter () =
      try
	let n = input chanin s 0 len in
	if n = 0 then
          ()
	else
          (
           Buffer.add_substring buf s 0 n;
           iter ()
          )
      with
	End_of_file -> ()
    in
    iter ();
    close_in chanin;
    Buffer.contents buf
  with
    Sys_error _ ->
      ""

class gen () =
  object (self)
    inherit Odoc_html.html as html

    method output_code in_title file code =
      try
        let chanout = open_out file in
        let html_code = self#html_of_code code in
	if !whole_page then
          output_string chanout ("<html>"^(self#header (self#inner_title in_title))^"<body>\n");
        output_string chanout html_code;
        if !whole_page then output_string chanout "</body></html>";
        close_out chanout
      with
        Sys_error s -> 
          incr Odoc_info.errors ;
          prerr_endline s

    method generate_module m =
      let file = !Odoc_info.Args.out_file in
      let s = input_file_as_string m.m_file in
      self#output_code m.m_name file s
      
    method generate (modules : Module.t_module list) =
      (* init the style *)
      self#init_style ;

      match modules with
	[] -> ()
      |	m :: _ -> self#generate_module m

  end

let generator = ((new gen ()) :> Odoc_args.doc_generator)


let op_nopage = ("-nopage", Arg.Clear whole_page, "Do not generate a page but only the HTML code for the given file")

let _ = Odoc_args.add_option op_nopage

let _ = Odoc_args.set_doc_generator (Some generator)
