/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: Filters.pmod,v 1.1.2.3 2004/04/07 13:52:32 vida Exp $
 */

/* This file helps you managing mail filters */

#include <camas/globals.h>

//! method: string encode_filter(string filter)
//!  Encode the filter, escaping any internal character used by the
//!  filter engine
string encode_filter(string filter)
{
  if(filter)
    return replace(filter, ({ ":", "\n", }),
      ({ "|||||||||||||", "************" }));
}

//! method: string decode_filter(string filter)
//!  Decode the filter, escaping any internal character used by the
//!  filter engine
string decode_filter(string filter)
{
  if(filter)
    return replace(filter, ({ "|||||||||||||", "************" }), 
      ({ ":", "\n" }));
}

//! method: array(string) get_filters(object id)
//!  returns an array contains all this user's filters
//!  the first element of the array is the name of the filter
//!  the second is the expression
//!  the third is the folder to move the mail to in case filterexpression match
array(array(string)) get_filters(object id)
{
  string namefield, filterexpression, filterfolder;
  array (string) filters =
    (sizeof(CSESSION->filterbook)>0)?(CAMAS.Tools.decode_pref(CSESSION->filterbook)/"\n"):({});
  filters -= ({ "" });
  array(array) result = ({ });
 
  for (int i = 0; i < sizeof(filters); i++)
  {
    if(sscanf(filters[i], "%s:%s:%s",
        namefield, filterexpression, filterfolder) == 3)
    {
      namefield = decode_filter(namefield);
      filterexpression = decode_filter(filterexpression);
      filterfolder = decode_filter(filterfolder);
      result += ({ ({ namefield, filterexpression, filterfolder }) });
    }
  }
  return result;
}

//! method: void set_filters(object id, array(array) filters)
//!   Write filters to the user session.  filters is the array as returned
//!   by get_filters()
void set_filters(object id, array(array) filters)
{
  string result = "";
  foreach(filters, array filter)
  {
    result += CAMAS.Tools.encode_pref(encode_filter(filter[0]) + ":" 
        + encode_filter(filter[1]) + ":"
        + encode_filter(filter[2]) + "\n");
  }
  CSESSION->filterbook = result;
  CSESSION->filters_have_changed = 1;
}

//! method: void add_filter(object id, string namefield, string filterexpression, string filterfolder)
//!  add a filter at the end of the user's filters
void add_filter(object id, string namefield, string filterexpression, string
    filterfolder)
{
  if(!CSESSION->filterbook)
    CSESSION->filterbook = "";
  string filter = CAMAS.Tools.encode_pref(encode_filter(namefield) + ":" 
      + encode_filter(filterexpression) + ":"
      + encode_filter(filterfolder) + "\n");
  CSESSION->filterbook += filter;
  CSESSION->filters_have_changed = 1;
}

//! method: void replace_filter_by_filternumber(object id, int filternumber, string namefield, string filterexpression, string filterfolder)
//!   Replace a filter given by the filternumber in the array returned by
//!   get_filters()
//!   If filternumber is out of range, the filter is added
void replace_filter_by_filternumber(object id, int filternumber, string namefield, string filterexpression, string filterfolder)
{
  array(array) filters = get_filters(id);
  if(filternumber < sizeof(filters) && filternumber >= 0)
  {
    array(string) filter = ({ namefield, filterexpression, filterfolder });
    filters[filternumber] = filter;
    set_filters(id, filters);
  }
  else
    add_filter(id, namefield, filterexpression, filterfolder);
}

// ! method: void replace_filter(object id, void|string namefield, void|string filterexpression, void|string filterfolder, string newnamefield, string newfilterexpression, string newfoldername)
//!   If namefield is not zero, replace a filter for which the namefield of the filter equal
//!   namefield 
//!   If filterexpression is not zero, replace a filter for which the 
//!   filterexpression equa filterexpression
//!   And so on 
//!   If you provide more than one filter type to replace, the matching filter
//!   is deleted if all the filters type matches (and'ed)
//!   newnamefield, newfilterexpression and newfoldername are the new values for the filter
//!   that will be replaced. If one of them if zero, the old value of this type is kept
void replace_filter(object id, void|string namefield, void|string filterexpression, void|string filterfolder, void|string newnamefield, string newfilterexpression, string newfoldername)
{
  array(array) filters = get_filters(id);
  for (int i = 0; i < sizeof(filters); i++)
  {
    array(string) filter = filters[i];
    int replaceable = 0;
    if(namefield)
      replaceable = (filter[0] == namefield)?1:0;
    if(filterexpression)
      replaceable = (filter[1] == filterexpression)?1:0;
    if(filterfolder)
      replaceable = (filter[2] == filterfolder)?1:0;
    if(replaceable)
    {
      string _namefield = newnamefield || filter[0];
      string _filterexpression = newfilterexpression || filter[1];
      string _foldername = newfoldername || filter[2];
      filters[i] = ({ _namefield, _filterexpression, _foldername });
    }
  }
  set_filters(id, filters);
}

// ! method: void delete_filter(object id, void|string namefield, void|string filterexpression, void|string filterfolder)
//!   If namefield is not zero, delete a filter for which the namefield of the filter equal
//!   namefield
//!   If filterexpression is not zero, delete a filter for which the
//!   filterexpression equa filterexpression
//!   And so on
//!   If you provide more than one filter type to delete, the matching filter
//!   is deleted if all the filters type matches (and'ed)
void delete_filter(object id, void|string namefield, void|string filterexpression, void|string filterfolder)
{
  array(array) filters = get_filters(id);
  array(array) result = allocate(sizeof(filters));
  int j = 0;
  for (int i = 0; i < sizeof(filters); i++)
  {
    array(string) filter = filters[i];
    int deleteable = 0;
    if(namefield)
      deleteable = (filter[0] == namefield)?1:0;
    if(filterexpression)
      deleteable = (filter[1] == filterexpression)?1:0;
    if(filterfolder)
      deleteable = (filter[2] == filterfolder)?1:0;
    if(deleteable)
      result = result[..sizeof(result)-2];
    else
      result[j++] = filters[i];
  }
  set_filters(id, result);
}

//! method: void delete_filter_by_filternumber(object id, int filternumber)
//!  Delete the filternumber from this user's filters list
void delete_filter_by_filternumber(object id, int filternumber)
{
  array(array) filters = get_filters(id);
  array(array) result = allocate(sizeof(filters));
  if(filternumber >= 0 && filternumber < sizeof(filters))
  {
    result = filters[..filternumber-1] + filters[filternumber+1..];
    set_filters(id, result);
  }
}

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
