/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * Copyright  1994-2001 Roxen Internet Software
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_language.pike,v 1.8.2.3.2.3 2001/09/24 11:10:57 kiwi Exp $
 */

#include <module.h>
inherit "module";

#include <camas/globals.h>

inherit "caudiumlib";

//
//! module: CAMAS: Language
//!  The Language Module for CAMAS.<br />
//!  This module handle all internationalization parts of CAMAS. It will
//!  translate internals messages into localized messages that CAMAS can handle.
//!  <br />This is the module where you specify the default language used in
//!  CAMAS.<br /><b>This module is automatically selected if you select "CAMAS:
//!  Main module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_language.pike,v 1.8.2.3.2.3 2001/09/24 11:10:57 kiwi Exp $
//

constant cvs_version = "$Id: camas_language.pike,v 1.8.2.3.2.3 2001/09/24 11:10:57 kiwi Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Language Module";
constant module_doc  = "The Language Module for CAMAS.<br />"
                       "This module handle all internationalization parts of "
                       "CAMAS. It will tranlater internals messages into "
                       "localized messages that CAMAS can handle.<br />"
                       "This is the module where you specify the default "
                       "language used in CAMAS.<br /><b>This module is "
                       "automatically selected if you select \"CAMAS: Main "
                       "module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;		// I think this module should be :)

#if constant(thread_create)
object global_lock = Thread.Mutex ();
#endif

mapping lang_progs = ([ "english" : 0 ]);
mapping lang_progs_short = ([ "en" : "english" ]);
mapping lang_charsets = ([ "english" : "iso-8859-1" ]);
mapping lang_replacement_strings = ([ "english" : "" ]);
int langs_loaded = 0;

void create () {
//  defvar ("debug", 0, "Debug", TYPE_FLAG,
//	  "When on, debug messages will be logged in Caudium's debug logfile. "
//	  "This information is very useful to the developers when fixing bugs.");
  
  defvar ("langfiledir", "modules/camas/languages/", "Language files directory", TYPE_STRING,
	  "The directory where you store the language files (camas_*.pike)."
	  "You can leave this empty if you're only going to use english.");

  defvar("defaultlang", "english", "Default user interface language",
         TYPE_STRING,
	 "The default language for users. The language at the login prompt."
         "This is the name of the language as it's presented at the user "
         "preferences screen.");

  defvar("browserlang", 1, "Default language from browser", TYPE_FLAG,
         "Decides wheter the default language should be selected from the browser's "
         "settings if possible. If all your users use the same langauge you may want "
         "to disable this, since many users don't bother changing their language "
         "preferences in their browsers.");
}

string check_variable (string var, mixed value) {
  switch (var) {
  case "langfiledir":
    if (sizeof (value)) {
      array stat = file_stat (value);
      if (!stat || stat[1] != -2)
	return "Must be a directory or empty string.";
    }
    langs_loaded = 0;
    break;
  }
}

void start () {
  if (!langs_loaded)
    load_langs ();
}

string status ()
{
  string out = "CAMAS Language Module.";
  out += "<br />Provides these languages :<ul>";
  foreach (sort (indices (lang_progs)), string lang)
    out += "<li>" + lang + "</li>";
  out += "</ul>";
  return out;
}

string query_provides ()
{
  return ("camas_lang");
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide v2, we
//!  *MUST* be backward compatible with v1.
//
int version ()
{
  return 1;
}

void load_langs () {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  if (!langs_loaded) {
    mixed err;
    lang_progs = ([ "english" : 0 ]);
    lang_progs_short = ([ "en" : "english" ]); 
    lang_charsets = ([ "english" : "iso-8859-1" ]);
    lang_replacement_strings = ([ "english": "" ]);

    if (sizeof (QUERY (langfiledir))) {
      array files = get_dir (QUERY (langfiledir));
//      if (QUERY (debug) && files)
        perror (sprintf ("\nCAMAS: Looking up languages files, there %s %d file%s...\n",
			 (sizeof (files) == 1) ? "is" : "are",
			 sizeof (files),
			 (sizeof (files) == 1) ? "" : "s"));
      if (files) {
	foreach (files, string file) {
	  if (file[0..5] == "camas_" && (lower_case (file[sizeof (file) - 5..]) == ".pike" ||
					 lower_case (file[sizeof (file) - 8..]) == ".pike.gz")) {
//	    if (QUERY (debug))
	      perror ("CAMAS: Considering lang file " + file + "\n");
	    object lang_prog;
	    string lang_file = combine_path (QUERY (langfiledir), file);
	    master()->set_inhibit_compile_errors ("");
	    err = catch (lang_prog = compile_string (Stdio.read_bytes (lang_file))());
	    // Does caudium work w.o. Gz?
	    // Yes, it certainly does. - david
#if constant(Gz.inflate)
	    if(err && (file[sizeof(file)-3..] == ".gz"))
	      err = catch (lang_prog = compile_string (Gz.inflate()->inflate (Stdio.read_bytes (lang_file)))());
#endif
	    master()->set_inhibit_compile_errors (0);
	    if (!err && lang_prog->imho_lang) {
//	      if (QUERY (debug))
		perror ("CAMAS: Loaded lang " + lang_prog->imho_lang() + " from file " + file + "\n");
	      lang_progs[lang_prog->imho_lang()] = lang_prog;
	      if (lang_prog->imho_lang_short)
		lang_progs_short[lang_prog->imho_lang_short()] = (lang_prog->imho_lang ());
	      if (lang_prog->imho_lang_charset)
		lang_charsets[lang_prog->imho_lang()] = (lang_prog->imho_lang_charset ());
	      if (lang_prog->imho_lang_replacement_string)
		lang_replacement_strings[lang_prog->imho_lang()] = (lang_prog->imho_lang_replacement_string ());

	      // FIXME: what is this?!
	      // Roxen 2.x dumps precompiled files as .o-files which pike
	      // loads a lot faster. Support for this in Caudium doesn't
	      // yet exist. -- david
#if 0
	      if (roxen->dump)
		roxen->dump (lang_file);
#endif
	    }
	    else
	      //if (err && QUERY (debug))
	      if (err)
		perror ("CAMAS: could not load language file: " + err[0] + "\n");
	  }
	}
      }
    }
  }
  langs_loaded = 1;
#if constant(thread_create)
  destruct(lock);
#endif
}

#include <camas/defaultlanguage.h>

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: langfiledir
//! The directory where you store the language files (camas_*.pike).You can leave this empty if you're only going to use english.
//!  type: TYPE_STRING
//!  name: Language files directory
//
//! defvar: defaultlang
//! The default language for users. The language at the login prompt.This is the name of the language as it's presented at the user preferences screen.
//!  type: TYPE_STRING
//!  name: Default user interface language
//
//! defvar: browserlang
//! Decides wheter the default language should be selected from the browser's settings if possible. If all your users use the same langauge you may want to disable this, since many users don't bother changing their language preferences in their browsers.
//!  type: TYPE_FLAG
//!  name: Default language from browser
//
