/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * Based on IMHO  Stefan Wallstrm and Bosse Lincoln.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: imapclient.pmod,v 1.36.4.8 2001/11/02 15:42:46 oliv3 Exp $
 */
/*
 * CAMAS IMAP client
 */
#include <camas/globals.h>
#include <camas/functions.h>
#include <camas/parser.h>


/*
  oliv3 FIXME: if !mboxencode -> save a filter fails (does not return
  back to the previous screen).

  oliv3 FIXME: if !mboxencode -> rename a folder fails
*/

#define MSG(m) sessobj->lang_module->msg (sessobj, m)
#define MSGA(m,a) sessobj->lang_module->msg (sessobj, m, a)

//
// IMAP-Client
//
#define MODE_LINE 0
#define MODE_PARSE 1
#define TIMEOUT1 60
//#define TIMEOUT1 10

#define IMAP_QSTRING(x) ("\""+replace(x,({"\"","\\"}),({"\\\"","\\\\"}) )+"\"")

class imapclient {
  // variables
  object imho;
  object parser;

  mapping mcache;

  int loggedin = 0;
  int abort=0;
  int result_sent=0;
  mapping sessobj;
  object prefsmailobj;
  object ebookmailobj;
  object id;
  string data="";
  string imapwritedata = "";
  string line=0;
  string tmp_line="";
  string dsn_comm="";
  int dsn=0;
  int parse_type;
  mixed parse_result=0;
  int tokens_to_get=0;
  array (mapping) commands;
  static mapping command=0;
  int command_result=0;
  int mode;
  mixed timeout_call_out=0;
  mixed imap_idle_call_out=0;
  object fd;
  int imap_seq=0;

  array folderstodelete=({ });
  int trycreate=0;
  int reconnects=0;
  array parserstate = 0;
  mixed cmd_result = 0;
  mixed cmd_tmp = 0;

  int command_seq=0;
  string foo,bar;
  array tmp_array;

  string selected_mailbox = 0;

  array (string) imaplinehist = ({ });

  string literal_string = 0;
  int literal_addcrlf = 0;
  int literal_continue = 0;

  // methods

  // Return data from an argument in a command structure
  // If arg is a string, it is retuned untouched
  // If arg is an array, the variable with name in arg[0] contains the 
  // data to be returned. If the name is "-", the data is taken 
  // from cmd_result instead (a kind of inter-command-pipe).
  inline mixed get_cmd_arg (mixed arg) {
    if (arg && sizeof (arg) > 0) {
      if (arrayp (arg) && (sizeof (arg) > 0) && intp (arg[0]) && (arg[0] == CMD_VAR_POINTER)) {
	string var = arg[1];
	if (var == "-")
	  return (cmd_result);
	else
	  return (sessobj[var]);
      }
      else
	return (arg);
    }
    return 0;
  }

  inline void set_cmd_arg (mixed arg, mixed data) {
    if (arg && sizeof (arg) > 0) {
      if (arrayp (arg) && (sizeof (arg) > 0) && intp (arg[0]) && (arg[0] == CMD_VAR_POINTER)) {
	string var = arg[1];
	if (var == "-")
	  cmd_result = data;
	else
	  sessobj[var] = data;
      }
      else
	perror ("set_cmd_arg: Wrong argument name!");
    }
  }
  
  string translate_frommbox (mapping sessobj, string mbox) {
    if (mbox == "INBOX")
      mbox = MSG(M_INBOX); 
    if (mbox == sessobj->trashfolder)
      mbox = MSG(M_BOXTRASH); 
    if (mbox == sessobj->sentfolder)
      mbox = MSG(M_BOXSENTMAIL); 
    if (mbox == sessobj->draftsfolder)
      mbox = MSG(M_BOXDRAFTS); 
    return (mbox);
  }
  
  string translate_tombox (mapping sessobj, string name) {
    if (name == MSG(M_INBOX))
      name = "INBOX";
    if (name == MSG(M_BOXTRASH))
      name = sessobj->trashfolder;
    if (name == MSG(M_BOXSENTMAIL))
      name = sessobj->sentfolder;
    if (name == MSG(M_BOXDRAFTS))
      name = sessobj->draftsfolder;
    return (name);
  }

  string fixmail (string mail) {
    // Fix mails which the MIME module cannot parse. This function is
    // called recursively for each mailpart

    int headend1 = search(mail, "\n\n");
    int headend2 = search(mail, "\r\n\r\n");
    int headend = (headend1<0? headend2 :
		   (headend2<0? headend1 :
		    (headend1<headend2? headend1 : headend2)));
    array (string) headers = (replace(mail[0..headend-1], ({"\r", "\t", "\n "}), ({ "", " ", " "})))/"\n";
    string newheaders = "", contenttype = "", boundary = "";

    foreach (headers, string h) {
      string key = "", data = "";
      if (sscanf (h, "%[!-9;-~]%*[ ]:%*[ ]%s", key, data) > 2) {
	if (lower_case (key) == "content-type") {
	  //write ("content-type: data= " + data + "\n");
	  string k1 = "", k2 = "", opts = "";
	  sscanf (data, "%s/%s;%s", contenttype, k2, opts);
	  foreach (opts/";", string opt) {
	    string l1 = "", l2 = "";
	    if (sscanf (opt, "%*[ ]%s=%s", l1, l2) > 1) { 
	      if (lower_case (l1) == "boundary") {
		boundary = l2;
		sscanf (boundary, "\"%s\"", boundary);
	      }
	    }
	  }
	}
      }
    }

    foreach (headers, string h) {
      string key = "", data = "";
      if (sscanf (h, "%[!-9;-~]%*[ ]:%*[ ]%s", key, data) > 2)
	{
	  string lc_key = lower_case (key);

	  if (lc_key == "content-transfer-encoding" && 
	      ((lower_case (contenttype) == "multipart") || (lower_case (data) == "8bits"))) {
	    data = "8bit";
	  }

	  if ((lc_key == "content-type" || 
	       lc_key == "content-description" ||
	       lc_key == "content-disposition") && sizeof (data) > 0)
	    while (data[-1] == ' ' || data[-1] == '\t' || data[-1] == ';')
	      data = data[0..sizeof(data)-2];
	  
	  if (lc_key == "content-disposition") {
	    array foo = data / " ";
	    for (int i = 1; i < sizeof (foo); i++)
	      if (!has_value (foo[i], "="))
		foo[i] += "=fixed";
	    data = foo * " ";
	  }

	  if (lc_key == "content-type") {
	    /*
	      yet another fix for those spam-bots:
	      "content-type: foo/bar, baz"   instead of
	      "content-type: foo/bar; baz"
	    */

	    /* RFC 2045, 5.1

	         content := "Content-Type" ":" type "/" subtype
		 *(";" parameter)
		 ; Matching of media type and subtype
		 ; is ALWAYS case-insensitive. */
	    string type, subtype, parameter;
	    if (sscanf (data, "%s/%s, %s", type, subtype, parameter) == 3) {
	      //write ("bad content-type: " + data + "...\n");
	      data = type + "/" + subtype + "; " + parameter;
	    }

	    /*
	      and here we go again...

	      MIME.Message failed: invalid parameter in Content-Type
	      content-type: data= text/plain;X-Mailer: cls_mail
	    */
	    string junk_header, junk_data;
	    if (sscanf (data, "%s/%s;%s:%s", type, subtype, junk_header, junk_data) == 4) {
	      //write ("Header: " + type + "/" + subtype + ";\n");
	      //write ("Junk: " + junk_header + ":" + junk_data + ";\n");
	      newheaders += junk_header + ":" + junk_data + ";\n";
	      data = type + "/" + subtype + ";";
	    }
	  }

	  newheaders += key + ": " + data + "\n";
	}
    }
    
    string newmail = "";
    if (sizeof(boundary) > 0) {
      array (string) mailparts = mail[headend+2..] / ("--"+boundary);
      newmail += newheaders+"\n"+mailparts[0];
      for (int i = 1; i < sizeof(mailparts)-1; i++)
	newmail += "--"+boundary+"\r\n"+fixmail(mailparts[i]);
      newmail += "--"+boundary+"--\r\n\r\n";
    }
    else
      newmail = newheaders + "\n" + mail[headend+2..];
    return(newmail);
  }

  int find_mailbox (mapping sessobj, string name)
  {
    int nr = Array.search_array(sessobj->mailboxes,
				lambda(array a,string mbox) 
				{
				  if(a[MB_DISPLAYNAME_IDX] == mbox)
				    return 1;
				  return 0;
				},
				name);
    return(nr);
  }
  
  void imapwrite (string data) { // Write data to a nonblocking socket
    if(imap_idle_call_out)
      remove_call_out(imap_idle_call_out);
    imap_idle_call_out=call_out(handle_imap_idle_timeout,sessobj->imapidlelogout);
    int res;
    if (sizeof(imapwritedata) > 0)
      imapwritedata += data;
    else {
      res = fd->write(data);
      if (res < sizeof(data))
	imapwritedata = data[res..];
    }
  }

  string encode_pref (string s) {
    if (!s)
      return "";
    return replace(s, ({ "<br", "\n", "\r" }), ({ "<br0", "<br>", "" }));
  }
  
  string decode_pref (string s) {
    if (!s)
      return "";
    return replace(s, ({ "<br0", "<br>", "\r" }), ({ "<br", "\n", "" }));
  }
  
  void init_parser (int pt) {
    mode = MODE_PARSE;
    parser->init ();
    line = 0;
    parse_type = pt;
  }
  
  string imap_tag () {
    return "camas" + sprintf ("%04d", imap_seq);
  }
  
  void imap_firsttag () {
    imap_seq = 0;
  }
  
  void imap_newtag () {
    imap_seq++;
  }

  void handle_error () {
    if(command && command->error && sizeof(command->error) > 0) {
      sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
      if (command && command->erroraction)
	sessobj->dialogactions = ({ command->erroraction });
      else
	sessobj->dialogactions = ({ "actionfolderlist" });
      sessobj->dialogtext = command->error;
      if (sessobj->displayimaperrors) 
	sessobj->dialogtext += "\n"+line;
      sessobj->status = DIALOGBOX;
      return;
    }

    if (command && (command->cmd == "low_login"))
      {
	// don't keep connection to imapd open
	// if login fails.
	catch 
	{
	  fd->close();
	};
	fd = 0;
	selected_mailbox = 0;
      }

    if (sessobj->status < 0)
      {
	if (command_result == -1)
	  {
	    imho->imho_log ("loginfailed", ([ "login": sessobj->address ]));
	    sessobj->status = LOGINFAILED;
	  }
	else
	  sessobj->status = LOGINFAILEDIMAP;
      }
    else
      {
	sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
	if (command && command->erroraction)
	  sessobj->dialogactions = ({ command->erroraction });
	else
	  sessobj->dialogactions = ({ "actionmailindex" });
	sessobj->dialogtext = MSG(M_IMAPERROR);
	imho->imho_log("imapfail", ([ "login" : sessobj->address, 
				      "command" : ((command) ? command->cmd : "unknown command"),
				      "line" : line,
				      "history" : imaplinehist * "\n" ]));
	if (sessobj->displayimaperrors) 
	  sessobj->dialogtext += "\n" + line;
	sessobj->status = DIALOGBOX;
      }
  }

  void handle_timeout () {
    handle_error ();
    command_result = -1;
    if (!command->stop)
      next_command ();
  }

  void timeout (int s) {
    if (timeout_call_out)
      remove_call_out (timeout_call_out);

    if (s)
      timeout_call_out = call_out (handle_timeout, s);
  }

  void handle_imap_idle_timeout () {
    catch 
    { 
      fd->close();  // Close connection to IMAP server
    };
    fd = 0;
    selected_mailbox = 0;
  }

  void send_result () {
    if (id && imho)
      id->send_result (imho->create_output (id));
  }

  inline void done () {
    timeout (0);
    if (!result_sent) {
      result_sent = 1;
      call_out (send_result, 0);
    }
  }

  void socket_close () {
    if (sessobj->debug)
      write ("\nLost mailbox lock on " + ctime (time ()));
    timeout (0);
    fd = 0;
    selected_mailbox = 0;
    if (!abort) {
      if (reconnects > 0) // Did we already try once?
	// other end closed unexpectedly
	handle_error ();
      else {
	if (command != 0) { 
	  reconnects++; // Try to connect again
	  commands = ({ command }) + commands;
	  next_command ();
	}
      }
    }
    else {
      if (!result_sent) {
	result_sent = 1;
	call_out (send_result, 0);
      }
    }
  }

  void imap_data_written () {
    int res;
    if (sizeof (imapwritedata) > 0)
      {
	res = fd->write (imapwritedata);
	if (res < sizeof (imapwritedata))
	  imapwritedata = imapwritedata[res..];
	else
	  imapwritedata = "";
      }
  }

  void got_data (mixed from, string input) {
    timeout (0);
    
    int going = 1; // true when rows are available
    data += input;
    
    while (going && !abort) {
      switch (mode) {
      case MODE_LINE:
	bar = "";
	if (sscanf (data, "%s%[\r\n]%s", foo, bar, data) && sizeof (bar)) {
	  line = tmp_line + foo;
	  tmp_line = "";
	}
	else {
	  tmp_line += foo;
	  line = 0;
	  going = 0;
	}
	break;
	
      case MODE_PARSE:
	parserstate = parser->parseimap (data, parse_type, tokens_to_get, parserstate);
	
	data = parserstate[2];
	if (parserstate[0]) { // Done!
	  parse_result = parserstate[1];
	  parserstate = 0;
	  mode = MODE_LINE;
	  handle_command ();
	}
	else
	  going = 0;
	break;
      }
      
      if (going && (line && sizeof (line)))
	handle_command ();
    }
  }
  
  string imapwrite_literal (string s, int addcrlf, int cont) {
    // addcrlf - if !0 : sends an additional \r\n after the string.
    //           usable when this string ends a command.
    // cont    - if !0 : calls handle_command after successfully
    //           sending the string
    //           NOTE! handle_command is always called if writing
    //           should fail.
    imapwrite ("{" + sizeof (s) + "}\r\n");
    literal_string = s;
    literal_addcrlf = addcrlf;
    literal_continue = cont;
  }


  //oliv3: start_ / handle_ functions
      
  // low_login
  void start_low_login () {
    loggedin = 0; // because we're NOT logged in :)
    timeout (TIMEOUT1);
  }
  
  void handle_low_login () {
    switch (command_seq) {
    case 0: // expect greeting
      if (sessobj->debug)
	perror ("imap server responding ?\n");
      if (line && sscanf (line, "* OK%*s")) {
	//oliv3: keep this ?
	//sscanf (line, "* OK %s %*s", sessobj->realimapserver);
	//write ("IMAP server: " + sessobj->realimapserver + "\n");
	command_seq = 1;
	imap_newtag ();
	imapwrite (imap_tag () + " LOGIN " + IMAP_QSTRING (sessobj->login) + " " 
		   + IMAP_QSTRING (sessobj->passwd) + "\r\n");
	if (sessobj->debug)
	  perror ("imap login: auth token sent.\n");
      }
      timeout (TIMEOUT1);
      break; // 0
      
    case 1: // wait for login result
      if (sscanf (line, imap_tag () + " %s", foo))
	{
	  if (foo[0..1] == "OK")
	    {
	      command_result = 0;
	      loggedin = 1; // now we're logged in :)
	    }
	  else
	    command_result = -1;
	  next_command ();
	}
      else
	timeout (TIMEOUT1);
      break; // 1
    }
  }
  
  // low_select
  void start_low_select () {
    if (command->mailbox == 0)
      command->mailbox = sessobj->mailbox[MB_FOLDERNAME_IDX];
    if (sessobj->debug)
      perror ("Select: " + command->mailbox + "\n");
    if (command->mailbox == selected_mailbox)
      {
	command_result = 0;
	next_command ();
      }
    else
      {
	if (sessobj->mboxencode)
	  imapwrite (imap_tag () + " SELECT " + IMAP_QSTRING (command->mailbox) + "\r\n");
	else
	  {
	    imapwrite (imap_tag () + " SELECT ");
	    imapwrite_literal (command->mailbox, 1, 0);
	  }
	timeout (TIMEOUT1);
      }
  }
  
  void handle_low_select () {
    if (sscanf (line, imap_tag () + " %s", foo))
      {
	if (foo[0..1] == "OK") {
	  command_result = 0;
	  selected_mailbox = command->mailbox;
	}
	else
	  {
	    command_result = -1;
	    selected_mailbox = 0;
	  }
	next_command ();
      }
    else
      timeout (TIMEOUT1);
  }
  
  // status
  void start_status () {
    if (sessobj->debug)
      perror ("status " + command->mailbox + ": ");
    mcache = sessobj->cache[command->mailbox];
    
    if (!mcache)
      {
	mcache = ([ ]);
	mcache->mails = ({ });
	mcache->new_mails = ({ });
	mcache->uid_validity = 0;
	mcache->messages = mcache->recent = mcache->unseen = 0;
	mcache->size = 0;
	mcache->uidnext = 0;
	mcache->changed = 1;
      }
    
    if (sessobj->mboxencode)
      imapwrite (imap_tag () + " STATUS " + IMAP_QSTRING (command->mailbox)
		 +" (MESSAGES RECENT UNSEEN UIDNEXT UIDVALIDITY)\r\n");
    else
      {
	imapwrite (imap_tag () + " STATUS ");
	imapwrite_literal (command->mailbox, 0, 1);
	command_seq = 2;
      }
    timeout (TIMEOUT1);
  }
  
  void handle_status () {
#define CACHE(reason) if (sessobj->debug) write ("Cache changed: " + reason + "\n")
    if (!sessobj->cache[command->mailbox])
      sessobj->cache[command->mailbox] = ([ ]);

    mcache = sessobj->cache[command->mailbox];
    
    switch (command_seq)
      {
      case 2: {
	// Continue STATUS command after literal
	imapwrite (" (MESSAGES RECENT UNSEEN UIDNEXT UIDVALIDITY)\r\n");
	command_seq = 0;
	timeout (TIMEOUT1);
      };
	break; // 2
	
      case 0:
	if (sscanf (line, "* STATUS {%s", foo) == 1)
	  command_seq = 5;
	else
	  if ((sscanf (line, "* STATUS %*s (%s", foo) == 2) || (sscanf (line, "* STATUS \"%*s\" (%s", foo) == 2)) {
	    data = "(" + foo + "\n" + data;
	    init_parser (PT_MAPPING);
	    command_seq = 1;
	  }
	timeout (TIMEOUT1);
	break; // 0

      case 5:
	if ((sscanf (line, "%*s (%s", foo) == 2) || (sscanf (line, "\"%*s\" (%s", foo) == 2)) {
	  data = "(" + foo + "\n" + data;
	  init_parser (PT_MAPPING);
	  command_seq = 1;
	}
	timeout (TIMEOUT1);
	break; // 5

      case 1:
	if (!parse_result) {
	  if (sessobj->debug)
	    perror ("CAMAS : IMAP parse error!\n");
	  command_result = -1;
	  next_command ();
	}
	else
	  {
	    foreach (indices (parse_result), string i)
	      parse_result[i] = (int)parse_result[i];
	    
	    if (parse_result->RECENT) {
	      CACHE("recent messages: " + parse_result->RECENT);
	      mcache->changed = 1;
	    }
	    
	    if (mcache->uidnext != parse_result->UIDNEXT) {
	      CACHE("next uid: " + parse_result->UIDNEXT);
	      mcache->changed = 1;
	      mcache->uidnext = parse_result->UIDNEXT;
	    }
	    
	    if (mcache->uid_validity != parse_result->UIDVALIDITY) {
	      CACHE ("uid validity: " + parse_result->UIDVALIDITY);
	      mcache->uid_validity = parse_result->UIDVALIDITY;
	      mcache->mails = ({ });
	      mcache->new_mails = ({ });
	      mcache->messages = mcache->recent = mcache->unseen = 0;
	      mcache->size = 0;
	      mcache->uidnext = 0;
	      mcache->changed = 1;
	    }
	    
	    if (parse_result->MESSAGES) {
	      if (mcache->messages != parse_result->MESSAGES) {
		CACHE("messages: " + parse_result->MESSAGES);
		mcache->changed = 1;
		mcache->messages = parse_result->MESSAGES;
	      }
	    }
	    else { // empty mailbox
	      mcache->mails = ({ });
	      mcache->new_mails = ({ });
	      mcache->messages = mcache->recent = mcache->unseen = 0;
	      mcache->size = 0;
	      mcache->changed = 0;
	    }
	    
	    mcache->unseen = parse_result->UNSEEN;

	    if ((command->mailbox == "INBOX") && mcache->changed && !sessobj->filters_have_changed) {
	      if (imho->query ("debug"))
		write ("new mail has arrived. 'apply_mail_filters' scheduled.\n");
	      sessobj->filters_have_changed = 1;
	    }
	    
	    data = line + "\n" + data;
	    parse_result = 0;
	    command_seq = 3;
	    timeout (TIMEOUT1);
	  }
	break; // 1
	
      case 3:
	if (sscanf (line, imap_tag () + " %s", foo)) {
	  if (foo[0..1] == "OK") {
	    if (mcache->changed) {
	      command_seq = 4;
	      imap_newtag ();
	      imapwrite (imap_tag () + " UID SEARCH ALL\r\n");
	      timeout (TIMEOUT1);
	    }
	    else {
	      if (sessobj->debug)
		write ("mailbox has not changed.\n");
	      //set_cmd_arg (command->output, mcache->mails);
	      command_result = 0;
	      next_command ();
	    }
	  }
	  else {
	    command_result = -1;
	    selected_mailbox = 0;
	    next_command ();
	  }
	}
	else
	  timeout (TIMEOUT1);
	break; // 3
	
      case 4: { // SEARCH result
	if (sscanf (line, "* SEARCH %s", foo) == 1) {
	  array new_uids = Array.map (foo / " ", lambda (string s) { return (int)s; });
	  array old_uids = Array.map (mcache->mails, lambda (mapping m) { return m->imap->UID; });
	  array new_mails = new_uids - old_uids;
	  
	  if (sizeof (new_mails)) {
	    mcache->new_mails |= new_mails;
	    if (sessobj->debug)
	      write (sprintf ("new mails: %d/%d\n", sizeof (new_mails), sizeof (mcache->new_mails)));
	  } 
	  else {
	    if (sessobj->debug)
	      write ("no new mails.\n");
	  }
	  new_mails = 0;
	  
	  array del_mails = old_uids - new_uids;
	  if (sizeof (del_mails)) {
	    if (sessobj->debug)
	      write (sprintf ("deleted mails: %d\n", sizeof (del_mails)));
	    array (mapping) deleted_mails = ({ });
	    foreach (mcache->mails, mapping m)
	      if (search (del_mails, m->imap->UID) != -1) {
		// write ("deleted UID: " + m->imap->UID + "\n");
		deleted_mails += ({ m });
		del_mails -= ({ m->imap->UID });
		if (!sizeof (del_mails))
		  break;
	      }
	    mcache->mails -= deleted_mails;
	  }
	  else {
	    if (sessobj->debug)
	      write ("no deleted mails.\n");
	  }
	  new_uids = old_uids = 0;
	  mcache->changed = 0;
	}
	else
	  if (sscanf (line, imap_tag () + " %s", foo)) {
	    command_result = (foo[0..1] == "OK") ? 0 : -1;
	    next_command ();
	  }
      }; break; // 4
      }
  }
  
  // low_list
  void start_low_list () {
    sessobj->mailboxes = ({ });
    imapwrite (imap_tag () + " LIST \"" + command->path + "\" *\r\n");
    timeout (TIMEOUT1);
  }
  
  void handle_low_list () {
    switch (command_seq) {
    case 0: // expect "* LIST" or command result
      if (sscanf (line, "* LIST %s", foo))
	{
	  data = foo + "\n" + data;
	  init_parser (PT_LIST);
	  command_seq = 1;
	}
      else
	if (sscanf (line, imap_tag () + " %s", foo))
	  {
	    if (foo[0..1] == "OK")
	      {
		command_result = 0;
		// there should always be an INBOX...
		if(Array.search_array(sessobj->mailboxes,
				      lambda(array a) 
				      {
					return (a[MB_FOLDERNAME_IDX] == "INBOX");
				      }) == -1) 
		  sessobj->mailboxes += ({ ({ "INBOX","INBOX",
					      MB_NOINFERIORS,0,
					      ({"INBOX"})  }) });
		// add folders that are missing in the hierarchy
		for (int i; i < sizeof (sessobj->mailboxes); i++)
		  {
		    if ((Array.search_array (sessobj->mailboxes,
					     lambda (array mbox_a, array hierarchy) 
					     {
					       if (sizeof (hierarchy) < 2)
						 return 1;
					       if (sizeof (hierarchy) - 1 != sizeof (mbox_a[MB_HIERARCHY_IDX]))
						 return 0;
					       int i;
					       for (i = 0; i < sizeof (hierarchy) - 1; i++)
						 if (hierarchy[i] != mbox_a[MB_HIERARCHY_IDX][i])
						   return 0;
					       return 1;
					     }, sessobj->mailboxes[i][MB_HIERARCHY_IDX])) == -1)
		      {
			// add implicit folder
			int foo = sizeof (sessobj->mailboxes[i][MB_SEPARATOR_IDX])+
			  sizeof (sessobj->mailboxes[i][MB_HIERARCHY_IDX][-1]);
			string fname = sessobj->mailboxes[i][MB_FOLDERNAME_IDX][0..sizeof(sessobj->mailboxes[i][MB_FOLDERNAME_IDX])-1-foo];
			string dname = sessobj->mailboxes[i][MB_DISPLAYNAME_IDX][0..sizeof(sessobj->mailboxes[i][MB_DISPLAYNAME_IDX])-1-foo];
			sessobj->mailboxes = sessobj->mailboxes + ({ ({ fname, dname, MB_NOSELECT | MB_IMPLICIT_FOLDER,sessobj->mailboxes[i][MB_SEPARATOR_IDX],sessobj->mailboxes[i][MB_HIERARCHY_IDX][0..sizeof(sessobj->mailboxes[i][MB_HIERARCHY_IDX])-2] }) });
		      }
		  }
		// sort mailboxes. INBOX should always be first.
		sessobj->mailboxes=Array.sort_array(sessobj->mailboxes,lambda(array a1,array a2) 
								       {
									 if(a1[MB_FOLDERNAME_IDX]=="INBOX")
									   return 0;
									 if(a2[MB_FOLDERNAME_IDX]=="INBOX")
									   return 1;
									 int s1=sizeof(a1[MB_HIERARCHY_IDX]);
									 int s2=sizeof(a2[MB_HIERARCHY_IDX]);
									 int i;
									 for(i=0;i<(s1<s2?s1:s2);i++)
									   if(a1[MB_HIERARCHY_IDX][i]>a2[MB_HIERARCHY_IDX][i])
									     return 1;
									 return(s1<s2);
								       });
	      }
	    else
	      command_result = -1;
	    next_command ();
	  }
	else
	  timeout(TIMEOUT1);
      break;
      
    case 1: // we should have a list of flags now
      if(!parse_result) {
	command_result=-1;
	next_command();
      }
      else {
	tmp_array=parse_result;
	command_seq=2;
	tokens_to_get=2;
	init_parser(PT_TOKENS);
	timeout(TIMEOUT1);
      }
      break; // 1
      
    case 2:
      if(!parse_result) {
	command_result=-1;
	next_command();
      }
      else {
	// ({ foldername, displayname, flags, separator, displayname/separator })
	string foldername = parse_result[1];
	if (upper_case (foldername) == "INBOX")
	  foldername = "INBOX";
	string separator = parse_result[0];
	int flags = 0;
	string displayname = foldername;
	sscanf (foldername, command->path + "%s", displayname);
	//oliv3 RFCCHECK sscanf (displayname, separator + "%s", displayname);
	// if the mailbox name was given as a quoted string, we may want to decode it
	if (sessobj->mboxencode)
	  displayname = imho->decode_mboxname (displayname);
	//write ("low_list: displayname= " + displayname + "\n");
	if ((displayname != "") && (sessobj->showprefsbox || (foldername != sessobj->prefsbox))
	    && (!sessobj->hidemboxes || ((separator ? displayname / separator : ({ displayname }))[-1][0..sizeof (sessobj->hidemboxes) - 1] != sessobj->hidemboxes))) {
	    array flag_a = Array.map (tmp_array, lambda (string s) { return lower_case (s); });
	    if (has_value (flag_a, "\\noselect"))
	      flags |= MB_NOSELECT;
	    if (has_value (flag_a, "\\noinferiors") || has_value (flag_a, "\\hasnochildren"))
	      flags |= MB_NOINFERIORS;
	    sessobj->mailboxes += ({ ({ foldername,
					displayname,
					flags,
					separator,
					separator ? ((displayname / separator) - ({ "" })) : ({ displayname }) }) });
	    //separator ? (displayname / separator) : ({ displayname }) }) });
	  }
	  timeout(TIMEOUT1);
	  command_seq=0;
      }
      break; // 2
    }
  }
  
  // low_logout
  void start_low_logout () {
    fd->set_close_callback (0);
    imapwrite (imap_tag () + " LOGOUT\r\n");
    catch { fd->close (); };
    fd = 0;
    selected_mailbox = 0;
    command_result = 0;
    abort = 1;
    sessobj->connected = 0;
    next_command ();
  }
  
  void handle_low_logout () {

  }
  
  // low_close
  void start_low_close () {
    imapwrite (imap_tag () + " CLOSE\r\n");
    timeout (TIMEOUT1);
  }
  
  void handle_low_close () {
    switch (command_seq) {
    case 0:
      if (sscanf (line, imap_tag () + " %s", foo)) {
	// whatever
	command_result = 0;
	selected_mailbox = 0;
	next_command ();
      }
      else
	timeout (TIMEOUT1);
      break; //0 
    }
  }
  
  // get_headers
  void start_get_headers () {
    set_cmd_arg (command->output, mcache->mails);
    
    if (sizeof (mcache->new_mails)) {
      if (!sizeof (mcache->mails)) {
	imap_newtag ();
	if (sessobj->debug)
	  write ("getting all headers (" + selected_mailbox + ")...\n");
	set_cmd_arg (command->output, ({  }));
	imapwrite (imap_tag () + " FETCH 1:* (INTERNALDATE ENVELOPE UID FLAGS RFC822.SIZE)\r\n");
	mcache->new_mails = ({ });
	timeout (TIMEOUT1);
      }
      else 
	if (sizeof (mcache->new_mails)) {
	  imap_newtag ();
	  if (sessobj->debug)
	    write ("getting some headers (" + selected_mailbox + ": " + sizeof (mcache->new_mails) + ")...\n");
	  string fuids = "";
	  int nuids = sizeof (mcache->new_mails);
	  foreach (mcache->new_mails, int uid) {
	    fuids += (string)uid;
	    if (--nuids)
	      fuids += ",";
	  }
	  imapwrite (imap_tag () + " UID FETCH " + fuids + " (INTERNALDATE ENVELOPE UID FLAGS RFC822.SIZE)\r\n");
	  mcache->new_mails = ({ });
	  timeout (TIMEOUT1);
	}
    }
    else {
      if (sessobj->debug)
	write ("unchanged mailbox\n");
      set_cmd_arg (command->output, mcache->mails);
      command_result=0;
      next_command();
    }
  }
  
  void handle_get_headers () {
    mcache = sessobj->cache[selected_mailbox];
    switch (command_seq) {
    case 0: // expect mailheaders or command result
      if (sscanf (line, "* %*d FETCH%s", foo) == 2) {
	data = foo + "\n" + data;
	init_parser (PT_MAPPING);
	command_seq = 1;
	timeout (TIMEOUT1);
      }
      else
	if (sscanf (line, imap_tag () + " %s", foo)) {
	  if(foo[0..1] == "OK") { // done
	    mcache->mails = get_cmd_arg (command->output);
	    mcache->new_mails = ({ });
	    mcache->changed = 0;
	    if (sessobj->debug)
	      write ("setting mail numbers ...\n");
	    int number = 0;
	    foreach (mcache->mails, mapping m)
	      m->number = number++;
	    
	    //oliv3: in camas.pike, always used
	    if (command->updatelistpos) { // change the maillist position
	      sscanf(sessobj->visiblemail, "%d", sessobj->visible);
	      int size = sizeof (mcache->mails);
	      if (sessobj->visible < size)
		sessobj->firstvisiblemail = size - sessobj->visible;
	      else
		sessobj->firstvisiblemail = 0;
	    }
 
	    command_result = 0;
	    next_command ();
	  }
	  else {
	    command_result = -1;
	    next_command ();
	  }
	}
	else
	  timeout (TIMEOUT1);
      break; //0
      
    case 1: // expect parsed headers
      if (!parse_result) {
	if (sessobj->debug)
	  perror ("CAMAS : IMAP parse error!\n");
	command_result = -1;
	next_command ();
      }
      else {
	data = line + "\n" + data;
	parse_result->UID = (int)parse_result->UID;
	parse_result["RFC822.SIZE"] = (int)parse_result["RFC822.SIZE"];
	set_cmd_arg (command->output, 
		     get_cmd_arg (command->output) + ({ ([ "imap": parse_result ]) }));
	set_cmd_arg (command->setsortcolumn, "num");
	command_seq = 0;
	timeout (TIMEOUT1);
      }
      break; // 1
    }
  }
  
  // apply_mail_filters
  void start_apply_mail_filters () {
    if (sessobj->mailbox[MB_FOLDERNAME_IDX] != "INBOX" || !sessobj->mails 
	|| !sizeof (sessobj->mails) || !strlen (sessobj->filterbook) || !sessobj->filters_have_changed) {
      if (!sessobj->filters_have_changed && imho->query ("debug"))
	write ("Filters have not changed or no new mail => nothing to do.\n");
      sessobj->filters_have_changed = 0;
      command_result = 0;
      next_command ();
    }
    else {
      if (sessobj->filters_have_changed && imho->query ("debug"))
	write ("Filters have changed or new mail has arrived => applying filters.\n");
      sessobj->filters_have_changed = 0;
      int mails = sizeof (sessobj->mails);
      array (mapping) deleted_mails = ({ });

      string namefield = "", filterexpression = "", filterfolder = "";
      array (array (string)) filters = ({ });
      foreach (((sessobj->filterbook / "\n") - ({ "" })), string line)
	if (sscanf (line, "%s:%s:%s", namefield, filterexpression, filterfolder) == 3
	    && (filterfolder != sessobj->mailbox[MB_FOLDERNAME_IDX]))
	  filters += ({ ({ namefield, filterexpression, filterfolder }) });

      if (sizeof (filters))
      for (int i = 0; i < mails; i++) {
	int match = 0;
	//write (sprintf ("mail #%d, date= %s\n", i, (string)sessobj["mails"][i]->imap->ENVELOPE[DATE_IDX]));
	foreach (filters, array filter) {
	  switch (filter[0]) {
	  case "From" :
	    if (sessobj["mails"][i]->imap->ENVELOPE[FROM_IDX])
	      if (has_value (fix_header (sessobj["mails"][i]->imap->ENVELOPE[FROM_IDX][0][0])+" "+fix_header (sessobj["mails"][i]->imap->ENVELOPE[FROM_IDX][0][2])+"@"+fix_header (sessobj["mails"][i]->imap->ENVELOPE[FROM_IDX][0][3]), filter[1])) { 
		match = 1; 
		//write ("matched on From\n"); 
	      }
	    break;

	  case "Subject" :
	    if (has_value (fix_header (sessobj["mails"][i]->imap->ENVELOPE[SUBJECT_IDX]), filter[1])) { 
	      match= 1; 
	      //write ("matched on Subject\n"); 
	    }
	    break;

	  case "Date" :
	    if (has_value (fix_header (sessobj["mails"][i]->imap->ENVELOPE[DATE_IDX]), filter[1])) { 
	      match= 1; 
	      //write ("matched on Date\n"); 
	    }
	    break;

	  case "To" :
	    if (sessobj["mails"][i]->imap->ENVELOPE[TO_IDX]) {
	      //write (sprintf ("size= %d\n", sizeof (sessobj["mails"][i]->imap->ENVELOPE[TO_IDX])));
	      int nb_to = sizeof (sessobj["mails"][i]->imap->ENVELOPE[TO_IDX]);
	      for (int to = 0; to < nb_to; to++) {
		string to_dest = fix_header (sessobj["mails"][i]->imap->ENVELOPE[TO_IDX][to][0])+" "+fix_header(sessobj["mails"][i]->imap->ENVELOPE[TO_IDX][to][2])+"@"+fix_header(sessobj["mails"][i]->imap->ENVELOPE[TO_IDX][to][3]);
		//write (sprintf ("checkTO: to= %d, dest= %s\n", to, to_dest));
		if (has_value (to_dest, filter[1])) {
		  match = 1;
		  //write ("matched on To\n"); 
		  break;
		}
	      }
	    }
	    break;

	  case "Cc" :
	    if (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX]) {
	      //write (sprintf ("size= %d\n", sizeof (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX])));
	      int nb_cc = sizeof (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX]);
	      for (int cc = 0; cc < nb_cc; cc++) {
		string cc_dest = fix_header (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX][cc][0])+" "+fix_header (sessobj["mails"][i]->imap->ENVELOPE[CC_IDX][cc][2])+"@"+fix_header(sessobj["mails"][i]->imap->ENVELOPE[CC_IDX][cc][3]);
		//write (sprintf ("checkCC: cc= %d, dest= %s\n", cc, cc_dest));
		if (has_value (cc_dest, filter[1])) {
		  match = 1;
		  //write ("matched on Cc\n"); 
		  break;
		}
	      }
	    }
	    break;
	  default: break;
	  }

	  if (match) {
	    // write ("match on #" + i + " => " + filter[2] + "\n");
	    commands += ({ imap_cmd ("copy",
				     "uids", ({ sessobj->mails[i]->imap->UID }),
				     "copytobox", filter[2],
				     "updatembox", command->updatembox),
			   imap_cmd ("delete",
				     "uids", ({ sessobj->mails[i]->imap->UID }),
				     "updatembox", command->updatembox,
				     "noexpunge", 1) });
	    deleted_mails += ({ sessobj->mails[i] });
	    break;
	  }
	}
      }

      if (sizeof (deleted_mails)) {
	commands += ({ imap_cmd ("expunge") });
	sessobj->mails -= deleted_mails;
	deleted_mails = ({ });
	sessobj->cache[selected_mailbox]->mails = sessobj->mails;
	set_cmd_arg (command->updatembox, sessobj->mails);
      }
      
      command_result = 0;
      next_command ();
    }
  }
    
  void handle_apply_mail_filters () {
    command_result = 0;
    next_command ();
  }

  // get_prefs_mail
  void start_get_prefs_mail () {
    command->uid = -1;
    foreach (indices (sessobj->prefsmails), int i)
      {
	object mail = sessobj->prefsmails[i];
	if (mail->imap->ENVELOPE[SUBJECT_IDX] == "CAMAS Preferences")
	  command->uid = sessobj->prefsmails[i]->imap->UID;
      }
    if ((command->uid == -1) && imho->query ("upgrading")) {
      foreach (indices (sessobj->prefsmails), int i)
	{
	  object mail = sessobj->prefsmails[i];
	  if (mail->imap->ENVELOPE[SUBJECT_IDX] == "IMHO Preferences")
	    command->uid = sessobj->prefsmails[i]->imap->UID;
	}
    }
    if (command->uid >= 0)
      {
	command->abortfail = 0;
	command->subcommands = ({ 
	  imap_cmd ("get_mail", "uid", command->uid,
		    "mailbox",sessobj->prefsbox,
		    "setprefsloaded", 1, "abortfail", 1, "output", imap_cmd_var ("prefsmailobj")) 
	});
	commands = ({ command }) + commands;
      }
    command_result = 0;
    next_command ();
  }
  
  // get_ebook_mail
  void start_get_ebook_mail () {
    command->uid = -1;
    foreach (indices (sessobj->prefsmails), int i)
      {
	object mail = sessobj->prefsmails[i];
	if (mail->imap->ENVELOPE[SUBJECT_IDX] == HEADER_EBOOK_SUBJECT_VALUE)
	  command->uid = sessobj->prefsmails[i]->imap->UID;
      }
    if (command->uid >= 0)
      {
	command->abortfail = 0;
	command->subcommands = ({ 
	imap_cmd ("get_mail", "uid", command->uid,
		  "mailbox", sessobj->prefsbox,
		  "setprefsloaded", 1, "abortfail", 1, "output", imap_cmd_var ("ebookmailobj")) 
	});
	commands = ({ command }) + commands;
    }
    command_result = 0;
    next_command ();
  }
  
  // get_mail
  void start_get_mail () {
    imapwrite (imap_tag () + " UID FETCH " + command->uid + " RFC822\r\n");
    timeout (TIMEOUT1);
  }
  
  void handle_get_mail () {
    switch (command_seq) {
    case 0: // expect data or command result
      if (sscanf (line, "* %*d FETCH%s", foo) == 2) {
	data = foo + "\n" + data;
	init_parser (PT_MAPPING);
	command_seq = 1;
	timeout (TIMEOUT1);
      }
      else
	if (sscanf (line, imap_tag () + " %s", foo)) {
	  if (foo[0..1] == "OK") {
            commands += ({ imap_cmd ("add_flag", "uid", command->uid, "flag", "\\Seen") });
            command_result = 0;
	    next_command ();
	  }
	  else {
	    command_result = -1;
	    next_command ();
	  }
	}
	else
	  timeout (TIMEOUT1);
      break; //0
      
    case 1: // expect parsed mapping
      if (!parse_result) {
	command_result = -1;
	next_command ();
      }
      else {
	data = line + "\n" + data;
	if (command->setprefsloaded)
	  sessobj->prefsloaded = 1;

	if (!zero_type (parse_result->RFC822)) {
	  //write ("got RFC822\n");
	  object mailobj = 0;
	  mixed mime_error = catch { 
	    mailobj = MIME.Message (parse_result->RFC822);
	    if (mailobj)
	      string test = mailobj->getdata ();
	  };

	  string mail = parse_result->RFC822;

	  if (mime_error || !mailobj) {
	    // Fix a number of problems that makes the MIME module fail.
	    write ("MIME.Message failed: " + mime_error[0]);
	    mail = fixmail (mail);
	    mime_error = catch { mailobj = MIME.Message (mail); };
	    if (mime_error)
	      write ("MIME.Message failed: " + mime_error[0]);
	  }

	  set_cmd_arg (command->output, mailobj);
	  parse_result = 0;
	}
	else
	  if (!zero_type (parse_result->FLAGS)) {
	    //oliv3 FIXME: update the flags
	    //write ("got FLAGS\n");
	    /*
	    write ("uid= " + command->uid + " ( " + typeof (command->uid) + " )\n");
	    write ("sizeof (sessobj->mails)= " + sizeof (sessobj->mails) + "\n");
	    write (sprintf ("FLAGS= %O\n", parse_result->FLAGS));
	    */
	    for (int i = 0; i < sizeof (sessobj->mails); i++) {
	      //write ("UID= " + sessobj->mails[i]->imap->UID + " ( " + typeof (sessobj->mails[i]->imap->UID) + " )\n");
	      if ((int)sessobj->mails[i]->imap->UID == (int)command->uid) {
		//write (sprintf ("old FLAGS= %O\n" + sessobj->mails[i]->imap->FLAGS));
		sessobj->mails[i]->imap->FLAGS |= parse_result->FLAGS;
		//write (sprintf ("new FLAGS= %O\n" + sessobj->mails[i]->imap->FLAGS));
		break;
	      }
	    }
	  }
	command_seq = 0;
	timeout (TIMEOUT1);
	}
      
      break; // 1
    }
  }

  // delete
  void start_delete () {
    array mails_to_delete = get_cmd_arg (command->uids);
    if (mails_to_delete)
      {
	int nuids = sizeof (mails_to_delete);
	if (!nuids)
	  {
	    command_result = 0;
	    next_command ();
	  }
	else
	  {
	    string duids = "";
	    foreach (mails_to_delete, mixed uid) {
	      duids += (string)uid;
	      if (--nuids)
		duids += ",";
	    }
	    //write ("DELETING: " + duids + "\n");
	    imapwrite (imap_tag () + " UID STORE " + duids + " +FLAGS.SILENT (\\Deleted)\r\n");
	    timeout (TIMEOUT1);
	  }
      }
    else
      {
	command_result = 0;
	next_command ();
      }
  }
  
  void handle_delete () {
    // expect store command result
    if (sscanf (line, imap_tag () + " %s", foo))
      if (foo[0..1] == "OK") {
	if (!command->noexpunge) {
	  // all mails flagged. now expunge
	  imap_newtag ();
	  imapwrite (imap_tag () + " EXPUNGE\r\n");
	  
	  array mails = get_cmd_arg (command->updatembox);
	  if (mails) {
	    array delmails = ({ });
	    array deletedmail = get_cmd_arg (command->uids);
	    
	    foreach (deletedmail, mixed uid)
	      if (intp (uid)) // deleting a single mail
		delmails += Array.filter (mails,
					  lambda (mapping m) {
					    return (m->imap->UID == uid);
					  });
	      else { // deleting several mails
		int start, end;
		sscanf (uid, "%d:%d", start, end);
		delmails += Array.filter (mails,
					  lambda (mapping m) {
					    return ((m->imap->UID >= start) && (m->imap->UID <= end));
					  });
	      }
	    
	    mails -= delmails;
	    sessobj->cache[selected_mailbox]->mails = mails;
	    set_cmd_arg (command->updatembox, mails);
	  }
	}

	command_result = 0;
	next_command ();
      }
      else {
	command_result = -1;
	next_command ();
      }
  }
  
  // move
  void start_move () {
    // a move is a copy + a delete
    command->subcommands = (command->copytobox) ?
      ({ imap_cmd ("copy",
		   "uids", command->uids,
		   "copytobox", command->copytobox,
		   "updatembox", command->updatembox) })
      : ({ imap_cmd ("copy",
		     "uids", command->uids,
		     "updatembox", command->updatembox) });
    command->subcommands += 
      ({ imap_cmd ("delete",
		   "uids", command->uids,
		   "updatembox", command->updatembox) });
    commands = ({ command }) + commands;
    command_result = 0;
    next_command ();
  }
  
  // copy
  void start_copy () {
    array mails_to_copy = get_cmd_arg (command->uids);
    if (command->copytobox)
      sessobj->copytobox = get_cmd_arg (command->copytobox);
    int nuids = sizeof (mails_to_copy);
    if (!nuids) {
      command_result = 0;
      next_command ();
    }
    else {
      string cuids = "";
      foreach (mails_to_copy, int uid) {
	cuids += (string)uid;
	if (--nuids)
	  cuids += ",";
      }
      //write ("COPYING: " + cuids + "\n");
      if (sessobj->mboxencode)
	imapwrite (imap_tag () + " UID COPY " + cuids + " " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
      else {
	imapwrite (imap_tag () + " UID COPY " + cuids + " ");
	imapwrite_literal (sessobj->copytobox, 1, 0);
      }
      trycreate = 0;
      timeout (TIMEOUT1);
    }
  }

  void handle_copy () {
    switch (command_seq) {
    case 0: // check COPY command result
      if (sscanf (line, imap_tag () + " %s", foo)) {
	if (foo[0..1] == "OK") {
	  command_result = 0;
	  next_command ();
	}
	else {
	  if (sscanf (foo, "NO %s", foo) && (trycreate || has_value (foo, "[TRYCREATE]"))) {
	    imap_newtag ();
	    if (sessobj->mboxencode)
	      imapwrite (imap_tag () + " CREATE " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
	    else {
	      imapwrite (imap_tag () + " CREATE ");
	      imapwrite_literal (sessobj->copytobox, 1, 0);
	    }
	    command_seq = 2;
	    timeout (TIMEOUT1);
	  }
	  else {
	    command_result = -1;
	    next_command ();
	  }
	}
      }
      else
	timeout (TIMEOUT1);
      break; //0
      
    case 1: // expect COPY command result
      if (sscanf (line, imap_tag () + " %s", foo)) {
	if (foo[0..1] == "OK") {
	  command_result = 0;
	  next_command ();
	}
	else {
	  command_result = -1;
	  next_command ();
	}
      }
      else
	timeout (TIMEOUT1);
      break; //1
      
    case 2: // Retry copying after CREATE
      if (sscanf (line, imap_tag () + " %s", foo)) {
	if (foo[0..1] == "OK") {
	  imap_newtag ();

	  array mails_to_copy = get_cmd_arg (command->uids);
	  int nuids = sizeof (mails_to_copy);

	  string cuids = "";
	  foreach (mails_to_copy, int uid) {
	    cuids += (string)uid;
	    if (--nuids)
	      cuids += ",";
	  }
	    
	  if (sessobj->mboxencode)
	    imapwrite (imap_tag () + " UID COPY " + cuids + " " + IMAP_QSTRING (sessobj->copytobox) + "\r\n");
	  else {
	    imapwrite (imap_tag () + " UID COPY " + cuids + " ");
	    imapwrite_literal (sessobj->copytobox, 1, 0);
	  }

	  command_seq = 1;
	  timeout (TIMEOUT1);
	}
	else {
	  command_result = -1;
	  next_command ();
	}
      }
      else
	timeout (TIMEOUT1);
      break; //2
    }
  }

  // low_create
  void start_low_create () {
    if (sessobj->mboxencode)
      imapwrite (imap_tag () + " CREATE " + IMAP_QSTRING (command->newmailbox) + "\r\n");
    else {
      imapwrite (imap_tag () + " CREATE ");
      imapwrite_literal (command->newmailbox, 1, 0);
    }
    timeout (TIMEOUT1);
  }
  
  void handle_low_create () {
    switch (command_seq) {
    case 0: // expect CREATE command result
      if (sscanf (line, imap_tag () + " %s", foo))
	{
	  if (foo[0..1] == "OK")
	    {
	      // ok. folder created
	      commands = ({ imap_cmd ("cache_init", "mailbox", command->newmailbox) }) + commands;
	      command_result = 0;
	      next_command ();
	    }
	  else
	    {
	      command_result=-1;
	      next_command();
	    }
	}
      else
	timeout (TIMEOUT1);
      break; //0
    }
  }
  
  // create
  void start_create () {
    string newmbox = command->newmailbox;
    string error = command->error;
    if (error)
      commands = ({ imap_cmd ("low_create", "newmailbox", newmbox, "error", error) }) + commands;
    else
      commands = ({ imap_cmd ("low_create", "newmailbox", newmbox, "abortfail", 0) }) + commands;
    command_result = 0;
    next_command ();
  }

  // delete_folders
  void start_delete_folders () {
    folderstodelete = ({ });
    foreach (get_cmd_arg (command->folders), string mbox)
      folderstodelete+=({ mbox });

    if (!sizeof (folderstodelete)) {
      command_result=0;
      next_command ();
    }
    else {
      if (sessobj->mboxencode)
	imapwrite (imap_tag () + " DELETE " + IMAP_QSTRING (folderstodelete[0]) + "\r\n");
	else {
	  imapwrite (imap_tag () + " DELETE ");
	  imapwrite_literal (folderstodelete[0], 1, 0);
	}
	timeout (TIMEOUT1);
    }
  }

  void handle_delete_folders () {
    // expect DELETE command result
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK") {
	// remove filter rules
	int changed = 0;
	if (sizeof (sessobj->filterbook)) {
	  array (string) filters = sessobj->filterbook/"\n";
	  for (int i = 0; i < sizeof (filters); i++) {
	    string namefield = "", filterexpression = "", filterfolder = "";
	    if (sscanf (filters[i], "%s:%s:%s", namefield, filterexpression, filterfolder) == 3) {
	      if (filterfolder == folderstodelete[0]) {
		if (sessobj->debug)
		  write ("removing filter rule: " + filters[i] + "\n");
		filters -= ({ filters[i] });
		changed = 1;
	      }
	    }
	  }
	  if (changed) {
	    sessobj->filterbook = filters * "\n";
	    if (sessobj->debug)
	      write ("scheduling saveuserprefs...\n");
	    commands += ({ imap_cmd("save_user_prefs") });
	  }
	}

	if (imho->query ("foldersinfo") && sessobj->foldersinfo[folderstodelete[0]]) {
	  sessobj->allfolderssize -= sessobj->foldersinfo[folderstodelete[0]][MB_SIZE_IDX];
	  m_delete (sessobj->foldersinfo, folderstodelete[0]);
        }
	m_delete (sessobj->cache, folderstodelete[0]);

	folderstodelete=folderstodelete[1..];
	if(sizeof(folderstodelete)) {
	  imap_newtag();
	  if(sessobj->mboxencode)
	    imapwrite(imap_tag()+" DELETE "+IMAP_QSTRING(folderstodelete[0])+"\r\n");
	  else {
	    imapwrite(imap_tag()+" DELETE ");
	    imapwrite_literal(folderstodelete[0],1,0);
	  }
	  timeout(TIMEOUT1);
	} else {
	  command_result=0;
	  next_command();
	}
      }
      else {
	command_result=-1;
	next_command();
      }
    }
    else
      timeout (TIMEOUT1);
  }

  // low_append
  void start_low_append () {
    if (sessobj->mboxencode) {
      imapwrite (imap_tag () + " APPEND " + IMAP_QSTRING (command->tobox) + " ");
      imapwrite_literal (get_cmd_arg (command->data), 1 ,0);
      command_seq = 1;
    }
    else {
      imapwrite (imap_tag() + " APPEND ");
      imapwrite_literal (command->tobox, 0, 1);
    }
    trycreate = 0;
    timeout (TIMEOUT1);
  }
  
  void handle_low_append () {
    switch(command_seq) {

    case 0: // send appenddata
      if(line[0..0]!="+") {
	command_seq=1;
	handle_command();
      }
      else {
	imapwrite(" ");
	imapwrite_literal(get_cmd_arg(command->data),1,0);
	command_seq=1;
      }
      break; //0
      
    case 1: // expect APPEND command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  command_result=0;
	  next_command();
	}
	else {
	  if(sscanf(foo,"NO %s",foo) && (trycreate || has_value (foo, "[TRYCREATE]"))) {
	    imap_newtag();
	    if(sessobj->mboxencode)
	      imapwrite(imap_tag()+" CREATE "+IMAP_QSTRING(command->tobox)+"\r\n");
	    else {
	      imapwrite(imap_tag()+" CREATE ");
	      imapwrite_literal(command->tobox,1,0);
	    }
	    command_seq=2;
	    timeout(TIMEOUT1);
	  }
	  else {
	    command_result=-1;
	    next_command();
	  }
	}
      }
      else { 
	if (search(line, "[TRYCREATE]")!=-1)
	  trycreate = 1;
	timeout(TIMEOUT1);
      }
      break; //1
      
    case 2: // expect create results
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  imap_newtag();
	  if(sessobj->mboxencode) {
	    imapwrite(imap_tag()+" APPEND "+IMAP_QSTRING(command->tobox)+" ");
	    imapwrite_literal(get_cmd_arg(command->data),1,0);
	    command_seq=4;
	  }
	  else {
	    imapwrite(imap_tag()+" APPEND ");
	    imapwrite_literal(command->tobox,0,1);
	    command_seq=3;
	  }
	  timeout(TIMEOUT1);
	}
	else {
	  command_result=-1;
	  next_command();
	}
      }
      else
	timeout(TIMEOUT1);
      break; //2
      
    case 3: // send appenddata
      if(line[0..0]!="+") {
	command_seq=4;
	handle_command();
      }
      else {
	imapwrite(" ");
	imapwrite_literal(get_cmd_arg(command->data),1,0);
	command_seq=1;
      }
      break; // 3
      
    case 4: // expect APPEND command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  command_result=0;
	  next_command();
	}
	else {
	  command_result=-1;
	  next_command();
	}
      }
      else
	timeout(TIMEOUT1);
	break; // 4
    }
  }

  // add_flag
  void start_add_flag () {
    imapwrite (imap_tag () + " UID STORE " + command->uid + " +FLAGS.SILENT (" + command->flag + ")\r\n");
    timeout (TIMEOUT1);
  }
      
  void handle_add_flag () {
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK") {
	command_result = 0;
	if (!command->stop)
	  next_command ();
      }
      else {
	command_result = -1;
	next_command ();
      }
    }
    else
      timeout (TIMEOUT1);
  }

  // expunge
  void start_expunge () {
    imapwrite (imap_tag () + " EXPUNGE\r\n");
    timeout (TIMEOUT1);
  }
      
  void handle_expunge () {
    if (sscanf (line, imap_tag () + " %s", foo)) {
      if (foo[0..1] == "OK")
	command_result = 0;
      else
	command_result = -1;
      next_command ();
    }
    else
      timeout (TIMEOUT1);
  }

  // save_user_prefs
  void start_save_user_prefs () {
    if(sessobj->usersetup) {
      if(sizeof(sessobj->prefsdir)) {
	command->subcommands=
	  ({ imap_cmd("create_prefs_mail","output",imap_cmd_var("-"),"dataonly",1),
	     imap_cmd("save_prefs_file","data",imap_cmd_var("-")) });
	if (imho->feature(FEAT_EXTENDEDABOOK)) {
	  command->subcommands += ({
	    imap_cmd("create_ebook_mail","output",imap_cmd_var("-"),"dataonly",1),
	    imap_cmd("save_ebook_file","data",imap_cmd_var("-"))
	  });
	}
      } else {
	if (imho->query ("buggyimap"))
	  {
	    if (sessobj->debug)
	      write ("buggyimap: creating " + sessobj->prefsbox + "\n");
	    command->subcommands = ({ imap_cmd ("create", "newmailbox", sessobj->prefsbox) });
	  }
	else
	  command->subcommands = ({ });

	//command->subcommands = ({ imap_cmd ("create", "newmailbox", sessobj->prefsbox) });

	command->subcommands += ({
	  imap_cmd("status","mailbox",sessobj->prefsbox),
	  imap_cmd("get_headers",
		   "mailbox",sessobj->prefsbox,
		   "output",imap_cmd_var("prefsmails"),
		   "abortfail", 0),
	  imap_cmd("create_prefs_mail",
		   "output",imap_cmd_var("-")),
	  imap_cmd("low_append",
		   "tobox",sessobj->prefsbox,
		   "data",imap_cmd_var("-")),
	  imap_cmd("delete",
		   "mailbox",sessobj->prefsbox,
		   "uids",imap_cmd_var("prefuids"))
	});

	if (imho->feature(FEAT_EXTENDEDABOOK)) {
	  command->subcommands += ({
	    imap_cmd("create_ebook_mail",
		     "output", imap_cmd_var("-")),
	    imap_cmd("low_append",
		     "tobox",sessobj->prefsbox,
		     "data",imap_cmd_var("-")),
	    imap_cmd("delete",
		     "mailbox",sessobj->prefsbox,
		     "uids",imap_cmd_var("ebookuids"))
	  });
	}
      }
      commands = ({ command }) + commands;
    }
    command_result = 0;
    next_command ();
  }

  // save_ebook_file
  void start_save_ebook_file () {
    object prefsfile = Stdio.File ();
    string filename = sessobj->prefsdir + "/" + lower_case (sessobj->login) + ".imhoebook";
    if(prefsfile->open(filename,"wct")) {
      prefsfile->write((string)ebookmailobj);
      prefsfile->close();
    }
    else 
      report_warning("CAMAS: Could not write ebookfile: "+filename+"\n");
    command_result=0;
    next_command();
  }
  
  // save_prefs_file
  void start_save_prefs_file () {
    object prefsfile=Stdio.File();
    if(prefsfile->open(sessobj->prefsdir+"/"+lower_case(sessobj->login)+".imhoprefs","wct")) {
      prefsfile->write(get_cmd_arg(command->data)||"");
      prefsfile->close();
    }
    else 
      report_warning("CAMAS: Could not write prefsfile: "+sessobj->prefsdir+"/"+lower_case(sessobj->login)+".imhoprefs"+"\n");
    command_result=0;
    next_command();
  }
  
  // create_ebook_mail
  void start_create_ebook_mail () {
    if (sessobj->extendedabook)
      {
	ebookmailobj = sessobj->extendedabook->export_mime();
	sessobj->ebookuids=({ });
	if (sessobj->prefsmails) {
	  foreach(indices(sessobj->prefsmails),int i) {
	    object mail=sessobj->prefsmails[i];
	    if (i < sizeof(sessobj->prefsmails) && 
		mail->imap->ENVELOPE[SUBJECT_IDX] == HEADER_EBOOK_SUBJECT_VALUE) {
	      sessobj->ebookuids += ({ mail->imap->UID });
	    }
	  }
	}
	set_cmd_arg(command->output,(string)ebookmailobj);
      }
    command_result = 0;
    next_command();
  }
  
  // create_prefs_mail
  void start_create_prefs_mail () {
    object prefsmailobj = 0;
    if(!command->dataonly) {
      prefsmailobj = MIME.Message("",
				  ([ "MIME-Version" : "1.0",
				     "Content-Type" : "text/plain; charset=UTF-8",
				     "User-Agent" : "CAMAS/"+imho->camas_version+" (Webmail for Caudium)" ]) );
      prefsmailobj->headers["to"]  = (sessobj)["address"];
      prefsmailobj->headers["from"] = (sessobj)["name"]+" <"+(sessobj)["address"]+">" ;
      prefsmailobj->headers["subject"] = "CAMAS Preferences";
      prefsmailobj->setencoding("8bit");
    }
    foo = "2\r\n"; // Version 2
    foreach (indices(imho->prefproperties), string prop) {
      if (sessobj[prop] && sessobj["usersetup"+prop])
	foo += "#"+ replace(prop +" = "+encode_pref((sessobj)[prop]),
			    ({ "\\", "#"}), ({ "\\1", "\\2" }))+"\r\n";
    }
    if(command->dataonly) {
      set_cmd_arg(command->output,foo);
      command_result=0;
      next_command();
    }
    else {
      sessobj->prefuids=({ });
      prefsmailobj->setdata(TO_UTF8(foo));
      if (sessobj->prefsmails) {
	foreach(indices(sessobj->prefsmails),int i) {
	  object mail=sessobj->prefsmails[i];
	  if (i < sizeof(sessobj->prefsmails) && 
	      mail->imap->ENVELOPE[SUBJECT_IDX] ==  "CAMAS Preferences") {
	    sessobj->prefuids += ({ mail->imap->UID });
	  }
	}
      }
      set_cmd_arg(command->output,(string)prefsmailobj);
      command_result=0;
      next_command();
    }
  }

  // load_user_prefs
  void start_load_user_prefs () {
    sessobj->prefsmailobj=0;
    if(sessobj->usersetup) { 
      if(!sizeof(sessobj->prefsdir)){
	command->subcommands = ({ 
	  imap_cmd("low_create", "newmailbox", sessobj->prefsbox, "abortfail", 0),
	  imap_cmd("status","mailbox",sessobj->prefsbox),
	  imap_cmd("get_headers",
		   "mailbox",sessobj->prefsbox,
		   "output",imap_cmd_var("prefsmails"),
		   "abortfail",0),
	  imap_cmd("get_prefs_mail",
		   "mailbox",sessobj->prefsbox,
		   "abortfail",0),
	  imap_cmd("set_prefs")
	});
	
	if (imho->feature(FEAT_EXTENDEDABOOK)) {
	  command->subcommands += ({
	    imap_cmd("status","mailbox",sessobj->prefsbox),
	    imap_cmd("get_headers",
		     "mailbox",sessobj->prefsbox,
		     "output",imap_cmd_var("prefsmails"),
		     "abortfail",0),
	    imap_cmd("get_ebook_mail",
		     "mailbox", sessobj->prefsbox,
		     "abortfail", 0),
	    imap_cmd("set_ebook")
	  });
	}
	commands = ({ command }) + commands;
      } else {
	string prefs=Stdio.read_bytes(sessobj->prefsdir+"/"+lower_case(sessobj->login)+".imhoprefs");
	if(prefs) {
	  sessobj->prefsmailobj=MIME.Message("");
	  sessobj->prefsmailobj->setdata(prefs);
	  sessobj->prefsloaded=1;
	}
	command->subcommands = ({ imap_cmd("set_prefs") });
	commands = ({ command }) + commands;
	if (imho->feature(FEAT_EXTENDEDABOOK)) {
	  prefs = Stdio.read_bytes(sessobj->prefsdir+"/" +
				   lower_case(sessobj->login) + ".imhoebook");
	  ebookmailobj = MIME.Message(prefs||"");
	  commands = ({ imap_cmd("set_ebook") }) + commands;
	}
      }
    }
    else
      commands=({ imap_cmd("set_prefs") })+commands;
    command_result=0;
    next_command();
  }
  
  // set_ebook
  void start_set_ebook () {
    sessobj->extendedabook = CAMAS.ebook.ebook ();
    if (sessobj->ebookmailobj)
      sessobj->extendedabook->import_mime (sessobj->ebookmailobj);
    sessobj->extendedabook->set_fields (imho->query ("extendedabookfields")/ ",");
    command_result = 0;
    next_command ();
  }

  // set_prefs
  void start_set_prefs () {
    if (sessobj->prefsmailobj) {
      if (sessobj->prefsmailobj->body_parts)
	sessobj->prefsmailobj = sessobj->prefsmailobj->body_parts[0];
      string data=replace(sessobj->prefsmailobj->getdata()||"","\r","");
      // UTF-8 encoded prefs starts with a '@' == prefs version 1
      // Prefs version 2 and up start with the version number
      // Prefs 2 lines start with #, and # is encoded \2 ,  \ is encoded \1
      ///array (string) lines = (data[0..0]=="@"?FROM_UTF8(data):data) / "\n";
      //FIXME: sscanf doesn't work on wide strings. decoding values separately instead.
      int utf8=data[0..0]=="@";
      int version = 0;
      if (utf8)
	version = 1;
      else
	version = (int) data[0..0];
      array (string) lines;
      if (version >= 2)
	lines= data / "\n#";
      else
	lines= data / "\n";
      foreach(indices(lines), int i) {
	string prop, value;
	string line = ((version >= 2)?
		       replace(lines[i],({ "\n","\\2", "\\1" }),
			       ({ "","#", "\\" })):lines[i]);
	if (sscanf(line, "%s = %s", prop, value ) > 0) {
	  if (imho->prefproperties[prop]) {
	    if (sessobj["usersetup"+prop])
	      (sessobj)[prop] = decode_pref((version>=1)?FROM_UTF8(value):value);
	  }
	}
      }
      if (!sessobj["usersetupmailpath"])
	sessobj["mailpath"] = sessobj["defaultmailpath"];
      if (!sessobj["usersetupaddress"])
	sessobj["address"] = sessobj["defaultaddress"];
      if (!sessobj->layout || (sessobj->layout == "0"))
	sessobj->layout = imho->query ("defaultlayout");
      if (sessobj["overridelayout"])
	sessobj["layout"] = sessobj["overridelayout"];
      }

    command_result = 0;
    next_command ();
  }

  // start
  void start_start () {
    imho->imho_log("login", ([ "login":sessobj->address ]));

    if(sessobj->prefsloaded || !sessobj->usersetup)
      (sessobj)["status"] = MAILINDEX;
    else
      if (imho->query("newbiesetup"))
	(sessobj)["status"] = SETUP;
      else
	(sessobj)["status"] = MAILINDEX;

    if (sessobj->debug)
      perror(sprintf("Entering status %d\n",sessobj["status"]));
    (sessobj)["loadfiles"] = 1;
    command_result=0;
    next_command();
  }

  // check_mailboxes
  void start_check_mailboxes () {
    if (sessobj->debug)
      write ("checkmailboxes: checking INBOX\n");
    commands = ({ imap_cmd ("low_close"), imap_cmd ("status", "mailbox", "INBOX") }) + commands;
    command_result = 0;
    next_command ();
  }

  // search
  void start_search () {
    if (command->searchtext && command->searchfield) {
      imapwrite(imap_tag()+" UID SEARCH CHARSET UTF-8 "+
		command->searchfield+" ");
      imapwrite_literal(TO_UTF8(command->searchtext),1,0);
      command_seq = 0;
      timeout(TIMEOUT1);
    }
    else {
      command_result = 0;
      next_command();
    }
  }

  void handle_search () {
    // expect SEARCH command result
    if(sscanf(line,imap_tag()+" %s",foo)) {
      if(foo[0..1]=="OK") {
	command_result=0;
	next_command();
      } else {
	command_result=-1;
	next_command();
      }
    } else {
      timeout(TIMEOUT1);
      foo = "";
      //oliv3: FIXED ? in testing
      if (sscanf (line,"* SEARCH %s", foo)) {
	array uids = Array.map (foo / " ", lambda (string s) { return (int)s; });
	array mails = get_cmd_arg (command->headers);
	mails = Array.filter (mails, lambda (mapping mail, array uids)
				     { return (search (uids, mail->imap->UID) != -1); }, uids);
	set_cmd_arg(command->headers, mails);
      }
    }
  }

  // scan_folders
  void start_scan_folders () {
    if (!sessobj->foldersinfo)
      sessobj->foldersinfo = ([ ]);
    
    foreach (sessobj->mailboxes, array mbox_a)
      if (!sessobj->foldersinfo[mbox_a[MB_FOLDERNAME_IDX]])
	sessobj->foldersinfo += ([ mbox_a[MB_FOLDERNAME_IDX]: ({ 0, 0, 0, 0, 0 }) ]);
    
    sessobj->folderstoscan = ({ });
    sessobj->nodefoldersinfo = ([ ]);
    foreach (sessobj->mailboxes, array mbox_a)
      if (!(mbox_a[MB_FLAGS_IDX] & MB_NOSELECT))
	sessobj->folderstoscan += ({ mbox_a[MB_FOLDERNAME_IDX] });
      else
	sessobj->nodefoldersinfo += ([ mbox_a[MB_FOLDERNAME_IDX]: ({ 0, 0, 0, 0 }) ]);
    
    commands = ({ imap_cmd ("start_scan") }) + commands;
    command_result = 0;
    next_command ();
  }
  
  // start_scan
  void start_start_scan () {
    string mbox2scan = sessobj->folderstoscan[0];
    imap_newtag ();
    if (sessobj->mboxencode) {
      imapwrite(imap_tag()+" STATUS "+IMAP_QSTRING(mbox2scan)+" (MESSAGES RECENT UNSEEN UIDNEXT)\r\n");
    }
    else {
      imapwrite(imap_tag()+" STATUS ");
      imapwrite_literal(mbox2scan,0,1);
      command_seq = 1;
    }
    timeout (TIMEOUT1);
  }
  
  void handle_start_scan () {
    //oliv3 FIXME reorder the switches...
    switch (command_seq) {
    case 1: // Continue STATUS command after literal
      imapwrite(" (MESSAGES RECENT UNSEEN UIDNEXT)\r\n");
      command_seq = 0;
      timeout (TIMEOUT1);
      break;
      
    case 0:
      if (sscanf (line, "* STATUS {%s", foo) == 1)
	command_seq = 6;
      else
	if ((sscanf (line, "* STATUS %*s (%s", foo) == 2) || (sscanf (line, "* STATUS \"%*s\" (%s", foo) == 2)) {
	  data = "(" + foo + "\n" + data;
	  init_parser (PT_MAPPING);
	  command_seq = 2;
	}
      timeout (TIMEOUT1);
      break; // 0
      
    case 6:
      if ((sscanf (line, "%*s (%s", foo) == 2) || (sscanf (line, "\"%*s\" (%s", foo) == 2)) {
	data = "(" + foo + "\n" + data;
	init_parser (PT_MAPPING);
	command_seq = 2;
      }
      timeout (TIMEOUT1);
      break; // 6

    case 2:
      if (!parse_result) {
	if (sessobj->debug)
	  perror ("CAMAS : IMAP parse error!\n");
	command_result = -1;
	next_command ();
      }
      else
	{
	  sessobj->exists_old = sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX];
	  sessobj->uidnext_old = sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX];
	  sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX] = (int)parse_result->MESSAGES;
	  sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_RECENT_IDX] = (int)parse_result->RECENT;
	  sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_UNSEEN_IDX] = (int)parse_result->UNSEEN;
	  sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX] = (int)parse_result->UIDNEXT;
	  command_seq = 3;
	  timeout (TIMEOUT1);
	}
      break; // 2
      
    case 3: // do EXAMINE only if (non-empty mailbox)
      if (sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX]) {
	imap_newtag ();
	string mb = sessobj->folderstoscan[0];
	if (sessobj->mboxencode)
	  imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(mb)+"\r\n");
	else {
	  imapwrite(imap_tag()+" EXAMINE ");
	  imapwrite_literal(mb,1,0);
	}
	command_seq = 4;
	timeout (TIMEOUT1);
      }
      else {
	sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_SIZE_IDX] = 0;
	commands = ({ imap_cmd ("end_scan") }) + commands;
	command_result = 0;
	next_command ();
      }
      break; // 3
      
    case 4: // EXAMINE result + send FETCH
      if (sscanf (line, imap_tag () + " OK [READ-ONLY]%s", foo)) {
	if ((sessobj->uidnext_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX]) || 
	    (sessobj->exists_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX])) {
	  if (sessobj->debug) {
 	    if (sessobj->uidnext_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_NEXTUID_IDX])
	      write (sessobj->folderstoscan[0]+": scheduled fetch due to UIDNEXT change...\n");
	    if (sessobj->exists_old != sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_MESSAGES_IDX])
	      write (sessobj->folderstoscan[0]+": scheduled fetch due to MESSAGES change...\n");
	  }
	  sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_SIZE_IDX] = 0;
	  imap_newtag ();
	  imapwrite(imap_tag()+" FETCH 1:* RFC822.SIZE\r\n");
	  sessobj->foldershavechanged = 1;
	  command_seq = 5;
	  timeout (TIMEOUT1);
	}
	else {
	  commands = ({ imap_cmd ("end_scan") }) + commands;
	  next_command ();
	}
      }	
      break; // 4
      
    case 5: // FETCH result
      int size;
      if (sscanf (line, "* %*d FETCH (RFC822.SIZE %d)", size)) {
	sessobj->foldersinfo[sessobj->folderstoscan[0]][MB_SIZE_IDX] += size;
      }
      else {
	commands = ({ imap_cmd ("end_scan") }) + commands;
	next_command ();
      }
      break; // 5
    }
  }

  // end_scan
  void start_end_scan () {
    if (sizeof (sessobj->folderstoscan) > 1) {
      sessobj->folderstoscan = sessobj->folderstoscan[1..];
      commands = ({ imap_cmd ("start_scan") }) + commands;			
      command_result = 0;
      next_command ();
    }
    else { // done
      m_delete (sessobj, "folderstoscan");
      
      if (sessobj->foldershavechanged) {
	//write ("updating folders info\n");
	sessobj->allfolderssize = 0;
	foreach (indices (sessobj->foldersinfo), string key)
	  if (sizeof (sessobj->foldersinfo[key]) == 5) // it's a leaf folder
	    sessobj->allfolderssize += sessobj->foldersinfo[key][MB_SIZE_IDX];
	
	foreach (sessobj->mailboxes, array mbox_a) {
	  if (!(mbox_a[MB_FLAGS_IDX] & MB_NOSELECT))
	    if (sizeof (mbox_a[MB_HIERARCHY_IDX]) > 1) {
	      // write (mbox_a[MB_FOLDERNAME_IDX] + " is a sub-mailbox\n");
	      for (int p_mbox = 0; p_mbox < (sizeof (mbox_a[MB_HIERARCHY_IDX]) - 1); p_mbox++) {
		// write ("parent mbox #" + p_mbox + " = ");
		string p_mbox_name = imho->query("defaultmailpath") + mbox_a[MB_HIERARCHY_IDX][0];
		for (int i = 1; i <= p_mbox; i++)
		  p_mbox_name += (mbox_a[MB_SEPARATOR_IDX] + mbox_a[MB_HIERARCHY_IDX][i]);
	        /*
		  write (p_mbox_name + "\n");
	          write (sprintf ("foldersinfo= %O\n", sessobj->foldersinfo));
	          write (sprintf ("nodefoldersinfo= %O\n", sessobj->nodefoldersinfo));
                */
		for (int k = 0; k < 4; k++)
		  sessobj->nodefoldersinfo[p_mbox_name][k] += sessobj->foldersinfo[mbox_a[MB_FOLDERNAME_IDX]][k];
	      }
	    }
	}
	
	foreach (indices (sessobj->nodefoldersinfo), string mbox)
	  sessobj->foldersinfo[mbox] = sessobj->nodefoldersinfo[mbox];
      }
      //else
      //write ("folders info has not changed\n");
      
      m_delete (sessobj, "nodefoldersinfo");
      m_delete (sessobj, "foldershavechanged");
      m_delete (sessobj, "exists_old");
      m_delete (sessobj, "uidnext_old");
      commands = ({ imap_cmd ("low_close") }) + commands;
      command_result=0;
      next_command ();
    }
  }

  // rename_folder
  void start_rename_folder () {
    if(!sessobj->oldfoldername || !sizeof(sessobj->oldfoldername) ||
       !sessobj->newfoldername || !sizeof(sessobj->newfoldername)) {
      command_result=-1;
      next_command();
    }
    else {
      if (sessobj->debug)
	write(sprintf("About to rename folder : %s => %s\n",sessobj->oldfoldername,sessobj->newfoldername));
      imap_newtag ();
      if(sessobj->mboxencode)
	imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(sessobj->newfoldername)+"\r\n");
      else {
	imapwrite(imap_tag()+" EXAMINE ");
	imapwrite_literal(sessobj->newfoldername,1,0);
      }
      timeout(TIMEOUT1);
    }
  }
  
  void handle_rename_folder () {
    switch(command_seq) {
    case 0: // expect EXAMINE command result (newfoldername)
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  // oops. mailbox does exist. skip.
	  sessobj->newfoldername = 0;
	  command_result=-1;
	  next_command();
	}
	else {
	  imap_newtag();
	  if(sessobj->mboxencode)
	    imapwrite(imap_tag()+" EXAMINE "+IMAP_QSTRING(sessobj->oldfoldername)+"\r\n");
	  else {
	    imapwrite(imap_tag()+" EXAMINE ");
	    imapwrite_literal(sessobj->oldfoldername,1,0);
	  }
	  command_seq=1;
	  timeout(TIMEOUT1);
	}
      }
      else
	timeout(TIMEOUT1);
      break; //0 
      
    case 1: // expect EXAMINE command result (oldfoldername)
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  imap_newtag();
	  if(sessobj->mboxencode) {
	    imapwrite(imap_tag()+" RENAME "+
		      IMAP_QSTRING(sessobj->oldfoldername)+" "+
		      IMAP_QSTRING(sessobj->newfoldername)+
		      "\r\n");
	  }
	  else {
	    //oliv3: FIXME (imapwrite_literal bug)
	    imapwrite(imap_tag()+" RENAME ");
	    //imapwrite_literal(sessobj->oldfoldername + " " + sessobj->newfoldername,1,0);
	    imapwrite_literal(sprintf ("%s %s", sessobj->oldfoldername, sessobj->newfoldername),1,0);
	  }
	  command_seq=2;
	  timeout(TIMEOUT1);
	}
      }
      else
	timeout(TIMEOUT1);
      break; //1
      
    case 2: // expect RENAME command result
      if(sscanf(line,imap_tag()+" %s",foo)) {
	if(foo[0..1]=="OK") {
	  // ok. folder renamed
	  if (imho->query ("foldersinfo")) {
	    sessobj->foldersinfo[sessobj->newfoldername] = sessobj->foldersinfo[sessobj->oldfoldername];
	    m_delete (sessobj->foldersinfo, sessobj->oldfoldername);
	  }
	  sessobj->cache[sessobj->newfoldername] = sessobj->cache[sessobj->oldfoldername];
	  m_delete (sessobj->cache, sessobj->oldfoldername);
	  
	  commands = ({ imap_cmd("update_filters") }) + commands;
	  command_result=0;
	  next_command();
	} else {
	  command_result=-1;
	  next_command();
	}
      }
      else
	timeout(TIMEOUT1);
      break; //2
    }
  }

  // update_filters
  void start_update_filters () {
    if (sizeof (sessobj->filterbook)) {
      array (string) filters = sessobj->filterbook / "\n";
      for (int i = 0; i < sizeof (filters); i++) {
	string namefield = "", filterexpression = "", filterfolder = "";
	if (sscanf (string_to_unicode (filters[i]), "%s\0:%s\0:%s", namefield, filterexpression, filterfolder) == 3) {
	  if (sessobj->debug)
	    write ("folder= " + unicode_to_string (filterfolder) + "\n");
	  if (unicode_to_string (filterfolder) == sessobj->oldfoldername) {
	    if (sessobj->debug)
	      write ("changing filter rule: old= " + filters[i] + "\n");
	    filters[i] = unicode_to_string (namefield) + ":" + unicode_to_string (filterexpression) + ":" + sessobj->newfoldername;
	    if (sessobj->debug)
	      write ("changing filter rule: new= " + filters[i] + "\n");
	  }
	}
      }
      sessobj->filterbook = filters * "\n";
      commands = ({ imap_cmd("save_user_prefs") }) + commands;
    }
    else
      {
	if (sessobj->debug)
	  write ("no filters -> no rename\n");
      }
    
    command_result = 0;
    next_command ();
  }
  
  // cache_init
  void start_cache_init () {
    mcache = sessobj->cache[command->mailbox];
    if (!mcache) {
      if (sessobj->debug)
	write ("cache_init (" + command->mailbox + ")\n");
      mcache = ([ ]);
      mcache->mails = ({ });
      mcache->new_mails = ({ });
      mcache->uid_validity = 0;
      mcache->messages = mcache->recent = mcache->unseen = 0;
      mcache->size = 0;
      mcache->uidnext = 0;
      mcache->changed = 1;

      if (imho->query ("foldersinfo"))
	sessobj->foldersinfo += ([ command->mailbox: ({ 0, 0, 0, 0, 0 }) ]);

      command_result = 0;
    }
    else {
      if (sessobj->debug)
	write ("oops: initializing a known folder (" + command->mailbox + ")\n");
      command_result = -1;
    }
    next_command ();
  }

  // check
  void start_check () {
    if (selected_mailbox) {
      imap_newtag ();
      imapwrite (imap_tag () + " CHECK\r\n");
      timeout (TIMEOUT1);
    }
    else {
      command_result = 0;
      next_command ();
    }
  }

  void handle_check () {
    if (sscanf (line, imap_tag () + " %s", foo))
      {
	if (foo[0..1] == "OK")
	  command_result = 0;
	else
	  command_result = -1;
	next_command ();
      }
    else
      timeout (TIMEOUT1);
  }

  // ------------------------------------------------------------------
  
  void start_command (mapping com) {
    if (sessobj->debug)
      perror("start command : " + com->cmd + "\n");
    command = com;
    command_seq = 0;
    imap_newtag ();

    if (this_object () ["start_" + com->cmd])
      this_object () ["start_" + com->cmd] ();
    else
      {
	perror ("CAMAS: Unimplemented command: start_" + com->cmd + "\n");
	command_result = -1;
	next_command ();
      }
  }


  void handle_command () {
    if (sessobj->debug && command) // && !((< "get_headers", "low_list", "status", "low_select" >) [command->cmd]))
      {
	perror ("CAMAS command: " + command->cmd + " (" + command_seq + ")\n");
	perror ("CAMAS line: " + line + "\n");
      }

    if (line && sizeof (line) > 0)
      {
	if (sizeof (imaplinehist) > 4)
	  imaplinehist -= ({ imaplinehist[4] });
	imaplinehist = ({ (sizeof(line) > 70) ? line[0..69] : line }) + imaplinehist;
      }

    if (literal_string) {
      // see imapwrite_string above
      if (line && line[0..0] == "+") {
	imapwrite (literal_string + (literal_addcrlf ? "\r\n" : ""));
	literal_string = 0;
	if (literal_continue)
	  handle_command ();
      }
      else {
	literal_string = 0;
	handle_command ();
      }
      return;
    }

    if (command) {
      if (this_object () ["handle_" + command->cmd])
	this_object () ["handle_" + command->cmd] ();
      else {
	perror ("CAMAS: Unimplemented command: handle_" + command->cmd + "\n");
	command_result = -1;
	next_command ();
      }
    }
    else {
      string response = "", code = "", text = "";
      if (sscanf (line, "* BYE%s", text) == 1) {
	// we'll have to reconnect :)
	loggedin = 0;
	if (sessobj->debug)
	  write ("got BYE response:" + text + "\n");
      }
      else 
	if (sscanf (line, "* %s [%s] %s", response, code, text) > 0) {
	  // Anything interesting ?
	  if (sessobj->debug)
	    write ("Response= " + response + "\nCode= " + code + "\nText= " + text + "\n");
	}
    }
  }


  void next_command () {
    if (command_result < 0)
      { 
	if (command->abortfail)
	  {
	    if (command->container)
	      {
		command = command->container; // Step back
		next_command ();
		return;
	      }
	    abort = 1;
	    commands = ({ });
	    handle_error ();
	  }
	if (command->subcommands)
	  command->subcommands = ({ });
      }

    if (sizeof (commands))
      {
	if (!fd || !loggedin)
	  {
	    fd = Stdio.File ();
	    fd->open_socket ();
	    fd->set_nonblocking (got_data, imap_data_written, socket_close);
	    mixed error = catch
	    { 
	      fd->connect (sessobj->imapserver, sessobj->imapport);
	    };
	    if (error)
	      {
		abort = 1;
		commands = ({ }); 
		handle_error (); 
		done ();
	      }
	    if (sessobj->debug)
	      write ("reconnecting to IMAP server ...\n");
	    start_command (imap_cmd ("low_login"));
	  }
	else
	  {
	    if (loggedin)
	      {
		mapping com;
		mapping container;
		// Find the next command. 
		com = commands[0];
		while (sizeof (commands) && com->subcommands)
		  {
		    com = commands[0];
		    container = 0;
		    while (com->subcommands && sizeof (com->subcommands))
		      { 
			container = com;
			com = com->subcommands[0];
		      }
		    if (com->subcommands)
		      {
			if (container)
			  container->subcommands = container->subcommands[1..];
			else
			  commands=commands[1..];
		      }
		  }
		if (sizeof (commands))
		  {
		    if (!com->noselect &&
			((com->mailbox && (com->mailbox != selected_mailbox)) ||
			 (!com->mailbox && (sessobj->mailbox[MB_FOLDERNAME_IDX] != selected_mailbox))))
		      {  // Make sure the right box is selected
			if (com->tryselect)
			  { // If we've already tried to select then
			    // fail the command that needed the select
			    if (container)
			      container->subcommands = container->subcommands[1..];
			    else
			      commands = commands[1..];
			    com->container = container;
			    command = com;
			    command_result = -1; // Muahaha
			    next_command ();
			  }
			else
			  {
			    com->tryselect = 1; // Try to select only once
			    start_command (imap_cmd ("low_select", "mailbox", com->mailbox, "abortfail", 0));
			  }
		      }
		    else
		      {
			if (container)
			  container->subcommands = container->subcommands[1..];
			else
			  commands = commands[1..];
			com->container = container;
			start_command (com);
		      }
		  }
		else
		  {
		    command = 0;
		    done ();
		  }
	      }
	  }
      }
    else
      {
	command = 0;
	done ();
      }
  }

  void create () {
    parser = CAMAS.parser.Parser ();
    fd = 0;
    command = 0;
  }
  
  void destroy () {
    if (fd && loggedin) {
      fd->set_close_callback (0);
      imapwrite (imap_tag () + " LOGOUT\r\n");
      catch { fd->close (); };
      fd = 0;
      selected_mailbox = 0;
      //write ("LOGOUT...\n");
    }

    destruct (parser);
    timeout (0);
    remove_call_out (imap_idle_call_out);

    if (fd) {
      fd->set_close_callback (0);
      fd->set_read_callback  (0);
      fd->set_write_callback (0);
      catch { 
	fd->close();
      };
    }
  }
  
  void imap_command (object imho_in, object id_in, mapping sessobj_in, array (mapping) commands_in) {
    cmd_result = 0;
    abort = 0;
    result_sent = 0;
    command_result = 0;
    reconnects = 0;
    imho = imho_in;
    id = id_in;
    sessobj = sessobj_in;

    if (commands)
      commands += commands_in;
    else
      commands = commands_in;

    mode = MODE_LINE;
    if (id)
      id->do_not_disconnect = 1;
    next_command ();
  }
}
