#! /bin/sh
# helper functions for the initial network setup on s390
#
# written by Gerhard Tonn, Stefan Gybas and Frank Kirschner
# maintained by the boot-floppies team, debian-boot@lists.debian.org
# copyright (c) 2001
# This is free software under the GNU General Public License.


# Read an input line from the user and return the answer
# Parameters: $1: the question to ask (required)
#             $2: default answer (optional), will be returned if user input
#                 is empty
# Returns:  $ans: the answer (as entered in mixed case letters)
readln_mixed () {
  if [ "$auto_setup" = "1" ]; then
    ans=$2
  else
    if [ ! -z "$2" ] ; then
      echo -n "$1 [$2] "
    else
      echo -n "$1 "
    fi
    IFS='@' read ans || exit 1
    ans=`echo $ans | sed -e 's/^ *//'`
    if [ ! -z "$2" -a -z "$ans" ]; then
      ans=$2
    fi
  fi
}


# Read an input line from the user and return the answer
# Parameters: $1: the question to ask (required)
#             $2: default answer (optional), will be returned iin lower case
#                 if user input is empty
# Returns:  $ans: the answer in lower case letters
readln () {
  readln_mixed "$1" "$2"
  ans=`echo $ans | tr [A-Z] [a-z]`
}


# Read an input line from the user (yes/no) and return the answer
# Parameters: $1: the question to ask (required)
#             $2: default answer (optional), will be returned if user input
#                 is empty - must be "yes" or "no"
# Returns:  $ans: the answer in lower case letters, either "yes" or "no"
yes_no () {
  while :; do
    readln "$1"
    case "$ans" in 
      y | yes) ans=yes; break;;
      n | no)  ans=no; break;;
    esac
    echo "Invalid answer! Please enter yes or no."
    auto_setup=0
  done
}


# Read an input line from the user (IPv4 address) and return the answer
# Parameters: $1: the question to ask (required)
#             $2: default answer (optional), will be returned if user input
#                 is empty - must be a valid IPv4 address
# Returns:  $ans: the answer from the user, might be empty!
readip4 () {
  while :; do
    readln "$1" "$2"
    [ -z "$ans" ] && break
    ip4_ok "$ans"
    [ $? -eq 0 ] && break
    echo "Invalid answer! Please enter an IPv4 address, e.g. 192.168.1.2"
    auto_setup=0
  done
}


# Echos the base name of a module (strips .o and the path)
# Parameters: $1: the complete path name to the module (required)
basemodulename () {
  local module=$1
  # strip off .o suffix
  module=${module%.o}
  # strip off path prefix
  module=${module##*/}
  echo $module
} 


# Build a list of all network devices on the initial RAM disk
# Returns: $network_module_list: the modules, sorted and separated by spaces
#          $chandev_module_list: chandev capable modules, sorted and separated
#                                by spaces
build_network_lists () {
  local module_list
  if [ -d "/lib/modules/drivers/s390" ]; then
    module_list=`find /lib/modules/drivers/s390 -name *.o`
    network_module_list="$(for file in $module_list; do grep -q "register_netdev" $file; if [ $? -eq 0 ]; then basemodulename $file; fi; done | sort | uniq)"
    chandev_module_list="$(for file in $module_list; do grep -q "chandev_register" $file; if [ $? -eq 0 ]; then basemodulename $file; fi; done | sort | uniq )"
    return 0;
  fi
  return 1
}


# Checks if an IPv4 address is valid
# Parameters: $1: the IP address to check, as string (required)
# Returns: 0 if the address is valid as IPv4, 1 otherwise
ip4_ok () {
  local chk_ip=$1
  echo $chk_ip | grep -q '^[0-2]\?[0-9]\?[0-9]\.[0-2]\?[0-9]\?[0-9]\.[0-2]\?[0-9]\?[0-9]\.[0-2]\?[0-9]\?[0-9]$'
  if [ $? != 0 ]; then
    return 1
  fi
  for i in 1 2 3 4; do
    nr=`echo $chk_ip | cut -d. -f $i`
    if [ $nr -gt 255 ]; then
      return 1
    fi
  done
  return 0
}


# Checks wheter a string is a member of a list
# Parameters: $1 : the string to search in the list (required)
#             $2-: the list (strings separated by spaces) (optional)
# Returns: 0 if the string was found, 1 otherwise
in_list () {
  local a=$1
  local i
  shift
 
  for i in $*; do
    if [ "$i" = "$a" ]; then
      return 0
    fi
  done
  return 1
} 

# Return an element from a list
# Parameters: $1 : the index postion to return (required)
#             $2-: the list (strings separated by spaces) (optional)
# Returns: 0 if the string was found, 1 otherwise
#          $element: the element at the index position
element_from_list () {
  local index=$1
  local i=1
  shift

  for element in $*; do
    if [ $i = $index ]; then
      return 0
    fi
    i=`expr $i + 1`
  done
  return 1
} 

# Return the number of elements in a list
# Parameters: $1 : the list (strings separated by spaces) (required)
# Returns: the number of elements in the list
list_length () {
  local element
  local i=0

  for element in $*; do
    shift
    i=`expr $i + 1`
  done
  return $i
}
