#! /bin/bash
# kernel and modules preparation for boot-floppies
#
# Enrique Zanardi, 1998
# maintained by the boot-floppies team, debian-boot@lists.debian.org
#   copyright (c) 1999 -- 2001
# This is free software under the GNU General Public License.

set -e

. ./common.sh

if [ $# -lt 3 ]; then
    echo "usage: $0 archive kernel-version subarch-or-flavor [packages-with-modules ...]" 1>&2
    cat 1>&2 << EOF

        archive: the directory where Debian binary packages will be downloaded
        kernel-version: version of the kernel
        subarch-or-flavor: subarchitecture or kernel flavor, or "" for none
        packages-with-modules: other package names containing needed modules
EOF
    exit -1
fi

archive=$1
kernel_version=$2
subarch=$3                      # subarchitecture or flavor
shift 3

# whether this arch supports PCMCIA
case "$arch/$subarch" in
    i386/*|powerpc/*pmac) HAS_PCMCIA=true ;;
    *) HAS_PCMCIA=false ;;
esac

# convenience variable which is the subarch prepended with -, if set at all
subarch_suffix=${subarch:+-$subarch}

# expand the list of packages with modules to include the
# subarch/flavor pkg suffix
packages_with_modules=""
for p in $*; do
    packages_with_modules="$packages_with_modules $p$subarch_suffix"
done

testroot

extractdir=${tmpdir}/boot-floppies/extract-tmp-$$
rm -rf $extractdir
make_tmpdir $extractdir

kernelpkg=kernel-image-${kernel_version}${subarch_suffix}
info "downloading $kernelpkg $packages_with_modules"
PACKAGE_PATHS=`grab_paths $kernelpkg $packages_with_modules`

if [ "$HAS_PCMCIA" = true ]; then
    pcmciamods=pcmcia-modules-${kernel_version}${subarch_suffix}
    info "downloading PCMCIA module package, '$pcmciamods'"
    PACKAGE_PATHS="$PACKAGE_PATHS `grab_paths $pcmciamods`"
fi

info "extracting kernel and modules (if any) to extract area, '$extractdir'"
for i in $PACKAGE_PATHS; do
    debug "extracting $i"
    dpkg-deb --extract $i $extractdir
done

info "creating module contents file, 'modcont$subarch'"
(cd $extractdir;
    for i in lib/modules/*/*; do
	if [ -d  $i ]; then
            echo -n "dir_content_`basename $i`=\"";
            for j in $i/*.o; do
		echo -n `basename $j .o` " " ;
            done ;
            echo \" ;
	fi ;
    done
) > modcont$subarch

relax

info "creating 'modules${subarch}.tgz'"
(cd $extractdir && tar clf - lib/modules | gzip -9) > modules${subarch}.tgz

if [ "$subarch" = "" ]; then
    info "appending to 'modcont$subarch'"
    (cd $extractdir/lib/modules/*;
	rm -rf block ipv? misc net pcmcia fs scsi *MODULES )

    (cd $extractdir;
	echo -n "rescue_content=\"" ;
	for i in `find lib/modules -name \*.o`; do
            echo -n `basename $i .o` " ";
	done ;
	echo \" ;
    ) >> modcont$subarch

    # FIXME: what's all this then?
    #	(cd $extractdir && tar clf - lib/modules | gzip -9) > rescmods$subarch.tgz
fi

info "creating 'linux$subarch.bin'"
case "$arch/$subarch" in
    */*pmac)
    	# vmlinux-<version> is the 'work' kernel
	# vmlinux.coff-<version> is the kernel for OF booting
	cp $extractdir/boot/vmlinux-* linux$subarch.bin
	cp $extractdir/boot/vmlinux.coff-* linux$subarch.coff
	;;
    hppa/*|mipsel/*)
        # From kernel-package 7.79 on, the kernel package contains 
        # only an ELF kernel for mipsel by default. We need ELF for
        # diskbooting and ECOFF for netbooting on DECstations, so we 
        # build an additional ECOFF image if elf2ecoff is installed.
        # Karsten Merker <merker@linuxtag.org>, 2002/01/20
    	cp $extractdir/boot/vmlinux-* linux$subarch.bin
	if [ -f $extractdir/boot/vmlinux.ecoff-* ]; then
            info "kernel package contains both ELF and ECOFF kernels"
            cp $extractdir/boot/vmlinux.ecoff-* linux$subarch.ecoff
        else
            if [ -x /usr/bin/elf2ecoff ]; then
                info "no ECOFF image in kernel package, running elf2ecoff"
                /usr/bin/elf2ecoff $extractdir/boot/vmlinux-* linux$subarch.ecoff
            else
                info "elf2ecoff not installed, cannot provide ECOFF kernel" 
            fi
	fi
	;;
    mips/*)
    	cp $extractdir/boot/vmlinux* linux$subarch.bin
	;;
    *)
    	cp $extractdir/boot/vmlinuz* linux$subarch.bin
	;;
esac

info "creating 'sys_map$subarch.gz'"
cat $extractdir/boot/System.map-* | gzip --no-name -9 > sys_map$subarch.gz

info "creating 'config$subarch.gz'"
# copy the config getting rid of comments and blank lines
cat $extractdir/boot/config-* | grep -vE '^(#|$)' | gzip --no-name -9 > config$subarch.gz

info "cleaning up extract area"
rm -rf $extractdir

exit 0
