#include <ctype.h>
#include <slang.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <wctype.h>
#include <assert.h>

#include "newt.h"
#include "newt_pr.h"

struct textbox
{
    char **lines;
    int numLines;
    int linesAlloced;
    int doWrap;
    newtComponent sb;
    int topLine;
    int textWidth;
};

static char *expandTabs (const char *text);
static void textboxDraw (newtComponent co);
static void addLine (newtComponent co, const char *s, int len);
static void doReflow (const char *text, char **resultPtr, int width,
                      int *badness, int *heightPtr);
static struct eventResult textboxEvent (newtComponent c, struct event ev);
static void textboxDestroy (newtComponent co);
static void textboxPlace (newtComponent co, int newLeft, int newTop);
static void textboxMapped (newtComponent co, int isMapped);

static struct componentOps textboxOps = {
    textboxDraw,
    textboxEvent,
    textboxDestroy,
    textboxPlace,
    textboxMapped,
};

static void
textboxMapped (newtComponent co, int isMapped)
{
    struct textbox *tb = co->data;

    co->isMapped = isMapped;
    if (tb->sb)
        tb->sb->ops->mapped (tb->sb, isMapped);
}

static void
textboxPlace (newtComponent co, int newLeft, int newTop)
{
    struct textbox *tb = co->data;

    co->top = newTop;
    co->left = newLeft;

    if (tb->sb)
        tb->sb->ops->place (tb->sb, co->left + co->width - 1, co->top);
}

void
newtTextboxSetHeight (newtComponent co, int height)
{
    co->height = height;
}

int
newtTextboxGetNumLines (newtComponent co)
{
    struct textbox *tb = co->data;

    return (tb->numLines);
}

newtComponent
newtTextboxReflowed (int left, int top, char *text, int width,
                     int flexDown, int flexUp, int flags)
{
    newtComponent co;
    char *reflowedText;
    int actWidth, actHeight;

    reflowedText = newtReflowText (text, width, flexDown, flexUp,
                                   &actWidth, &actHeight);

    co = newtTextbox (left, top, actWidth, actHeight, NEWT_FLAG_WRAP);
    newtTextboxSetText (co, reflowedText);
    free (reflowedText);

    return co;
}

newtComponent
newtTextbox (int left, int top, int width, int height, int flags)
{
    newtComponent co;
    struct textbox *tb;

    co = malloc (sizeof (*co));
    tb = malloc (sizeof (*tb));
    co->data = tb;

    co->ops = &textboxOps;

    co->height = height;
    co->top = top;
    co->left = left;
    co->takesFocus = 0;
    co->width = width;

    tb->doWrap = flags & NEWT_FLAG_WRAP;
    tb->numLines = 0;
    tb->linesAlloced = 0;
    tb->lines = NULL;
    tb->topLine = 0;
    tb->textWidth = width;

    if (flags & NEWT_FLAG_SCROLL)
    {
        co->width += 2;
        tb->sb = newtVerticalScrollbar (co->left + co->width - 1, co->top,
                                        co->height, COLORSET_TEXTBOX,
                                        COLORSET_TEXTBOX);
    }
    else
    {
        tb->sb = NULL;
    }

    return co;
}

static char *
expandTabs (const char *text)
{
    int bufAlloced = strlen (text) + 40;
    char *buf, *dest;
    const char *src;
    int bufUsed = 0;
    int linePos = 0;
    int i;

    /* this one needs rewriting - Edmund */
    buf = malloc (bufAlloced + 1);
    for (src = text, dest = buf; *src; src++)
    {
        if ((bufUsed + 10) > bufAlloced)
        {
            bufAlloced += strlen (text) / 2;
            buf = realloc (buf, bufAlloced + 1);
            dest = buf + bufUsed;
        }
        if (*src == '\t')
        {
            i = 8 - (linePos & 8);
            memset (dest, ' ', i);
            dest += i, bufUsed += i, linePos += i;
        }
        else
        {
            if (*src == '\n')
                linePos = 0;
            else
                linePos++;

            *dest++ = *src;
            bufUsed++;
        }
    }

    *dest = '\0';
    return buf;
}

static void doReflow(const char * text, char ** resultPtr, int width, 
		     int * badness, int * heightPtr) {
    wchar_t * buf;
    int bufn;
    int nn = 0;
    char * result = 0;
    int howbad = 0;
    int height = 0;
    int n, pos, resultn, i, j, k, w;
    wchar_t wc;

    bufn = strlen(text) + 1000;
    buf = malloc(bufn*sizeof(wchar_t));
    n = 0;

    if (resultPtr) {
	resultn = strlen(text) + 1 + 1000;
	result = malloc(resultn);
	nn = 0;
    }

    mbtowc(0, 0, 0);
    while ((k = mbtowc(&wc, text, -1)) > 0) {
	text += k;
	if (n >= bufn) {
	    bufn += width;
	    buf = realloc(buf, bufn*sizeof(wchar_t));
	}
	buf[n++] = wc;
    }

    for (i = 0; i < n; ) {
	pos = 0;
	for (j = i; j < n && buf[j] != '\n'; j++) {
	    if ((w = wcwidth(buf[j])) != -1)
		if ((pos += w) > width)
		    break;
	}
	if (j < n && buf[j] != '\n') {
	    for (k = j; i < k && !iswspace(buf[k]); k--) ;
	    for (; i < k && iswspace(buf[k-1]); k--) ;
	    j = (i == k) ? j : k;
	}
	pos = 0;
	for (k = i; k < j; k++) {
	    if ((w = wcwidth(buf[k])) != -1) {
		if (result) {
		    if (nn + MB_LEN_MAX*2 + 1 > resultn) {
			resultn += width + MB_LEN_MAX*2;
			result = realloc(result, resultn);
		    }
		    nn += wctomb(result+nn, buf[k]);
		}
		pos += w;
	    }
	}
	nn += wctomb(result+nn, '\n');
	++height;
	if (j < n && buf[j] != '\n')
	    howbad += width - pos;
	else if (pos < width/2)
	    howbad += ((width)/2 - pos)/2;
	if (j < n && buf[j] == '\n')
	    i = j + 1;
	else
	    for (i = j; i < n && iswspace(buf[i]); i++) ;
    }

    if (result)
	result[nn] = '\0';
    free(buf);
    if (badness) *badness = howbad;
    if (resultPtr) *resultPtr = result;
    if (heightPtr) *heightPtr = height;
}


#if 0
static void
doReflow (const char *text, char **resultPtr, int width,
          int *badness, int *heightPtr)
{
    wchar_t *buf;
    int *bufw;
    char *result = 0;
    int howbad = 0;
    int height = 0;
    int n, pos, bufn, resultn, k, i, w, m, m1, nn;
    wchar_t wc;

    bufn = width;
    buf = malloc (bufn * sizeof (wchar_t));
    bufw = malloc (bufn * sizeof (int));

    if (resultPtr)
    {
        resultn = strlen (text);
        result = malloc (resultn);
        nn = 0;
    }

    n = pos = 0;
    for (;;)
    {
        while ((k = mbtowc (&wc, text, -1)) > 0)
        {
            text += k;
            if ((w = wcwidth (wc)) == -1)
                break;
            if (n + 1 >= bufn)
            {
                bufn += width;
                buf = realloc (buf, bufn * sizeof (wchar_t));
                bufw = realloc (bufw, bufn * sizeof (wchar_t));
            }
            bufw[n] = pos;
            buf[n++] = wc;
            pos += w;
            if (pos > width)
                break;
        }
        if (k > 0 && w != -1)
        {
            for (m1 = n; m1 > 0 && !iswspace (buf[m1 - 1]); m1--)
                ;
            if (m1 > 0)
                for (m = m1 - 1; m > 0 && iswspace (buf[m - 1]); m--)
                    ;
            else
            {
                m = n - 1;
                m1 = n;
                bufw[n] = bufw[m];
                buf[n++] = bufw[m];
            }
        }
        else
            m = m1 = n;
        buf[m++] = '\n';
        if (result)
            for (i = 0; i < m; i++)
            {
                if (nn + MB_LEN_MAX >= resultn)
                {
                    resultn += width;
                    result = realloc (result, resultn);
                }
                nn += wctomb (result + nn, buf[i]);
            }
        ++height;
        howbad += width - pos;
        if (m1 < n)
        {
            n -= m1;
            buf[0] = buf[m1];
            for (i = 1; i < n; i++)
            {
                bufw[i] = bufw[i + m1] - bufw[0];
                buf[i] = buf[i + m1];
            }
            pos = bufw[n] - bufw[0];
        }
        else if (m <= m1)
        {
            while ((k = mbtowc (&wc, text, -1)) > 0)
            {
                text += k;
                if (!iswspace (wc))
                    break;
            }
            if ((pos = wcwidth (wc)) == -1)
                break;
            buf[0] = wc;
            n = 1;
        }
        else
            break;
    }
    if (result)
        result[nn] = '\0';

    if (badness)
        *badness = howbad;
    if (resultPtr)
        *resultPtr = result;
    if (heightPtr)
        *heightPtr = height;
}
#endif

#if 0

static void
doReflow (const char *text, char **resultPtr, int width,
          int *badness, int *heightPtr)
{
    char *result = NULL;
    const char *chptr, *end;
    int howbad = 0;
    int height = 0;

    if (resultPtr)
    {
        resultn = strlen (text) + 1000;
        result = malloc (resultn);
    }

    mbrtowc (0, 0, 0, &ps1);
    wcrtomb (0, 0, &ps2);
    for (;;)
    {
        for (;;)
        {
            ps0 = ps1, text0 = text;
            if ((k = mbrtowc (&wc, text, -1, &ps1)) <= 0)
                break;
            text += k;
            if (wc == '\n')
                break;
            if ((w = wcwidth (wc)) == -1)
                continue;

            if (iswspace (wc))
            {
                if (break1)
                {
                    break1 = 0;
                    break2 = text2, breakps2 = ps2;
                }
                else if (!break2)
                    continue;
            }
            else
            {
                if (!break1)
                    break1 = text0, breakps1 = ps0;
            }

            if (pos + w > width)
            {
                if (break1)
                    ps1 = breakps1, text = break1;
                if (break2)
                    ps2 = breakps2, text2 = break2;
                break;
            }
            text2 += wcrtomb (text2, wc, &ps2);
            pos += w;
        }
        if (k != -1 || pos)
            wcrtomb (text2, '\n', &ps2);
        if (k == -1)
            break;
        pos = 0;
    }
}

static void
doReflow (const char *text, char **resultPtr, int width,
          int *badness, int *heightPtr)
{
    char *result = NULL;
    const char *chptr, *end;
    int howbad = 0;
    int height = 0;

    if (resultPtr)
    {
        /* XXX I think this will work */
        result = malloc (strlen (text) + (strlen (text) / width) + 2);
        *result = '\0';
    }

    while (*text)
    {
        end = strchr (text, '\n');
        if (!end)
            end = text + strlen (text);

        while (*text && text <= end)
        {
            if (end - text < width)
            {
                if (result)
                {
                    strncat (result, text, end - text);
                    strcat (result, "\n");
                    height++;
                }

                if (end - text < (width / 2))
                    howbad += ((width / 2) - (end - text)) / 2;
                text = end;
                if (*text)
                    text++;
            }
            else
            {
                chptr = text + width - 1;
                while (chptr > text && !isspace (*chptr))
                    chptr--;
                while (chptr > text && isspace (*chptr))
                    chptr--;
                chptr++;

                if (chptr - text == 1 && !isspace (*chptr))
                    chptr = text + width - 1;

                if (chptr > text)
                    howbad += width - (chptr - text) + 1;
                if (result)
                {
                    strncat (result, text, chptr - text);
                    strcat (result, "\n");
                    height++;
                }

                if (isspace (*chptr))
                    text = chptr + 1;
                else
                    text = chptr;
                while (isspace (*text))
                    text++;
            }
        }
    }

    if (badness)
        *badness = howbad;
    if (resultPtr)
        *resultPtr = result;
    if (heightPtr)
        *heightPtr = height;
}

#endif

char *
newtReflowText (char *text, int width, int flexDown, int flexUp,
                int *actualWidth, int *actualHeight)
{
    int min, max;
    int i;
    char *result;
    int minbad, minbadwidth, howbad;
    char *expandedText;

    expandedText = expandTabs (text);

    if (flexDown || flexUp)
    {
        min = width - flexDown;
        max = width + flexUp;

        minbad = -1;
        minbadwidth = width;

        for (i = min; i <= max; i++)
        {
            doReflow (expandedText, NULL, i, &howbad, NULL);

            if (minbad == -1 || howbad < minbad)
            {
                minbad = howbad;
                minbadwidth = i;
            }
        }

        width = minbadwidth;
    }

    doReflow (expandedText, &result, width, NULL, actualHeight);
    free (expandedText);
    if (actualWidth)
        *actualWidth = width;
    return result;
}

void
newtTextboxSetText (newtComponent co, const char *text)
{
    const char *start, *end;
    struct textbox *tb = co->data;
    char *reflowed, *expanded;
    int badness, height;

    if (tb->lines)
    {
        free (tb->lines);
        tb->linesAlloced = tb->numLines = 0;
    }

    expanded = expandTabs (text);

    if (tb->doWrap)
    {
        doReflow (expanded, &reflowed, tb->textWidth, &badness, &height);
        free (expanded);
        expanded = reflowed;
    }

    for (start = expanded; *start; start++)
        if (*start == '\n')
            tb->linesAlloced++;

    /* This ++ leaves room for an ending line w/o a \n */
    tb->linesAlloced++;
    tb->lines = malloc (sizeof (char *) * tb->linesAlloced);

    start = expanded;
    while ((end = strchr (start, '\n')))
    {
        addLine (co, start, end - start);
        start = end + 1;
    }

    if (*start)
        addLine (co, start, strlen (start));

    free (expanded);
}

/* This assumes the buffer is allocated properly! */
static void
addLine (newtComponent co, const char *s, int len)
{
    struct textbox *tb = co->data;
    const char *t;
    wchar_t wc;
    int k, w, pos = 0;

    for (t = s;
         (k = mbtowc (&wc, t, len)) != -1
         && wc
         && (w = wcwidth (wc)) != -1
         && pos + w <= tb->textWidth; t += k, len -= k, pos += w)
        ;
    tb->lines[tb->numLines] = malloc (t - s + tb->textWidth - pos + 1);
    memcpy (tb->lines[tb->numLines], s, t - s);
    memset (tb->lines[tb->numLines] + (t - s), ' ', tb->textWidth - pos);
    tb->lines[tb->numLines++][t - s + tb->textWidth - pos] = '\0';
}

static void
textboxDraw (newtComponent c)
{
    int i;
    struct textbox *tb = c->data;
    int size;

    if (tb->sb)
    {
        size = tb->numLines - c->height;
        newtScrollbarSet (tb->sb, tb->topLine, size ? size : 0);
        tb->sb->ops->draw (tb->sb);
    }

    SLsmg_set_color (NEWT_COLORSET_TEXTBOX);

    for (i = 0; (i + tb->topLine) < tb->numLines && i < c->height; i++)
    {
        newtGotorc (c->top + i, c->left);
        SLsmg_write_string (tb->lines[i + tb->topLine]);
    }
}

static struct eventResult
textboxEvent (newtComponent co, struct event ev)
{
    struct textbox *tb = co->data;
    struct eventResult er;

    er.result = ER_IGNORED;

    if (ev.when == EV_EARLY && ev.event == EV_KEYPRESS && tb->sb)
    {
        switch (ev.u.key)
        {
            case NEWT_KEY_UP:
                if (tb->topLine)
                    tb->topLine--;
                textboxDraw (co);
                er.result = ER_SWALLOWED;
                break;

            case NEWT_KEY_DOWN:
                if (tb->topLine < (tb->numLines - co->height))
                    tb->topLine++;
                textboxDraw (co);
                er.result = ER_SWALLOWED;
                break;

            case NEWT_KEY_PGDN:
                tb->topLine += co->height;
                if (tb->topLine > (tb->numLines - co->height))
                {
                    tb->topLine = tb->numLines - co->height;
                    if (tb->topLine < 0)
                        tb->topLine = 0;
                }
                textboxDraw (co);
                er.result = ER_SWALLOWED;
                break;

            case NEWT_KEY_PGUP:
                tb->topLine -= co->height;
                if (tb->topLine < 0)
                    tb->topLine = 0;
                textboxDraw (co);
                er.result = ER_SWALLOWED;
                break;
        }
    }
    if (ev.when == EV_EARLY && ev.event == EV_MOUSE && tb->sb)
    {
        /* Top scroll arrow */
        if (ev.u.mouse.x == co->width && ev.u.mouse.y == co->top)
        {
            if (tb->topLine)
                tb->topLine--;
            textboxDraw (co);

            er.result = ER_SWALLOWED;
        }
        /* Bottom scroll arrow */
        if (ev.u.mouse.x == co->width &&
            ev.u.mouse.y == co->top + co->height - 1)
        {
            if (tb->topLine < (tb->numLines - co->height))
                tb->topLine++;
            textboxDraw (co);

            er.result = ER_SWALLOWED;
        }
    }
    return er;
}

static void
textboxDestroy (newtComponent co)
{
    int i;
    struct textbox *tb = co->data;

    for (i = 0; i < tb->numLines; i++)
        free (tb->lines[i]);
    free (tb->lines);
    free (tb);
    free (co);
}
