#!	/bin/sh
# Low mem Boot Disk maker. Enrique Zanardi, June 1998
# This is free software under the GNU General Public License.

. ./common.sh

# Print a usage message and exit if the argument count is wrong.
if [ $# != 6 ]; then
echo "Usage: "$0" revision-extension kernel-package initrd-image blocks debian-version system" 1>&2
	cat 1>&2 << EOF

	revision-extension: the revision extension of the kernel.
	kernel-package: the Debian package containing the kernel.
	initrd-image: a compressed disk image to load in ramdisk and mount as temporary root.
	blocks: the size of the floppy in 1K blocks, use 720, 1200, or 1440.
	debian-version: version of the Debian release.
	system: name of the target system.
EOF

	exit -1
fi

revext=$1

# Set this to the location of the package archive.
archive=$2

# Set this to the location of the initrd filesystem image
initrdimage=$3

# Set to the number of blocks per floppy.
blocks=$4

#Set this to the Debian version
debianversion=$5

# Set this to the system
system=$6

# Set this to the block device for the floppy disk.
floppy=$(tempfile -p fimg)

mnt=${tmpdir}/boot-floppies/mnt.lowmem

loopdevice=/dev/loop3

scripts=`pwd`/scripts/lowmem

# Make sure the kernel package is available.
if [ ! -f $archive ]; then
	echo "Can't find $archive" 1>&2
	exit -1
fi

# Try to unmount anything that should not be mounted. Aborted runs of this
# script may have left things mounted.
(umount /mnt; umount $mnt; umount $floppy; true)2>/dev/null >/dev/null

make_tmpdir $mnt

# Zero the entire disk, so that when I compress the raw disk image,
# unused blocks will compress well.
dd if=/dev/zero of=$floppy bs=1k count=$blocks

# Get the kernel version from its package name.
KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/_.*//'`
ARCHITECTURE=${arch}
DEBIAN_KERNEL_IMAGE=`basename $archive .deb | sed -e s/_$ARCHITECTURE\$//`

# Make the filesystem using the loop device, so that the floppy image file
# won't be truncated if the filesystem is shorter than the image file.

case "$arch" in
    alpha)
	echo "No support (yet) for Alpha!"
	exit 1
    ;;
    arm)
	echo "No support (yet) for ARM!"
	exit 1
    ;;
    i386)
	gunzip</usr/lib/syslinux/img$blocks""k.gz > $floppy
	nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
    ;;
    m68k)
	  echo "Experimental support (yet) for m68k!"
          case "$system" in
              amiga|atari)
		  mkdosfs -r 112 -F 12 $floppy $blocks
		  nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
              ;;
	      mac)
		  mkdosfs -r 112 -F 12 $floppy $blocks	# just for resc1440; 
		  nlfmt="recode $charset..applemac"    # Mac newline convention
	      ;;
              mvme|bvme)
	          echo "No support (yet) for VME!"
	          exit 1
              ;;
          esac
    ;;
    mips)
	echo "No support (yet) for Mips!"
	exit 1
    ;;
    powerpc)
	echo "No support (yet) for PowerPC!"
	exit 1
    ;;
    sparc)
	echo "No support (yet) for Sparc!"
	exit 1
    ;;
esac

# mount the disk image to install additional files
mount -o fat=12,loop=$loopdevice -t msdos $floppy $mnt

# For atari copy bootloader and linux on disk
if [ "$system" = atari ]; then
	mkdir $mnt/auto
	# hardcoded bootargs; kernel is in \, bootstrap in \auto
	cat <<EOF > $mnt/auto/bootargs
-s -k \linux -r \lowmemrd.bin root=/dev/ram init=/linuxrc load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/auto/bootstra.prg
	# both bootstrap and kernel/ramdisk in \
	cat <<EOF > $mnt/bootargs
-s -k linux -r lowmemrd.bin root=/dev/ram init=/linuxrc load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/bootstra.ttp
fi

cp linux$revext $mnt/linux

cp $initrdimage $mnt/lowmemrd.bin

# Install the greeting and help messages (f6.txt f7.txt f8.txt contain
#   no information at all, so don't copy them to save some space.
charset=latin1
for i in debian.txt readme.txt f1.txt f10.txt f2.txt f3.txt \
                f4.txt f5.txt f9.txt ; do
        cat $scripts/$i | \
                m4 -P -D__kernel_version__=$KERNEL_VERSION \
                                -D__kernel_image__=$DEBIAN_KERNEL_IMAGE \
                                -D__debianversion__=$debianversion \
                        | $nlfmt > $mnt/$i
done

cp $scripts/syslinux.cfg $mnt

echo lowmem >$mnt/type.txt

umount $mnt
rmdir $mnt

(rm -f -r ${tmpdir}/boot-floppies/extract-tmp-$$ lowmem.bin;true)
mv $floppy lowmem$revext.bin
ls -l lowmem$revext.bin
exit 0
