/*
 * Created on 19-Apr-2004
 * Created by Paul Gardner
 * Copyright (C) 2004, 2005, 2006 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SAS au capital de 46,603.30 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package org.gudy.azureus2.plugins.ui;

import java.io.File;
import java.net.URL;

import org.gudy.azureus2.plugins.PluginView;
import org.gudy.azureus2.plugins.torrent.Torrent;
import org.gudy.azureus2.plugins.ui.model.*;
import org.gudy.azureus2.plugins.ui.SWT.SWTManager;
import org.gudy.azureus2.plugins.ui.menus.MenuManager;
import org.gudy.azureus2.plugins.ui.tables.TableManager;

/**
 * Management tools for the user interface.
 * 
 * @author parg
 */
public interface 
UIManager 
{
		/**
		 * Gets a basic plugin view model that supports simple plugin requirements
		 * After getting the model create the view using createPluginView
		 * @param name name
		 * @return BasicPluginViewModel
		 * @deprecated Use createBasicPluginViewModel 
		 */
	
	public BasicPluginViewModel
	getBasicPluginViewModel(
		String			name );

		/**
		 * Creates a view from the model. It is then necessary to add it to the plugin
		 * as any other PluginView
		 * @param model
		 * @return PluginView
		 * @deprecated Use createBasicPluginViewModel
		 */
	
	public PluginView
	createPluginView(
		PluginViewModel	model );
	
	/**
	 * 
	 * @param section_name
	 * @return BasicPluginConfigModel
	 * @since 2.1.0.0
	 */
	public BasicPluginConfigModel
	createBasicPluginConfigModel(
		String		section_name );
	

	/**
	 * Creates a basic plugin view model and adds it to the plugin in one step.
	 * 
	 * @param parent_section
	 * @param section_name  see {@link org.gudy.azureus2.plugins.ui.config.ConfigSection}.SECTION_*
	 * @return BasicPluginConfigModel
	 * @since 2.1.0.0
	 */
	public BasicPluginConfigModel
	createBasicPluginConfigModel(
		String		parent_section,
		String		section_name );

	/**
	 * 
	 * @return
	 * @since 2.5.0.1
	 */
	public PluginConfigModel[] getPluginConfigModels();

	/**
	 * Creates a basic plugin view model and adds it to the plugin in one step.
	 * view is placed inside the plugins section of the configuration page.
	 * 
	 * @param name The name for the view (this should be localised text, rather
	 *     than a message string).
	 * @return BasicPluginViewModel
	 * @since 2.1.0.2
	 */
	public BasicPluginViewModel
	createBasicPluginViewModel(
		String			name );
	
	/**
	 * 
	 * @param data
	 * @throws UIException
	 * @since 2.1.0.0
	 */
	public void
	copyToClipBoard(
		String		data )
	
		throws UIException;

	/**
	 * Retrieve the Table Manager
	 * 
	 * @return Table management functions
	 * @since 2.1.0.0
	 */
	public TableManager getTableManager();

	/**
	 * 
	 * @param title_resource
	 * @param message_resource
	 * @param contents
	 * @since 2.3.0.5
	 */
	public void
	showTextMessage(
		String		title_resource,
		String		message_resource,
		String		contents );
	
	/**
	 * 
	 * @param title_resource
	 * @param message
	 * @param message_map - see UIManagerEvent.MT_x
	 * @return selected value
	 * @since 3.0.5.3
	 */
	
	public long
	showMessageBox(
		String					title_resource,
		String					message_resource,
		long					message_map );
	
		/**
		 * @since 2.3.0.6
		 * @param url
		 */
	
	public void
	openURL(
		URL		url )
	
		throws UIException;
	
		/**
		 * 
		 * @param torrent
		 * @since 3.0.5.3
		 */
	
	public void
	openTorrent(
		Torrent		torrent );
	
	/** Retrieve a class of SWT specific functions 
	 * 
	 * @deprecated 
	 * @return SWTManager
	 * 
	 * @since 2.1.0.0
	 */
	
	public SWTManager getSWTManager();
  
	
	/**
	 * Open Config View to the section specified
	 * 
	 * @param sectionID ID of section to open to.
	 *         {@link org.gudy.azureus2.plugins.ui.config.ConfigSection}.SECTION_* constants
	 * @return true-Section opened; false-Section invalid or UI does not support config views
	 * 
	 * @since 2.3.0.7
	 */
	public boolean showConfigSection(String sectionID);

	/**
	 * Retrieve the menu manager.
	 * 
	 * @return Menu management functions
	 * @since 3.0.0.7
	 */
    public MenuManager getMenuManager();
  
		/**
		 * UIs should support generic UI-agnostic views such as the basic config model by default. The can also
		 * expose a UI-specific plugin interface to plugins via the UIInstance (see interface for details).
		 * To get access to this it is necessary to use the UIManagerListener 
		 */
	
	/**
	 * attach a new UI
	 *   
	 * @param factory
	 * @throws UIException
	 * 
	 * @since 2.3.0.5
	 */
	
	public void
	attachUI(
		UIInstanceFactory		factory )
	
		throws UIException;
	
	/**
	 * detach a UI - can fail if the UI doesn't support detaching
	 * 
	 * @param factory
	 * @throws UIException
	 * 
	 * @since 2.3.0.5
	 */
	
	public void
	detachUI(
		UIInstanceFactory		factory )
	
		throws UIException;

	/**
	 * 
	 * @param listener
	 * 
	 * @since 2.3.0.5
	 */
  	public void
  	addUIListener(
  		UIManagerListener listener );

	/**
	 * 
	 * @param listener
	 * 
	 * @since 2.3.0.5
	 */
 	public void
  	removeUIListener(
  		UIManagerListener listener );
 	
	/**
	 * 
	 * @param listener
	 * 
	 * @since 2.3.0.5
	 */
 	public void
  	addUIEventListener(
  		UIManagerEventListener listener );
  	
	/**
	 * 
	 * @param listener
	 * 
	 * @since 2.3.0.5
	 */
 	public void
  	removeUIEventListener(
  		UIManagerEventListener listener );
 	
 	/**
 	 * Returns <tt>true</tt> if there any user interfaces currently attached to
 	 * Azureus.
 	 * 
 	 * Note that this value may change over time, and that at the point when a
 	 * plugin is being initialised, there is no guarantee over whether if the
 	 * user interfaces will have been created yet or not.
 	 * 
 	 * If you want to monitor what user interfaces are being attached / detached
 	 * from Azureus, you should add a <tt>UIListener</tt> to this object.
 	 * 
 	 * @since 3.0.0.7
 	 */
 	public boolean hasUIInstances();
 	
 	/**
 	 * Returns an array of all currently attached user interfaces.
 	 * 
 	 * Note that this value may change over time, and that at the point when a
 	 * plugin is being initialised, there is no guarantee over whether if the
 	 * user interfaces will have been created yet or not.
 	 * 
 	 * If you want to monitor what user interfaces are being attached / detached
 	 * from Azureus, you should add a <tt>UIListener</tt> to this object.
 	 * 
 	 * @since 3.0.0.7
 	 */
 	public UIInstance[] getUIInstances();
 	
 	/**
 	 * Retrieves a {@link UIInputReceiver} from any interface available, or
 	 * returns <tt>null</tt> if one is not available. This is a convenience
 	 * method to allow you to grab an instance without having to iterate over
 	 * any attached interfaces.
 	 * 
 	 * @see UIInstance#getInputReceiver()
 	 * @since 3.0.5.3
 	 */
 	public UIInputReceiver getInputReceiver();

 	/**
 	 * Retrieves a {@link UIMessage} from any interface available, or
 	 * returns <tt>null</tt> if one is not available. This is a convenience
 	 * method to allow you to grab an instance without having to iterate over
 	 * any attached interfaces.
 	 * 
 	 * @see UIInstance#createMessage()
 	 * @since 3.0.5.3
 	 */
 	public UIMessage createMessage();
 	
 	/**
 	 * Opens up the file using the associated application. 
 	 * 
 	 * @param file The file to open.
 	 * @since 3.0.5.3
 	 */
 	public void openFile(File file);

 	/**
 	 * Shows the file in a file explorer application in its parent folder. 
 	 * 
 	 * @param file The file to show.
 	 * @since 3.0.5.3
 	 */
 	public void showFile(File file);
 	
}
