/*  -*- Mode: C -*-  */

/* snprintfv.h --- printf clone for argv arrays */

/* Author:	       Gary V. Vaughan <gvv@techie.com>
 * Maintainer:	       Gary V. Vaughan <gvv@techie.com>
 * Created:	       Fri Nov 13 16:51:38 1998
 * Last Modified:      Wed Jul 14 15:08:37 1999
 *            by:      Gary V. Vaughan <gvv@techie.com>
 * ---------------------------------------------------------------------
 * @(#) $Id: snprintfv.in,v 1.3 2000/07/27 15:46:01 bkorb Exp $
 * ---------------------------------------------------------------------
 */

/* Copyright (C) 1998, 1999 Gary V. Vaughan */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * As a special exception to the GNU General Public License, if you
 * distribute this file as part of a program that also links with and
 * uses the libopts library from AutoGen, you may include it under
 * the same distribution terms used by the libopts library.
 */

/* Code: */

#ifndef SNPRINTFV_H
#define SNPRINTFV_H 1

#include <snprintfv/compat.h>
#include <snprintfv/stream.h>

/**
 * SNV_UNLIMITED:
 * Used to denote that there is no upper limit to the number of characters
 * that can safely be written to a stream.
 **/
#define SNV_UNLIMITED		~(0L)

#ifdef __cplusplus
extern "C" {
#endif

/**
 * stream_printfv:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_printfv PARAMS((STREAM *stream, const char *format, snv_constpointer const args[]));

/**
 * stream_vprintf:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to @stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_vprintf PARAMS((STREAM *stream, const char *format, va_list ap));

/**
 * stream_printf:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_printf SNV_GNUC_PRINTF((STREAM *stream, const char *format, ...), 2, 3);


/**
 * fdputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a file descriptor.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream, or -1 in case of
 * an error (errno will be set to indicate the type of error).
 **/
extern int fdputc PARAMS((int ch, STREAM *stream));

/**
 * snv_dprintf:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_dprintf SNV_GNUC_PRINTF((int fd, const char *format, ...), 2, 3);


/**
 * snv_vdprintf:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vdprintf PARAMS((int fd, const char *format, va_list ap));

/**
 * dprintfv:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int dprintfv PARAMS((int fd, const char *format, snv_constpointer const args[]));

/**
 * fileputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a FILE*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
extern int fileputc PARAMS((int ch, STREAM *stream));

/**
 * snv_printf:  
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_printf SNV_GNUC_PRINTF((const char *format, ...), 1, 2);


/**
 * snv_vprintf:  
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vprintf PARAMS((const char *format, va_list ap));

/**
 * printfv:  
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to the string @format,
 * and write the result to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int printfv PARAMS((const char *format, snv_constpointer const args[]));

/**
 * snv_fprintf:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_fprintf SNV_GNUC_PRINTF((FILE *file, const char *format, ...), 2, 3);


/**
 * snv_vfprintf:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vfprintf PARAMS((FILE *file, const char *format, va_list ap));

/**
 * fprintfv:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @file.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int fprintfv PARAMS((FILE *file, const char *format, snv_constpointer const args[]));

/**
 * bufputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a char buffer.
 * 
 * Return value:
 * The value of @ch that has been put in @stream.
 **/
extern int bufputc PARAMS((int ch, STREAM *stream));

/**
 * snv_sprintf:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_sprintf SNV_GNUC_PRINTF((char buffer[], const char *format, ...), 2, 3);


/**
 * snv_vsprintf:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vsprintf PARAMS((char buffer[], const char *format, va_list ap));

/**
 * sprintfv:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int sprintfv PARAMS((char buffer[], const char *format, snv_constpointer const args[]));

/**
 * snv_snprintf:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_snprintf SNV_GNUC_PRINTF((char buffer[], unsigned long limit, const char *format, ...), 3, 4);


/**
 * snv_vsnprintf:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vsnprintf PARAMS((char buffer[], unsigned long limit, const char *format, va_list ap));

/**
 * snprintfv:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snprintfv PARAMS((char buffer[], unsigned long limit, const char *format, snv_constpointer const args[]));

/**
 * filputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a filament*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
extern int filputc PARAMS((int ch, STREAM *stream));

/**
 * snv_asprintf:  
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to an internally allocated buffer.
 * 
 * Return value:
 * The address of the memory which holds the formatted string is
 * returned (and should be freed by the caller), unless there is
 * an error, when NULL is returned.
 **/
extern char* snv_asprintf SNV_GNUC_PRINTF((const char *format, ...), 1, 2);


/**
 * snv_vasprintf:  
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to an internally allocated buffer.
 * 
 * Return value:
 * The address of the memory which holds the formatted string is
 * returned (and should be freed by the caller), unless there is
 * an error, when NULL is returned.
 **/
extern char* snv_vasprintf PARAMS((const char *format, va_list ap));

/**
 * asprintfv:  
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to an internally allocated buffer.
 * 
 * Return value:
 * The address of the memory which holds the formatted string is
 * returned (and should be freed by the caller), unless there is
 * an error, when NULL is returned.
 **/
extern char* asprintfv PARAMS((const char *format, snv_constpointer const args[]));


#ifdef __cplusplus
}
#endif

/* If you don't want to use snprintfv functions for *all* of your string
   formatting API, then define COMPILING_SNPRINTFV_C and use the snv_
   prefix for the entry points below. */

#ifndef COMPILING_SNPRINTFV_C
#define printf		snv_printf
#define vprintf		snv_vprintf
#define dprintf		snv_dprintf
#define vdprintf	snv_vdprintf
#define fprintf		snv_fprintf
#define vfprintf	snv_vfprintf
#define sprintf		snv_sprintf
#define vsprintf	snv_vsprintf
#define snprintf	snv_snprintf
#define vsnprintf	snv_vsnprintf
#define asprintf	snv_asprintf
#define vasprintf	snv_vasprintf
#endif /* !COMPILING_SNPRINTFV_C */

#endif /* SNPRINTFV_H */

/* snprintfv.h ends here */
