------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                    A S I S _ U L . U T I L I T I E S                     --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2007-2008, AdaCore                     --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

with Asis.Compilation_Units; use Asis.Compilation_Units;
with Asis.Declarations;      use Asis.Declarations;
with Asis.Definitions;       use Asis.Definitions;
with Asis.Elements;          use Asis.Elements;
with Asis.Expressions;       use Asis.Expressions;
with Asis.Extensions;        use Asis.Extensions;
with Asis.Iterator;          use Asis.Iterator;
with Asis.Statements;        use Asis.Statements;
with Asis.Text;              use Asis.Text;

with Asis.Set_Get;           use Asis.Set_Get;

with Atree;                  use Atree;
with Einfo;                  use Einfo;
with Sinfo;                  use Sinfo;
with Types;                  use Types;

package body ASIS_UL.Utilities is

   -----------------------
   -- Local subprograms --
   -----------------------

   function Contains
     (Outer : Element;
      Inner : Element)
      return  Boolean;
   --  Checks if Outer contains Inner. At the moment this function is
   --  implemented for explicit elements only, or, more precisely, for the
   --  situation when for both arguments Is_Text_Available. If at least one of
   --  the parameters does not have a text properties available, False is
   --  returned.
   --
   --  Note, that the current implementation assumes that both arguments are
   --  from the same Compilation_Unit!

   function Is_Static_Subtype (E : Element) return Boolean;
   --  Checks if the argument is a static subtype indication or a static
   --  A_Discrete_Subtype_Definition. This function is supposed to be applied
   --  to discrete subtype indications (and definitions) of the form
   --  subtype_mark [constraint].

   ----------------------------
   -- Adds_New_Nesting_Level --
   ----------------------------

   function Adds_New_Nesting_Level
     (El_Kind : Flat_Element_Kinds)
      return    Boolean
   is
      Result : Boolean := False;
   begin

      case El_Kind is
         when A_Procedure_Body_Declaration       |
              A_Function_Body_Declaration        |
              A_Package_Declaration              |
              A_Package_Body_Declaration         |
              A_Task_Body_Declaration            |
              A_Protected_Body_Declaration       |
              An_Entry_Body_Declaration          |
              A_Generic_Package_Declaration      |
              An_If_Statement                    |
              A_Case_Statement                   |
              A_Loop_Statement                   |
              A_While_Loop_Statement             |
              A_For_Loop_Statement               |
              A_Block_Statement                  |
              An_Accept_Statement                |
              A_Selective_Accept_Statement       |
              A_Timed_Entry_Call_Statement       |
              A_Conditional_Entry_Call_Statement |
              An_Asynchronous_Select_Statement   =>
            Result := True;
         when others =>
            null;
      end case;

      return Result;
   end Adds_New_Nesting_Level;

   --------------------------------
   -- Can_Have_Elaboration_Calls --
   --------------------------------

   function Can_Have_Elaboration_Calls (El : Asis.Element) return Boolean is
      Arg_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (El);
      Tmp_El   :          Asis.Element       := Nil_Element;
      Result   :          Boolean            := False;
   begin

      --  Note that we do not have to care about evaluating default paramenets
      --  for IN formal objects for package instantiations- the corresponding
      --  default expression will be processed as a part of processing the
      --  expanded generic. But we have to care about subprogram
      --  instantiations

      case Arg_Kind is
         when A_Variable_Declaration =>
            Result := Is_Nil (Initialization_Expression (El));

         when A_Function_Call            |
              A_Procedure_Call_Statement =>

            --  Calls to operator functions and to attribute subprograms cannot
            --  evaluate any default expressions:

            if Arg_Kind = A_Procedure_Call_Statement then
               Tmp_El := Called_Name (El);
            elsif Arg_Kind = A_Function_Call then
               Tmp_El := Prefix (El);
            else
               Tmp_El := Nil_Element;
            end if;

            if Expression_Kind (Tmp_El) = A_Selected_Component then
               Tmp_El := Selector (Tmp_El);
            end if;

            Result := not (Expression_Kind (Tmp_El) = An_Attribute_Reference
                  or else
                           Expression_Kind (Tmp_El) = An_Operator_Symbol);

         when A_Procedure_Instantiation  |
              A_Function_Instantiation   |
              An_Allocation_From_Subtype |
              An_Entry_Call_Statement    =>

            Result := True;
         when others =>
            null;
      end case;

      return Result;
   end Can_Have_Elaboration_Calls;

   --------------
   -- Contains --
   --------------

   function Contains
     (Outer : Element;
      Inner : Element)
      return  Boolean
   is
      Outer_Span : Span;
      Inner_Span : Span;
      Result     : Boolean := False;
   begin

      if Is_Text_Available (Outer) and then
         Is_Text_Available (Inner)
      then
         Outer_Span := Element_Span (Outer);
         Inner_Span := Element_Span (Inner);

         if (Outer_Span.First_Line < Inner_Span.First_Line
            or else
             (Outer_Span.First_Line = Inner_Span.First_Line and then
              Outer_Span.First_Column <= Inner_Span.First_Column))

         and then

            (Outer_Span.Last_Line  > Inner_Span.Last_Line
            or else
             (Outer_Span.Last_Line = Inner_Span.Last_Line and then
              Outer_Span.Last_Column >= Inner_Span.Last_Column))
         then
            Result := True;
         end if;

      end if;

      return Result;
   end Contains;

   ---------------------------------
   -- Does_Not_Add_New_Components --
   ---------------------------------

   function Does_Not_Add_New_Components (El : Asis.Element) return Boolean is
      Result : Boolean := False;
      Tmp    : Asis.Element;
   begin

      if Asis.Elements.Type_Kind (El) =
           A_Derived_Record_Extension_Definition
      then

         Tmp := Asis.Definitions.Record_Definition (El);

         if Definition_Kind (Tmp) = A_Null_Record_Definition then
            Result := True;
         elsif Record_Components (Tmp)'Length = 1
             and then
               Definition_Kind (Record_Components (Tmp) (1)) = A_Null_Component
         then
            Result := True;
         end if;

      end if;

      return Result;
   end Does_Not_Add_New_Components;

   ----------------
   -- First_Name --
   ----------------

   function First_Name (Dcl : Asis.Element) return Asis.Element is
      Name_List : constant Asis.Element_List := Names (Dcl);
   begin
      return Name_List (Name_List'First);
   end First_Name;

   -------------------------
   -- Get_Call_Parameters --
   -------------------------

   function Get_Call_Parameters
     (Call       : Asis.Element;
      Normalized : Boolean := False)
      return       Asis.Element_List
   is
   begin

      if Expression_Kind (Call) = A_Function_Call then
         return Function_Call_Parameters (Call, Normalized);
      else
         return Call_Statement_Parameters (Call, Normalized);
      end if;

   end Get_Call_Parameters;

   ---------------------------
   -- Get_Subtype_Structure --
   ---------------------------

   function Get_Subtype_Structure (Def : Asis.Element) return Asis.Element is
      Result   : Asis.Element := Def;
   begin
      Result := Asis.Definitions.Subtype_Mark  (Result);
      Result := Normalize_Reference            (Result);
      Result := Corresponding_Name_Declaration (Result);
      Result := Get_Type_Structure             (Result);

      return Result;
   end Get_Subtype_Structure;

   ------------------------
   -- Get_Type_Structure --
   ------------------------

   function Get_Type_Structure (Decl : Asis.Element) return Asis.Element is
      Arg_Kind : constant Declaration_Kinds := Declaration_Kind (Decl);
      Result   :          Asis.Element      := Decl;
      Tmp      :          Asis.Element;
   begin
      --  Should be replaced with regular kind check!
      pragma Assert
        (Arg_Kind in
           An_Ordinary_Type_Declaration .. A_Subtype_Declaration
              or else
         Arg_Kind = A_Formal_Type_Declaration);

      --  We cannot use Asis,Definitions queries Corresponding_Root_Type or
      --  Corresponding_Type_Structure - they unwind derivations, so we can
      --  miss extension components

      case Arg_Kind is
         when A_Task_Type_Declaration              |
              A_Protected_Type_Declaration         |
              An_Incomplete_Type_Declaration       |
              A_Tagged_Incomplete_Type_Declaration |
              A_Formal_Type_Declaration            =>
            null;
         when An_Ordinary_Type_Declaration =>
            Tmp := Type_Declaration_View (Result);

            if Asis.Elements.Type_Kind (Tmp) = A_Derived_Type_Definition
              or else
                (Asis.Elements.Type_Kind (Tmp) =
                   A_Derived_Record_Extension_Definition
               and then
                Does_Not_Add_New_Components (Tmp))
            then
               Result := Parent_Subtype_Indication (Tmp);
               Result := Get_Subtype_Structure     (Result);
            end if;

         when A_Private_Type_Declaration |
              A_Private_Extension_Declaration =>
            Result :=
              Get_Type_Structure (Corresponding_Type_Declaration (Result));

         when A_Subtype_Declaration =>
            Result := Type_Declaration_View (Result);
            Result := Get_Subtype_Structure (Result);
         when others =>
            pragma Assert (False);
            null;

      end case;

      return Result;
   end Get_Type_Structure;

   -------------------------------------
   -- Is_Call_To_Attribute_Subprogram --
   -------------------------------------

   function Is_Call_To_Attribute_Subprogram
     (El   : Asis.Element)
      return Boolean
   is
      Result      : Boolean      := False;
      Call_Prefix : Asis.Element := Nil_Element;
   begin

      case Flat_Element_Kind (El) is
         when A_Procedure_Call_Statement =>
            Call_Prefix := Called_Name (El);

         when A_Function_Call =>

            if Is_Prefix_Call (El) then
               Call_Prefix := Prefix (El);
            end if;

         when others =>
            null;
      end case;

      if Expression_Kind (Call_Prefix) = An_Attribute_Reference then
         Result := True;
      end if;

      return Result;

   end Is_Call_To_Attribute_Subprogram;

   ------------------------
   -- Is_Executable_Body --
   ------------------------

   function Is_Executable_Body (El : Element) return Boolean is
      El_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (El);
      Result  : Boolean                     := False;
   begin

      case El_Kind is

         when A_Procedure_Body_Declaration |
              A_Function_Body_Declaration  |
              A_Task_Body_Declaration      |
              An_Entry_Body_Declaration    =>
            Result := True;
         when A_Package_Body_Declaration =>
            Result := Body_Statements (El)'Length > 0;
         when others =>
            null;
      end case;

      return Result;
   end Is_Executable_Body;

   ----------------------------
   -- Is_Imported_Subprogram --
   ----------------------------

   function Is_Imported_Subprogram (El : Asis.Element) return Boolean is
      Argument : Asis.Element := El;
      Result   : Boolean      := False;
   begin

      if Declaration_Kind (Argument) in
         A_Procedure_Instantiation .. A_Function_Instantiation
      then
         Argument := Normalize_Reference (Generic_Unit_Name (Argument));
         Argument := Corresponding_Name_Declaration (Argument);
      end if;

      if Declaration_Kind (Argument) in
           A_Procedure_Declaration .. A_Function_Declaration
       or else
         Declaration_Kind (Argument) in
           A_Generic_Procedure_Declaration .. A_Generic_Function_Declaration
      then
         Argument := Corresponding_Body (Argument);
         Result   := Pragma_Kind (Argument) = An_Import_Pragma;
      end if;

      return Result;
   end Is_Imported_Subprogram;

   ------------------------
   -- Is_In_Private_Part --
   ------------------------

   function Is_In_Private_Part
     (Decl  : Element;
      Scope : Element)
      return  Boolean
   is
      Result : Boolean := False;
   begin

      if Element_Kind (Decl) = A_Declaration
        and then
         (Declaration_Kind (Scope) = A_Package_Declaration
         or else
          Declaration_Kind (Scope) = A_Generic_Package_Declaration)
      then

         declare
            Private_Dls : constant Asis.Element_List :=
              Private_Part_Declarative_Items (Scope);

            First_Private_Line : Asis.Text.Line_Number_Positive;
            Last_Private_Line  : Asis.Text.Line_Number_Positive;
         begin

            if not Is_Nil (Private_Dls) then

               First_Private_Line :=
                 Element_Span (Private_Dls (Private_Dls'First)).First_Line;
               Last_Private_Line :=
                 Element_Span (Private_Dls (Private_Dls'Last)).Last_Line;

               Result := Element_Span (Decl).First_Line in
                            First_Private_Line .. Last_Private_Line;
            end if;

         end;

      end if;

      return Result;
   end Is_In_Private_Part;

   ------------------------
   -- Is_In_Visible_Part --
   ------------------------

   function Is_In_Visible_Part
     (Decl  : Element;
      Scope : Element)
      return  Boolean
   is
      Result : Boolean := False;
   begin

      if Element_Kind (Decl) = A_Declaration
        and then
         (Declaration_Kind (Scope) = A_Package_Declaration
         or else
          Declaration_Kind (Scope) = A_Generic_Package_Declaration)
      then

         declare
            Visible_Dls : constant Asis.Element_List :=
              Visible_Part_Declarative_Items (Scope);

            First_Public_Line : Asis.Text.Line_Number_Positive;
            Last_Public_Line  : Asis.Text.Line_Number_Positive;
         begin

            if not Is_Nil (Visible_Dls) then

               First_Public_Line :=
                 Element_Span (Visible_Dls (Visible_Dls'First)).First_Line;
               Last_Public_Line :=
                 Element_Span (Visible_Dls (Visible_Dls'Last)).Last_Line;

               Result := Element_Span (Decl).First_Line in
                           First_Public_Line .. Last_Public_Line;
            end if;

         end;

      end if;

      return Result;
   end Is_In_Visible_Part;

   ---------------------------
   -- Is_Indefinite_Subtype --
   ---------------------------

   function Is_Indefinite_Subtype (SM : Asis.Element) return Boolean is
      Result    : Boolean      := False;
      SM_Entity : Entity_Id;
   begin

      if Expression_Kind (SM) = A_Selected_Component
       or else
         Expression_Kind (SM) = An_Identifier
      then
         SM_Entity := Entity (R_Node  (SM));

         pragma Assert (Ekind (SM_Entity) in Einfo.Type_Kind);

         Result := not Is_Constrained (SM_Entity);

      end if;

      return Result;
   end Is_Indefinite_Subtype;

   ---------------------------------
   -- Is_Non_Structural_Statement --
   ---------------------------------

   function Is_Non_Structural_Statement
     (Stmt         : Element;
      Exit_Is_Goto : Boolean := True)
      return         Boolean
   is
      Arg_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (Stmt);

      Result  : Boolean          := False;
      Control : Traverse_Control := Continue;

      Target_Stmt : Element;

      procedure Pre_Operation
        (Element :        Asis.Element;
         Control : in out Traverse_Control;
         State   : in out Boolean);
      --  This procedure does most of the job. It checks if the element being
      --  visited does transfer the control outside Stmt. If this is really so
      --  it sets Result to True and terminates the traversal

      procedure Post_Operation
        (Element :        Asis.Element;
         Control : in out Traverse_Control;
         State   : in out Boolean);

      procedure Check_Statement is new
        Traverse_Element (Boolean, Pre_Operation, Post_Operation);

      procedure Pre_Operation
        (Element :        Asis.Element;
         Control : in out Traverse_Control;
         State   : in out Boolean)
      is
         Arg_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (Element);
      begin

         case Arg_Kind is
            when Flat_Path_Kinds                    |
                 An_If_Statement                    |
                 A_Case_Statement                   |
                 A_Loop_Statement                   |
                 A_While_Loop_Statement             |
                 A_For_Loop_Statement               |
                 A_Block_Statement                  |
                 A_Selective_Accept_Statement       |
                 A_Timed_Entry_Call_Statement       |
                 A_Conditional_Entry_Call_Statement |
                 An_Asynchronous_Select_Statement   =>

               --  We may control transfer inside such a construct. So just
               --  continue...
               null;

            when A_Return_Statement                |
                 A_Raise_Statement                 |
                 A_Terminate_Alternative_Statement =>

               State   := True;
               Control := Terminate_Immediately;

            when An_Exit_Statement =>

               if Exit_Is_Goto then
                  Target_Stmt := Corresponding_Loop_Exited (Element);

                  if not Contains (Outer => Stmt, Inner => Target_Stmt) or else
                     Is_Equal (Stmt, Target_Stmt)
                  then
                     State   := True;
                     Control := Terminate_Immediately;
                  end if;

               end if;

            when A_Goto_Statement =>

               Target_Stmt := Corresponding_Destination_Statement (Element);

               if not Contains (Outer => Stmt, Inner => Target_Stmt) or else
                  Is_Equal (Stmt, Target_Stmt)
               then
                  State   := True;
                  Control := Terminate_Immediately;
               end if;

            when others =>
               --  Nothing interesting inside...
               Control := Abandon_Children;
         end case;

      end Pre_Operation;

      procedure Post_Operation
        (Element :        Asis.Element;
         Control : in out Traverse_Control;
         State   : in out Boolean)
      is
      begin
         pragma Unreferenced (Element);
         pragma Unreferenced (Control);
         pragma Unreferenced (State);
         null;
      end Post_Operation;

   begin

      if Arg_Kind = An_If_Statement                    or else
         Arg_Kind = A_Case_Statement                   or else
         Arg_Kind = A_Loop_Statement                   or else
         Arg_Kind = A_While_Loop_Statement             or else
         Arg_Kind = A_For_Loop_Statement               or else
         Arg_Kind = A_Selective_Accept_Statement       or else
         Arg_Kind = A_Timed_Entry_Call_Statement       or else
         Arg_Kind = A_Conditional_Entry_Call_Statement or else
         Arg_Kind = An_Asynchronous_Select_Statement
      then
         Check_Statement (Stmt, Control, Result);
      end if;

      return Result;
   end Is_Non_Structural_Statement;

   ---------------------
   -- Is_Program_Unit --
   ---------------------

   function Is_Program_Unit (El : Element) return Boolean is
      El_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (El);
      Result  : Boolean                     := False;
   begin

      Result := False
        or else El_Kind = A_Task_Type_Declaration
        or else El_Kind = A_Protected_Type_Declaration
        or else El_Kind = A_Single_Task_Declaration
        or else El_Kind = A_Single_Protected_Declaration
        or else El_Kind = A_Package_Declaration
        or else El_Kind = A_Generic_Package_Declaration
        or else El_Kind = A_Package_Body_Declaration
        or else El_Kind = A_Procedure_Body_Declaration
        or else El_Kind = A_Function_Body_Declaration
        or else El_Kind = A_Task_Body_Declaration
        or else El_Kind = A_Protected_Body_Declaration
        or else El_Kind = An_Entry_Body_Declaration;

      if not Result then

         Result := False
           or else El_Kind = A_Procedure_Declaration
           or else El_Kind = A_Function_Declaration
           or else El_Kind = A_Generic_Procedure_Declaration
           or else El_Kind = A_Generic_Function_Declaration
           or else El_Kind = A_Package_Instantiation
           or else El_Kind = A_Procedure_Instantiation
           or else El_Kind = A_Function_Instantiation
           or else El_Kind = A_Package_Renaming_Declaration
           or else El_Kind = A_Procedure_Renaming_Declaration
           or else El_Kind = A_Function_Renaming_Declaration
           or else El_Kind = A_Generic_Package_Renaming_Declaration
           or else El_Kind = A_Generic_Procedure_Renaming_Declaration
           or else El_Kind = A_Generic_Function_Renaming_Declaration;

         Result :=
           Result and then
           Is_Equal (El, Unit_Declaration (Enclosing_Compilation_Unit (El)));
      end if;

      return Result;

   end Is_Program_Unit;

   ------------------------
   -- Is_RM_Program_Unit --
   ------------------------

   function Is_RM_Program_Unit (El : Element) return Boolean is
      Result  : Boolean                     := False;
      El_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (El);
   begin

      Result := False
        or else El_Kind = A_Task_Type_Declaration
        or else El_Kind = A_Protected_Type_Declaration
        or else El_Kind = A_Single_Task_Declaration
        or else El_Kind = A_Single_Protected_Declaration
        or else El_Kind = A_Procedure_Declaration
        or else El_Kind = A_Function_Declaration
        or else El_Kind = A_Procedure_Body_Declaration
        or else El_Kind = A_Function_Body_Declaration
        or else El_Kind = A_Package_Declaration
        or else El_Kind = A_Package_Body_Declaration
        or else El_Kind = A_Task_Body_Declaration
        or else El_Kind = A_Protected_Body_Declaration
        or else El_Kind = An_Entry_Body_Declaration
        or else El_Kind = A_Procedure_Body_Stub
        or else El_Kind = A_Function_Body_Stub
        or else El_Kind = A_Package_Body_Stub
        or else El_Kind = A_Task_Body_Stub
        or else El_Kind = A_Protected_Body_Stub
        or else El_Kind = A_Generic_Procedure_Declaration
        or else El_Kind = A_Generic_Function_Declaration
        or else El_Kind = A_Generic_Package_Declaration;

      if El_Kind = An_Entry_Declaration then
         Result :=
           Definition_Kind (Enclosing_Element (El)) = A_Protected_Definition;
      end if;

      return Result;

   end Is_RM_Program_Unit;

   ------------------------------
   -- Is_Publically_Accessible --
   ------------------------------

   function Is_Publically_Accessible (Decl : Element) return Boolean is
      Enclosing_CU : constant Asis.Compilation_Unit :=
        Enclosing_Compilation_Unit (Decl);
      Enclosing_Unit : Asis.Element;
      Local_Pkg      : Asis.Element;

      Result : Boolean := False;

   begin

      if Element_Kind (Decl) = A_Declaration
        and then
         (Unit_Kind (Enclosing_CU) = A_Package
          or else
          Unit_Kind (Enclosing_CU) = A_Generic_Package)
        and then
         Unit_Class (Enclosing_CU) = A_Public_Declaration
      then
         Enclosing_Unit := Unit_Declaration (Enclosing_CU);

         Result := Is_In_Visible_Part (Decl, Enclosing_Unit);

         if Result then
            --  Decl could be in the private part of some local package
            Local_Pkg := Enclosing_Element (Decl);

            while not Is_Equal (Local_Pkg, Enclosing_Unit) loop

               if Is_In_Private_Part (Decl, Local_Pkg) then
                  Result := False;
                  exit;
               end if;

               Local_Pkg := Enclosing_Element (Local_Pkg);
            end loop;

         end if;

      end if;

      return Result;
   end Is_Publically_Accessible;

   --------------------
   -- Is_Static_Loop --
   --------------------

   function Is_Static_Loop (Loop_Stmt : Element) return Boolean
   is
      Param_Definition : Element;

      Result : Boolean := False;
   begin

      if Flat_Element_Kind (Loop_Stmt) = A_For_Loop_Statement then

         Param_Definition :=
           Specification_Subtype_Definition
             (For_Loop_Parameter_Specification (Loop_Stmt));

         case Flat_Element_Kind (Param_Definition) is

            when A_Discrete_Subtype_Indication_As_Subtype_Definition =>

               Result := Is_Static_Subtype (Param_Definition);
               --  Is_Static_Subtype (Subtype_Constraint (Param_Definition));

            when A_Discrete_Range_Attribute_Reference_As_Subtype_Definition =>
               Result := Is_Static (Param_Definition);

            when A_Discrete_Simple_Expression_Range_As_Subtype_Definition =>

               Result := Is_Static (Lower_Bound (Param_Definition)) and then
                         Is_Static (Upper_Bound (Param_Definition));

            when others =>
               null;
         end case;

      end if;

      return Result;
   end Is_Static_Loop;

   -----------------------
   -- Is_Static_Subtype --
   -----------------------

   function Is_Static_Subtype (E : Element) return Boolean is
      Result   : Boolean                     := False;
      Arg_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (E);

      Def_Name    : Element;
      Type_Entity : Entity_Id;
      Tmp         : Element;
   begin
      --  Note, that this NOT an ASIS secondary query, some routines from
      --  Einfo are used.

      --  First, return False for any non-expected or definitely non-static
      --  result

      if not (Arg_Kind = A_Subtype_Indication          or else
              Arg_Kind = A_Discrete_Subtype_Indication or else
              Arg_Kind = A_Discrete_Subtype_Indication_As_Subtype_Definition)
      then
         return False;
      end if;

      Tmp := Asis.Definitions.Subtype_Mark (E);

      if Flat_Element_Kind (Tmp) = A_Selected_Component then
         Tmp := Selector (Tmp);
      end if;

      Def_Name    := Corresponding_Name_Definition (Tmp);
      Type_Entity := Node (Def_Name);

      if Is_Non_Static_Subtype (Type_Entity) or else
         Ekind (Type_Entity) not in Discrete_Kind
      then
         return False;
      end if;

      --  If we are here, we are sure that we are processing some discrete
      --  subtype indication

      Tmp := Subtype_Constraint (E);

      if not Is_Nil (Tmp) then

         if Flat_Element_Kind (Tmp) = A_Range_Attribute_Reference then
            Result := Is_Static (Tmp);
         else
            Result := Is_Static (Lower_Bound (Tmp)) and then
                      Is_Static (Upper_Bound (Tmp));
         end if;

         if not Result then
            --  The constraint is not static. No chance to be a static
            --  subtype...
            return False;
         end if;

      end if;

      --  If we are here, the constraint is either absent or static. So,
      --  checking the subtype mark

      Tmp := Type_Declaration_View (Enclosing_Element (Def_Name));

      if Flat_Element_Kind (Tmp) = A_Subtype_Indication then
         Result := Is_Static_Subtype (Tmp);
      else
         --  that is, here we have a type definition

         case Flat_Element_Kind (Tmp) is

            when A_Derived_Type_Definition =>
               Result := Is_Static_Subtype (Parent_Subtype_Indication (Tmp));

            when An_Enumeration_Type_Definition   |
                 A_Signed_Integer_Type_Definition |
                 A_Modular_Type_Definition =>
               Result := True;

            when others =>
               Result := False;
         end case;

      end if;

      return Result;
   end Is_Static_Subtype;

   -------------------------
   -- Normalize_Reference --
   -------------------------

   function Normalize_Reference (Ref : Asis.Element) return Asis.Element is
      Result : Asis.Element := Ref;
   begin
      case Expression_Kind (Ref) is
         when A_Selected_Component =>
            Result := Selector (Ref);
         when An_Attribute_Reference =>
            Result := Normalize_Reference (Prefix (Ref));
         when others =>
            null;
      end case;

      return Result;
   end Normalize_Reference;

end ASIS_UL.Utilities;
