/* $Id: ArkCollision.h,v 1.13 2003/03/09 21:51:00 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_COLLISION_H
#define ARK_COLLISION_H

#include <Ark/Ark.h>
#include <Ark/ArkRay.h>
#include <Ark/ArkFactory.h>
#include <Ark/ArkModelState.h>

namespace Ark
{

   class Entity;

   /**
    * This class  stores the results of a collision between a bounding
    * box and another object (model or world), or the result of a raytracing.
    */

   class ARK_DLL_API Collision
   {
      public:
	 /**
	  * The collision flags. They are used as bitmask in m_Flags to specify
	  * which data are present (have been computed : POSITION, PLANE), to
	  * tell if a collision occured (COLLIDE), and what the ray/bbox hit
	  * (WORLD/ENTITY).
	  * They are used to call RayTrace and TestCollision to tell which
	  * infos are  needed, and if only one (the first) collision should
	  * be given (ONESHOT).
	  *
	  * POTENTIAL has a special meaning. When this flag is set, the
	  * collision didn't really occur, but the two entities are very close
	  * to each other (test done with big bounding boxes).
	  */
	 enum Flags
	 {
	    COLLIDE  = (1<<0),
	    WORLD    = (1<<1),
	    ENTITY   = (1<<2),
	    TYPEMASK = ENTITY|WORLD,
	    POSITION = (1<<3),
	    PLANE    = (1<<4),
	    ONESHOT  = (1<<5),
	    POTENTIAL = (1<<6)
	 };
	 
      public:
	 /// Initialise this collision
	 Collision ()
	 {m_Flags = 0;}
	 
	 /// Get the type of the collision. (with ground or with an entity ?)
	 int GetType () const
	 {return m_Flags & TYPEMASK;}
      
	 /// The entity if this collision is object-object.
	 Entity *m_Entity;
	    
	 /// The ground if this collision is world-object.
	 Material *m_Material;
	 
	 /** The plane of the collision. */
	 Plane m_Plane;
	 
	 /** The position the collision occured at. */
	 Vector3 m_Pos;
	 
	 /// Collision flags (or'ed Flags values)
	 int m_Flags;
   };

   struct ARK_DLL_API ColTriangle
   {
	 Vector3 m_Coords[3];
	 Material *m_Material;
   };

   /**
    * This class is used to store the results of a collision between two
    * models.
    */
   struct ARK_DLL_API ColPair
   {
	 /**
	  * The triangle at which the collision occured (in the first
	  * model).
	  */
	 ColTriangle m_Triangle0;

	 /**
	  * The bodypart (ie name of the submodel) at which the collision
	  * occured, in the first model.
	  */
	 String m_BodyPart0;

	 /**
	  * The triangle at which the collision occured (in the second
	  * model).
	  */
	 ColTriangle m_Triangle1;

	 /**
	  * The bodypart (ie name of the submodel) at which the collision
	  * occured, in the second model.
	  */
	 String m_BodyPart1;
   };

   /**
    * A Collision model. The functions have to be implemented by a
    * derived class. A collision model is generally created by a collision
    * system, through the use of its CreateModel() member function.
    */

   class ARK_DLL_API ColModel
   {
      public:

	 /// Destroy the collision model;
	 virtual ~ColModel() {}
   };

   /**
    * A class to detect model-model collisions and ray-model intersections.
    * The various functions are implemented using a specialized library.
    * To be able to do collision detection with two models, it is compulsory
    * that those models have been loaded using the same cache, as this one
    * carries a pointer to a specific collision system, which may be
    * incompatible whith another cache's collision system.
    */
   class ARK_DLL_API ColSystem
   {
      public:

	 /**
	  * Destroy the given collision system. The collision models won't
	  * be destroyed, so you have to be careful to delete them before
	  * you call this destructor. Otherwise the effects are unpredictable.
	  */
         virtual ~ColSystem () {}

	 /**
	  * Returns true if a collision occurs between the two models
	  * given as arguments. In such a case, the function fills the fields
	  * of the pair structure.
	  *
	  * Remark : only the first collision is taken into account.
	  */
	 virtual bool TestCollision (ModelState &m1, ModelState &m2,
				     ColPair &pair) = 0;

	 /**
	  * Returns true if the ray hits the model given as first argument,
	  * and fill the collision structure.
	  */
	 virtual bool RayTrace (ModelState &m1, const Ray &ray,
				Collision &collision) = 0;

	 /** Return true if the given bounding box hits the model.
	  */
	 // FIXME do this later
	 //virtual bool BoxTest (ModelState &m1, const BBox &box) = 0;


	 /** Create a collision model from the given model. */
	 virtual ColModel *CreateModel (Model *from) = 0;
   };

   class ARK_DLL_API ColSystemFactory : public Factory
   {
      public:
	 ColSystemFactory();

	 virtual ~ColSystemFactory() {}
	 virtual ColSystem *NewColSystem() = 0;
	 static ColSystem *CreateColSystem(const String &implname,
					   FactoryList *factlist = 0);
   };
}


#endif
