/* file-tag.c
 *
 ****************************************************************
 * Copyright (C) 2001, 2002  Tom Lord
 * 
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/os/errno.h"
#include "hackerlab/char/char-class.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/cmd/main.h"
#include "file-utils/ftag.h"



static t_uchar * program_name = "file-tag";
static t_uchar * usage = "file-tag [options] file ...";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
				   "\n"
				   "Copyright 2001, 2002 Tom Lord\n"
				   "\n"
				   "This is free software; see the source for copying conditions.\n"
				   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
				   "PARTICULAR PURPOSE.\n"
				   "\n"
				   "Report bugs to <lord@regexps.com>.\n"
				   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_implicit, 0, "implicit", 0, \
      "Use the implicit tagging method (default).") \
  OP (opt_explicit, 0, "explicit", 0, \
      "Use the explicit tagging method.") \
  OP (opt_names, 0, "names", 0, \
      "Use the names tagging method.") \
  OP (opt_silent, 0, "silent", 0, \
      "No output -- exit status only.")

static t_uchar long_help[] = ("Report the inventory tag for FILE.\n"
			      "See the \"arch User's Guide\" for more information about inventory tags.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)  
};

struct opt_desc opts[] = 
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
main (int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int silent;
  enum ftag_method method;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  silent = 0;
  method = ftag_implicit;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, long_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
	break;
      switch (o)
	{
	default:
	  safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
	  panic ("internal error parsing arguments");

	usage_error:
	  opt_usage (2, argv[0], program_name, usage, 1);
	  exit (1);

	/* bogus_arg: */
	  safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
	  goto usage_error;

	case opt_implicit:
	  method = ftag_implicit;
	  break;

	case opt_explicit:
	  method = ftag_explicit;
	  break;

	case opt_names:
	  method = ftag_names;
	  break;

	case opt_silent:
	  silent = 1;
	  break;
	}
    }

  if (argc != 2)
    goto usage_error;


  {
    int errn;
    t_uchar * answer;

    answer = file_tag (&errn, 0, method, argv[1]);

    if (!answer)
      {
	if (errn == 0)
	  {
	    if (!silent)
	      {
		safe_printfmt (2, "\n");
		safe_printfmt (2, "file-tag: untagged file\n");
		safe_printfmt (2, "  %s\n", argv[1]);
		safe_printfmt (2, "\n");
	      }
	    exit (1);
	  }
	else
	  {
	    safe_printfmt (2, "\n");
	    safe_printfmt (2, "file-tag: error while computing file tag\n");
	    safe_printfmt (2, "  file: %s\n", argv[1]);
	    safe_printfmt (2, "  error: %s\n", errno_to_string (errn));
	    safe_printfmt (2, "\n");
	    exit (2);
	  }
      }
    else if (!silent)
      safe_printfmt (1, "%s\t%s\n", argv[1], answer);
  }

  return 0;
}

/* tag: Tom Lord Sun Dec  9 05:34:34 2001 (file-tag.c)
 */
