# update-distributions.sh: build or update an FTP area
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "build or update an FTP area\\n"
                printf "usage: update-distributions [options] [archive/]revision dist-name ftp-dir\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
		printf "  --cache DIR                  cache directory for locally cached\\n"
                printf "                                 revisions\\n"
                printf "\\n"
                printf " --config NAME                 after checking out the revision, pass\\n"
                printf "                                NAME to \"build-config\" to complete\\n"
                printf "                                the source tree.\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Ensure that FTP-DIR exists and is populated with the indicated revision\\n"
		printf "from the archive.\\n"
                printf "\\n"
                printf "The tar file will be named:\\n"
                printf "\\n"
                printf "	DIST-NAME-\${VERSION}\${SUFFIX}\\n"
                printf "\\n"
                printf "where VERSION is the version number of the revision being packaged, and\\n"
                printf "SUFFIX is derived from the patch-level name of the revision.\\n"
		printf "\\n"
                printf "The value of \${SUFFIX} depends on the revision's patch level, as in\\n"
                printf "these examples:\\n"
                printf "\\n"
                printf "	revision:				tar file:\\n"
                printf "\\n"
                printf "	arch--release--1.0--base-0		arch-1.0pre0.tar.gz\\n"
                printf "	arch--release--1.0--patch-3		arch-1.0pre3.tar.gz\\n"
                printf "	arch--release--1.0--version-0		arch-1.0.tar.gz\\n"
                printf "	arch--release--1.0--versionfix-5	arch-1.0.5.tar.gz\\n"
                printf "\\n"
                printf "The top-level directory in the tar file has the same name as the tar\\n"
                printf "file (but without the \".tar.gz\" suffix).\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
cache_dir=

config=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=
__restart=

while test $# -ne 0 ; do

  case "$1" in 

    ----restart) shift
    		__restart=----restart
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "update-distributions: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "update-distributions: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "update-distributions: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --cache)            shift
                        if test $# -eq 0 ; then
                          printf "update-distributions: --cache requires an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        cache_dir="$1"
                        shift
                        ;;

    --config)           shift
                        if test $# -eq 0 ; then
                          printf "update-distributions: --config requires an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        config="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;

    -*)                 printf "update-distributions: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 3 ; then
  printf "usage: update-distributions [options] [archive/]revision dist-name ftp-dir\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

arch_rev_spec="$1"
shift

dist_name="$1"
shift

ftp_dir="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 
  
here="`pwd`"
mkdir -p "$ftp_dir"
cd "$ftp_dir"
ftp_dir="`pwd`"
cd "$here"

arch_rev="`larch indicated-revision -R \"$archroot\" -A \"$archive\" -- \"$arch_rev_spec\"`"

archive="`larch parse-package-name --arch \"$arch_rev\"`"
revision="`larch parse-package-name --non-arch \"$arch_rev\"`"

################################################################
# Greetings
# 

if test "(" -z "$__restart" -a  ! -z "$quiet" ")" -o ! -z "$report" ; then
  larch heading "update-distributions\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "update-distributions start time: %s\\n" "`date`"
  larch heading --sub "ftp dir: %s\\n" "$ftp_dir"
  larch heading --sub "archive: %s\\n" "$archive"
  larch heading --sub "version: %s\\n" "$version"
fi


################################################################
# Make a temp directory
# 

tmpdir="$here/,,update-distributions.$$"

bail()
{
  cd "$here"
  rm -rf "$tmpdir"
  exit 1
}

trap "printf \"update-distributions: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

mkdir "$tmpdir"

################################################################
# What Revision is the Latest?
# 


full_dist_name="`larch distribution-name --dist \"$dist_name\" \"$revision\"`"
tar_file="$full_dist_name.tar.gz"

################################################################
# Is it Already There?
# 

if test -e "$ftp_dir/$tar_file" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "latest revision already available\\n"
    printf "%s\n" "$tar_file" | larch body-indent --sub
  fi

else

  ################################################################
  # Get a Copy of the Revision
  # 
  
  cd "$tmpdir"

  if test ! -z "$quiet" ; then
    larch heading --sub "getting revision\\n"
  fi

  larch nested --sub \
       larch get \
	        $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
	        --no-pristine \
	       "$archive/$revision" "$full_dist_name"  

  ################################################################
  # Build a Configuration, if Requested
  # 
  if test ! -z "$config" ; then
    if test ! -z "$quiet" ; then
      larch heading --sub "building configuration: %s\\n" "$config"
    fi

    larch nested --sub \
	 larch build-config \
	        $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
 		--dir "$tmpdir/$full_dist_name" \
		"$config"
  fi

  ################################################################
  # Make tar File
  # 
  
  cd "$tmpdir"
  larch inventory --source --all --nested "$full_dist_name" > ,,files
  
  if test ! -z "$report" ; then
    larch heading --sub "tar file"
    tar -T ,,files --no-recursion -zcvf "$tar_file" "$full_dist_name" \
    | larch body-indent --sub
  else
    tar -T ,,files --no-recursion -zcf  "$tar_file" "$full_dist_name" 
  fi
  
  mv "$tar_file" "$ftp_dir"

  cd "$here"
  rm -rf "$tmpdir"

fi

################################################################
# Bye
# 

if test ! -z "$quiet" ; then
  larch heading --sub "update-distributions finish time: %s\\n" "`date`"
fi
  
# tag: Tom Lord Thu Jan 10 06:26:49 2002 (web/update-distributions.sh)
#
