# 
# lock-pristine.sh: lock a pristine revision locking
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "lock a pristine revision locking\\n"
		printf "usage: lock-pristine [options] [archive/]revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
		printf " -D --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf " -u --unlock                   unlock the revision\\n"
		printf "\\n"
		printf "Lock the indicated pristine REVISION in the project tree containing\\n"
		printf "DIR (or the current directory).  This prevents the revision from being\\n"
		printf "automatically recycled.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
dir=
unlock=

while test $# -ne 0 ; do

  case "$1" in 

    -u|--unlock)	shift
    			unlock=--unlock
			;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "lock-pristine: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "lock-pristine: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -D|--dir)           shift
                        if test $# -eq 0 ; then
                          printf "lock-pristine: -D and --dir require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        dir="$1"
                        shift
                        ;;

    --)			shift
			break
			;;


    -*)			printf "lock-pristine.shs: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: lock-pristine [options] [archive/]revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

archive_revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"
wdroot="`larch tree-root`"

larch valid-package-name -e lock-pristine --patch-level "$archive_revision"

archive=`larch parse-package-name --arch "$archive_revision"`
revision=`larch parse-package-name --non-arch "$archive_revision"`

category=`larch parse-package-name --basename $revision`
branch=`larch parse-package-name --package $revision`
version=`larch parse-package-name --package-version $revision`

if test -z "$unlock" ; then
  from_state=unlocked
  to_state=locked
else
  from_state=locked
  to_state=unlocked
fi
pristine_path_unlocked="$wdroot/{arch}/++pristine-trees/unlocked/$category/$branch/$version/$archive/$revision"
pristine_path_locked="$wdroot/{arch}/++pristine-trees/locked/$category/$branch/$version/$archive/$revision"


from="$wdroot/{arch}/++pristine-trees/$from_state/$category/$branch/$version/$archive/$revision"
to="$wdroot/{arch}/++pristine-trees/$to_state/$category/$branch/$version/$archive/$revision"

if test -d "$to" ; then
  printf "pristine revision already %s\\n" $to_state
  printf "  project tree: %s\\n" "$wdroot"
  printf "  revision: %s\\n" $revision
  printf "\\n"
  exit 0
fi

if test ! -d "$from" ; then
  printf "lock-pristine: pristine revision not %s\\n" $from_state 1>&2
  printf "  project tree: %s\\n" "$wdroot" 1>&2
  printf "  revision: %s\\n" $revision 1>&2
  printf "\\n" 1>&2
  exit 1
fi



################################################################
# Lock It
# 

mkdir -p "`dirname \"$to\"`"
mv "$from" "$to"

# tag: Tom Lord Mon Dec 10 22:10:18 2001 (local-cache/lock-pristine.sh)
#
