# my-revision-library.sh: the user's revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "print or set your revision library path\\n"
		printf "usage: my-revision-library [options] [dir]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname prog             specify program name for errors\\n"
		printf "\\n"
		printf " -s --silent                   don't print a reassuring message\\n"
		printf " -d --delete                   unspecify your library location\\n"
		printf "\\n"
		printf "With no argument, and without -d, print the path to your revision\\n"
		printf "library.\\n"
		printf "\\n"
		printf "With an argument, record DIR as the path to your revision library\\n"
		printf "in ~/.arch-params/=revision-library\\n"
		printf "\\n"
		printf "With the option -d (--delete) and no argument, ensure that\\n"
		printf "you do not have a revision library path set in ~/.arch-params.\\n"
		printf "\\n"
		printf "If no revision library is set, the program exits with status 1,\\n"
		printf "printing an error message unless the -s (--silent) option is given.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

silent=
delete=
errname=my-revision-library

while test $# -ne 0 ; do

  case "$1" in 

    -e|--errname)	shift
			if test $# -eq 0 ; then
			  printf "my-revision-library: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -s|--silent)	shift
    			silent=--silent
			;;

    -d|--delete)	shift
    			delete=--delete
			;;

    -*)			printf "my-revision-library: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test ! -z "$delete" ; then

  if test $# -ne 0 ; then
    printf "usage: my-revision-library [options] -d|--delete\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
  fi

  type=delete

else

  case $# in 

    0)	type=print
	;;

    1)	type=set
  	newdflt="$1"
	;;

    *)	if test $# -gt 1 ; then
	  printf "usage: my-revision-library [options] [dir]\\n" 1>&2
	  printf "try --help\\n" 1>&2
	  exit 1
	fi
	;;
  esac

fi


################################################################
# Sanity Check
# 

if test $type = set ; then
  if ! test -d "$newdflt" ; then
    printf "\\n" 1>&2
    printf "my-revision-library: path is not a directory\\n" 1>&2
    printf "  path: %s\\n" "$newdflt" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
fi


################################################################
# Do It
# 

case $type in

  delete)	mkdir -p ~/.arch-params
  		cd ~/.arch-params

		if test -e =revision-library ; then
		  rm =revision-library
 		fi

		if test -z "$silent" ; then
		  printf "revision library location removed\\n"
		fi

		exit 0
		;;

  set)		mkdir -p ~/.arch-params/
  		printf "%s\\n" "$newdflt" > ~/.arch-params/=revision-library

		if test -z "$silent" ; then
		  printf "revision library location set (%s)\\n" "$newdflt"
		fi

		exit 0
		;;

  print)	if test ! -e ~/.arch-params/=revision-library ; then
		  if test -z "$silent" ; then
		    printf "\\n" 1>&2
		    printf "%s: no revision library location set\\n" "$errname" 1>&2
		    printf "  try \"larch my-revision-library --help\"\\n" 1>&2
		    printf "\\n" 1>&2
		  fi
		  exit 1
		fi

		cat ~/.arch-params/=revision-library
		;;


esac

# tag: Tom Lord Sat Jan 19 21:43:52 2002 (library/my-revision-library.sh)
#
