# update-config.sh: update a multi-project tree
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "update a multi-project tree\\n"
		printf "usage: update-config [options] config-name\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf " --config-dir CFG              get config from directory CFG\\n"
		printf "\\n"
		printf " --force                       pass the --force flag to update\\n"
		printf "\\n"
		printf "Update the named configuration based on the specification\\n"
		printf "in the CFG directory (or \"configs\" subdirectory\\n"
		printf "at the root of the project tree containing DIR).\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.
config_dir=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

force_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --force)    shift
		force_opt=--force
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "update-config: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -d|--dir)		shift
    			if test $# -eq 0 ; then
			  printf "update-config: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir=$1
			shift
			;;

    --config-dir)       shift
			if test $# -eq 0 ; then
			  printf "update-config: --config-dir requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			config_dir="$1"
			shift
			;;

    --)			shift
    			break
			;;

    -*)			printf "update-config: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: update-config [options] name\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

name="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

wdroot="`larch tree-root --accurate`"
cd "$wdroot"

config_file="`larch indicated-config-file -e update-config --config-dir \"$config_dir\" --dir \"$wdroot\" --exists \"$name\"`"

################################################################
# Greetings
# 

if test ! -z "$quiet" ; then
  larch heading "update-config\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "update-config start time: %s\\n" "`date`"
  larch heading --sub "project tree: %s\\n" "$wdroot"
  larch heading --sub "configuration: %s\\n" "$name"
  cat "$config_file" | larch body-indent --sub
fi

################################################################
# Update Requested Directories
# 

cd "$wdroot"

cat "$config_file" \
| larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
| sort -k 1 \
| awk -v silent_opt="$silent_opt" \
      -v quiet_opt="$quiet_opt" \
      -v report_opt="$report_opt" \
      -v debug_opt="$debug_opt" \
      -v force_opt="$force_opt" \
      '{
          printf("larch nested --sub larch update %s %s %s %s %s  --in-place -- %s %s\n", force_opt, silent_opt, quiet_opt, report_opt, debug_opt, $1, $2);
       }' \
| $ARCH_SHELL

# tag: Tom Lord Sat Dec 29 23:20:42 2001 (configurations/update-configuration.sh)
#
