# record-config.sh: record a revision-specific configuration
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "record a revision-specific configuration\\n"
		printf "usage: record-config [options] config-name new-config-name\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf " --config-dir CFG              get config from directory CFG\\n"
		printf "\\n"
		printf " -f --force                    overwrite an existing config\\n"
		printf "\\n"
		printf "Update the named configuration based on the specification\\n"
		printf "in the \"configs\" subdirectory at the root of the project\\n"
		printf "tree.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.
config_dir=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

force=

while test $# -ne 0 ; do

  case "$1" in 

    -f|--force) shift
    		force=--force
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "record-config: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -d|--dir)		shift
    			if test $# -eq 0 ; then
			  printf "record-config: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir=$1
			shift
			;;

    --config-dir)       shift
			if test $# -eq 0 ; then
			  printf "record-config: --config-dir requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			config_dir="$1"
			shift
			;;

    --)			shift
    			break
			;;

    -*)			printf "record-config: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
  printf "usage: record-config [options] config-name new-config-name\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

name="$1"
shift
new_name="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

wdroot="`larch tree-root --accurate`"
cd "$wdroot"

config_file="`larch indicated-config-file -e record-config --config-dir \"$config_dir\" --dir \"$wdroot\" --exists \"$name\"`"
new_config_file="`larch indicated-config-file -e record-config --config-dir \"$config_dir\" --dir \"$wdroot\" --new $force \"$new_name\"`"

################################################################
# Greetings
# 

if test ! -z "$quiet" ; then
  larch heading "record-config\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "record-config start time: %s\\n" "`date`"
  larch heading --sub "project tree: %s\\n" "$wdroot"
  larch heading --sub "input configuration: %s\\n" "$name"
  cat "$config_file" | larch body-indent --sub
fi

################################################################
# Compute the New Configuration
# 

cd "$wdroot"

if test ! -z "$quiet" ; then
  larch heading --sub "output configuration: %s\\n" "$new_name"
fi

rm -f "$new_config_file"

( \
  printf "#\\n" ; \
  printf "# automatically generated configuration\\n" ; \
  printf "#\\n" ; \
  printf "#  derived from %s\\n" "$name" ; \
  printf "#\\n" ; \
  printf "\\n" ; \
  
  cat "$config_file" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | sort -k 1 \
  | awk -v here="$wdroot" \
       '{
          print "cd \"" here "\"" ; 
	  print "printf \"%-30s\\t\" \"" $1 "\""  ;
	  print "cd \"" $1 "\"" ; 
 	  print "x=\"`larch log-ls --full --reverse \\\"" $2 "\\\"`\"" ; 
	  print "if test -z \"$x\" ; then"
	  print "  printf \"record-config: no log entries for %s\\\\n\" \"" $2 "\" 1>&2"
	  print "  exit 1"
	  print "else"
	  print "  larch log-ls --full --reverse \"" $2 "\" | head -1"
	  print "fi"
        }' \
  | $ARCH_SHELL -e \
) \
| tee "$new_config_file" \
| larch body-indent --sub

if test ! -z "$quiet" ; then
  larch heading --sub "record-config finish time: %s\\n" "`date`"
fi


# tag: Tom Lord Sun Dec 30 17:46:10 2001 (configurations/record-config.sh)
#
