# 
# archive-uncache-revision.sh - remove a cached revision
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "remove a cached revision from an archive\\n"
                printf "usage: archive-uncache-revision [options] [[archive/]revision]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Remove the cached form of REVISION from its archive\\n."
		printf "\\n"
		printf "Also see \"larch archive-cache-revision --help\"\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

__restart=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    ----restart) shift
    		__restart=----restart
		;;

    --no-lint)  shift
    		no_lint=--no-lint
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "archive-uncache-revision: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "archive-uncache-revision: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "archive-uncache-revision: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -*)                 printf "archive-uncache-revision: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: archive-uncache-revision [options] [[archive/]revision]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  rvnspec="$1"
  shift
else
  rvnspec=
fi


################################################################
# Sanity Check and Process Defaults
# 

archive_revision=`larch indicated-revision -e archive-uncache-revision -R "$archroot" -A "$archive" "$rvnspec"`

if test -z "$rvnspec" ; then
  rvnspec="`larch tree-version`"
fi

larch valid-package-name --tolerant "$rvnspec"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$rvnspec\"`"
revision_spec="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --non-arch \"$rvnspec\"`"
category=`larch parse-package-name --basename $revision_spec`
branch=`larch parse-package-name --package $revision_spec`


################################################################
# Greetings
# 

if test "(" -z "$__restart" -a ! -z "$quiet" ")" -o ! -z "$report" ; then
  larch heading "archive-uncache-revision\\n"
  printf "arguments: %s\\n"  "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "archive-uncache-revision start time: %s\\n" "`date`"
  larch heading --sub "archive: %s\\n" "$archive"
  larch heading --sub "revision spec: %s\\n" "$revision_spec"
fi


################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "restarting with connection to archive\\n"
  fi

  exec larch with-archive -A "$archive"  \
	larch archive-uncache-revision \
				      $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
				      ----restart \
				      "$archive/$revision_spec"

fi


################################################################
# Which Revision are We Caching?
# 

archive_revision="`larch indicated-revision -e archive-cache-revision \"$archive/$revision_spec\"`"
revision="`larch parse-package-name --non-arch \"$archive_revision\"`"

if test ! -z "$quiet" ; then
  larch heading --sub "revision: %s\\n" "$revision" 
fi

vsn=`larch parse-package-name --vsn $revision`
lvl=`larch parse-package-name --lvl $revision`


################################################################
# Delete the Cached Revision
# 

wftp-home
wftp-cd $category/$branch/$branch--$vsn/$lvl
wftp-delete $revision.tar.gz

if test ! -z "$quiet" ; then
  larch heading --sub "archive-uncache-revision finish time: %s\\n" "`date`"
fi
