// apt.cc
//
//  Copyright 1999,2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Handles basic apt bookeeping that apt-pkg doesn't :)

#include "apt.h"
#include "../aptitude.h"
#include "undo.h"
#include "config_signal.h"
#include "pkg_hier.h"
#include "tasks.h"

#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/error.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/init.h>
#include <apt-pkg/depcache.h>

#include <assert.h>
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>

aptitudeCacheFile *apt_cache_file=NULL;
signalling_config *aptcfg=NULL;
pkgRecords *apt_package_records=NULL;
undo_list *apt_undos=NULL;
pkg_hier *user_pkg_hier=NULL;

string *pendingerr=NULL;
bool erroriswarning=false;

static Configuration theme_config;

SigC::Signal0<void> cache_closed, cache_reloaded, cache_reload_failed;
SigC::Signal0<void> hier_reloaded;

static void reload_user_pkg_hier()
{
  delete user_pkg_hier;
  user_pkg_hier=new pkg_hier;

  user_pkg_hier->input_file(PKGDATADIR "/function_groups");

  const char *cfgloc=getenv("HOME");
  if(cfgloc)
    {
      string user_hier=cfgloc+string("/.aptitude/function_pkgs");
      if(access(user_hier.c_str(), R_OK)==0)
	user_pkg_hier->input_file(user_hier);
      else
	user_pkg_hier->input_file(PKGDATADIR "/function_pkgs");
    }
}

void apt_preinit()
{
  signal(SIGPIPE, SIG_IGN);

  ReadConfigFile(theme_config, PKGDATADIR "/aptitude-defaults");

#ifndef HAVE_LIBAPT_PKG3
  pkgInitialize(*_config);
  // Magic Happens.. :)
#else
  if(pkgInitConfig(*_config))
    pkgInitSystem(*_config, _system);
#endif

  // Allow a user-specific customization file.
  const char *HOME=getenv("HOME");

  if(HOME)
    {
      string cfgloc=HOME;
      cfgloc+="/.aptitude/config";

      // This attempts to test whether the file is available
      if(access(cfgloc.c_str(), R_OK)==0)
	ReadConfigFile(*_config, cfgloc);
    }

  aptcfg=new signalling_config(_config, &theme_config);

  apt_undos=new undo_list;
}

// Hacked from the APT sources
void apt_dumpcfg(const char *root)
{
  const char *HOME=getenv("HOME");

  if(!HOME)
    return;

  string cfgloc=HOME;
  cfgloc+="/.aptitude";

  if(mkdir(cfgloc.c_str(), 0700)<0 && errno!=EEXIST)
    {
      _error->Errno("mkdir", "%s", cfgloc.c_str());
      return;
    }

  cfgloc+="/config";

  FILE *f=fopen((cfgloc+".new").c_str(), "w");

  if(!f)
    {
      _error->Errno(_("Unable to open %s for writing"), cfgloc.c_str());
      return;
    }

  /* Write out all of the configuration directives by walking the 
     configuration tree */
  const Configuration::Item *Top = _config->Tree(root), *Curr=Top;
  if(Curr)
    do
      {
	// Exclude theme configuration from being saved.
	// (note: this is just a specific instance of a general problem,
	// that we save configuration items that the user hasn't messed with)
	if(strcasecmp(Curr->FullTag().c_str(), PACKAGE "::Themes"))
	  {
	    fprintf(f, "%s \"%s\";\n", Curr->FullTag().c_str(), Curr->Value.c_str());
      
	    if (Curr->Child != 0)
	      {
		Curr = Curr->Child;
		continue;
	      }
	  }
      
	while (Curr != 0 && Curr->Next == 0 && Curr->Parent!=Top)
	  Curr = Curr->Parent;
	if (Curr != 0)
	  Curr = Curr->Next;
      } while(Curr && Curr!=Top);

  fclose(f);

  if(rename((cfgloc+".new").c_str(), cfgloc.c_str())!=0)
    {
      _error->Errno(_("Unable to replace %s with new configuration file"), cfgloc.c_str());
      return;
    }
}

// Revert back to the default set of options.  Is it a hack?  Hmmm...
void apt_revertoptions()
{
  _config=new Configuration;

#ifndef HAVE_LIBAPT_PKG3
  pkgInitialize(*_config);
  // Magic Happens.. :)
#else
  // ick?
  if(pkgInitConfig(*_config))
    pkgInitSystem(*_config, _system);
#endif

  delete aptcfg->setcfg(_config);
}

void apt_init(OpProgress *progress_bar, bool do_initselections,
	      const char *status_fname)
{
  if(!apt_cache_file)
    apt_reload_cache(progress_bar, do_initselections, status_fname);
}

void apt_reload_cache(OpProgress *progress_bar, bool do_initselections,
		      const char * status_fname)
{
  cache_closed();

  reset_tasks();

  if(apt_package_records)
    {
      delete apt_package_records;
      apt_package_records=NULL;
    }

  if(apt_cache_file)
    {
      delete apt_cache_file;
      apt_cache_file=NULL;
    }

  aptitudeCacheFile *new_file=new aptitudeCacheFile;

  bool open_failed=!new_file->Open(*progress_bar, do_initselections,
				   (getuid()==0), status_fname)
    || _error->PendingError();

  if(open_failed && getuid()==0)
    {
      _error->Discard();

      open_failed=!new_file->Open(*progress_bar, do_initselections,
				  false, status_fname) ||
	_error->PendingError();

      if(!open_failed)
	_error->Warning(_("Warning: could not lock the cache file.  Opening in read-only mode"));
    }

  if(open_failed)
    {
      delete new_file;
      cache_reload_failed();
      return;
    }

  apt_cache_file=new_file;
  apt_package_records=new pkgRecords(*apt_cache_file);

  // Um, good time to clear our undo info.
  apt_undos->clear_items();

  load_tasks();

  if(user_pkg_hier)
    {
      reload_user_pkg_hier();
      hier_reloaded();
    }

  cache_reloaded();
}

pkg_hier *get_user_pkg_hier()
{
  if(!user_pkg_hier)
    reload_user_pkg_hier();

  return user_pkg_hier;
}

pkg_action_state find_pkg_state(pkgCache::PkgIterator pkg)
{
  aptitudeDepCache::StateCache &state=(*apt_cache_file)[pkg];
  aptitudeDepCache::aptitude_state &extstate=(*apt_cache_file)->get_ext_state(pkg);

  if((!state.Install() && state.NowBroken()) ||
     (state.Install() && state.InstBroken()))
    return pkg_broken;
  else if(state.Delete())
    {
      if(extstate.installremove_reason==aptitudeDepCache::manual)
	return pkg_remove;
      else if(extstate.installremove_reason==aptitudeDepCache::unused)
	return pkg_unused_remove;
      else
	return pkg_auto_remove;
    }
  else if(state.Install())
    {
      if(!pkg.CurrentVer().end())
	{
	  if(state.iFlags&pkgDepCache::ReInstall)
	    return pkg_reinstall;
	  else if(state.Downgrade())
	    return pkg_downgrade;
	  else if(state.Upgrade())
	    return pkg_upgrade;
	  else
	    // FOO!  Should I abort here?
	    return pkg_install;
	}
      else if(extstate.installremove_reason!=aptitudeDepCache::manual)
	return pkg_auto_install;
      else
	return pkg_install;
    }

  else if(state.Status==1 &&
	  state.Keep())
    {
      if(!(state.Flags & pkgDepCache::AutoKept))
	return pkg_hold;
      else
	return pkg_auto_hold;
    }

  else if(state.iFlags&pkgDepCache::ReInstall)
    return pkg_reinstall;

  return pkg_unchanged;
}

bool pkg_obsolete(pkgCache::PkgIterator pkg)
{
  if(pkg.CurrentVer().end())
    return false;
  else
    {
      pkgCache::VerIterator ver=pkg.VersionList();
      ver++;

      if(!ver.end())
	return false;
      else // Ok, there's only one version.  Good.
	{
	  pkgCache::VerFileIterator files=pkg.VersionList().FileList();
	  if(!files.end())
	    {
	      files++;
	      if(!files.end())
		return false; // Nope, more than one file
	    }
	}
    }

  return true;
}
